﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Result.h>
#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Abort.h>
#include <nn/dbg/dbg_Api.h>
#include <cctype>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_Host.h>
#include <nn/fs/fs_SdCardForDebug.h>
#include <nn/fs/fs_SdCardPrivate.h>
#include <nn/fs/fs_IEventNotifier.h>
#include <nn/fs.h>
#include <cstdio>

#include "dmnt_Server.h"

//==============================================================================

#if DEBUG
#define DMNT_TARGETIO_TRACE( channel, ... ) NN_SDK_LOG( "[" ); NN_SDK_LOG( channel ); NN_SDK_LOG( "] - " ); NN_SDK_LOG( __VA_ARGS__ ); NN_SDK_LOG( "\n" )
#define DMNT_TARGETIO_LOG(...) NN_SDK_LOG( "[dmnt] - " ); NN_SDK_LOG( __VA_ARGS__ ); NN_SDK_LOG( "\n" )
#else
#define DMNT_TARGETIO_TRACE( channel, ... )
#define DMNT_TARGETIO_LOG(...)
#endif

enum
{
    MAX_TARGET_IO_PATH = nn::fs::EntryNameLengthMax,
};

//==============================================================================

static const char sdRoot[] = "sdmc";

//==============================================================================
// Card insertion state variables.
static nn::os::SystemEventType s_SdCardDetectionEvent;
static std::unique_ptr<nn::fs::IEventNotifier> s_SdCardDetectionEventNotifier( nullptr );
static bool s_SystemInitialized = false;
static bool s_CardIsInserted = false;

//==============================================================================

static bool initSystem()
{
    if( s_SystemInitialized == false )
    {
        if( !s_SdCardDetectionEventNotifier )
        {
#if 0 // DeviceDetection 権限が必要
            nn::Result result = nn::fs::OpenSdCardDetectionEventNotifier( &s_SdCardDetectionEventNotifier );
            if( result.IsFailure() )
            {
                DMNT_TARGETIO_LOG( "initSystem:  OpenSdCardDetectionEventNotifier FAILED - %x",  result.GetInnerValueForDebug() );
                s_SdCardDetectionEventNotifier.reset();
            }
#endif
        }
        if( s_SdCardDetectionEventNotifier )
        {
            nn::Result result = s_SdCardDetectionEventNotifier->BindEvent( &s_SdCardDetectionEvent, nn::os::EventClearMode_ManualClear );
            if (result.IsSuccess())
            {
                s_CardIsInserted = nn::fs::IsSdCardInserted();

                // If we can, mount
                if( s_CardIsInserted )
                {
                    result = nn::fs::MountSdCardForDebug( sdRoot );
                    if( result.IsFailure() )
                    {
                        DMNT_TARGETIO_LOG( "initSystem:  MountSdCardForDebug FAILED - %x",  result.GetInnerValueForDebug() );
                    }
                }

                s_SystemInitialized = true;
                DMNT_TARGETIO_LOG( "initSystem:  Card is %s",  s_CardIsInserted == true ? "IN" : "OUT");
            }
            else
            {
                DMNT_TARGETIO_LOG( "initSystem:  BindEvent FAILED - %x",  result.GetInnerValueForDebug() );
            }
        }
    }
    return s_SystemInitialized;
}

//==============================================================================

nn::Result TargetIO_Finalize() NN_NOEXCEPT
{
    DMNT_TARGETIO_LOG( "TargetIO_Finalize" );

    if( s_SystemInitialized == true )
    {
        nn::os::DestroySystemEvent( &s_SdCardDetectionEvent );
        if( s_CardIsInserted )
        {
            nn::fs::Unmount( sdRoot );
            s_CardIsInserted = false;
        }
        s_SystemInitialized = false;
    }

    return nn::ResultSuccess();
}
//==============================================================================

static bool onCardEvent( )
{
    if( s_CardIsInserted == true )
    {
        //=========================================================================
        // If the card WAS inserted, then it was removed at least once.  Unmount.
        //========================================================================
        nn::fs::Unmount( sdRoot );
    }

    //======================================================================================
    // We might have had more than one event since last we checked.  Get our current state.
    //======================================================================================
    s_CardIsInserted = nn::fs::IsSdCardInserted();

    if( s_CardIsInserted == true )
    {
        //=========================================================================
        // Now it's inserted, so mount.
        //========================================================================
        nn::fs::MountSdCardForDebug( sdRoot );
    }

    DMNT_TARGETIO_LOG( "onCardEvent:  Card is %s",  s_CardIsInserted == true ? "IN" : "OUT");

    return s_CardIsInserted;
}

//==============================================================================

static bool checkCard()
{
    if( initSystem() )
    {
        if( nn::os::TryWaitSystemEvent( &s_SdCardDetectionEvent ) )
        {
            DMNT_TARGETIO_LOG( "checkCard:  Received card event");
            nn::os::ClearSystemEvent( &s_SdCardDetectionEvent );
            onCardEvent( );
        }
    }
    return s_CardIsInserted;
}

//==============================================================================

nn::Result checkCardState() NN_NOEXCEPT
{
    // Did the card get jerked out?
    if( checkCard() == false )
    {
        DMNT_TARGETIO_LOG( "TargetIO_CheckState:  Card is %s",  s_CardIsInserted == true ? "IN" : "OUT");
        return nn::fs::ResultSdCardAccessFailed();
    }

    return nn::ResultSuccess();
}

//==============================================================================

#define CHECK_STATE() { nn::Result StateRes = checkCardState(); if( StateRes.IsFailure() ) return StateRes; }

//==============================================================================

static char* nameToValidPath( const char* pName, char* pToPath, size_t SizeOfBuffer )
{
    // Start with our root
    strcpy( pToPath, sdRoot );
    strcat( pToPath, ":/" );

    // Remove any starting slashes, we've already added all we need
    char* pFromName = (char*)pName;
    while( (*pFromName != NULL) && ( (*pFromName == '\\') || (*pFromName == '/') ) )
    {
        pFromName += 1;
    }

    // Convert the slashes to something we understand
    size_t Length = strlen( pToPath );
    char* pWriteTo = pToPath + Length;
    const char* pSource = pFromName;
    size_t AtIndex = Length;
    while( *pSource != 0 && (AtIndex < ( SizeOfBuffer - 1 ) ) )
    {
        if( *pSource == '\\')
        {
            *pWriteTo = '/';
        }
        else
        {
            *pWriteTo = *pSource;
        }
        pWriteTo += 1;
        pSource += 1;
        AtIndex += 1;
    }

    *pWriteTo = 0;
    return pToPath;
}

//==============================================================================

enum CreationDispositionEnum : int32_t
{
    TARGET_IO_CREATE_ALWAYS = 2,
    TARGET_IO_CREATE_NEW = 1,
    TARGET_IO_OPEN_ALWAYS   = 4,
    TARGET_IO_OPEN_EXISTING = 3,
    TARGET_IO_TRUNCATE_EXISTING = 5,
};

nn::Result Server::TargetIO_FileOpen( const nn::sf::InBuffer& FileName, uint32_t OpenMode, int32_t create, const nn::sf::OutBuffer& pOutHandle ) NN_NOEXCEPT
{
    char* pFileName = (char*)FileName.GetPointerUnsafe();
    DMNT_TARGETIO_TRACE( "TargetIO_FileOpen", "Open %s", pFileName );
    CHECK_STATE();

    nn::fs::FileHandle* pHandle = (nn::fs::FileHandle*)pOutHandle.GetPointerUnsafe();

    nn::Result res = nn::ResultSuccess();
///    nn::fs::OpenMode openMode = (nn::fs::OpenMode)OpenMode;

    char tempPath[MAX_TARGET_IO_PATH];
    char* pOpenName = nameToValidPath( pFileName, tempPath, sizeof(tempPath) );

    if( create == TARGET_IO_CREATE_ALWAYS)
    {
        // Always create a new file.
        // If the specified file exists and is writable, the function overwrites the file.
        // If the specified file does not exist and is a valid path, a new file is created.
        nn::fs::DeleteFile( pOpenName );
        res = nn::fs::CreateFile( pOpenName, 0 );
    }

    else if( create == TARGET_IO_CREATE_NEW )
    {
        // Creates a new file, only if it does not already exist.
        // If the specified file exists, the function fails and the last-error code is set to ERROR_FILE_EXISTS (80).
        // If the specified file does not exist and is a valid path to a writable location, a new file is created.
        res = nn::fs::CreateFile( pOpenName, 0 );
    }

    if( res.IsFailure() )
    {
        return res;
    }

    res = nn::fs::OpenFile( pHandle, pOpenName, OpenMode );

    if( res.IsFailure() )
    {
        if( create == TARGET_IO_OPEN_ALWAYS )
        {
            // Opens a file, always.
            // If the specified file exists, the function succeeds and the last-error code is set to ERROR_ALREADY_EXISTS (183).
            // If the specified file does not exist and is a valid path to a writable location, the function creates a file and the last-error code is set to zero.
            nn::fs::CreateFile( pOpenName, 0 );
            res = nn::fs::OpenFile( pHandle, pOpenName, OpenMode );
        }
        else if( ( create == TARGET_IO_OPEN_EXISTING ) || ( create == TARGET_IO_TRUNCATE_EXISTING ) )
        {
            // Opens a file or device, only if it exists.
            // If the specified file or device does not exist, the function fails and the last-error code is set to ERROR_FILE_NOT_FOUND (2).
            return res;
        }
    }

    if( create == TARGET_IO_TRUNCATE_EXISTING )
    {
        // Opens a file and truncates it so that its size is zero bytes, only if it exists.
        // If the specified file does not exist, the function fails and the last-error code is set to ERROR_FILE_NOT_FOUND (2).
        // The calling process must open the file with the GENERIC_WRITE bit set as part of the dwDesiredAccess parameter.
        res = nn::fs::SetFileSize( *pHandle, 0 );
    }

    return res;
}

//==============================================================================

nn::Result Server::TargetIO_FileClose( const nn::sf::InBuffer& Handle ) NN_NOEXCEPT
{
    DMNT_TARGETIO_TRACE( "TargetIO_FileClose", "FileClose" );
    CHECK_STATE();
    nn::fs::FileHandle* pHandle = (nn::fs::FileHandle*)Handle.GetPointerUnsafe();
    nn::fs::CloseFile( *pHandle );
    return nn::ResultSuccess();
}

//==============================================================================

nn::Result Server::TargetIO_FileRead( const nn::sf::InBuffer& Handle, const nn::sf::OutBuffer& pBuffer, nn::sf::Out<int32_t> pNumberOfBytesRead, int64_t Offset ) NN_NOEXCEPT
{
    DMNT_TARGETIO_TRACE( "TargetIO_FileRead", "FileRead %d", pBuffer.GetSize() );
    CHECK_STATE();
    nn::fs::FileHandle* pHandle = (nn::fs::FileHandle*)Handle.GetPointerUnsafe();
    void* pReadBuffer = pBuffer.GetPointerUnsafe();

    size_t amountRead = 0;
    nn::Result res = nn::fs::ReadFile( &amountRead, *pHandle, Offset, pReadBuffer, pBuffer.GetSize() );
    pNumberOfBytesRead.Set( amountRead );

    return res;
}

//==============================================================================

nn::Result Server::TargetIO_FileWrite( const nn::sf::InBuffer& Handle, const nn::sf::InBuffer& pBuffer, nn::sf::Out<int32_t> pNumberOfBytesWritten, int64_t Offset ) NN_NOEXCEPT
{
    DMNT_TARGETIO_TRACE( "TargetIO_FileWrite", "FileWrite %d", pBuffer.GetSize() );
    CHECK_STATE();
    nn::fs::FileHandle* pHandle = (nn::fs::FileHandle*)Handle.GetPointerUnsafe();
    void* pWriteBuffer = (void*)pBuffer.GetPointerUnsafe();

    nn::fs::WriteOption writeOption = { nn::fs::WriteOptionFlag_Flush };
    nn::Result res = nn::fs::WriteFile( *pHandle, Offset, pWriteBuffer, (size_t)pBuffer.GetSize(), writeOption );
    if( res.IsSuccess() )
    {
        // Given this API, this is our best guess.
        pNumberOfBytesWritten.Set( pBuffer.GetSize() );
    }

    return res;
}

//==============================================================================

nn::Result Server::TargetIO_FileSetAttributes( const nn::sf::InBuffer& FileName, const nn::sf::InBuffer& pAttribs ) NN_NOEXCEPT
{
    char* pFileName = (char*)FileName.GetPointerUnsafe();
    DMNT_TARGETIO_TRACE( "TargetIO_FileSetAttributes", "TargetIO_FileSetAttributes %s", pFileName );
    CHECK_STATE();
    return nn::ResultSuccess();
}

//==============================================================================

nn::Result Server::TargetIO_FileGetInformation( const nn::sf::InBuffer& FileName, nn::sf::Out<int32_t> pIsDirectory, const nn::sf::OutBuffer& pData ) NN_NOEXCEPT
{
    char* pFileName = (char*)FileName.GetPointerUnsafe();
    DMNT_TARGETIO_TRACE( "TargetIO_FileGetInformation", "TargetIO_FileGetInformation%s", pFileName );
    CHECK_STATE();

    // Default to Not a directory
    pIsDirectory.Set(0);

    // Clear our size and times to known values
    uint64_t* pStats = (uint64_t*)pData.GetPointerUnsafe();
    pStats[0] = 0;
    pStats[1] = 0; // Create Time
    pStats[2] = 0; // Last Access Time
    pStats[3] = 0; // Modified Time

    char tempPath[MAX_TARGET_IO_PATH];
    char* pName = nameToValidPath( pFileName, tempPath, sizeof(tempPath) );

    nn::fs::FileHandle fileHandle;
    nn::Result res = nn::fs::OpenFile( &fileHandle, pName, nn::fs::OpenMode_Read );

    if( res.IsSuccess() )
    {
        int64_t FileSize = 0;
        nn::Result SizeRes = nn::fs::GetFileSize( &FileSize, fileHandle );
        nn::fs::CloseFile( fileHandle );
        if( SizeRes.IsSuccess() )
        {
            pStats[0] = FileSize;
        }
        else
        {
            DMNT_TARGETIO_TRACE( "TargetIO_FileGetInformation", "nn::fs::GetFileSize FAILED for %s, error = %x", pFileName, SizeRes.GetInnerValueForDebug() );
        }
        //TODO:
        pStats[1] = 0; // Create Time
        pStats[2] = 0; // Last Access Time
        pStats[3] = 0; // Modified Time
    }
    else ///if( res.IsFailure() )
    {
        nn::fs::DirectoryHandle dirHandle;
        res = nn::fs::OpenDirectory( &dirHandle, pName, nn::fs::OpenDirectoryMode_All );
        if( res.IsSuccess() )
        {
            pIsDirectory.Set(1);
            nn::fs::CloseDirectory( dirHandle );
        }
    }
    return res;
}

//==============================================================================

nn::Result Server::TargetIO_FileSetTime( const nn::sf::InBuffer& FileName, uint64_t CreateTime, uint64_t AccessTime, uint64_t ModifyTime ) NN_NOEXCEPT
{
    char* pFileName = (char*)FileName.GetPointerUnsafe();
    DMNT_TARGETIO_TRACE( "TargetIO_FileSetTime", "TargetIO_FileSetTime %s", pFileName );
    CHECK_STATE();
    return nn::ResultSuccess();
}

//==============================================================================

nn::Result Server::TargetIO_FileSetSize( const nn::sf::InBuffer& FileName, int64_t size ) NN_NOEXCEPT
{
    char* pFileName = (char*)FileName.GetPointerUnsafe();
    DMNT_TARGETIO_TRACE( "TargetIO_FileSetSize", "TargetIO_FileSetSize %s", pFileName );
    CHECK_STATE();

    char tempPath[MAX_TARGET_IO_PATH];
    char* pName = nameToValidPath( pFileName, tempPath, sizeof(tempPath) );

    nn::fs::FileHandle fileHandle;
    nn::Result res = nn::fs::OpenFile( &fileHandle, pName, nn::fs::OpenMode_Write );

    if( res.IsSuccess() )
    {
        res = nn::fs::SetFileSize( fileHandle, size );
        nn::fs::CloseFile( fileHandle );
    }

    return res;
}

//==============================================================================

nn::Result Server::TargetIO_FileDelete( const nn::sf::InBuffer& FileName ) NN_NOEXCEPT
{
    char* pFileName = (char*)FileName.GetPointerUnsafe();
    DMNT_TARGETIO_TRACE( "TargetIO_FileDelete", "TargetIO_FileDelete %s", pFileName );
    CHECK_STATE();
    char tempPath[MAX_TARGET_IO_PATH];
    char* pName = nameToValidPath( pFileName, tempPath, sizeof(tempPath) );
    return nn::fs::DeleteFile( pName );
}

//==============================================================================

nn::Result Server::TargetIO_FileMove( const nn::sf::InBuffer& FromName, const nn::sf::InBuffer& ToName ) NN_NOEXCEPT
{
    char* pFromName = (char*)FromName.GetPointerUnsafe();
    char* pToName = (char*)ToName.GetPointerUnsafe();
    DMNT_TARGETIO_TRACE( "TargetIO_FileMove", "FileMove %s to %s", pFromName, pToName );
    CHECK_STATE();
    char fromPath[MAX_TARGET_IO_PATH];
    char* pOrigName = nameToValidPath( pFromName, fromPath, sizeof(fromPath) );
    char toPath[MAX_TARGET_IO_PATH];
    char* pNewName = nameToValidPath( pToName, toPath, sizeof(toPath) );
    return nn::fs::RenameFile( pOrigName, pNewName );
}

//==============================================================================
// Directory functionality
//==============================================================================

nn::Result Server::TargetIO_DirectoryCreate( const nn::sf::InBuffer& DirectoryName )  NN_NOEXCEPT
{
    char* pPath = (char*)DirectoryName.GetPointerUnsafe();
    DMNT_TARGETIO_TRACE( "TargetIO_DirectoryCreate", "DirectoryCreate %s", pPath );
    CHECK_STATE();
    char tempPath[MAX_TARGET_IO_PATH];
    char* pName = nameToValidPath( pPath, tempPath, sizeof(tempPath) );
    return nn::fs::CreateDirectory( pName );
}

//==============================================================================

nn::Result Server::TargetIO_DirectoryDelete( const nn::sf::InBuffer& DirectoryName )  NN_NOEXCEPT
{
    char* pPath = (char*)DirectoryName.GetPointerUnsafe();
    DMNT_TARGETIO_TRACE( "TargetIO_DirectoryDelete", "DirectoryDelete %s", pPath );
    CHECK_STATE();
    char tempPath[MAX_TARGET_IO_PATH];
    char* pName = nameToValidPath( pPath, tempPath, sizeof(tempPath) );
    return nn::fs::DeleteDirectoryRecursively( pName );
}

//==============================================================================

nn::Result Server::TargetIO_DirectoryRename( const nn::sf::InBuffer& FromName, const nn::sf::InBuffer& ToName )  NN_NOEXCEPT
{
    char* pFromPath = (char*)FromName.GetPointerUnsafe();
    char* pToPath = (char*)ToName.GetPointerUnsafe();
    DMNT_TARGETIO_TRACE( "TargetIO_DirectoryRename", "DirectoryRename %s to %s", pFromPath, pToPath );
    CHECK_STATE();
    char tempFromName[MAX_TARGET_IO_PATH];
    char* pFromValidName = nameToValidPath( pFromPath, tempFromName, sizeof(tempFromName) );
    char tempToName[MAX_TARGET_IO_PATH];
    char* pToValidFileName = nameToValidPath( pToPath, tempToName, sizeof(tempToName) );
    return nn::fs::RenameDirectory( pFromValidName, pToValidFileName );
}

//==============================================================================

static nn::Result openDirectory( char* pPath, nn::fs::DirectoryHandle* pDirHandle )
{
    char validPathBuffer[MAX_TARGET_IO_PATH];
    char* pSearchPath = nameToValidPath( pPath, validPathBuffer, sizeof(validPathBuffer) );
    return nn::fs::OpenDirectory( pDirHandle, pSearchPath, nn::fs::OpenDirectoryMode_All );
}

//==============================================================================

nn::Result Server::TargetIO_DirectoryGetCount( const nn::sf::InBuffer& DirectoryName, nn::sf::Out<int32_t> pNumberOfEntries )  NN_NOEXCEPT
{
    char* pPath = (char*)DirectoryName.GetPointerUnsafe();
    DMNT_TARGETIO_TRACE( "TargetIO_DirectoryGetCount", "DirectoryGetCount %s", pPath );
    CHECK_STATE();

    pNumberOfEntries.Set( 0 );

    nn::fs::DirectoryHandle dirHandle;
    nn::Result res = openDirectory( pPath, &dirHandle );
    if( res.IsSuccess() )
    {
        // Start our search
        int64_t numEntries = 0;
        res = nn::fs::GetDirectoryEntryCount( &numEntries, dirHandle );
        if( res.IsSuccess() )
        {
            pNumberOfEntries.Set( numEntries );
        }

        nn::fs::CloseDirectory( dirHandle );
    }

    return res;
}

//==============================================================================

nn::Result Server::TargetIO_DirectoryOpen( const nn::sf::InBuffer& DirectoryName, const nn::sf::OutBuffer& pHandle )  NN_NOEXCEPT
{
    char* pPath = (char*)DirectoryName.GetPointerUnsafe();
    DMNT_TARGETIO_TRACE( "TargetIO_DirectoryOpen", "DirectoryOpen %s", pPath );
    CHECK_STATE();
    nn::fs::DirectoryHandle* pDirHandle = (nn::fs::DirectoryHandle*)pHandle.GetPointerUnsafe();

    char validPathBuffer[MAX_TARGET_IO_PATH];
    char* pSearchPath = nameToValidPath( pPath, validPathBuffer, sizeof(validPathBuffer) );

    nn::Result res = nn::fs::OpenDirectory( pDirHandle, pSearchPath, nn::fs::OpenDirectoryMode_All );

    return res;
}

//==============================================================================

nn::Result Server::TargetIO_DirectoryGetNext( const nn::sf::InBuffer& pHandle, const nn::sf::OutBuffer& pEntry )  NN_NOEXCEPT
{
    DMNT_TARGETIO_TRACE( "TargetIO_DirectoryGetNext", "DirectoryGetNext" );
    CHECK_STATE();
    nn::fs::DirectoryHandle* pDirHandle = (nn::fs::DirectoryHandle*)pHandle.GetPointerUnsafe();
    nn::fs::DirectoryEntry* pDirEntry = (nn::fs::DirectoryEntry*)pEntry.GetPointerUnsafe();

    int64_t numEntries = 0;
    nn::Result res = nn::fs::ReadDirectory( &numEntries, pDirEntry, *pDirHandle, 1 );
    if( res.IsSuccess() )
    {
        if( numEntries <= 0 )
        {
            res = nn::fs::ResultOutOfRange();
        }
    }

    return res;
}

//==============================================================================

nn::Result Server::TargetIO_DirectoryClose( const nn::sf::InBuffer& pHandle )  NN_NOEXCEPT
{
    DMNT_TARGETIO_TRACE( "TargetIO_SearchClose", "SearchClose" );
    CHECK_STATE();
    nn::fs::DirectoryHandle* pDirHandle = (nn::fs::DirectoryHandle*)pHandle.GetPointerUnsafe();
    nn::fs::CloseDirectory( *pDirHandle );

    return nn::ResultSuccess();
}

//==============================================================================
// Drive functionality
//==============================================================================

nn::Result Server::TargetIO_GetFreeSpace( const nn::sf::OutBuffer& pFreeSpace )  NN_NOEXCEPT
{
    DMNT_TARGETIO_TRACE( "TargetIO_GetFreeSpace", "GetFreeSpace" );
    CHECK_STATE();
    int64_t* pSpace = (int64_t*)pFreeSpace.GetPointerUnsafe();

    pSpace[0] = 0;
    pSpace[1] = 0;
    pSpace[2] = 0;

    int64_t size = 0;
    nn::Result res = nn::fs::GetSdCardUserAreaSize( &size );
    if( res.IsSuccess() )
    {
        // Total number of bytes
        pSpace[1] = size;
        res = nn::fs::GetSdCardProtectedAreaSize( &size );
        if( res.IsSuccess() )
        {
            pSpace[1] = pSpace[1] + size;
        }
    }

    //FIXME - How do we get free bytes available?

    return res;
}

//==============================================================================

nn::Result Server::TargetIO_GetVolumeInformation( const nn::sf::OutBuffer& pVolumeNameBuffer, const nn::sf::OutBuffer& pFileSystemNameBuffer,
        nn::sf::Out<int32_t> pVolumeSerialNumber, nn::sf::Out<int32_t> pMaximumComponentLength, nn::sf::Out<int32_t> pFileSystemFlags )  NN_NOEXCEPT
{
    DMNT_TARGETIO_TRACE( "TargetIO_GetVolumeInformation", "GetVolumeInformation" );
    CHECK_STATE();

    char* pVolumeName = pVolumeNameBuffer.GetPointerUnsafe();
    char* pFileSystemName = pFileSystemNameBuffer.GetPointerUnsafe();
    memset( pVolumeName, 0, pVolumeNameBuffer.GetSize() );
    memset( pFileSystemName, 0, pFileSystemNameBuffer.GetSize() );

    // Let's read the CID:
    char CIDBuffer[16];
    nn::Result res = nn::fs::GetSdCardCid( CIDBuffer, sizeof(CIDBuffer) );

    if( res.IsSuccess() )
    {
        /************************************************************************************************************************************
            The following information is stored in the CID:
                 Name               - Bits  - Location  -  Description
            Manufacturer ID         -   8   - [127:120] - Assigned by SD-3C, LLC.
            OEM/Application ID      -  16   - [119:104] - Identifies the card OEM and/or the card contents. Assigned by SD-3C, LLC.
            Product Name            -  40   - [103: 64] - 5 characters long (ASCII)
            Product Revision        -   8   - [ 63: 56] - Two binary coded decimal (BCD) digits. Each is four bits. The PRV is in the form x.y.
            Serial Number           -  32   - [ 55: 24] - This 32 bit field is intended to be read as an unsigned integer
            Reserved                -   4   - [ 23: 20] - This 32 bit field is intended to be read as an unsigned integer
            Manufacture Date Code   -  12   - [ 19:  8] - Manufacture date is stored in the form yym (offset from 2000)
            CRC7 checksum           -   7   - [  7:  1] - 7 bit code used for checking errors in the card register
            Unused                  -   1   - [  0:  0] - Not used, always 1
        *************************************************************************************************************************************/

        //Manufacturer's ID is the last two bytes, reversed:
        int32_t ManufacturerID  = CIDBuffer[14] | (CIDBuffer[15] << 8);
        char    OEMID[3];
        OEMID[0]                = CIDBuffer[13];
        OEMID[1]                = CIDBuffer[12];
        OEMID[2]                = 0;

        char    pProductName[6];
        pProductName[0]         = CIDBuffer[11];
        pProductName[1]         = CIDBuffer[10];
        pProductName[2]         = CIDBuffer[9];
        pProductName[3]         = CIDBuffer[8];
        pProductName[4]         = CIDBuffer[7];
        pProductName[5]         = 0;
        char    ProductRev      = CIDBuffer[6];
        int32_t Serial          = CIDBuffer[2] | (CIDBuffer[3] << 8) | (CIDBuffer[4] << 8) | (CIDBuffer[5] << 8);
        char    YearA           = (CIDBuffer[1] & 0xF);
        char    YearB           = (CIDBuffer[0] >> 4);
        char    Month           = (CIDBuffer[0] & 0xF);
        int     Year            = 2000 + (YearA * 10) + YearB;
        sprintf( pVolumeName, "Serial %d, Manufacture date %d/%d", Serial, Month, Year );
        pVolumeSerialNumber.Set( Serial );
    }
    else
    {
        strncpy( pVolumeName, "NX VOLUME", pVolumeNameBuffer.GetSize() );
        pVolumeSerialNumber.Set(0);
    }

    //FIXME
    strncpy( pFileSystemName, "FAT 32", pFileSystemNameBuffer.GetSize() );
    pMaximumComponentLength.Set( 123456 );
    pFileSystemFlags.Set( 123456 );

    return nn::ResultSuccess();
}

//==============================================================================


