﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Result.h>
#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Abort.h>
#include <nn/dbg/dbg_Api.h>
#include <nn/fs.h>
#include <cctype>

#include "dmnt_Server.h"
#include "..\..\TargetTools\SnapShotDumper\coredump\coredump_Format.h"
#include "..\..\TargetTools\SnapShotDumper\coredump\coredump_Data.h"
#include "..\..\Libraries\tmagent\dbg\dbg_RegisterDefs.h"
#include <nn/nn_SystemThreadDefinition.h>

//========================================
// Defined in dmnt_ServerMain.cpp:
extern void*    Alloc( size_t size );
extern void     Dealloc( void* Ptr, size_t Size );
extern bool     InitFS();

enum
{
    MAX_NUMBER_OF_MODULES = nn::dbg::MaxNsoModules + nn::dbg::MaxNroModules,
    MAX_NUMBER_OF_THREADS = 128,
    MAX_PATH = 1024
};

// Statically-declared coredump writer.  Done this way to reduce memory fragmentation
static coredump::coredump_writer s_Writer;

// Buffers used for writing core dumps.  Done this way to reduce memory fragmentation.
static unsigned char s_CoredumpMemoryBuffer[COREDUMP_MEMORY_SECTION_BUFFER_SIZE];
static unsigned char s_CoredumpCompressionBuffer[LZ4_COMPRESSBOUND(sizeof(s_CoredumpMemoryBuffer))];
static coredump::coredump_compiled_data_thread s_CoredumpThreadDetailBuffer[MAX_NUMBER_OF_THREADS];
static nn::dbg::ModuleInfo s_pModuleBuffer[MAX_NUMBER_OF_MODULES];

//==============================================================================

#if DEBUG
#define DMNT_CORE_DUMPER_LOG(...) NN_SDK_LOG( "[dmnt:CoreDump] - " ); NN_SDK_LOG( __VA_ARGS__ ); NN_SDK_LOG( "\n" )
#else
#define DMNT_CORE_DUMPER_LOG(...)
#endif

#define DMNT_CORE_DUMPER_ERROR(...) NN_SDK_LOG( "[dmnt:CoreDump] - ERROR - " ); NN_SDK_LOG( __VA_ARGS__ ); NN_SDK_LOG( "\n" )

#define SNAP_SHOT_DUMPER_EXTENSION ".nxdmp"

enum
{
    OUTPUT_FILE_NAME_SIZE       = 0x100,
    ARM64_REGISTER_SP           = 31,
    ARM64_REGISTER_LR           = 30,
    ARM64_REGISTER_PC           = 32, // TEMP HACK: Beyond bounds of cpuRegisters[]
    ARM64_REGISTER_CONTROL      = 33, // TEMP HACK: Beyond bounds of cpuRegisters[]
    ARM64_FPU_REGISTER_START    = ((ARM64_REGISTER_CONTROL) + 1),

    ARM32_REGISTER_SP           = 13,
    ARM32_REGISTER_LR           = 14,
    ARM32_REGISTER_PC           = 15,
    ARM32_REGISTER_CONTROL      = 16,// TEMP HACK: Beyond bounds of cpuRegisters[]
    ARM32_FPU_REGISTER_START    = ((ARM32_REGISTER_CONTROL) + 1),

    CF_REG_IP                   = -3, // Call Frame aliases for registers
    CF_REG_FA                   = -2,
    CF_REG_RA                   = -1,
};

//==============================================================================

bool Open( nn::svc::Handle debugHandle, const coredump::coredump_compiled_data* pDetails, char* pOutputName, coredump::coredump_writer** pWriter )
{
    DMNT_CORE_DUMPER_LOG( "Open %s", pOutputName );

    //===============================================================
    // We have to insure this file exists before we can open it...
    //===============================================================
    nn::fs::DeleteFile( pOutputName );
    nn::fs::CreateFile( pOutputName, 0 );

    //Create and write our header
    coredump::coredump_file_header Header;

    Header.m_ProcessId = pDetails->m_ProcessId;
    strcpy( Header.m_ProcessName, pDetails->m_ProcessName );
    Header.m_ExceptionNumber = pDetails->m_ExceptionId;
    memcpy( Header.m_Args, pDetails->m_Args, sizeof(pDetails->m_Args) );
    strcpy( Header.m_Architecture, "NX" );
    strcpy( Header.m_OSVersion,    "1.0.0" );

    *pWriter = &s_Writer;
    return s_Writer.Init( &Header, pOutputName );
}

//============================================================================================

void AddTTY( nn::svc::Handle debugHandle, const coredump::coredump_compiled_data* pDetails, coredump::coredump_writer* pWriter )
{
    DMNT_CORE_DUMPER_LOG( "AddTTY" );
    //char* pTTY = "Can't get TTY at the moment.\n"; //SnapShotDumper::System::GetProcessTTY( (s32)ProcessId );
    if( pDetails->m_pTTY != NULL )
    {
        pWriter->AddTTY( pDetails->m_pTTY, strlen(pDetails->m_pTTY) );
    }
}

//============================================================================================

bool AddRegisterDefinitions( nn::svc::Handle debugHandle, const coredump::coredump_compiled_data* pDetails, coredump::coredump_writer* pWriter )
{
    DMNT_CORE_DUMPER_LOG( "AddRegisterDefinitions" );
    bool Ret = false;
    uint32_t RegisterDefinitionSize = 0;
    tma::dbg::GetRegisterDefinitions ( pDetails->m_Is64Bit, pDetails->m_Is64BitAddressSpace, RegisterDefinitionSize, (void*)&RegisterDefinitionSize );
    DMNT_CORE_DUMPER_LOG("AddRegisterDefinitions:  GetRegisterDefinitions needs %d bytes", RegisterDefinitionSize);

    if( RegisterDefinitionSize > 0 )
    {
        void* pRegisterDefinitions = Alloc( RegisterDefinitionSize );
        if( pRegisterDefinitions != NULL )
        {
            tma::dbg::GetRegisterDefinitions( pDetails->m_Is64Bit, pDetails->m_Is64BitAddressSpace, RegisterDefinitionSize, pRegisterDefinitions );
            Ret = pWriter->AddRegisterDefinitions( pRegisterDefinitions, RegisterDefinitionSize ) == coredump::result::RESULT_COREDUMP_OK;
            Dealloc( pRegisterDefinitions, RegisterDefinitionSize );
        }
    }
    return Ret;
}

//============================================================================================

bool AddApplicationId( nn::svc::Handle debugHandle, const coredump::coredump_compiled_data* pDetails, coredump::coredump_writer* pWriter )
{
    pWriter->AddApplicationId( pDetails->m_ProgramId );
    DMNT_CORE_DUMPER_LOG( "AddApplicationID added Id %llx", pDetails->m_ProgramId );

    return true;
}

//============================================================================================

bool AddModules( nn::svc::Handle debugHandle, const coredump::coredump_compiled_data* pDetails, coredump::coredump_compiled_data_module* pModules, coredump::coredump_writer* pWriter )
{
    DMNT_CORE_DUMPER_LOG( "Adding %d modules from process %lld", pDetails->m_NumberOfModules, pDetails->m_ProcessId );

    for(int ModuleIndex = 0; ModuleIndex < pDetails->m_NumberOfModules; ModuleIndex += 1 )
    {
        if( pWriter->AddModule( pModules[ModuleIndex].m_Name, pModules[ModuleIndex].m_Id, pModules[ModuleIndex].m_Address, pModules[ModuleIndex].m_Size )
            != coredump::result::RESULT_COREDUMP_OK )
        {
            return false;
        }
    }
    return true;
}

//============================================================================================

bool AddModules(nn::svc::Handle debugHandle, const coredump::coredump_compiled_data* pDetails, coredump::coredump_writer* pWriter)
{
    bool Ret = true;
    int nModules = 0;
    nn::os::ProcessId Id;
    Id.value = pDetails->m_ProcessId;

    nn::Result result = nn::dbg::GetProcessModuleInfo(&nModules, s_pModuleBuffer, MAX_NUMBER_OF_MODULES, Id );

    if (result.IsSuccess())
    {
        if ( nModules != pDetails->m_NumberOfModules )
        {
            // Not a fatal error, but let's report it anyway.
            DMNT_CORE_DUMPER_ERROR("AddModules - Found %d modules, but was supposed to be %d", nModules, pDetails->m_NumberOfModules );
        }

        DMNT_CORE_DUMPER_LOG("Adding %d modules from process %lld", nModules, pDetails->m_ProcessId);

        nn::svc::PageInfo PageInfo;
        nn::svc::MemoryInfo MemInfo;
        char PathBuffer[MAX_PATH + 1];
        for (int ModuleIndex = 0; ModuleIndex < nModules; ModuleIndex += 1)
        {
            nn::dbg::ModuleInfo* pAddModule = &s_pModuleBuffer[ModuleIndex];
            memset(PathBuffer, 0, sizeof(PathBuffer));

            DMNT_CORE_DUMPER_LOG( "Adding module %d (address = 0x%llx, size = 0x%llx).", ModuleIndex, pAddModule->address, pAddModule->size );
            //=================================================================================================================
            // Dig into the NSO to pull out the ELF file information -Wl,--nx-debuglink
            // Get EX size to determine the start of the RO section. First entry should be a Pascal string with the .nss path
            //=================================================================================================================
            nn::Result queryResult = nn::dbg::QueryDebugProcessMemory(&MemInfo, &PageInfo, debugHandle, pAddModule->address );
            if (queryResult.IsSuccess())
            {
                coredump::u64 Buffer = 0;
                coredump::u64 Address = MemInfo.baseAddress + MemInfo.size;
                queryResult = nn::dbg::ReadDebugProcessMemory((uintptr_t)&Buffer, debugHandle, (uintptr_t)Address, sizeof(Buffer));
                coredump::u64 Length = Buffer >> 32;
                Length |= (Buffer << 32);

                if (queryResult.IsSuccess() && Length > 0 && Length < MAX_PATH)
                {
                    queryResult = nn::dbg::ReadDebugProcessMemory((uintptr_t)PathBuffer, debugHandle, (uintptr_t)(Address + sizeof(coredump::u64)), Length);
                    PathBuffer[Length] = '\0';
                    if (queryResult.IsFailure())
                    {
                        memset(PathBuffer, 0, sizeof(PathBuffer));
                    }
                }
            }
            else
            {
                // Not a fatal error, but let's report it anyway.
                DMNT_CORE_DUMPER_ERROR("AddModules - Getting module path FAILED - nn::dbg::QueryDebugProcessMemory returned 0x%x", queryResult.GetInnerValueForDebug());
            }

            if ( pWriter->AddModule(PathBuffer, (char*)pAddModule->moduleId, pAddModule->address, pAddModule->size) != coredump::result::RESULT_COREDUMP_OK )
            {
                Ret = false;
                break;
            }
        }
    }
    else
    {
        DMNT_CORE_DUMPER_ERROR("AddModules - nn::dbg::GetProcessModuleInfo returned 0x%x", result.GetInnerValueForDebug());
        Ret = false;
    }

    return Ret;
}

//========================================================================================================

enum
{
    HIGHEST_APP_PRIORITY    = 28,
    APP_PRIORITY_RANGE      = 32,
    LOWEST_APP_PRIORITY     = (HIGHEST_APP_PRIORITY + APP_PRIORITY_RANGE),
};

//============================================================================================

static coredump::s16 GetThreadPriority(  nn::svc::Handle debugHandle, nn::Bit64 ThreadId )
{
    coredump::s16 Priority = nn::os::DefaultThreadPriority;

    // Try to read the thread priority from the system.
    nn::Bit64 r1 = 0;
    nn::Bit32 r2 = 0;
    nn::Result result = nn::dbg::GetDebugThreadParam( &r1, &r2, debugHandle, ThreadId, nn::svc::DebugThreadParam_Priority );
    if( result.IsSuccess() )
    {
        //Cleanse this for public consumption.
        Priority = r2 - HIGHEST_APP_PRIORITY;
    }

    return Priority;
}

//==============================================================================

static coredump::s16 GetThreadStatus( nn::svc::Handle debugHandle, nn::Bit64 ThreadId )
{
    //Default to runnable suspended
    coredump::s16 Ret = 'S';

    nn::Bit64 r1 = 0;
    nn::Bit32 r2 = 0;
    nn::Result result = nn::dbg::GetDebugThreadParam( &r1, &r2, debugHandle, ThreadId, nn::svc::DebugThreadParam_State );
    if( result.IsSuccess() )
    {
        switch( r2 )
        {
            default:
            case nn::svc::ThreadState::ThreadState_Runnable:
                Ret = ((r1 & nn::svc::ThreadSuspend_Debug)? 'S': 'R');
                break;

            case nn::svc::ThreadState::ThreadState_Wait:
                Ret = 'W';
                break;

            case nn::svc::ThreadState::ThreadState_Terminated:
                Ret = 'M';
                break;
        }
    }

    return Ret;
}

//==============================================================================

static coredump::s16 GetThreadCore(  nn::svc::Handle debugHandle, nn::Bit64 ThreadId )
{
    coredump::s16 Core      = 0;

    nn::Bit64 r1 = 0;
    nn::Bit32 r2 = 0;
    nn::Result result = nn::dbg::GetDebugThreadParam( &r1, &r2, debugHandle, ThreadId, nn::svc::DebugThreadParam_CurrentCore );
    if( result.IsSuccess() )
    {
        Core = r2;
        //Get the affinity mask.
        result = nn::dbg::GetDebugThreadParam( &r1, &r2, debugHandle, ThreadId, nn::svc::DebugThreadParam_AffinityMask );
        if( result.IsSuccess() )
        {
            nn::Bit32 affinityMask = (nn::Bit32)r1;

            // This goes in the upper 8 bits of this field.
            coredump::s16 temp = ( affinityMask & 0xFF );
            Core |= (temp << 8);
        }
    }

    return Core;
}

//========================================================================================================

static nn::svc::ThreadContext sContext;

enum
{
    NUMBER_OF_CONTEXT_GP_REGISTERS = sizeof(sContext.r) / sizeof(sContext.r[0]),
    NUMBER_OF_CONTEXT_FP_REGISTERS = 64, //sizeof(sContext.v) / sizeof(sContext.v[0]),
};

static coredump::u64 FPRegisters[NUMBER_OF_CONTEXT_FP_REGISTERS];
static coredump::u64 GPRegisters[NUMBER_OF_CONTEXT_GP_REGISTERS + 4];

//========================================================================================================

bool AddThread( nn::svc::Handle debugHandle, const coredump::coredump_compiled_data* pDetails, const coredump::coredump_compiled_data_thread* pThread, coredump::coredump_writer* pWriter )
{
    DMNT_CORE_DUMPER_LOG( "AddThread %lld", pThread->m_Id );
    coredump::coredump_thread_info Info;
    nn::Result result = nn::dbg::GetDebugThreadContext( &sContext, debugHandle, pThread->m_Id, ALL_REGISTERS );
    if( result.IsFailure() )
    {
        DMNT_CORE_DUMPER_ERROR( "dmnt_CoreDump::AddThread nn::dbg::GetDebugThreadContext failed." );
        return false;
    }

#if defined(NN_BUILD_CONFIG_CPU_ARM_V8A)

    if( sContext.sp == 0 ) // !pProcess->Is64Bit() ) // Fixup the context info
    {
        sContext.sp  = sContext.r[ ARM32_REGISTER_SP ]; // Copy registers to named locations
        sContext.lr  = sContext.r[ ARM32_REGISTER_LR ];
        sContext.r[ ARM32_REGISTER_PC      ]  = sContext.pc; // Copy registers to numeric locations
        sContext.r[ ARM32_REGISTER_CONTROL ]  = sContext.pstate;
    }
#endif

    Info.m_ThreadId         = pThread->m_Id;
    Info.m_CurrentThread    = (pDetails->m_CurrentThreadId == pThread->m_Id);

    Info.m_Priority         = GetThreadPriority( debugHandle, pThread->m_Id );
    Info.m_Core             = GetThreadCore( debugHandle, pThread->m_Id );
    Info.m_IP               = sContext.pc;
    Info.m_SP               = sContext.sp;
    Info.m_Status           = GetThreadStatus( debugHandle, pThread->m_Id );
    strcpy(Info.m_Name, pThread->m_Name);

    //Save our registers
    Info.m_NumberOfGPRegisters = NUMBER_OF_CONTEXT_GP_REGISTERS + 4;
    int RegIndex = 0;
    for( RegIndex = 0; RegIndex < NUMBER_OF_CONTEXT_GP_REGISTERS; RegIndex += 1)
    {
        GPRegisters[ RegIndex ] = sContext.r[ RegIndex ];
    }

#if defined(NN_BUILD_CONFIG_CPU_ARM_V8A)
    GPRegisters[ RegIndex + 0 ]   = sContext.fp;
    GPRegisters[ RegIndex + 1 ]   = sContext.lr;
    GPRegisters[ RegIndex + 2 ]   = sContext.sp;
    GPRegisters[ RegIndex + 3 ]   = sContext.pc;
#elif defined(NN_BUILD_CONFIG_CPU_ARM_V7A)
    GPRegisters[ RegIndex + 0 ]   = sContext.lr;
    GPRegisters[ RegIndex + 1 ]   = sContext.sp;
    GPRegisters[ RegIndex + 2 ]   = sContext.pc;
#endif

    Info.m_pGPRegisters = GPRegisters;

    Info.m_NumberOfGPControlRegisters = 1;
#if defined(NN_BUILD_CONFIG_CPU_ARM_V8A)
    coredump::u64 GPControlRegisters[1] = { sContext.pstate };
#elif defined(NN_BUILD_CONFIG_CPU_ARM_V7A)
    coredump::u64 GPControlRegisters[1] = { sContext.cpsr };
#endif
    Info.m_pGPControlRegisters = GPControlRegisters;

    Info.m_NumberOfFPRegisters = NUMBER_OF_CONTEXT_FP_REGISTERS;

#if defined(NN_BUILD_CONFIG_CPU_ARM_V8A)
    memcpy( (void*)FPRegisters, (void*)&sContext.v, sizeof(FPRegisters) );
#endif
    Info.m_pFPRegisters = FPRegisters;

    Info.m_NumberOfFPControlRegisters = 2;

#if defined(NN_BUILD_CONFIG_CPU_ARM_V8A)
    coredump::u64 FPControlRegisters[2] = { sContext.fpcr, sContext.fpsr };
#elif defined(NN_BUILD_CONFIG_CPU_ARM_V7A)
    coredump::u64 FPControlRegisters[2] = { sContext.fpscr, sContext.fpexc };
#endif
    Info.m_pFPControlRegisters = FPControlRegisters;
    return pWriter->AddThread( &Info, sContext.tpidr ) == coredump::result::RESULT_COREDUMP_OK;
}

//=======================================================================================

bool AddThreads( nn::svc::Handle debugHandle, const coredump::coredump_compiled_data* pDetails, coredump::coredump_compiled_data_thread* pThreads, coredump::coredump_writer* pWriter )
{
    DMNT_CORE_DUMPER_LOG( "Adding %d threads from process %lld", pDetails->m_NumberOfThreads, pDetails->m_ProcessId );

    for(int ThreadIndex = 0; ThreadIndex < pDetails->m_NumberOfThreads; ThreadIndex += 1 )
    {
        if( AddThread( debugHandle, pDetails, &pThreads[ThreadIndex], pWriter ) != true )
        {
            return false;
        }
    }
    return true;
}

//=======================================================================================

bool AddMemory( nn::svc::Handle debugHandle, coredump::MemoryType Type, coredump::u64 SizeOfData, coredump::u64 Address, coredump::coredump_writer* pWriter )
{
    //Create the memory section.
    DMNT_CORE_DUMPER_LOG( "AddMemory:  0x%llx (%lld bytes)", Address, SizeOfData );
    bool Ret = false;
    if( pWriter->CreateMemorySection( Address, Type, SizeOfData ) == coredump::RESULT_COREDUMP_OK )
    {
        size_t MemorySectionBufferSize = ( SizeOfData > COREDUMP_MEMORY_SECTION_BUFFER_SIZE ) ? COREDUMP_MEMORY_SECTION_BUFFER_SIZE : SizeOfData;

        //Add all our data.
        unsigned char* pCopyBuffer = (unsigned char*)Alloc( MemorySectionBufferSize );

        if( pCopyBuffer != NULL )
        {
            coredump::u64 SizeOfCompressionBuffer = pWriter->CompressionBufferSizeRequired( MemorySectionBufferSize );
            unsigned char* pCompressionBuffer = (unsigned char*)Alloc( SizeOfCompressionBuffer );

            if( pCompressionBuffer != NULL )
            {
                //Start our copy loop.
                Ret = true;

                uint64_t ReadFromAddress = Address;
                uint64_t AmountToCopy = SizeOfData;

                while( AmountToCopy > 0 )
                {
                    uint64_t SizeToCopy = ( AmountToCopy > MemorySectionBufferSize ) ? MemorySectionBufferSize : AmountToCopy;
                    nn::Result Result = nn::dbg::ReadDebugProcessMemory( (uintptr_t)pCopyBuffer, debugHandle, ReadFromAddress, SizeToCopy );
                    if( Result.IsFailure() )
                    {
                        memset( pCopyBuffer, 0xFF, SizeToCopy );
                    }

                    DMNT_CORE_DUMPER_LOG( "AddMemoryBlock:  Adding memory data size = %lld", SizeToCopy );
                    if( pWriter->AddMemoryData( pCopyBuffer, SizeToCopy, pCompressionBuffer, SizeOfCompressionBuffer ) != coredump::RESULT_COREDUMP_OK )
                    {
                        Ret = false;
                        break;
                    }
                    AmountToCopy    -= SizeToCopy;
                    ReadFromAddress += SizeToCopy;
                }
                Dealloc( pCompressionBuffer, SizeOfCompressionBuffer );
            }
            //Done with our buffer.
            Dealloc( pCopyBuffer, MemorySectionBufferSize );
        }

        DMNT_CORE_DUMPER_LOG( "AddMemoryBlock:  Closing memory section" );
        pWriter->CloseMemorySection( Address );
    }
    return Ret;
}

//=======================================================================================

bool AddMemoryBlock( nn::svc::Handle debugHandle, nn::svc::MemoryInfo* pBlockInfo, coredump::coredump_writer* pWriter )
{
    DMNT_CORE_DUMPER_LOG( "AddMemoryBlock" );

    //Assume we can read this.
    coredump::MemoryType Type = coredump::MemoryType_Read;

    //Figure out our memory type.
    switch( pBlockInfo->permission )
    {
        default:
            Type = coredump::MemoryType_All;
            break;
        case nn::svc::MemoryPermission_Read:
            Type = coredump::MemoryType_Read;
            break;
        case nn::svc::MemoryPermission_Write:
            Type = coredump::MemoryType_Write;
            break;
        case nn::svc::MemoryPermission_Execute:
            Type = coredump::MemoryType_Execute;
            break;
        case nn::svc::MemoryPermission_ReadWrite:
            Type = coredump::MemoryType_ReadWrite;
            break;
        case nn::svc::MemoryPermission_ReadExecute:
            Type = coredump::MemoryType_ReadExecute;
            break;
    }

    //Read the data.
    return AddMemory( debugHandle, Type, (coredump::u64)pBlockInfo->size, (coredump::u64)pBlockInfo->baseAddress, pWriter );
}

//=======================================================================================

void AddMemory( nn::svc::Handle debugHandle, const coredump::coredump_compiled_data* pDetails, coredump::coredump_compiled_data_thread* pThreads, coredump::coredump_writer* pWriter )
{
    DMNT_CORE_DUMPER_LOG( "AddMemory" );
    // Find the memory blocks with these permissions
    int SaveMemoryPermissionType = ( nn::svc::MemoryPermission_Read | nn::svc::MemoryPermission_Write | nn::svc::MemoryPermission_Execute );
    if( pDetails->m_QuickDump == true )
    {
        //============================================================
        // If it's a mini dump, we just want the Execute memory blocks
        // at this point - We'll add specific areas of memory later.
        //============================================================
        SaveMemoryPermissionType = nn::svc::MemoryPermission_Execute;
    }

    for (uintptr_t v = 0;;)
    {
        nn::svc::MemoryInfo blockInfo;
        nn::svc::PageInfo   pageInfo;
        nn::Result result = nn::dbg::QueryDebugProcessMemory( &blockInfo, &pageInfo, debugHandle, v );
        if( result.IsFailure() )
        {
            break;
        }
        if( blockInfo.state == nn::svc::MemoryState_Inaccessible )
        {
            break;
        }

        if( blockInfo.baseAddress + blockInfo.size < v )
        {
            break;
        }

        if( ( blockInfo.permission & SaveMemoryPermissionType ) != 0 && (blockInfo.state != nn::svc::MemoryState_Io) )
        {
            if( AddMemoryBlock( debugHandle, &blockInfo, pWriter ) == false )
            {
                DMNT_CORE_DUMPER_LOG( "AddMemory FAILED." );
                return;
            }
        }

        v = blockInfo.baseAddress + blockInfo.size;
    }

    //=====================================================================
    // If a quick dump, we'll need to cherry-pick our memory -
    //=====================================================================
    if( pDetails->m_QuickDump == true)
    {
        DMNT_CORE_DUMPER_LOG( "AddMemory: Adding %d thread stacks for mini dump.",  pDetails->m_NumberOfThreads );

        // We'll want the stacks for each of our threads.
        for(int ThreadIndex = 0; ThreadIndex < pDetails->m_NumberOfThreads; ThreadIndex += 1 )
        {
            coredump::coredump_compiled_data_thread* pThread = &pThreads[ThreadIndex];
            if( pThread->m_StackSize > 0 )
            {
                DMNT_CORE_DUMPER_LOG( "AddMemory: Adding thread %d stack:  0x%llx (%lld bytes).",  pThread->m_Id, pThread->m_StackAddress, pThread->m_StackSize );
                if( AddMemory( debugHandle, coredump::MemoryType_ReadWrite, pThread->m_StackSize, pThread->m_StackAddress, pWriter ) == false )
                {
                    break;
                }
            }
            else
            {
                DMNT_CORE_DUMPER_LOG( "AddMemory: Unable to add thread %d stack:  0x%llx (%lld bytes).",  pThread->m_Id, pThread->m_StackAddress, pThread->m_StackSize );
            }
        }
    }
}

//================================================================================================================

struct memory_block_writer
{
    //================================================================================================================

    memory_block_writer()
    {
        Init( coredump::MemoryType_All, 0, 0);
    }

    //================================================================================================================

    void Init(nn::svc::MemoryInfo* pBlockInfo)
    {
        coredump::MemoryType Type = coredump::MemoryType_Read;

        // Figure out our memory type.
        switch (pBlockInfo->permission)
        {
        default:
            Type = coredump::MemoryType_All;
            break;
        case nn::svc::MemoryPermission_Read:
            Type = coredump::MemoryType_Read;
            break;
        case nn::svc::MemoryPermission_Write:
            Type = coredump::MemoryType_Write;
            break;
        case nn::svc::MemoryPermission_Execute:
            Type = coredump::MemoryType_Execute;
            break;
        case nn::svc::MemoryPermission_ReadWrite:
            Type = coredump::MemoryType_ReadWrite;
            break;
        case nn::svc::MemoryPermission_ReadExecute:
            Type = coredump::MemoryType_ReadExecute;
            break;
        }

        Init(Type, (coredump::u64)pBlockInfo->baseAddress, (coredump::u64)pBlockInfo->size);
    }

    //================================================================================================================

    void Init( coredump::MemoryType Type, coredump::u64 Address, coredump::u64 Size )
    {
        //DMNT_CORE_DUMPER_LOG("[memory_block_writer::Init] Type %d at 0x%llx (%lld bytes)", Type, Address, Size);
        m_Type = Type;
        m_Address = Address;
        m_ReadAddress = Address;
        m_AmountLeft = Size;
        m_Open = false;
    }

    //============================================================================================

    bool Continue( nn::svc::Handle debugHandle, unsigned char* pReadBuffer, size_t ReadBufferSize, unsigned char* pCompressionBuffer, size_t CompressionBufferSize, coredump::coredump_writer* pWriter, float* pRunningTotal )
    {
        // Open this section, if we haven't yet
        if( m_Open == false)
        {
            //DMNT_CORE_DUMPER_LOG("[memory_block_writer] Creating memory section at 0x%llx", m_Address );
            coredump::result res = pWriter->CreateMemorySection(m_Address, m_Type, m_AmountLeft);
            if(res != coredump::RESULT_COREDUMP_OK )
            {
                DMNT_CORE_DUMPER_ERROR("[memory_block_writer] CreateMemorySection returned FAIL:  %d", (int)res);
                return false;
            }

            m_Open = true;
        }

        // Already completed?
        if( m_AmountLeft <= 0 )
        {
            //DMNT_CORE_DUMPER_LOG("[memory_block_writer] DONE:  m_AmountLeft <= 0  (%lld)", m_AmountLeft );
            return false;
        }

        uint64_t SizeToCopy = ( m_AmountLeft > ReadBufferSize ) ? ReadBufferSize : m_AmountLeft;
        nn::Result Result = nn::dbg::ReadDebugProcessMemory( (uintptr_t)pReadBuffer, debugHandle, m_ReadAddress, SizeToCopy );

        if( Result.IsFailure() )
        {
            memset( pReadBuffer, 0xFF, SizeToCopy );
        }

        bool Ret = true;
        coredump::result res = pWriter->AddMemoryData(pReadBuffer, SizeToCopy, pCompressionBuffer, CompressionBufferSize);
        if(res != coredump::RESULT_COREDUMP_OK )
        {
            DMNT_CORE_DUMPER_ERROR("[memory_block_writer] AddMemoryData returned FAIL:  %d", (int)res);
            Ret = false;
        }

        m_AmountLeft    -= SizeToCopy;
        m_ReadAddress   += SizeToCopy;
        *pRunningTotal  += (float)SizeToCopy;

        if( m_AmountLeft <= 0 )
        {
            //DMNT_CORE_DUMPER_LOG( "[memory_block_writer] Closing memory section at 0x%llx", m_Address);
            res = pWriter->CloseMemorySection( m_Address );
            if (res != coredump::RESULT_COREDUMP_OK)
            {
                DMNT_CORE_DUMPER_ERROR("[memory_block_writer] CloseMemorySection returned FAIL:  %d", (int)res);
            }
        }

        return Ret;
    }

    //============================================================================================

    bool IsFinished()
    {
        return (m_AmountLeft <= 0 );
    }

    //============================================================================================

    bool                    m_Open;
    coredump::MemoryType    m_Type;
    coredump::u64           m_Address;
    coredump::u64           m_AmountLeft;
    coredump::u64           m_ReadAddress;
};

//============================================================================================

struct dmnt_core_dump
{
    dmnt_core_dump( nn::svc::Handle debugHandle, const coredump::coredump_compiled_data* pDetails, coredump::coredump_compiled_data_thread* pThreads, char* pOutputName )
    {
        m_DebugHandle = debugHandle;
        strcpy( m_OutputName, pOutputName );
        m_pWriter = NULL;
        m_AmountToWrite = 1000;
        m_AmountWritten = 0;
        m_WroteThreads = false;
        m_WroteModules = false;
        m_CompressionBufferSize = sizeof(s_CoredumpCompressionBuffer);
        m_pCompressionBuffer = s_CoredumpCompressionBuffer;

        // Init our memory writing tracking variables
        m_MemoryBlockIterator = 0;
        m_CompletedMemoryIteration = false;
        m_ThreadStackIterator = 0;
        m_CompletedThreadStacks = false;

        memcpy( &m_Details, pDetails, sizeof(m_Details) );

        //=======================================================================
        // We extract the module information ourselves, so mark this as NULL so
        // we don't use it or try to delete it.
        //=======================================================================
        m_Details.m_pModules = NULL;

        if( pDetails->m_NumberOfThreads > 0 )
        {
            DMNT_CORE_DUMPER_LOG("[dmnt_core_dump::dmnt_core_dump] Allocating %d for threads", sizeof(coredump::coredump_compiled_data_thread) * pDetails->m_NumberOfThreads);
            m_Details.m_pThreads = s_CoredumpThreadDetailBuffer;
            m_Details.m_NumberOfThreads = pDetails->m_NumberOfThreads < MAX_NUMBER_OF_THREADS ? pDetails->m_NumberOfThreads : MAX_NUMBER_OF_THREADS;
            memcpy( m_Details.m_pThreads, pThreads, sizeof(coredump::coredump_compiled_data_thread) * m_Details.m_NumberOfThreads);
        }
    }

    ~dmnt_core_dump()
    {
        DMNT_CORE_DUMPER_LOG( "[dmnt_core_dump::~dmnt_core_dump]" );
        if( m_pWriter != NULL )
        {
            m_pWriter->Close();
            //No longer need to delete this guy.
            // delete m_pWriter;
        }
        m_Details.m_pThreads = NULL;
        m_Details.m_pModules = NULL;
        DMNT_CORE_DUMPER_LOG( "[dmnt_core_dump::~dmnt_core_dump] COMPLETE" );
    }

    //Backwards-compatability:  One-stop shop for writing.
    nn::Result  Write       ( char*         pOutputName );

    // Cancellable-version (Siglo-56112)
    nn::Result  Begin       ( );
    nn::Result  Continue    ( );
    nn::Result  AddTTY      ( char*         pData,
                              size_t        AmountOfData );

    nn::Result  AddImage    ( char*         pData,
                              size_t        AmountOfData );

    // Object functionality.
    int32_t     GetProgress             ();
    bool        ContinueMemoryWrite     ();
    bool        WriteNextMemoryBlock    ();

    void        StartMemoryBlock();

    // Data
    nn::svc::Handle m_DebugHandle;
    char    m_OutputName[1024];

    coredump::coredump_compiled_data m_Details;
    coredump::coredump_writer* m_pWriter;
    bool  m_WroteThreads;
    bool  m_WroteModules;

    // Memory writing tracking
    size_t                  m_CompressionBufferSize;
    unsigned char*          m_pCompressionBuffer;

    // Memory writing tracking variables
    memory_block_writer     m_MemoryWriter;
    uintptr_t               m_MemoryBlockIterator;
    bool                    m_CompletedMemoryIteration;
    int                     m_ThreadStackIterator;
    bool                    m_CompletedThreadStacks;

    float m_AmountToWrite;
    float m_AmountWritten;
};

//============================================================================================

int32_t dmnt_core_dump::GetProgress()
{
    float Percent = m_AmountWritten / m_AmountToWrite;
    int32_t Ret = (int32_t)(Percent * 100.0f);
    return Ret;
}

//============================================================================================

void dmnt_core_dump::StartMemoryBlock()
{
    int SaveMemoryPermissionType = (nn::svc::MemoryPermission_Read | nn::svc::MemoryPermission_Write | nn::svc::MemoryPermission_Execute);
    if (m_Details.m_QuickDump == true)
    {
        //============================================================
        // If it's a mini dump, we just want the Execute memory blocks
        // at this point - We'll add specific areas of memory later.
        //============================================================
        SaveMemoryPermissionType = nn::svc::MemoryPermission_Execute;
    }

    while ( m_CompletedMemoryIteration != true )
    {
        nn::svc::MemoryInfo blockInfo;
        nn::svc::PageInfo   pageInfo;
        nn::Result result = nn::dbg::QueryDebugProcessMemory(&blockInfo, &pageInfo, m_DebugHandle, m_MemoryBlockIterator );
        if (result.IsFailure())
        {
            m_CompletedMemoryIteration = true;
            DMNT_CORE_DUMPER_LOG("[StartMemoryBlock] MemoryIteration COMPLETE:  nn::dbg::QueryDebugProcessMemory returned %d", result.GetInnerValueForDebug());
            break;
        }
        if (blockInfo.state == nn::svc::MemoryState_Inaccessible)
        {
            DMNT_CORE_DUMPER_LOG("[StartMemoryBlock] MemoryIteration COMPLETE:  blockInfo.state == nn::svc::MemoryState_Inaccessible");
            m_CompletedMemoryIteration = true;
            break;
        }

        if ( blockInfo.baseAddress + blockInfo.size < m_MemoryBlockIterator )
        {
            DMNT_CORE_DUMPER_LOG("[StartMemoryBlock] MemoryIteration COMPLETE:  blockInfo.baseAddress + blockInfo.size < m_MemoryBlockIterator");
            m_CompletedMemoryIteration = true;
            break;
        }

        // Increment our iterator
        m_MemoryBlockIterator = blockInfo.baseAddress + blockInfo.size;

        if (((blockInfo.permission & SaveMemoryPermissionType) != 0) && (blockInfo.state != nn::svc::MemoryState_Io))
        {
            //===========================================================================
            // This new operation adds this block to the m_pAllMemoryBlockWriters list.
            // We'll free it up when we are deleted.
            //===========================================================================
            //DMNT_CORE_DUMPER_LOG("[StartMemoryBlock] Starting memory section at 0x%llx", blockInfo.baseAddress);
            m_MemoryWriter.Init(&blockInfo);
            return;
        }
    }

    //=====================================================================
    // If a quick dump, we'll need to cherry-pick our memory -
    //=====================================================================
    if (m_Details.m_QuickDump == true && m_ThreadStackIterator < m_Details.m_NumberOfThreads )
    {
        // We'll want the stacks for each of our threads.
        while ( m_ThreadStackIterator < m_Details.m_NumberOfThreads )
        {
            coredump::coredump_compiled_data_thread* pThread = &m_Details.m_pThreads[m_ThreadStackIterator];
            m_ThreadStackIterator += 1;
            if (pThread->m_StackSize > 0)
            {
                //DMNT_CORE_DUMPER_LOG("[StartMemoryBlock] Starting memory section for thread at 0x%llx", pThread->m_StackAddress);
                m_MemoryWriter.Init( coredump::MemoryType_ReadWrite, pThread->m_StackAddress, pThread->m_StackSize );
                return;
            }
        }
    }
    DMNT_CORE_DUMPER_LOG("[StartMemoryBlock] COMPLETED");
}

//============================================================================================

bool dmnt_core_dump::WriteNextMemoryBlock()
{
    if ( m_MemoryWriter.IsFinished() )
    {
        StartMemoryBlock();
    }

    if ( m_MemoryWriter.IsFinished() == false )
    {
        return m_MemoryWriter.Continue(m_DebugHandle, s_CoredumpMemoryBuffer, sizeof(s_CoredumpMemoryBuffer), m_pCompressionBuffer, m_CompressionBufferSize, m_pWriter, &m_AmountWritten);
    }

    return false;
}

//============================================================================================

bool dmnt_core_dump::ContinueMemoryWrite()
{
    // Make sure we have buffers.
    if( m_pCompressionBuffer == NULL )
    {
        return false;
    }

    //=============================================================
    // Let's write blocks until our progress measureably changes.
    //=============================================================
    int32_t StartingProgress = GetProgress();
    bool Ret = WriteNextMemoryBlock();

    while( Ret )
    {
        if( GetProgress() != StartingProgress )
        {
            break;
        }
        Ret = WriteNextMemoryBlock();
    }

    return Ret;
}

//============================================================================================

nn::Result dmnt_core_dump::Write( char* pOutputName )
{
    nn::Result res = nn::ResultSuccess();
    DMNT_CORE_DUMPER_LOG( "Write %s", pOutputName );
    if( m_DebugHandle.IsValid() == true )
    {
        if( Open( m_DebugHandle, &m_Details, pOutputName, &m_pWriter ) )
        {
            if( AddRegisterDefinitions( m_DebugHandle, &m_Details, m_pWriter ) )
            {
                if( AddModules( m_DebugHandle, &m_Details, m_pWriter ) )
                {
                    if( AddThreads( m_DebugHandle, &m_Details, m_Details.m_pThreads, m_pWriter ) )
                    {
                        AddMemory( m_DebugHandle, &m_Details, m_Details.m_pThreads, m_pWriter );
                    }
                }
            }
        }
    }
    else
    {
    }

    return res;
}

//========================================================================================================

nn::Result dmnt_core_dump::Begin()
{
    DMNT_CORE_DUMPER_LOG( "[dmnt_core_dump::Begin]" );
    //======================================
    // Calculate how much we need to write.
    //======================================

    //===========================================================================================
    // NOTICE that these sizes for threads and modules don't have to be EXACTLY how much we are
    // going to write, but a relative representation of how much data we need to go through.
    //===========================================================================================
    m_AmountToWrite = (float)(m_Details.m_NumberOfThreads * sizeof( coredump::coredump_compiled_data_thread ));
    m_AmountToWrite += (float)(m_Details.m_NumberOfModules * sizeof( coredump::coredump_compiled_data_module ));

    //================================================================
    // Calculate our memory write requirements with more precision
    //================================================================
    int SaveMemoryPermissionType = ( nn::svc::MemoryPermission_Read | nn::svc::MemoryPermission_Write | nn::svc::MemoryPermission_Execute );
    if( m_Details.m_QuickDump == true )
    {
        //============================================================
        // If it's a mini dump, we just want the Execute memory blocks
        // at this point - We'll add specific areas of memory later.
        //============================================================
        SaveMemoryPermissionType = nn::svc::MemoryPermission_Execute;
    }

    for (uintptr_t v = 0;;)
    {
        nn::svc::MemoryInfo blockInfo;
        nn::svc::PageInfo   pageInfo;
        nn::Result result = nn::dbg::QueryDebugProcessMemory( &blockInfo, &pageInfo, m_DebugHandle, v );
        if( result.IsFailure() )
        {
            break;
        }
        if( blockInfo.state == nn::svc::MemoryState_Inaccessible )
        {
            break;
        }

        if( blockInfo.baseAddress + blockInfo.size < v )
        {
            break;
        }

        if( ((blockInfo.permission & SaveMemoryPermissionType) != 0) && (blockInfo.state != nn::svc::MemoryState_Io) )
        {
            m_AmountToWrite += (float)blockInfo.size;
        }

        v = blockInfo.baseAddress + blockInfo.size;
    }

    //=====================================================================
    // If a quick dump, we'll need to cherry-pick our memory -
    //=====================================================================
    if( m_Details.m_QuickDump == true)
    {
        // We'll want the stacks for each of our threads.
        for(int ThreadIndex = 0; ThreadIndex < m_Details.m_NumberOfThreads; ThreadIndex += 1 )
        {
            coredump::coredump_compiled_data_thread* pThread = &m_Details.m_pThreads[ThreadIndex];
            if( pThread->m_StackSize > 0 )
            {
                m_AmountToWrite += (float)pThread->m_StackSize;
            }
        }
    }

    // Init our starting amount
    m_AmountWritten = 0;

    return nn::ResultSuccess();
}

//==============================================================================

nn::Result dmnt_core_dump::Continue()
{
    //DMNT_CORE_DUMPER_LOG( "[dmnt_core_dump::Continue]" );
    if( InitFS() == false )
    {
        DMNT_CORE_DUMPER_ERROR( "[dmnt_core_dump::Continue] - Unable to allocate file system.\n" );
        return nn::fs::ResultUnexpected();
    }

    nn::Result Ret = nn::ResultSuccess();
    if( m_pWriter == NULL )
    {
        if( Open( m_DebugHandle, &m_Details, m_OutputName, &m_pWriter ))
        {
            AddRegisterDefinitions(  m_DebugHandle, &m_Details, m_pWriter );
            AddApplicationId( m_DebugHandle, &m_Details, m_pWriter );
        }
        else
        {
            DMNT_CORE_DUMPER_ERROR("[dmnt_core_dump::Continue] - Unable to open output file.\n");
            m_AmountWritten = m_AmountToWrite;
            return nn::fs::ResultUnexpected();
        }
    }
    // Have we added our modules yet?
    else if(m_WroteModules == false )
    {
        AddModules( m_DebugHandle, &m_Details, m_pWriter );
        m_AmountWritten += m_Details.m_NumberOfModules * sizeof( coredump::coredump_compiled_data_module );
        m_WroteModules = true;
    }
    else if( m_WroteThreads == false )
    {
        AddThreads( m_DebugHandle, &m_Details, m_Details.m_pThreads, m_pWriter );
        m_AmountWritten += m_Details.m_NumberOfThreads * sizeof( coredump::coredump_compiled_data_thread );
        m_WroteThreads = true;
    }
    else if (ContinueMemoryWrite() == false)
    {
        DMNT_CORE_DUMPER_LOG("[dmnt_core_dump::Continue] COMPLETE:  ContinueMemoryWrite() == false ");
        m_AmountWritten = m_AmountToWrite;
    }

    return Ret;
}

//==============================================================================

nn::Result dmnt_core_dump::AddTTY( char* pData, size_t AmountOfData )
{
    if( m_pWriter )
    {
        //If we're finished, just add it now
        if( m_AmountWritten == m_AmountToWrite )
        {
            m_pWriter->AddTTY( pData, AmountOfData );
        }
        else
        {
            // Save it for the end.
        }
    }

    return nn::ResultSuccess();
}

//==============================================================================

nn::Result dmnt_core_dump::AddImage( char* pData, size_t AmountOfData )
{
    return nn::ResultSuccess();
}

//========================================================================================================

static bool StringsEqual( const char* pStr1, const char* pStr2 )
{
    int Length = strlen( pStr1 );
    if ( strlen( pStr2 ) != Length )
    {
        return false;
    }

    for( int Index = 0; Index < Length; Index += 1 )
    {
        if( tolower(pStr1[Index]) !=  (unsigned int) tolower(pStr2[Index]) )
        {
            return false;
        }
    }

    return true;
}

//============================================================================================

static bool FormatOutputFileName( char* OutputFileName )
{
    bool Ret = false;
    if( OutputFileName != NULL && strlen( OutputFileName ) > 0 )
    {
        //Does it have a valid extension?
        char* pExtension = strrchr ( OutputFileName, '.' );
        if( pExtension == NULL || StringsEqual(pExtension, SNAP_SHOT_DUMPER_EXTENSION) == false )
        {
            strcat( OutputFileName, SNAP_SHOT_DUMPER_EXTENSION );
        }
        Ret = true;
    }

    return Ret;
}

//===========================================================================================================
// Backwards-compatible, old-school version.  Called via the debug task service and it can't be cancelled.
//===========================================================================================================

nn::Result Server::CreateCoreDump( const nn::sf::InBuffer& FileName, const nn::sf::InBuffer Details, const nn::sf::InBuffer Modules, const nn::sf::InBuffer Threads, nn::svc::Handle debugHandle ) NN_NOEXCEPT
{
    if( InitFS() == false )
    {
        DMNT_CORE_DUMPER_ERROR( "Unable to allocate file system.\n" );
        return nn::fs::ResultUnexpected();
    }

    coredump::coredump_compiled_data* pDetails  = (coredump::coredump_compiled_data*)Details.GetPointerUnsafe();
    coredump::coredump_compiled_data_module* pModules = (coredump::coredump_compiled_data_module*)Modules.GetPointerUnsafe();
    coredump::coredump_compiled_data_thread* pThreads = (coredump::coredump_compiled_data_thread*)Threads.GetPointerUnsafe();
    char* pOutputName = (char*)FileName.GetPointerUnsafe();

    //=======================================================
    // Insure the file name ends correctly.
    //=======================================================
    char OutputFileName[OUTPUT_FILE_NAME_SIZE];
    strcpy( OutputFileName, pOutputName );

    FormatOutputFileName( OutputFileName );

    if( InitFS() == false )
    {
        DMNT_CORE_DUMPER_ERROR( "Unable to allocate file system.\n" );
        return nn::fs::ResultUnexpected();
    }

    dmnt_core_dump* pNewDump = new dmnt_core_dump( debugHandle, pDetails, pThreads, pOutputName );

    nn::Result res = pNewDump->Write( pOutputName );
    delete pNewDump;

    return res;
}

//===========================================================================================================


//===========================================================================================================
// Cancel-able, progress-reporting version of the core dump.  Called via the coredump service.
//===========================================================================================================

nn::Result Server::InitiateCoreDump( const nn::sf::InBuffer& FileName, const nn::sf::InBuffer Details, const nn::sf::InBuffer Modules, const nn::sf::InBuffer Threads, nn::svc::Handle debugHandle, nn::sf::Out<uint64_t> pDumpHandle ) NN_NOEXCEPT
{
    DMNT_CORE_DUMPER_LOG( "[dmnt][InitiateCoreDump]" );
    coredump::coredump_compiled_data* pDetails  = (coredump::coredump_compiled_data*)Details.GetPointerUnsafe();
    coredump::coredump_compiled_data_module* pModules = (coredump::coredump_compiled_data_module*)Modules.GetPointerUnsafe();
    coredump::coredump_compiled_data_thread* pThreads = (coredump::coredump_compiled_data_thread*)Threads.GetPointerUnsafe();
    char* pOutputName = (char*)FileName.GetPointerUnsafe();

    //=======================================================
    // Insure the file name ends correctly.
    //=======================================================
    char OutputFileName[OUTPUT_FILE_NAME_SIZE];
    strcpy( OutputFileName, pOutputName );

    FormatOutputFileName( OutputFileName );

    DMNT_CORE_DUMPER_LOG( "[InitiateCoreDump] new dmnt_core_dump" );
    dmnt_core_dump* pNewDump = new dmnt_core_dump( debugHandle, pDetails, pThreads, OutputFileName );
    nn::Result ret = pNewDump->Begin();
    if( ret.IsSuccess() )
    {
        pDumpHandle.Set( (uint64_t)pNewDump );
    }
    else
    {
        DMNT_CORE_DUMPER_ERROR( "Unable to create core dump session.\n" );
        delete pNewDump;
    }

    DMNT_CORE_DUMPER_LOG( "[InitiateCoreDump] COMPLETE - 0x%llx", pNewDump );
    return ret;
}

//==============================================================================

nn::Result Server::ContinueCoreDump( uint64_t DumpHandle, nn::sf::Out<int32_t> pProgress ) NN_NOEXCEPT
{
    if( DumpHandle != 0 )
    {
        dmnt_core_dump* pDump = (dmnt_core_dump*)DumpHandle;
        pDump->Continue();
        int32_t Progress = pDump->GetProgress();
        pProgress.Set( Progress );

        //DMNT_CORE_DUMPER_LOG( "ContinueCoreDump returning %d progress (%f of %f)", Progress, pDump->m_AmountWritten, pDump->m_AmountToWrite );

        return nn::ResultSuccess();
    }
    return nn::fs::ResultInvalidHostHandle();
}

//==============================================================================

nn::Result Server::AddTTYToCoreDump( const nn::sf::InBuffer Data, uint64_t DumpHandle ) NN_NOEXCEPT
{
    DMNT_CORE_DUMPER_LOG( "AddTTYToCoreDump" );
    if( DumpHandle != 0 )
    {
        dmnt_core_dump* pDump = (dmnt_core_dump*)DumpHandle;
        return pDump->AddTTY( (char*)Data.GetPointerUnsafe(), Data.GetSize() );
    }
    return nn::fs::ResultInvalidHostHandle();
}

//==============================================================================

nn::Result Server::AddImageToCoreDump( const nn::sf::InBuffer Data, uint64_t DumpHandle ) NN_NOEXCEPT
{
    DMNT_CORE_DUMPER_LOG( "AddImageToCoreDump" );
    if( DumpHandle != 0 )
    {
        dmnt_core_dump* pDump = (dmnt_core_dump*)DumpHandle;
        return pDump->AddImage( (char*)Data.GetPointerUnsafe(), Data.GetSize() );
    }
    return nn::fs::ResultInvalidHostHandle();
}

//==============================================================================

nn::Result Server::CloseCoreDump( uint64_t DumpHandle ) NN_NOEXCEPT
{
    DMNT_CORE_DUMPER_LOG( "CloseCoreDump" );
    dmnt_core_dump* pDump = (dmnt_core_dump*)DumpHandle;
    if( pDump != NULL )
    {
        delete pDump;
    }
    DMNT_CORE_DUMPER_LOG( "CloseCoreDump COMPLETE" );
    return nn::ResultSuccess();
}

//==============================================================================
