﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkLog.h>

#include <nn/svc/svc_Base.h>
#include <nn/svc/svc_Dmnt.h>

#include <nn/crypto/crypto_Sha256Generator.h>

#include "creport_ThreadInfo.h"

namespace nn { namespace creport {

    bool ThreadInfo::GetThreadInfo(svc::Handle handle, Bit64 tid, bool is64bit) NN_NOEXCEPT
    {
        nn::Bit64 param1;
        nn::Bit32 param2;

        m_ThreadId = tid;
        std::memset(&m_Context, 0, sizeof(m_Context));

        auto result = svc::GetDebugThreadParam(&param1, &param2, handle, m_ThreadId, svc::DebugThreadParam_State);
        if (result.IsFailure())
        {
            return false;
        }

        bool isValidRegs = ((param2 == svc::ThreadState::ThreadState_Runnable) || (param2 == svc::ThreadState::ThreadState_Wait));
        if (!isValidRegs)
        {
            return false;
        }

        result = svc::GetDebugThreadContext(&m_Context, handle, m_ThreadId,
            svc::ContextFlag_General | svc::ContextFlag_Control | svc::ContextFlag_Fpu | svc::ContextFlag_FpuControl);
        if (result.IsFailure())
        {
            return false;
        }

        for (size_t i = 0; i < 29; i++)
        {
            NN_SDK_LOG("    X[%.2d]   = %016lx\n", i, m_Context.r[i]);
        }
        NN_SDK_LOG("    X[%.2d]   = %016lx\n", 29, m_Context.fp);
        NN_SDK_LOG("    X[%.2d]   = %016lx\n", 30, m_Context.lr);
        NN_SDK_LOG("    SP      = %016lx\n", m_Context.sp);
        NN_SDK_LOG("    PC      = %016lx\n", m_Context.pc);
        NN_SDK_LOG("    PSTATE  = %08x\n", m_Context.pstate);

        // 32bit の場合、フレームポインタが残っていないのでバックトレースを諦める
        if (is64bit == false)
        {
            // レジスタは取れているので true を返す
            return true;
        }

        // スタック領域を残す
        GetStackRegion(handle);

        // バックトレースを保存
        NN_SDK_LOG("Backtrace\n");
        NN_SDK_LOG("    %016lx\n", m_Context.pc);
        NN_SDK_LOG("    %016lx\n", m_Context.lr);
        uintptr_t frameAddress = static_cast<uintptr_t>(m_Context.fp);
        for (int i = 0; i < MaxStackTraceCount; i++)
        {
            if (!frameAddress)
            {
                break;
            }
            if (frameAddress & 0xF)
            {
                break;
            }
            struct
            {
                uint64_t frame;
                uint64_t lr;
            } frame;

            result = svc::ReadDebugProcessMemory(reinterpret_cast<uintptr_t>(&frame), handle, frameAddress, sizeof(frame));
            if (result.IsFailure())
            {
                break;
            }

            NN_SDK_LOG("    %016lx\n", frame.lr);
            m_StackTrace[m_StackTraceCount++] = frame.lr;

            frameAddress = frame.frame;
        }
        return true;
    }

    void ThreadInfo::CalculateHash(const ModuleManager& module) NN_NOEXCEPT
    {
        Bit64 trace[MaxStackTraceCount + 2];

        std::memset(trace, 0, sizeof(trace));

        int count = 0;
        trace[count++] = module.GetOffset(m_Context.pc);
        trace[count++] = module.GetOffset(m_Context.lr);
        for (int i = 0; i < m_StackTraceCount && count < NN_ARRAY_SIZE(trace); ++i)
        {
            trace[count++] = module.GetOffset(m_StackTrace[i]);
        }

        nn::crypto::GenerateSha256Hash(m_Hash, sizeof(m_Hash), trace, sizeof(trace));
    }

    void ThreadInfo::GetStackRegion(svc::Handle handle) NN_NOEXCEPT
    {
        svc::MemoryInfo block;
        svc::PageInfo   page;

        //
        // SP の指す領域がスタックであれば、その領域を記録する
        // スタックでなければその次の領域を調べる。その次の次は調べない。
        //
        auto result = svc::QueryDebugProcessMemory(&block, &page, handle, m_Context.sp);
        if (result.IsFailure())
        {
            return;
        }

        if (block.state == svc::MemoryState_Stack)
        {
            m_StackTop = block.baseAddress;
            m_StackBottom = block.baseAddress + block.size;
        }
        else
        {
            // Stack 以外なら直下の領域をスタック領域とみなす（=スタックオーバーフローしている）
            auto result = svc::QueryDebugProcessMemory(&block, &page, handle, block.baseAddress + block.size);
            if (result.IsFailure() || block.state != svc::MemoryState_Stack)
            {
                // SVC に失敗するか、領域がスタックでなければ諦める
                return;
            }

            m_StackTop = block.baseAddress;
            m_StackBottom = block.baseAddress + block.size;
        }
    }

    void ThreadInfo::AppendToBuffer(InfoBuffer* pOutBuffer) NN_NOEXCEPT
    {
        pOutBuffer->Append(&m_ThreadId, sizeof(m_ThreadId));
        pOutBuffer->Append(&m_Context.r, sizeof(m_Context.r));
        pOutBuffer->Append(&m_Context.fp, sizeof(m_Context.fp));
        pOutBuffer->Append(&m_Context.lr, sizeof(m_Context.lr));
        pOutBuffer->Append(&m_Context.sp, sizeof(m_Context.sp));
        pOutBuffer->Append(&m_Context.pc, sizeof(m_Context.pc));
        pOutBuffer->Append(&m_Context.pstate, sizeof(m_Context.pstate));
        pOutBuffer->Append(&m_StackTop, sizeof(m_StackTop));
        pOutBuffer->Append(&m_StackBottom, sizeof(m_StackBottom));
        pOutBuffer->Append(&m_StackTraceCount, sizeof(m_StackTraceCount));
        for (Bit8 i = 0; i < m_StackTraceCount; ++i)
        {
            pOutBuffer->Append(&m_StackTrace[i], sizeof(m_StackTrace[i]));
        }
    }



    void ThreadManager::CollectThreadInformation(svc::Handle handle, bool is64bit) NN_NOEXCEPT
    {
        int32_t threadCount;
        Bit64   threadId[MaxThreadCount];

        auto result = svc::GetThreadList(&threadCount, threadId, MaxThreadCount, handle);
        if (result.IsFailure())
        {
            m_ThreadCount = 0;
            return;
        }

        if (threadCount > MaxThreadCount)
        {
            threadCount = MaxThreadCount;
        }

        m_ThreadCount = 0;
        for (int index = 0; index < threadCount; index++)
        {
            bool isValid = m_Threads[m_ThreadCount].GetThreadInfo(handle, threadId[index], is64bit);
            if (isValid)
            {
                m_ThreadCount++;
            }
        }
    }

    void ThreadManager::AppendToBuffer(InfoBuffer* pOutBuffer) NN_NOEXCEPT
    {
        pOutBuffer->Append(&m_ThreadCount, sizeof(m_ThreadCount));

        NN_SDK_LOG("[creport] m_ThreadCount = %d\n", m_ThreadCount);

        for (int index = 0; index < m_ThreadCount; index++)
        {
            m_Threads[index].AppendToBuffer(pOutBuffer);
        }
    }

}} // nn::creport
