﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/svc/svc_Base.h>
#include <nn/svc/svc_Dmnt.h>

#include <nn/erpt.h>
#include <nn/crypto/crypto_Aes128CtrEncryptor.h>

#include "creport_Common.h"
#include "creport_Constant.h"
#include "creport_ThreadInfo.h"
#include "creport_ModuleInfo.h"

namespace nn { namespace creport {

    class CrashReport
    {
    public:
        CrashReport() NN_NOEXCEPT : m_DebugHandle(svc::INVALID_HANDLE_VALUE), m_ProgramId(0), m_Is64Bit(true), m_IsApplication(false), m_Result(ResultInvalidResult()),
                                    m_ExceptionCode(0), m_ExceptionAddress(0), m_ExceptionExtra(0), m_DyingMessageSize(0)
        {
            std::memset(m_AesKey, 0, sizeof(m_AesKey));
            std::memset(m_DyingMessage, 0, sizeof(m_DyingMessage));
        }

        bool Attach(Bit64 pid, bool isDetailed) NN_NOEXCEPT
        {
            m_IsDetailed = isDetailed;
            auto result = svc::DebugActiveProcess(&m_DebugHandle, pid);
            return result.IsSuccess() ? true : false;
        }

        void Detach() NN_NOEXCEPT
        {
            if (m_DebugHandle != svc::INVALID_HANDLE_VALUE)
            {
                svc::CloseHandle(m_DebugHandle);
            }
        }

        void CollectDebugEvent() NN_NOEXCEPT;

        void CollectDyingMessage() NN_NOEXCEPT;

        Result CreateErrorReport() NN_NOEXCEPT;

        Result GetResult() const NN_NOEXCEPT
        {
            return m_Result;
        }

        bool IsUserBreak() const NN_NOEXCEPT
        {
            return (m_ExceptionCode == nn::svc::DebugException_UserBreak);
        }

        bool IsApplication() const NN_NOEXCEPT
        {
            return m_IsApplication;
        }

    private:
        void RecordException(Result result, Bit64 extra, const svc::DebugEventInfo& dei) NN_NOEXCEPT;
        void HandleDebugEventException(const svc::DebugEventInfo& dei) NN_NOEXCEPT;
        void HandleDebugEventCreateProcess(const svc::DebugEventInfo& dei) NN_NOEXCEPT;
        void AppendToBuffer(InfoBuffer* pOutBuffer) NN_NOEXCEPT;
        void GenerateAesKey() NN_NOEXCEPT;
        void GetEncryptedKey(Bit8* pOutKey, size_t keySize) NN_NOEXCEPT;
        void EraseAesKey() NN_NOEXCEPT;
        void Encrypt(Bit8* source, size_t sourceSize) NN_NOEXCEPT;
        Result AddEncryptedDebugInformation(erpt::Context* context) NN_NOEXCEPT;

        Result GetUserBreakResult(uintptr_t address, size_t size) NN_NOEXCEPT;

        template <typename T, size_t N>
        bool FindArray(const T(&list)[N], const T& target) NN_NOEXCEPT;

    private:
        svc::Handle m_DebugHandle;
        Bit64       m_ProgramId;
        bool        m_Is64Bit;
        bool        m_IsApplication;
        bool        m_IsDetailed;

        Result      m_Result;
        Bit8        m_ExceptionCode;
        Bit64       m_ExceptionAddress;
        Bit64       m_ExceptionExtra;
        Bit64       m_ExceptionThreadId;
        ThreadInfo  m_ExceptionThread;

        ModuleManager   m_Modules;
        ThreadManager   m_Threads;

        Bit8        m_DyingMessage[DyingMessageMaxSize];
        Bit64       m_DyingMessageAddress;
        size_t      m_DyingMessageSize;

        InfoBuffer  m_Buffer;
        Bit8        m_EncryptedAesKey[256];

        static const size_t AesKeySize = nn::crypto::Aes128CtrEncryptor::BlockSize;
        static const size_t AesIvSize = nn::crypto::Aes128CtrEncryptor::IvSize;
        Bit8 m_AesKey[AesKeySize];
        Bit8 m_AesIv[AesIvSize];
    };

}}
