﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/applet/applet_FundamentalTypes.h>
#include <nn/util/util_BitFlagSet.h>

#include <nn/os/os_Mutex.h>
#include <nn/os/os_SdkMutex.h>

namespace nn { namespace btm {

//!< 管理するアプレットリソースユーザの最大数
const size_t AppletResourceUserEntryCountMax = 32;

//!< アプレットリソースユーザを管理するクラス
class AppletManager final
{
    NN_DISALLOW_COPY(AppletManager);
    NN_DISALLOW_MOVE(AppletManager);

public:
    //!< コンストラクタ
    AppletManager();

    //!< デストラクタ
    ~AppletManager();

    //!< アプレットリソースユーザを表す構造体です
    struct AppletResourceUser final
    {
        nn::applet::AppletResourceUserId    aruid;          //!< ARUID
        uint32_t                            appletId;       //!< Applet ID
    };

    //!< アプレットリソースユーザのエントリを表す構造体です
    struct AppletResourceUserEntry final
    {
        AppletResourceUser  id;             //!< アプレットリソースユーザ
        bool                isRegistered;   //!< 登録されているか否か
    };

    //!< アプレットリソースユーザを登録します
    bool RegisterAppletResourceUser(const nn::applet::AppletResourceUserId& aruid, uint32_t appletId) NN_NOEXCEPT;

    //!< アプレットリソースユーザの登録を削除します
    bool UnregisterAppletResourceUser(const nn::applet::AppletResourceUserId& aruid) NN_NOEXCEPT;

    //!< 現在インフォーカス状態にあるアプレットリソースユーザを設定します
    bool SetInFocusedAppletResourceUser(const nn::applet::AppletResourceUserId& aruid) NN_NOEXCEPT;

    //!< 現在インフォーカス状態にあるアプレットリソースユーザを取得します
    AppletResourceUser GetInFocusedAppletResourceUser() NN_NOEXCEPT;

private:
    //!< 指定した ARUID が登録されているかをチェックし、エントリのインデックスを返します
    int IsAppletResourceUserIdRegistered(const nn::applet::AppletResourceUserId& aruid) NN_NOEXCEPT;

    AppletResourceUser  m_InFocusedAppletResourceUser;     //!< 現在インフォーカス状態にあるアプレットリソースユーザ

    AppletResourceUserEntry  m_AppletResourceUserEntry[AppletResourceUserEntryCountMax];      //!< アプレットリソースユーザのエントリ

    nn::os::SdkRecursiveMutexType m_AppletResourceUserEntryMutex;   //!< アプレットリソースユーザのエントリ操作を保護するMutex
};

}}  // namespace nn::btm
