﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/gpio/driver/gpio_PadAccessor.h>
#include <nn/os.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_TimeSpan.h>
#include <nn/result/result_HandlingUtility.h>
#include <nne/../../Sources/Libraries/bq2419x/detail/bq2419x_Util.h>
#include <nne/../../Sources/Libraries/bq2419x/detail/bq2419x_Spec.h>
#include <nne/../../Sources/Libraries/bq2419x/detail/bq2419x_TargetSpec.h>

#include "boot_ChargerDriver.h"
#include "boot_I2cHelper.h"

namespace nn { namespace boot {

nn::Result ChargerDriver::Initialize() NN_NOEXCEPT
{
    return Initialize(true);
}

nn::Result ChargerDriver::Initialize(bool initializeInputCurrentLimit) NN_NOEXCEPT
{
    /* Set charge voltage limit */
    NN_RESULT_DO(SetChargeVoltageLimit(4208));

    /* Set FastCharge Limit */
    NN_RESULT_DO(SetFastChargeCurrentLimit(512));

    /* Disable Force 20 Percent Charge Current */
    NN_RESULT_DO(Force20PercentChargeCurrent(false));

    /* Set PreCharge Current Limit */
    NN_RESULT_DO(SetPreChargeCurrentLimit(128));

    /* Set Termination Current Limit */
    NN_RESULT_DO(SetTerminationCurrentLimit(128));

    /* Set Minimum System Voltage Limit */
    NN_RESULT_DO(SetMinimumSystemVoltageLimit(3000));

    /* Disable Watchdog Timer (set timer to 0 secs) */
    NN_RESULT_DO(SetI2CWatchdogTimerSetting(0));

    /* Disable Safety Timer */
    NN_RESULT_DO(SwitchChargingSafetyTimer(false));

    /* Reset I2C Watchdog Timer */
    NN_RESULT_DO(ResetI2CWatchdogTimer());

    // 以降
    // Programs\Eris\Sources\Libraries\psm\driver\detail\psm_ChargerDriver-hardware.nx.cpp
    // での初期設定項目
    if (initializeInputCurrentLimit)
    {
        NN_RESULT_DO(SetInputCurrentLimit(500));
    }
    NN_RESULT_DO(SetBoostModeCurrentLimit(500));
    NN_RESULT_DO(SwitchHiZ(false));

    NN_RESULT_SUCCESS;
}

nn::Result ChargerDriver::SetChargeEnabled(bool isEnabled) NN_NOEXCEPT
{
    if (isEnabled)
    {
        nn::gpio::driver::SetValue(&m_ChargeEnableSession, nn::gpio::GpioValue_Low);
    }
    else
    {
        nn::gpio::driver::SetValue(&m_ChargeEnableSession, nn::gpio::GpioValue_High);
    }
    // 電源断で充電できない状態にならないように GPIO に関係なく I2C レジスタは充電許可にしておく
    NN_RESULT_DO(ConfigureCharger(nne::bq2419x::ChargerType_ChargeBattery));

    NN_RESULT_SUCCESS;
}

nn::Result ChargerDriver::SwitchHiZ(bool enable) NN_NOEXCEPT
{
    uint8_t data;

    if (enable)
        data = nne::bq2419x::detail::RegisterMap::InputSourceControlDetail::EnHiZ_Enable;
    else
        data = nne::bq2419x::detail::RegisterMap::InputSourceControlDetail::EnHiZ_Disable;

    NN_RESULT_DO(Update(nne::bq2419x::detail::RegisterMap::InputSourceControl,
        nne::bq2419x::detail::RegisterMap::InputSourceControlDetail::Mask_EnHiZ, data));

    NN_RESULT_SUCCESS;
}

nn::Result ChargerDriver::SetInputCurrentLimit(uint32_t current) NN_NOEXCEPT
{
    NN_RESULT_DO(Update(nne::bq2419x::detail::RegisterMap::InputSourceControl,
        nne::bq2419x::detail::RegisterMap::InputSourceControlDetail::Mask_InputCurrentLimit,
        nne::bq2419x::detail::RegisterMap::InputSourceControlDetail::InputCurrentLimit_mA2Reg(current)));

    NN_RESULT_SUCCESS;
}

nn::Result ChargerDriver::GetInputCurrentLimit(uint32_t *current) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(current);

    uint8_t data;

    NN_RESULT_DO(Read(nne::bq2419x::detail::RegisterMap::InputSourceControl, &data));

    data &= nne::bq2419x::detail::RegisterMap::InputSourceControlDetail::Mask_InputCurrentLimit;

    *current = nne::bq2419x::detail::RegisterMap::InputSourceControlDetail::InputCurrentLimit_Reg2mA(data);

    NN_RESULT_SUCCESS;
}

nn::Result ChargerDriver::ResetI2CWatchdogTimer() NN_NOEXCEPT
{
    NN_RESULT_DO(Update(nne::bq2419x::detail::RegisterMap::PowerOnConfiguration,
        nne::bq2419x::detail::RegisterMap::PowerOnConfigurationDetail::Mask_I2CWatchdogTimerReset,
        nne::bq2419x::detail::RegisterMap::PowerOnConfigurationDetail::I2CWatchdogTimerReset_Reset));

    NN_RESULT_SUCCESS;
}

nn::Result ChargerDriver::ConfigureCharger(nne::bq2419x::ChargerType chargerType) NN_NOEXCEPT
{
    uint8_t data;

    if (chargerType == nne::bq2419x::ChargerType_ChargeBattery)
        data = nne::bq2419x::detail::RegisterMap::PowerOnConfigurationDetail::ChargerConfiguration_ChargeBattery;
    else if (chargerType == nne::bq2419x::ChargerType_OTG)
        data = nne::bq2419x::detail::RegisterMap::PowerOnConfigurationDetail::ChargerConfiguration_OTG;
    else
        data = nne::bq2419x::detail::RegisterMap::PowerOnConfigurationDetail::ChargerConfiguration_ChargeDisable;

    NN_RESULT_DO(Update(nne::bq2419x::detail::RegisterMap::PowerOnConfiguration,
        nne::bq2419x::detail::RegisterMap::PowerOnConfigurationDetail::Mask_ChargerConfiguration, data));

    NN_RESULT_SUCCESS;
}

nn::Result ChargerDriver::SetMinimumSystemVoltageLimit(uint32_t voltage) NN_NOEXCEPT
{
    NN_RESULT_DO(Update(nne::bq2419x::detail::RegisterMap::PowerOnConfiguration,
        nne::bq2419x::detail::RegisterMap::PowerOnConfigurationDetail::Mask_MinimumSystemVoltageLimit,
        nne::bq2419x::detail::RegisterMap::PowerOnConfigurationDetail::MinimumSystemVoltageLimit_mV2Reg(voltage)));

    NN_RESULT_SUCCESS;
}

nn::Result ChargerDriver::GetMinimumSystemVoltageLimit(uint32_t *voltage) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(voltage);

    uint8_t data;

    NN_RESULT_DO(Read(nne::bq2419x::detail::RegisterMap::PowerOnConfiguration, &data));

    data &= nne::bq2419x::detail::RegisterMap::PowerOnConfigurationDetail::Mask_MinimumSystemVoltageLimit;

    *voltage = nne::bq2419x::detail::RegisterMap::PowerOnConfigurationDetail::MinimumSystemVoltageLimit_Reg2mV(data);

    NN_RESULT_SUCCESS;
}

nn::Result ChargerDriver::SetBoostModeCurrentLimit(uint32_t current) NN_NOEXCEPT
{
    NN_RESULT_DO(Update(nne::bq2419x::detail::RegisterMap::PowerOnConfiguration,
        nne::bq2419x::detail::RegisterMap::PowerOnConfigurationDetail::Mask_BoostModeCurrentLimit,
        nne::bq2419x::detail::RegisterMap::PowerOnConfigurationDetail::BoostModeCurrentLimit_mA2Reg(current)));

    NN_RESULT_SUCCESS;
}

nn::Result ChargerDriver::GetBoostModeCurrentLimit(uint32_t *current) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(current);

    uint8_t data;

    NN_RESULT_DO(Read(nne::bq2419x::detail::RegisterMap::PowerOnConfiguration, &data));

    data &= nne::bq2419x::detail::RegisterMap::PowerOnConfigurationDetail::Mask_BoostModeCurrentLimit;

    *current = nne::bq2419x::detail::RegisterMap::PowerOnConfigurationDetail::BoostModeCurrentLimit_Reg2mA(data);

    NN_RESULT_SUCCESS;
}

nn::Result ChargerDriver::SetFastChargeCurrentLimit(uint32_t current) NN_NOEXCEPT
{
    NN_RESULT_DO(Update(nne::bq2419x::detail::RegisterMap::ChargeCurrentControl,
        nne::bq2419x::detail::RegisterMap::ChargeCurrentControlDetail::Mask_FastChargeCurrentLimit,
        nne::bq2419x::detail::RegisterMap::ChargeCurrentControlDetail::FastChargeCurrentLimit_mA2Reg(current)));

    NN_RESULT_SUCCESS;
}

nn::Result ChargerDriver::GetFastChargeCurrentLimit(uint32_t *current) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(current);

    uint8_t data;

    NN_RESULT_DO(Read(nne::bq2419x::detail::RegisterMap::ChargeCurrentControl, &data));

    data &= nne::bq2419x::detail::RegisterMap::ChargeCurrentControlDetail::Mask_FastChargeCurrentLimit;

    *current = nne::bq2419x::detail::RegisterMap::ChargeCurrentControlDetail::FastChargeCurrentLimit_Reg2mA(data);

    NN_RESULT_SUCCESS;
}

nn::Result ChargerDriver::Force20PercentChargeCurrent(bool enable) NN_NOEXCEPT
{
    uint8_t data;

    if (enable)
        data = nne::bq2419x::detail::RegisterMap::ChargeCurrentControlDetail::Force20Percent_ChargeCurrent20Percent;
    else
        data = nne::bq2419x::detail::RegisterMap::ChargeCurrentControlDetail::Force20Percent_ChargeCurrent100Percent;

    NN_RESULT_DO(Update(nne::bq2419x::detail::RegisterMap::ChargeCurrentControl,
        nne::bq2419x::detail::RegisterMap::ChargeCurrentControlDetail::Mask_Force20Percent, data));

    NN_RESULT_SUCCESS;
}

nn::Result ChargerDriver::SetPreChargeCurrentLimit(uint32_t current) NN_NOEXCEPT
{
    NN_RESULT_DO(Update(nne::bq2419x::detail::RegisterMap::PreChargeTerminationCurrentControl,
        nne::bq2419x::detail::RegisterMap::PreChargeTerminationCurrentControlDetail::Mask_PreChargeCurrentLimit,
        nne::bq2419x::detail::RegisterMap::PreChargeTerminationCurrentControlDetail::PreChargeCurrentLimit_mA2Reg(current)));

    NN_RESULT_SUCCESS;
}

nn::Result ChargerDriver::GetPreChargeCurrentLimit(uint32_t *current) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(current);

    uint8_t data;

    NN_RESULT_DO(Read(nne::bq2419x::detail::RegisterMap::PreChargeTerminationCurrentControl, &data));

    data &= nne::bq2419x::detail::RegisterMap::PreChargeTerminationCurrentControlDetail::Mask_PreChargeCurrentLimit;

    *current = nne::bq2419x::detail::RegisterMap::PreChargeTerminationCurrentControlDetail::PreChargeCurrentLimit_Reg2mA(data);

    NN_RESULT_SUCCESS;
}

nn::Result ChargerDriver::SetTerminationCurrentLimit(uint32_t current) NN_NOEXCEPT
{
    NN_RESULT_DO(Update(nne::bq2419x::detail::RegisterMap::PreChargeTerminationCurrentControl,
        nne::bq2419x::detail::RegisterMap::PreChargeTerminationCurrentControlDetail::Mask_TerminationCurrentLimit,
        nne::bq2419x::detail::RegisterMap::PreChargeTerminationCurrentControlDetail::TerminationCurrentLimit_mA2Reg(current)));

    NN_RESULT_SUCCESS;
}

nn::Result ChargerDriver::GetTerminationCurrentLimit(uint32_t *current) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(current);

    uint8_t data;

    NN_RESULT_DO(Read(nne::bq2419x::detail::RegisterMap::PreChargeTerminationCurrentControl, &data));

    data &= nne::bq2419x::detail::RegisterMap::PreChargeTerminationCurrentControlDetail::Mask_TerminationCurrentLimit;

    *current = nne::bq2419x::detail::RegisterMap::PreChargeTerminationCurrentControlDetail::TerminationCurrentLimit_Reg2mA(data);

    NN_RESULT_SUCCESS;
}

nn::Result ChargerDriver::SetChargeVoltageLimit(uint32_t voltage) NN_NOEXCEPT
{
    NN_RESULT_DO(Update(nne::bq2419x::detail::RegisterMap::ChargeVoltageControl,
        nne::bq2419x::detail::RegisterMap::ChargeVoltageControlDetail::Mask_ChargeVoltageLimit,
        nne::bq2419x::detail::RegisterMap::ChargeVoltageControlDetail::ChargeVoltageLimit_mV2Reg(voltage)));

    NN_RESULT_SUCCESS;
}

nn::Result ChargerDriver::GetChargeVoltageLimit(uint32_t *voltage) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(voltage);

    uint8_t data;

    NN_RESULT_DO(Read(nne::bq2419x::detail::RegisterMap::ChargeVoltageControl, &data));

    data &= nne::bq2419x::detail::RegisterMap::ChargeVoltageControlDetail::Mask_ChargeVoltageLimit;

    *voltage = nne::bq2419x::detail::RegisterMap::ChargeVoltageControlDetail::ChargeVoltageLimit_Reg2mV(data);

    NN_RESULT_SUCCESS;
}

nn::Result ChargerDriver::SetI2CWatchdogTimerSetting(uint32_t seconds) NN_NOEXCEPT
{
    NN_RESULT_DO(Update(nne::bq2419x::detail::RegisterMap::ChargeTerminationTimerControl,
        nne::bq2419x::detail::RegisterMap::ChargeTerminationTimerControlDetail::Mask_I2CWatchdogTimerSetting,
        nne::bq2419x::detail::RegisterMap::ChargeTerminationTimerControlDetail::I2CWatchdogTimerSetting_Sec2Reg(seconds)));

    NN_RESULT_SUCCESS;
}

nn::Result ChargerDriver::GetI2CWatchdogTimerSetting(uint32_t *seconds) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(seconds);

    uint8_t data;

    NN_RESULT_DO(Read(nne::bq2419x::detail::RegisterMap::ChargeTerminationTimerControl, &data));

    data &= nne::bq2419x::detail::RegisterMap::ChargeTerminationTimerControlDetail::Mask_I2CWatchdogTimerSetting;

    *seconds = nne::bq2419x::detail::RegisterMap::ChargeTerminationTimerControlDetail::I2CWatchdogTimerSetting_Reg2Sec(data);

    NN_RESULT_SUCCESS;
}

nn::Result ChargerDriver::SwitchChargingSafetyTimer(bool enable) NN_NOEXCEPT
{
    uint8_t data;

    if (enable)
        data = nne::bq2419x::detail::RegisterMap::ChargeTerminationTimerControlDetail::ChargingSafetyTimerEnable_Enable;
    else
        data = nne::bq2419x::detail::RegisterMap::ChargeTerminationTimerControlDetail::ChargingSafetyTimerEnable_Disable;

    NN_RESULT_DO(Update(nne::bq2419x::detail::RegisterMap::ChargeTerminationTimerControl,
        nne::bq2419x::detail::RegisterMap::ChargeTerminationTimerControlDetail::Mask_ChargingSafetyTimerEnable, data));

    NN_RESULT_SUCCESS;
}

nn::Result ChargerDriver::GetChargeStatus(nne::bq2419x::ChargeStatusType *chargeStatus)
{
    NN_ABORT_UNLESS_NOT_NULL(chargeStatus);

    uint8_t data;

    NN_RESULT_DO(Read(nne::bq2419x::detail::RegisterMap::SystemStatus, &data));

    data &= nne::bq2419x::detail::RegisterMap::SystemStatusDetail::Mask_ChargeStatus;

    if (data == nne::bq2419x::detail::RegisterMap::SystemStatusDetail::ChargeStatus_ChargeTerminationDone)
        *chargeStatus = nne::bq2419x::ChargeStatusType_ChargeTerminationDone;
    else if (data == nne::bq2419x::detail::RegisterMap::SystemStatusDetail::ChargeStatus_FastCharging)
        *chargeStatus = nne::bq2419x::ChargeStatusType_FastCharging;
    else if (data == nne::bq2419x::detail::RegisterMap::SystemStatusDetail::ChargeStatus_PreCharge)
        *chargeStatus = nne::bq2419x::ChargeStatusType_PreCharge;
    else
        *chargeStatus = nne::bq2419x::ChargeStatusType_NotCharging;

    NN_RESULT_SUCCESS;
}

nn::Result ChargerDriver::Read(const uint8_t addr, uint8_t *data) NN_NOEXCEPT
{
    return ReadI2cRegister(data, m_I2cSession, &addr, sizeof(addr), sizeof(*data));
}

nn::Result ChargerDriver::Write(const uint8_t addr, uint8_t data) NN_NOEXCEPT
{
    return WriteI2cRegister(m_I2cSession, &addr, sizeof(addr), &data, sizeof(data));
}

nn::Result ChargerDriver::Update(const uint8_t addr, uint8_t mask, uint16_t data) NN_NOEXCEPT
{
    uint8_t temp;

    NN_RESULT_DO(Read(addr, &temp));

    temp &= ~mask;
    temp |= (mask & data);

    NN_RESULT_DO(Write(addr, temp));

    NN_RESULT_SUCCESS;
}

}} // namespace nn::boot
