﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/spl/spl_Types.h>

#include "boot_BootReason.h"
#include "boot_PmicDriver.h"
#include "boot_RtcDriver.h"
#include "boot_SplAccessor.h"

namespace nn { namespace boot {

namespace {

nn::spl::BootReason g_BootReason;
bool g_IsBootReasonDetected;

nn::spl::BootReason DetectBootReason(uint8_t onOffIrq, uint8_t rtcInt, uint8_t nverc, bool isAcOk)
{
    const uint8_t onOffIrqEn0Mask  = 0x08;
    const uint8_t onOffIrqAcOkMask = 0x80;
    const uint8_t rtcIntRtcA1Mask  = 0x02;
    const uint8_t nvercMbuMask     = 0x40;
#if !defined(NN_BOOT_BUILD_TYPE_SAFE_MODE)
    const uint8_t rtcIntRtcA2Mask  = 0x04;
    const uint8_t nvercRstInMask   = 0x80;
#endif

    if ((onOffIrq & onOffIrqEn0Mask) != 0)
    {
        return nn::spl::BootReason_OnKey;
    }
    if ((rtcInt & rtcIntRtcA1Mask) != 0)
    {
        return nn::spl::BootReason_RtcAlarm1;
    }
    if ((onOffIrq & onOffIrqAcOkMask) != 0)
    {
        return nn::spl::BootReason_AcOk;
    }
#if !defined(NN_BOOT_BUILD_TYPE_SAFE_MODE)
    if ((rtcInt & rtcIntRtcA2Mask) != 0)
    {
        if (nverc != nvercRstInMask)
        {
            if (!IsRecoveryBoot())
            {
                return nn::spl::BootReason_RtcAlarm2;
            }
        }
    }
#endif
    if ((nverc & nvercMbuMask) != 0)
    {
        if (isAcOk)
        {
            return nn::spl::BootReason_AcOk;
        }
    }

    return nn::spl::BootReason_Unknown;
}

}

void DetectBootReason() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!g_IsBootReasonDetected);

    PmicDriver pmicDriver;
    uint8_t onOffIrq;
    uint8_t nverc;
    bool isAcOk;
    NN_ABORT_UNLESS_RESULT_SUCCESS(pmicDriver.GetOnOffIrq(&onOffIrq));
    NN_ABORT_UNLESS_RESULT_SUCCESS(pmicDriver.GetNverc(&nverc));
    NN_ABORT_UNLESS_RESULT_SUCCESS(pmicDriver.GetAcOk(&isAcOk));

    RtcDriver rtcDriver;
    uint8_t rtcInt;
    uint8_t rtcIntM;
    NN_ABORT_UNLESS_RESULT_SUCCESS(rtcDriver.GetRtcInt(&rtcInt));
    NN_ABORT_UNLESS_RESULT_SUCCESS(rtcDriver.GetRtcIntM(&rtcIntM));

    g_BootReason = DetectBootReason(onOffIrq, rtcInt & ~rtcIntM, nverc, isAcOk);

    nn::spl::BootReasonValue bootReason;
    bootReason.onOffIrq = onOffIrq;
    bootReason.nverc = nverc;
    bootReason.rtcInt = rtcInt & ~rtcIntM;
    bootReason.bootReason = g_BootReason;
    SetBootReason(bootReason);

    g_IsBootReasonDetected = true;
}

nn::spl::BootReason GetBootReason() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_IsBootReasonDetected);

    return g_BootReason;
}

}} // namespace nn::boot
