﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "data_types.h"
#include "bt_types.h"
#include "bsa_api.h"

#include <nn/settings/system/settings_BluetoothDevices.h>


/*
 * Definitions
 */

enum {APP_MAX_NB_REMOTE_STORED_DEVICES = 10};

// Information mask of the HH devices
enum {
    APP_HH_DEV_OPENED = 0x01   // Indicates if this device is connected
};

/*****************************************************************************
 **  New types
 *****************************************************************************/
typedef struct
{
    BOOLEAN enable; /* Is Bluetooth enabled/disabled */
    BOOLEAN discoverable; /* Is bluetooth device discoverable (InquiryScan) */
    BOOLEAN connectable; /* Is bluetooth device Connectable (PageScan) */
    BD_ADDR bd_addr; /* Local BdAddr */
    BD_NAME name; /* Local Name */
    DEV_CLASS class_of_device;/* Class Of Device */
    PIN_CODE pin_code;
    char pin_len;
    tBSA_SEC_IO_CAP io_cap;
} tAPP_XML_CONFIG;

/* remote device */
typedef struct
{
    BOOLEAN in_use;
    BD_ADDR bd_addr;
    BD_NAME name;
    DEV_CLASS class_of_device;
    tBSA_SERVICE_MASK available_services;
    tBSA_SERVICE_MASK trusted_services;
    BOOLEAN is_default_hs;
    BOOLEAN stored;
    BOOLEAN link_key_present;
    LINK_KEY link_key;
    unsigned char key_type;
    tBSA_SEC_IO_CAP io_cap;
    UINT16 pid;
    UINT16 vid;
    UINT16 version;
    BD_FEATURES features;
    UINT8 lmp_version;

    // from HH
    UINT8 sub_class;
    UINT16 attr_mask;
    UINT16 descriptor_size;
    UINT8 descriptor[128];
    UINT16 brr_size;
    UINT8 brr[9];
    UINT16 ssr_max_latency;
    UINT16 ssr_min_tout;
    UINT16 supervision_tout;

    // from app_hh
    UINT8 handle;
    UINT8 info_mask;

    tBT_DEVICE_TYPE device_type;
} tAPP_XML_REM_DEVICE;



/*
 * Global Variables
 */
extern tAPP_XML_REM_DEVICE app_xml_remote_devices_db[APP_MAX_NB_REMOTE_STORED_DEVICES];


void app_xml_init();
int  app_xml_add_db(const nn::settings::system::BluetoothDevicesSettings* pBds);
int  app_xml_remove_db(const BD_ADDR bd_addr);


tAPP_XML_REM_DEVICE *app_xml_find_dev_db(const BD_ADDR bd_addr);
int app_xml_update_key_db(const BD_ADDR bd_addr, const LINK_KEY link_key, unsigned char key_type);
int app_xml_add_trusted_services_db(const BD_ADDR bd_addr, tBSA_SERVICE_MASK trusted_services);
int app_xml_update_name_db(const BD_ADDR bd_addr, const BD_NAME name);
int app_xml_update_cod_db(const BD_ADDR bd_addr, DEV_CLASS class_of_device);
tAPP_XML_REM_DEVICE *app_xml_find_by_handle(tBSA_HH_HANDLE handle);
int app_xml_display_devices();
int app_read_xml_config(tAPP_XML_CONFIG *p_xml_config);
int app_write_xml_config(const tAPP_XML_CONFIG *p_xml_config);

const char* toString(const uint8_t *data, int len);

/*******************************************************************************
 **
 ** Function        isSameBdAddr
 **
 ** Description     Return true if the two addresses match.
 **
 *******************************************************************************/
static inline bool isSameBdAddr(const uint8_t* addr1, const uint8_t* addr2)
{
    return memcmp(addr1, addr2, sizeof(BD_ADDR)) == 0;
}


/*******************************************************************************
 **
 ** Function        copyBdAddr
 **
 ** Description     copy a BD_ADDR
 **
 *******************************************************************************/
static inline void copyBdAddr(uint8_t* addr1, const uint8_t* addr2)
{
    memcpy(addr1, addr2, sizeof(BD_ADDR));
}



