﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 * @file
 * @brief
 */

#include <nn/nn_Common.h>
#include <nn/os.h>

//--------------------------------------------------------------------------
//  C/C++
//--------------------------------------------------------------------------

typedef struct {
    uint8_t cbType;
    uint16_t dataLen;
    nn::os::Tick createTime;
    uint8_t buf[];
} HAL_PACKET_CONTEXT;


class CircBufferC
{
    NN_DISALLOW_COPY(CircBufferC);
    NN_DISALLOW_MOVE(CircBufferC);

private:
    uint8_t*    m_bufferPtr;  // pointer to start of buffer
    uint8_t*    m_writePtr;   // pointer to next write location
    size_t      m_bufferSize; // size of buffer
    const char* m_name;       // name, for debug only
    size_t      m_minSpace;   // track maximum buffer utilization
    nn::os::MutexType m_mutex; // to protect and allow for multiple writers


    bool  _write(uint8_t type, uint16_t size, void* dataPtr);
    void* _read();

public:
    uint8_t*    m_readPtr;    // pointer to next read location

    CircBufferC() NN_NOEXCEPT : m_bufferPtr(0), m_writePtr(0), m_bufferSize(0) {}

    static nn::os::EventType m_event; // for signaling events between writer and reader

    static const uint8_t DUMMY_PACKET_TYPE = 0xFF;

    size_t qBufferSize() const NN_NOEXCEPT
    {
        return m_bufferSize;
    }

    void qInitialize(uint8_t* buffer, size_t size, const char* name) NN_NOEXCEPT;
    void qFinalize() NN_NOEXCEPT;

    // get if the queue is initialized
    bool qIsInitialized() NN_NOEXCEPT;

    // get the number of bytes writeable in the buffer.
    size_t qWriteableSize() const NN_NOEXCEPT;

    // add to the queue, with timestamp
    // return 0 if successful, non-zero on failure (e.g. full)
    bool qWrite(uint8_t cbType, uint16_t size, void* dataPtr) NN_NOEXCEPT;

    // read from the queue (get pointer to the data), without freeing.
    // this function blocks if the queue is empty
    // returns NULL on failure, non-NULL otherwise
    void* qRead() NN_NOEXCEPT;

    // free a message from the queue
    // return 0 if successful, non-zero on failure (e.g. empty)
    bool qFree() NN_NOEXCEPT;
};
