﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkLog.h>
#include <nn/os.h>
#include "bluetooth.h"
#include "bluetooth_le.h"
#include "bluetooth_le_client.h"
#include "bluetooth_le_disc.h"
#include "bluetooth_le_server.h"
#include "bluetooth_le_interface.h"
#include "bluetooth_InternalTypes.h"
#include "bluetooth_queue.h"

static nn::os::MutexType g_MutexForGattDataPathReg;

static nn::bluetooth::BluetoothLeStatus RegisterGattDataPath(const GattAttributeUuid& uuid, uint8_t type)
{
    if (type != HAL_QUEUE_BLE && type != HAL_QUEUE_BLE_CORE && type != HAL_QUEUE_BLE_HID)
    {
        NN_SDK_LOG("[bluetooth] %s: Error. Invalid data path. %d\n", __func__, type);
        return nn::bluetooth::BluetoothLeStatus::BT_BLE_ERR_INVALID_PARAM;
    }

    // Search if already registered
    for (int i = 0; i < GATT_DATA_PATH_FILTER_NUM_MAX; ++i)
    {
        if (g_GattDataPathFilter[i].inUse == true && g_GattDataPathFilter[i].uuid == uuid)
        {
            if (g_GattDataPathFilter[i].type == type)
            {
                NN_SDK_LOG("[bluetooth] %s: This service is already registered to the same path %d.\n", __func__, g_GattDataPathFilter[i].type);
                return nn::bluetooth::BluetoothLeStatus::BT_OK;
            }
            else
            {
                NN_SDK_LOG("[bluetooth] %s: Error. This service is already registered to path to %d\n", __func__, g_GattDataPathFilter[i].type);
                return nn::bluetooth::BluetoothLeStatus::BT_ERR_DB_FULL;
            }
        }
    }

    // Search free space to register
    for (int i = 0; i < GATT_DATA_PATH_FILTER_NUM_MAX; ++i)
    {
        if (g_GattDataPathFilter[i].inUse == false)
        {
            nn::os::LockMutex(&g_MutexForGattDataPathReg);
            g_GattDataPathFilter[i].uuid = uuid;
            g_GattDataPathFilter[i].type = type;
            g_GattDataPathFilter[i].inUse = true;
            nn::os::UnlockMutex(&g_MutexForGattDataPathReg);

            return nn::bluetooth::BluetoothLeStatus::BT_OK;
        }
    }

    NN_SDK_LOG("[bluetooth] %s: Error. There is no space available to register GATT data path.\n", __func__);

    return nn::bluetooth::BluetoothLeStatus::BT_ERR_DB_FULL;
}

static BluetoothLeStatus UnregisterGattDataPath(const GattAttributeUuid& uuid, uint8_t type)
{
    BTHAL_IF_BLE_DEBUG("called");

    if (type != HAL_QUEUE_BLE && type != HAL_QUEUE_BLE_CORE && type != HAL_QUEUE_BLE_HID)
    {
        NN_SDK_LOG("[bluetooth] %s: Error. Invalid data path. %d\n", __func__, type);
        return nn::bluetooth::BluetoothLeStatus::BT_BLE_ERR_INVALID_PARAM;
    }

    // Search if already registered
    for (int i = 0; i < GATT_DATA_PATH_FILTER_NUM_MAX; ++i)
    {
        if (g_GattDataPathFilter[i].inUse == true && g_GattDataPathFilter[i].uuid == uuid && g_GattDataPathFilter[i].type == type)
        {
            nn::os::LockMutex(&g_MutexForGattDataPathReg);
            g_GattDataPathFilter[i].inUse = false;
            nn::os::UnlockMutex(&g_MutexForGattDataPathReg);
            break;
        }
    }

    // Even if not found, it's ok
    return nn::bluetooth::BluetoothLeStatus::BT_OK;
}

static nn::bluetooth::BluetoothLeStatus ConvertHciStatusToLeStatus(uint8_t hciStatus)
{
    nn::bluetooth::BluetoothLeStatus result = BT_OK;

    switch (hciStatus)
    {
    case HCI_SUCCESS:                                           // 0x00, HCI_PENDING
        result = nn::bluetooth::BT_OK;
        break;
    /* Bad parameter errors */
    case HCI_ERR_ILLEGAL_COMMAND:               // 0x01
    case HCI_ERR_COMMAND_DISALLOWED:            // 0x0C
    case HCI_ERR_UNSUPPORTED_VALUE:             // 0x11
    case HCI_ERR_PARAM_OUT_OF_RANGE:            // 0x30
    case HCI_ERR_UNACCEPT_CONN_INTERVAL:        // 0x3B
        result = nn::bluetooth::BT_ERR_BAD_PARAM;
        break;
    /* Connection errors */
    case HCI_ERR_NO_CONNECTION:                 // 0x02
    case HCI_ERR_CONNECTION_TOUT:               // 0x08
    case HCI_ERR_MAX_NUM_OF_CONNECTIONS:        // 0x09
    case HCI_ERR_CONNECTION_EXISTS:             // 0x0B
    case HCI_ERR_PEER_POWER_OFF:                // 0x15
    case HCI_ERR_CONN_CAUSE_LOCAL_HOST:         // 0x16
    case HCI_ERR_CONN_TOUT_DUE_TO_MIC_FAILURE:  // 0x3D
    case HCI_ERR_CONN_FAILED_ESTABLISHMENT:     // 0x3E
    case HCI_ERR_MAC_CONNECTION_FAILED:         // 0x3F
        result = nn::bluetooth::BT_ERR_CONNECTION;
        break;
    /* Security errors */
    case HCI_ERR_AUTH_FAILURE:                  // 0x05
    case HCI_ERR_KEY_MISSING:                   // 0x06
    case HCI_ERR_HOST_REJECT_SECURITY:          // 0x0E
    case HCI_ERR_PAIRING_NOT_ALLOWED:           // 0x18
    case HCI_ERR_INSUFFCIENT_SECURITY:          // 0x2F
    case HCI_ERR_SIMPLE_PAIRING_NOT_SUPPORTED:  // 0x37
        result = nn::bluetooth::BT_ERR_SECURITY;
        break;
    /* No resource/memory full errors */
    case HCI_ERR_MEMORY_FULL:                   // 0x07
    case HCI_ERR_HOST_REJECT_RESOURCES:         // 0x0D
    case HCI_ERR_PEER_LOW_RESOURCES:            // 0x14
        result = nn::bluetooth::BT_ERR_NO_RES;
        break;
    /* Busy/not ready errors */
    case HCI_ERR_REPEATED_ATTEMPTS:             // 0x17
    case HCI_ERR_DIFF_TRANSACTION_COLLISION:    // 0x2A
    case HCI_ERR_HOST_BUSY_PAIRING:             // 0x38
    case HCI_ERR_CONTROLLER_BUSY:               // 0x3A
        result = nn::bluetooth::BT_ERR_BUSY;
        break;
    default:
        result = nn::bluetooth::BT_ERR;
        NN_SDK_LOG("[bluetooth] %s: Unknown HCI status returned: %d \n", NN_CURRENT_FUNCTION_NAME, hciStatus);
        break;
    }

    BTHAL_IF_BLE_DEBUG("Converted HCI status %d to BLE Stateus %d\n", hciStatus, result);

    return result;
}

namespace nn {
namespace bluetooth {

/*******************************************************************************
 **
 ** Function        ConvertUuid
 **
 ** Description     Helper function to convert GattAttributeUuid type to tBT_UUID type
 **                 that also checks for proper uuid length
 **
 ** Parameters      None
 **
 ** Returns         BluetoothLeStatus
 **
 *******************************************************************************/
BluetoothLeStatus ConvertUuid(GattAttributeUuid uuid, tBT_UUID *bsaUuid)
{
    if ((uuid.length != 16) && (uuid.length != 4) && (uuid.length != 2))
    {
        NN_SDK_LOG("[bluetooth] %s: Error. Invalid UUID length: %d\n", __func__, uuid.length);
        return BT_BLE_ERR_INVALID_PARAM;
    }

    memcpy(bsaUuid, &uuid, sizeof(tBT_UUID));

    return BT_OK;
}

/*******************************************************************************
 **
 ** Function        InitializeBle
 **
 ** Description     Initialize server/client callbacks
 **
 ** Parameters      None
 **
 ** Returns         BluetoothLeStatus
 **
 *******************************************************************************/
BluetoothLeStatus BluetoothLeInterfaceC::InitializeBle(BluetoothLeCallbacks* callbacks)
{
    BluetoothLeStatus status = BT_OK;
    BTHAL_IF_BLE_DEBUG("called");
    BtHalLeInitialize();
    ::BtHalCallbackRegisterLowEnergy(callbacks);

    nn::os::InitializeMutex(&g_MutexForGattDataPathReg, false, 0);

    return status;
}

/*******************************************************************************
**
** Function        InitializeBle
**
** Description     Initialize server/client callbacks
**
** Parameters      None
**
** Returns         BluetoothLeStatus
**
*******************************************************************************/
void BluetoothLeInterfaceC::CleanupBle()
{
    BTHAL_IF_BLE_DEBUG("called");
    BtHalLeInitialize();

    BluetoothLeCallbacks invalidateCallbacks;

    memset(&invalidateCallbacks, 0, sizeof(BluetoothLeCallbacks));
    invalidateCallbacks.size = sizeof(BluetoothLeCallbacks);
    ::BtHalCallbackRegisterLowEnergy(&invalidateCallbacks);

    nn::os::FinalizeMutex(&g_MutexForGattDataPathReg);

    NN_SDK_LOG("[bluetooth] CleanupBle complete\n");
}

/*******************************************************************************
 **
 ** Function        EnableBle
 **
 ** Description     Initialize bluetooth low energy services
 **
 ** Parameters      None
 **
 ** Returns         BluetoothLeStatus
 **
 *******************************************************************************/
BluetoothLeStatus BluetoothLeInterfaceC::EnableBle()
{
    BluetoothLeStatus status = BT_OK;
    BTHAL_IF_BLE_DEBUG("called");
    status = BtHalLeEnable();
    return status;
}

/*******************************************************************************
 **
 ** Function        DisableBle
 **
 ** Description     Disables bluetooth low energy
 **
 ** Parameters      None
 **
 ** Returns         BluetoothLeStatus
 **
 *******************************************************************************/
BluetoothLeStatus BluetoothLeInterfaceC::DisableBle()
{
    BluetoothLeStatus status = BT_OK;
    BTHAL_IF_DEBUG("called");
    status = BtHalLeDisable();
    return status;
}

/*******************************************************************************
 **
 ** Function        SetLeVisibility
 **
 ** Description     Set BLE discoverable/connectable
 **
 ** Parameters      discoverable: FALSE if not discoverable
 **                 connectable: FALSE if not connectable
 **
 ** Returns         BluetoothLeStatus
 **
 *******************************************************************************/
BluetoothLeStatus BluetoothLeInterfaceC::SetLeVisibility(bool discoverable, bool connectable)
{
    BluetoothLeStatus status = BT_OK;
    BTHAL_IF_BLE_DEBUG("called");
    status = BtHalLeSetVisibility(discoverable, connectable);
    return status;
}

/*******************************************************************************
 **
 ** Function        SetLeVisibility
 **
 ** Description     Set BLE connection parameters
 **
 ** Parameters      pBdAddr:
 **                 minConnInt:
 **                 maxConnInt:
 **                 slaveLatency:
 **                 supervisionTimeout:
 **
 ** Returns         BluetoothLeStatus
 **
 *******************************************************************************/
BluetoothLeStatus BluetoothLeInterfaceC::SetLeConnectionParameter(LeConnectionParams connectionParams)
{
    BluetoothLeStatus status = BT_OK;
    tBSA_DM_BLE_CONN_PARAM connParam;
    BTHAL_IF_BLE_DEBUG("called");

    connParam.min_conn_int     = connectionParams.minConnectionInterval;
    connParam.max_conn_int     = connectionParams.maxConnectionInterval;
    connParam.min_ce_len       = connectionParams.minConnectionEventLen;
    connParam.max_ce_len       = connectionParams.maxConnectionEventLen;
    connParam.slave_latency    = connectionParams.slaveLatency;
    connParam.supervision_tout = connectionParams.supervisionTimeout;
    connParam.preference       = connectionParams.preference;
    memcpy(connParam.bd_addr, connectionParams.bluetoothAddress.address, sizeof(Btbdaddr));
    status = BtHalLeSetConnectionParameter(&connParam);
    return status;
}

/*******************************************************************************
 **
 ** Function        SetLeVisibility
 **
 ** Description     Set BLE default connection parameters
 **
 ** Parameters      pBdAddr:
 **                 minConnInt:
 **                 maxConnInt:
 **                 slaveLatency:
 **                 supervisionTimeout:
 **
 ** Returns         BluetoothLeStatus
 **
 *******************************************************************************/
BluetoothLeStatus BluetoothLeInterfaceC::SetLeDefaultConnectionParameter(LeConnectionParams connectionParams)
{
    BluetoothLeStatus status = BT_OK;
    tBSA_DM_BLE_CONN_PARAM connParam;
    BTHAL_IF_BLE_DEBUG("called");

    connParam.min_conn_int     = connectionParams.minConnectionInterval;
    connParam.max_conn_int     = connectionParams.maxConnectionInterval;
    connParam.min_ce_len       = connectionParams.minConnectionEventLen;
    connParam.max_ce_len       = connectionParams.maxConnectionEventLen;
    connParam.slave_latency    = connectionParams.slaveLatency;
    connParam.supervision_tout = connectionParams.supervisionTimeout;
    connParam.preference       = connectionParams.preference;
    memcpy(connParam.bd_addr, connectionParams.bluetoothAddress.address, sizeof(Btbdaddr));
    status = BtHalLeSetDefaultConnectionParameter(&connParam);
    return status;
}

/*******************************************************************************
 **
 ** Function        SetLeAdvertiseData
 **
 ** Description     Set BLE advertisement data
 **
 ** Parameters      advertiseData:
 **
 ** Returns         BluetoothLeStatus
 **
 *******************************************************************************/
BluetoothLeStatus BluetoothLeInterfaceC::SetLeAdvertiseData(LeAdvertiseData advertiseData)
{
    BluetoothLeStatus status = BT_OK;
    tBSA_DM_BLE_ADV_CONFIG advData;
    BTHAL_IF_BLE_DEBUG("called");

    advData.len = advertiseData.len;
    advData.flag = advertiseData.flag;
    advData.adv_data_mask = advertiseData.dataMask;
    advData.appearance_data = advertiseData.appearanceData;
    advData.num_service = advertiseData.numServices;
    advData.service_data_len = advertiseData.serviceDataLen;
    advData.service_data_uuid.uu.uuid16 = advertiseData.serviceDataUuid;
    advData.is_scan_rsp = advertiseData.isScanResponse;
    advData.tx_power = advertiseData.txPower;
    memcpy(advData.p_val, advertiseData.data, sizeof(advertiseData.data));

    for (int i = 0; i < ServiceUuidMax; ++i)
    {
        advData.uuid_val[i] = advertiseData.uuid[i].uu.uuid16;
    }

    memcpy(advData.service_data_val, advertiseData.serviceData, sizeof(advertiseData.serviceData));
    status = BtHalLeSetAdvertiseData(&advData);
    return status;
}

/*******************************************************************************
 **
 ** Function        SetLeAdvertiseParameter
 **
 ** Description     Set BLE advertisement parameters
 **
 ** Parameters      pBdAddr:
 **                 minAdvInt:
 **                 maxAdvInt:
 **
 ** Returns         BluetoothLeStatus
 **
 *******************************************************************************/
BluetoothLeStatus BluetoothLeInterfaceC::SetLeAdvertiseParameter(Btbdaddr *pBdAddr, uint16_t minAdvInt, uint16_t maxAdvInt)
{
    BluetoothLeStatus status = BT_OK;
    tBSA_DM_BLE_ADV_PARAM advParam;
    BTHAL_IF_BLE_DEBUG("called");

    advParam.adv_int_min = minAdvInt;
    advParam.adv_int_max = minAdvInt;
    memcpy(advParam.dir_bda.bd_addr, pBdAddr->address, sizeof(Btbdaddr));
    status = BtHalLeSetAdvertiseParam(&advParam);
    return status;
}

/*******************************************************************************
 **
 ** Function        StartLeScan
 **
 ** Description     Start BLE Scan
 **
 ** Parameters      None
 **
 ** Returns         BluetoothLeStatus
 **
 *******************************************************************************/
BluetoothLeStatus BluetoothLeInterfaceC::StartLeScan()
{
    BluetoothLeStatus status = BT_OK;
    BTHAL_IF_BLE_DEBUG("called");
    status = BtHalLeStartScan(0);
    return status;
}

/*******************************************************************************
 **
 ** Function        StopLeScan
 **
 ** Description     Stop BLE Scan
 **
 ** Parameters      None
 **
 ** Returns         BluetoothLeStatus
 **
 *******************************************************************************/
BluetoothLeStatus BluetoothLeInterfaceC::StopLeScan()
{
    BluetoothLeStatus status = BT_OK;
    BTHAL_IF_BLE_DEBUG("called");
    status = BtHalLeStopScan();
    return status;
}

/*******************************************************************************
 **
 ** Function        SetLeScanParameter
 **
 ** Description     Set BLE Scan parameters
 **
 ** Parameters      scanInterval:   Actual time is caluculated by multiplying 0.625 msec
 **                 scanWindow  :   Actual time is calcullated by myltiplying 0.625 msec
 **
 ** Returns         BluetoothLeStatus
 **
 *******************************************************************************/
BluetoothLeStatus BluetoothLeInterfaceC::SetLeScanParameter(uint16_t scanInterval, uint16_t scanWindow)
{
    BluetoothLeStatus status = BT_OK;
    BTHAL_IF_BLE_DEBUG("called");
    status = BtHalLeSetScanParameter(scanInterval, scanWindow);
    return status;
}

/*******************************************************************************
**
** Function        AddLeScanFilterCondition
**
** Description     Add BLE Scan filter condition. All of them are ORed at the moment.
**
** Parameters      [in] Pointer to a filter data
**
** Returns         BluetoothLeStatus
**
*******************************************************************************/
BluetoothLeStatus BluetoothLeInterfaceC::AddLeScanFilterCondition(const BleAdvertiseFilter* pFilter)
{
    BluetoothLeStatus status = BT_OK;
    BTHAL_IF_BLE_DEBUG("called");

    status = BtHalLeClientAddScanFilterCondition(pFilter->filterIndex, pFilter->structure.adType, pFilter->structure.data, pFilter->structure.length - 1, pFilter->mask, pFilter->maskeLength);
    return status;

}

/*******************************************************************************
**
** Function        DeleteLeScanFilterCondition
**
** Description     Delete BLE Scan filter condition.
**
** Parameters      [in] Pointer to a filter data
**
** Returns         BluetoothLeStatus
**
*******************************************************************************/
BluetoothLeStatus BluetoothLeInterfaceC::DeleteLeScanFilterCondition(const BleAdvertiseFilter* pFilter)
{
    BluetoothLeStatus status = BT_OK;
    BTHAL_IF_BLE_DEBUG("called");

    status = BtHalLeClientDeleteScanFilterCondition(pFilter->filterIndex, pFilter->structure.adType, pFilter->structure.data, pFilter->structure.length - 1, pFilter->mask, pFilter->maskeLength);
    return status;

}

/*******************************************************************************
**
** Function        DeleteLeScanFilter
**
** Description     Delete a BLE scan filter
**
** Parameters      None
**
** Returns         BluetoothLeStatus
**
*******************************************************************************/
BluetoothLeStatus BluetoothLeInterfaceC::DeleteLeScanFilter(uint8_t filterIndex)
{
    BluetoothLeStatus status = BT_OK;
    BTHAL_IF_BLE_DEBUG("called");

    status = BtHalLeClientDeleteScanFilter(filterIndex);
    return status;

}

/*******************************************************************************
**
** Function        ClearLeScanFilter
**
** Description     Clear all the BLE scan filters
**
** Parameters      None
**
** Returns         BluetoothLeStatus
**
*******************************************************************************/
BluetoothLeStatus BluetoothLeInterfaceC::ClearLeScanFilters()
{
    BluetoothLeStatus status = BT_OK;
    BTHAL_IF_BLE_DEBUG("called");

    status = BtHalLeClientClearScanFilters();
    return status;
}

/*******************************************************************************
**
** Function        ClearLeScanFilter
**
** Description     Stop BLE Scan
**
** Parameters      None
**
** Returns         BluetoothLeStatus
**
*******************************************************************************/
BluetoothLeStatus BluetoothLeInterfaceC::EnableLeScanFilter(bool enable)
{
    BluetoothLeStatus status = BT_OK;
    BTHAL_IF_BLE_DEBUG("called");

    status = BtHalLeClientEnableScanFilter(enable);
    return status;
}


/*******************************************************************************
 **
 ** Function        RegisterLeClient
 **
 ** Description     Register BLE GATT client
 **
 ** Parameters      None
 **
 ** Returns         BluetoothLeStatus
 **
 *******************************************************************************/
BluetoothLeStatus BluetoothLeInterfaceC::RegisterLeClient(GattAttributeUuid uuid)
{
    tBT_UUID bsaUuid;
    BluetoothLeStatus status = BT_OK;
    BTHAL_IF_BLE_DEBUG("called");
    status = ConvertUuid(uuid, &bsaUuid);
    if (status != BTHH_OK)
    {
        return status;
    }
    status = BtHalLeClientRegister(bsaUuid);
    return status;
}

/*******************************************************************************
**
** Function        UnregisterLeClient
**
** Description     Unregister BLE GATT client
**
** Parameters      clientIf ... BLE Client IF
**
** Returns         BluetoothLeStatus
**
*******************************************************************************/
BluetoothLeStatus BluetoothLeInterfaceC::UnregisterLeClient(uint8_t clientIf)
{
    BluetoothLeStatus status = BT_OK;
    BTHAL_IF_BLE_DEBUG("called");
    status = BtHalLeClientDeregister(clientIf);
    return status;
}

/*******************************************************************************
**
** Function        UnregisterLeClientAll
**
** Description     Unregister all BLE GATT clients
**
** Parameters      None
**
** Returns         BluetoothLeStatus
**
*******************************************************************************/
BluetoothLeStatus BluetoothLeInterfaceC::UnregisterLeClientAll()
{
    BluetoothLeStatus status = BT_OK;
    BTHAL_IF_BLE_DEBUG("called");
    status = BtHalLeClientDeregisterAll();
    return status;
}

/*******************************************************************************
 **
 ** Function        LeClientConnect
 **
 ** Description     Connect to a BLE GATT server
 **
 ** Parameters      pBdAddr - BD_ADDR of the device we are opening a connection with
 **
 ** Returns         BluetoothLeStatus
 **
 *******************************************************************************/
BluetoothLeStatus BluetoothLeInterfaceC::LeClientConnect(uint8_t clientIf, Btbdaddr *pBdAddr, bool isDirect)
{
    BluetoothLeStatus status = BT_OK;
    BTHAL_IF_BLE_DEBUG("called");
    status = BtHalLeClientOpenConnection(clientIf, pBdAddr, isDirect);
    return status;
}

/*******************************************************************************
 **
 ** Function        LeClientCancelConnection
 **
 ** Description     Cancel connection request to a BLE GATT server
 **
 ** Parameters      pBdAddr - BD_ADDR of the device we are opening a connection with
 **
 ** Returns         BluetoothLeStatus
 **
 *******************************************************************************/
BluetoothLeStatus BluetoothLeInterfaceC::LeClientCancelConnection(uint8_t clientIf, Btbdaddr* pBdAddr, bool isDirect)
{
    BluetoothLeStatus status = BT_OK;
    BTHAL_IF_BLE_DEBUG("called");
    status = BtHalLeClientCancelOpenConnection(clientIf, pBdAddr, isDirect);
    return status;
}

/*******************************************************************************
 **
 ** Function        LeClientDisconnect
 **
 ** Description     Disconnect from a BLE GATT server
 **
 ** Parameters      pBdAddr - BD_ADDR of the device being disconnected
 **
 ** Returns         BluetoothLeStatus
 **
 *******************************************************************************/
BluetoothLeStatus BluetoothLeInterfaceC::LeClientDisconnect(uint32_t connId)
{
    BluetoothLeStatus status = BT_OK;
    BTHAL_IF_BLE_DEBUG("called");
    status = BtHalLeClientCloseConnection(connId);  // TODO: This may close the connection to the server instead of a device
    return status;
}

/*******************************************************************************
 **
 ** Function        LeClientGetAttributes
 **
 ** Description     Get the attributes of a device
 **
 ** Parameters      pBdAddr - BD_ADDR of the device being disconnected
 **
 ** Returns         BluetoothLeStatus
 **
 *******************************************************************************/
BluetoothLeStatus BluetoothLeInterfaceC::LeClientGetAttributes(uint32_t connId, Btbdaddr *pBdAddr)
{
    BluetoothLeStatus status = BT_OK;
    BTHAL_IF_BLE_DEBUG("called");
    status = BtHalLeClientLoadAttributeCache(connId, pBdAddr);
    return status;
}

/*******************************************************************************
 **
 ** Function        LeClientDiscoverService
 **
 ** Description     Discover a BLE GATT service
 **
 ** Parameters      pBdAddr - BD_ADDR of the device being disconnected
 **
 ** Returns         BluetoothLeStatus
 **
 *******************************************************************************/
BluetoothLeStatus BluetoothLeInterfaceC::LeClientDiscoverService(uint32_t connId, GattAttributeUuid filterUuid)
{
    tBT_UUID bsaUuid;
    BluetoothLeStatus status = BT_OK;
    BTHAL_IF_BLE_DEBUG("called");
    status = ConvertUuid(filterUuid, &bsaUuid);
    if (status != BT_OK)
    {
        return status;
    }
    status = BtHalLeClientDiscoverService(connId, bsaUuid);
    return status;
}

/*******************************************************************************
 **
 ** Function        LeClientReadCharacteristic
 **
 ** Description     Read a BLE GATT characteristic from a service
 **
 ** Parameters      None
 **
 ** Returns         BluetoothLeStatus
 **
 *******************************************************************************/
BluetoothLeStatus BluetoothLeInterfaceC::LeClientReadCharacteristic(uint32_t connId, GattId serviceId, bool isPrimary, GattId charId, uint8_t authType)
{
    tBT_UUID bsaServUuid;
    tBT_UUID bsaCharUuid;
    BluetoothLeStatus status = BT_OK;
    BTHAL_IF_BLE_DEBUG("called");
    status = ConvertUuid(serviceId.uuid, &bsaServUuid);
    if (status != BT_OK)
    {
        return status;
    }
    status = ConvertUuid(charId.uuid, &bsaCharUuid);
    if (status != BT_OK)
    {
        return status;
    }
    status = BtHalLeClientReadCharacteristic(connId, bsaServUuid, serviceId.instanceId, isPrimary, bsaCharUuid, charId.instanceId, authType);
    return status;
}

/*******************************************************************************
 **
 ** Function        LeClientReadDescriptor
 **
 ** Description     Read a BLE GATT descriptor from a service
 **
 ** Parameters      None
 **
 ** Returns         BluetoothLeStatus
 **
 *******************************************************************************/
BluetoothLeStatus BluetoothLeInterfaceC::LeClientReadDescriptor(uint32_t connId, GattId serviceId, bool isPrimary, GattId charId, GattId descrId, uint8_t authType)
{
    tBT_UUID bsaServUuid;
    tBT_UUID bsaCharUuid;
    tBT_UUID bsaDescUuid;
    BluetoothLeStatus status = BT_OK;
    BTHAL_IF_BLE_DEBUG("called");
    status = ConvertUuid(serviceId.uuid, &bsaServUuid);
    if (status != BT_OK)
    {
        return status;
    }
    status = ConvertUuid(charId.uuid, &bsaCharUuid);
    if (status != BT_OK)
    {
        return status;
    }
    status = ConvertUuid(descrId.uuid, &bsaDescUuid);
    if (status != BT_OK)
    {
        return status;
    }
    status = BtHalLeClientReadDescriptor(connId, bsaServUuid, isPrimary, bsaCharUuid, bsaDescUuid, authType);
    return status;
}

/*******************************************************************************
 **
 ** Function        LeClientWriteCharacteristic
 **
 ** Description     Write a value to a BLE GATT service characteristic
 **
 ** Parameters      None
 **
 ** Returns         BluetoothLeStatus
 **
 *******************************************************************************/
BluetoothLeStatus BluetoothLeInterfaceC::LeClientWriteCharacteristic(uint32_t connId, GattId serviceId, bool isPrimary, GattId charId,  const uint8_t* pData, uint16_t len, uint8_t authType, bool withResponse)
{
    tBT_UUID bsaServUuid;
    tBT_UUID bsaCharUuid;
    BluetoothLeStatus status = BT_OK;
    BTHAL_IF_BLE_DEBUG("called");
    status = ConvertUuid(serviceId.uuid, &bsaServUuid);
    if (status != BT_OK)
    {
        return status;
    }
    status = ConvertUuid(charId.uuid, &bsaCharUuid);
    if (status != BT_OK)
    {
        return status;
    }
    status = BtHalLeClientWriteCharacteristic(connId, bsaServUuid, serviceId.instanceId, isPrimary, bsaCharUuid, charId.instanceId, pData, len, authType, withResponse);
    return status;
}

/*******************************************************************************
 **
 ** Function        LeClientWriteDescriptor
 **
 ** Description     Write a value to a BLE GATT service descriptor
 **
 ** Parameters      None
 **
 ** Returns         BluetoothLeStatus
 **
 *******************************************************************************/
BluetoothLeStatus BluetoothLeInterfaceC::LeClientWriteDescriptor(uint32_t connId, GattId serviceId, bool isPrimary, GattId charId, GattId descrId, const uint8_t* pData, uint16_t len, uint8_t authType)
{
    tBT_UUID bsaServUuid;
    tBT_UUID bsaCharUuid;
    tBT_UUID bsaDescUuid;
    BluetoothLeStatus status = BT_OK;
    BTHAL_IF_BLE_DEBUG("called");
    status = ConvertUuid(serviceId.uuid, &bsaServUuid);
    if (status != BT_OK)
    {
        return status;
    }
    status = ConvertUuid(charId.uuid, &bsaCharUuid);
    if (status != BT_OK)
    {
        return status;
    }
    status = ConvertUuid(descrId.uuid, &bsaDescUuid);
    if (status != BT_OK)
    {
        return status;
    }
    status = BtHalLeClientWriteDescriptor(connId, bsaServUuid, isPrimary, bsaCharUuid, bsaDescUuid, pData, len, authType);
    return status;
}

/*******************************************************************************
 **
 ** Function        LeClientRegisterNotification
 **
 ** Description     Register to receive notifications for a given characteristic
 **
 ** Parameters      None
 **
 ** Returns         BluetoothLeStatus
 **
 *******************************************************************************/
BluetoothLeStatus BluetoothLeInterfaceC::LeClientRegisterNotification(uint32_t connId, GattId serviceId, bool isPrimary, GattId charId)
{
    tBT_UUID bsaServUuid;
    tBT_UUID bsaCharUuid;
    BluetoothLeStatus status = BT_OK;
    BTHAL_IF_BLE_DEBUG("called");
    status = ConvertUuid(serviceId.uuid, &bsaServUuid);
    if (status != BT_OK)
    {
        return status;
    }
    status = ConvertUuid(charId.uuid, &bsaCharUuid);
    if (status != BT_OK)
    {
        return status;
    }
    status = BtHalLeClientRegisterNotification(connId, bsaServUuid, serviceId.instanceId, isPrimary, bsaCharUuid, charId.instanceId);
    return status;
}

/*******************************************************************************
 **
 ** Function        LeClientDeregisterNotificationImpl
 **
 ** Description     Deregister a previous request for notifications/indications
 **
 ** Parameters      None
 **
 ** Returns         BluetoothLeStatus
 **
 *******************************************************************************/
BluetoothLeStatus BluetoothLeInterfaceC::LeClientDeregisterNotification(uint32_t connId, GattId serviceId, bool isPrimary, GattId charId)
{
    tBT_UUID bsaServUuid;
    tBT_UUID bsaCharUuid;
    BluetoothLeStatus status = BT_OK;
    BTHAL_IF_BLE_DEBUG("called");
    status = ConvertUuid(serviceId.uuid, &bsaServUuid);
    if (status != BT_OK)
    {
        return status;
    }
    status = ConvertUuid(charId.uuid, &bsaCharUuid);
    if (status != BT_OK)
    {
        return status;
    }

    status = BtHalLeClientDeregisterNotification(connId, bsaServUuid, serviceId.instanceId, isPrimary, bsaCharUuid, charId.instanceId);
    return status;
}

BluetoothLeStatus BluetoothLeInterfaceC::LeClientConfigureMtu(uint32_t connId, uint16_t mtu)
{
    BluetoothLeStatus status = BT_OK;
    BTHAL_IF_BLE_DEBUG("called");

    status = BtHalLeClientConfigureMtu(connId, mtu);

    return status;
}

/*******************************************************************************
 **
 ** Function        RegisterLeServer
 **
 ** Description     Register BLE GATT server
 **
 ** Parameters      None
 **
 ** Returns         BluetoothLeStatus
 **
 *******************************************************************************/
BluetoothLeStatus BluetoothLeInterfaceC::RegisterLeServer(GattAttributeUuid uuid)
{
    tBT_UUID bsaUuid;
    BluetoothLeStatus status = BT_OK;
    BTHAL_IF_BLE_DEBUG("called");
    status = ConvertUuid(uuid, &bsaUuid);
    NN_SDK_LOG("[bluetooth] %s: server uuid:0x%x, bsaUuid:0x%x, UUID length: %d\n", __func__, uuid.uu.uuid16, bsaUuid.uu.uuid16, uuid.length);
    if (status != BT_OK)
    {
        return status;
    }
    status = BtHalLeServerRegister(bsaUuid);
    return status;
}

/*******************************************************************************
 **
 ** Function        UnregisterLeServer
 **
 ** Description     Deregister BLE GATT server
 **
 ** Parameters      None
 **
 ** Returns         BluetoothLeStatus
 **
 *******************************************************************************/
BluetoothLeStatus BluetoothLeInterfaceC::UnregisterLeServer(uint8_t serverIf)
{
    BluetoothLeStatus status = BT_OK;
    BTHAL_IF_BLE_DEBUG("called");
    status = BtHalLeServerDeregister(serverIf);
    return status;
}


/*******************************************************************************
 **
 ** Function        LeServerConnect
 **
 ** Description     Connect to a BLE GATT client
 **
 ** Parameters      pBdAddr - BD_ADDR of the device we are opening a connection with
 **
 ** Returns         BluetoothLeStatus
 **
 *******************************************************************************/
BluetoothLeStatus BluetoothLeInterfaceC::LeServerConnect(uint8_t serverIf, Btbdaddr *pBdAddr, bool isDirect)
{
    BluetoothLeStatus status = BT_OK;
    BTHAL_IF_BLE_DEBUG("called");
    status = BtHalLeServerOpenConnection(serverIf, pBdAddr, isDirect);
    return status;
}

/*******************************************************************************
 **
 ** Function        LeServerDisconnect
 **
 ** Description     Disconnect from a BLE GATT client
 **
 ** Parameters      pBdAddr - BD_ADDR of the device being disconnected
 **
 ** Returns         BluetoothLeStatus
 **
 *******************************************************************************/
BluetoothLeStatus BluetoothLeInterfaceC::LeServerDisconnect(uint8_t serverIf, Btbdaddr *pBdAddr)
{
    BluetoothLeStatus status = BT_OK;
    BTHAL_IF_BLE_DEBUG("called");
    status = BtHalLeServerCloseConnection(serverIf);  // TODO: This may close the connection to the server instead of a device
    return status;
}
/*******************************************************************************
 **
 ** Function        CreateLeService
 **
 ** Description     Initialize a BLE GATT server service
 **
 ** Parameters      None
 **
 ** Returns         BluetoothLeStatus
 **
 *******************************************************************************/
BluetoothLeStatus BluetoothLeInterfaceC::CreateLeService(uint8_t serverIf, GattAttributeUuid uuid, uint8_t handleNum, bool isPrimaryService)
{
    tBT_UUID bsaUuid;
    BluetoothLeStatus status = BT_OK;
    BTHAL_IF_BLE_DEBUG("called");
    status = ConvertUuid(uuid, &bsaUuid);
    if (status != BTHH_OK)
    {
        return status;
    }
    status = BtHalLeServerCreateService(serverIf, bsaUuid, handleNum, isPrimaryService); // TODO: Primary service true for now
    return status;
}

/*******************************************************************************
 **
 ** Function        StartLeService
 **
 ** Description     Start a local BLE GATT server service
 **
 ** Parameters      None
 **
 ** Returns         BluetoothLeStatus
 **
 *******************************************************************************/
BluetoothLeStatus BluetoothLeInterfaceC::StartLeService(uint8_t serverIf, GattAttributeUuid uuid)
{
    tBT_UUID bsaUuid;
    BluetoothLeStatus status = BT_OK;
    BTHAL_IF_BLE_DEBUG("called");
    status = ConvertUuid(uuid, &bsaUuid);
    if (status != BT_OK)
    {
        return status;
    }
    status = BtHalLeServerStartService(serverIf, bsaUuid);
    return status;
}

/*******************************************************************************
 **
 ** Function        AddLeCharacteristic
 **
 ** Description     Add a BLE GATT characteristic to a service
 **
 ** Parameters      None
 **
 ** Returns         BluetoothLeStatus
 **
 *******************************************************************************/
BluetoothLeStatus BluetoothLeInterfaceC::AddLeCharacteristic(uint8_t serverIf, GattAttributeUuid serviceUuid, GattAttributeUuid charUuid, uint16_t permissions, uint8_t properties)
{
    tBT_UUID bsaServUuid;
    tBT_UUID bsaCharUuid;
    BluetoothLeStatus status = BT_OK;
    BTHAL_IF_BLE_DEBUG("called");
    status = ConvertUuid(serviceUuid, &bsaServUuid);
    if (status != BT_OK)
    {
        return status;
    }
    status = ConvertUuid(charUuid, &bsaCharUuid);
    if (status != BT_OK)
    {
        return status;
    }
    status = BtHalLeServerAddCharacteristic(serverIf, bsaServUuid, bsaCharUuid, permissions, properties);
    return status;
}

/*******************************************************************************
 **
 ** Function        AddLeDescriptor
 **
 ** Description     Add a BLE GATT descriptor to a service
 **
 ** Parameters      None
 **
 ** Returns         BluetoothLeStatus
 **
 *******************************************************************************/
BluetoothLeStatus BluetoothLeInterfaceC::AddLeDescriptor(uint8_t serverIf, GattAttributeUuid serviceUuid, GattAttributeUuid descUuid, uint16_t permissions)
{
    tBT_UUID bsaServUuid;
    tBT_UUID bsaDescUuid;
    BluetoothLeStatus status = BT_OK;
    BTHAL_IF_BLE_DEBUG("called");
    status = ConvertUuid(serviceUuid, &bsaServUuid);
    if (status != BT_OK)
    {
        return status;
    }
    status = ConvertUuid(descUuid, &bsaDescUuid);
    if (status != BT_OK)
    {
        return status;
    }
    status = BtHalLeServerAddDescriptor(serverIf, bsaServUuid, bsaDescUuid, permissions);
    return status;
}

/*******************************************************************************
**
** Function        SetLeResponse
**
** Description     Set attribute value for response
**
** Parameters      None
**
** Returns         BluetoothLeStatus
**
*******************************************************************************/
BluetoothLeStatus BluetoothLeInterfaceC::SetLeResponse(uint8_t serverIf, GattAttributeUuid serviceUuid, GattAttributeUuid attrUuid, const uint8_t* pData, uint16_t len)
{
    tBT_UUID bsaServUuid;
    tBT_UUID bsaAttrUuid;
    BluetoothLeStatus status = BT_OK;
    BTHAL_IF_BLE_DEBUG("called");

    status = ConvertUuid(serviceUuid, &bsaServUuid);
    if (status != BT_OK)
    {
        return status;
    }
    status = ConvertUuid(attrUuid, &bsaAttrUuid);
    if (status != BT_OK)
    {
        return status;
    }
    status = BtHalLeServerSetResponse(serverIf, bsaServUuid, bsaAttrUuid, pData, len);
    return status;
}

/*******************************************************************************
 **
 ** Function        LeSendIndication
 **
 ** Description     Send indication/notification to client
 **
 ** Parameters      None
 **
 ** Returns         BluetoothLeStatus
 **
 *******************************************************************************/
BluetoothLeStatus BluetoothLeInterfaceC::LeSendIndication(uint8_t serverIf, GattAttributeUuid serviceUuid, GattAttributeUuid charUuid, uint16_t len, const uint8_t* pData, bool confirm)
{
    tBT_UUID bsaServUuid;
    tBT_UUID bsaCharUuid;
    BluetoothLeStatus status = BT_OK;
    BTHAL_IF_BLE_DEBUG("called");
    status = ConvertUuid(serviceUuid, &bsaServUuid);
    if (status != BT_OK)
    {
        return status;
    }
    status = ConvertUuid(charUuid, &bsaCharUuid);
    if (status != BT_OK)
    {
        return status;
    }
    if (confirm)
    {
        status = BtHalLeServerSendIndication(serverIf, bsaServUuid, bsaCharUuid, pData, len);
    }
    else
    {
        status = BtHalLeServerSendNotification(serverIf, bsaServUuid, bsaCharUuid, pData, len);
    }
    return status;
}

/*******************************************************************************
 **
 ** Function        LeGetFirstCharacteristic
 **
 ** Description     Get the first characteristic from a service
 **
 ** Parameters      None
 **
 ** Returns         BluetoothLeStatus
 **
 *******************************************************************************/
BluetoothLeStatus BluetoothLeInterfaceC::LeGetFirstCharacteristic(GattId *charOut, uint8_t *property, uint32_t connId, GattId serviceId, bool isPrimary, GattAttributeUuid charFilter)
{
    tBT_UUID bsaServUuid;
    tBT_UUID bsaCharUuid;
    tBT_UUID bsaCharFilterUuid;
    BluetoothLeStatus status = BT_OK;
    BTHAL_IF_BLE_DEBUG("called");
    status = ConvertUuid(serviceId.uuid, &bsaServUuid);
    if (status != BT_OK)
    {
        return status;
    }
    status = ConvertUuid(charFilter, &bsaCharFilterUuid);
    if (status != BT_OK)
    {
        return status;
    }

    status = BtHalLeGetFirstCharacteristic(&bsaCharUuid, &(charOut->instanceId), property, connId, bsaServUuid, serviceId.instanceId, isPrimary, bsaCharFilterUuid);

    memcpy(&charOut->uuid, &bsaCharUuid, sizeof(tBT_UUID));

    return status;
}

/*******************************************************************************
 **
 ** Function        LeGetNextCharacteristic
 **
 ** Description     Get the subsequent characteristic from a service
 **
 ** Parameters      None
 **
 ** Returns         BluetoothLeStatus
 **
 *******************************************************************************/
BluetoothLeStatus BluetoothLeInterfaceC::LeGetNextCharacteristic(GattId *charOut, uint8_t *property, uint32_t connId, GattId serviceId, bool isPrimary, GattId charIn, GattAttributeUuid charFilter)
{
    tBT_UUID bsaServUuid;
    tBT_UUID bsaCharUuid;
    tBT_UUID bsaCharPreviousUuid;
    tBT_UUID bsaCharFilterUuid;
    BluetoothLeStatus status = BT_OK;
    BTHAL_IF_BLE_DEBUG("called");

    status = ConvertUuid(serviceId.uuid, &bsaServUuid);
    if (status != BT_OK)
    {
        return status;
    }
    status = ConvertUuid(charIn.uuid, &bsaCharPreviousUuid);
    if (status != BT_OK)
    {
        return status;
    }
    status = ConvertUuid(charFilter, &bsaCharFilterUuid);
    if (status != BT_OK)
    {
        return status;
    }

    status = BtHalLeGetNextCharacteristic(&bsaCharUuid, &(charOut->instanceId), property, connId, bsaServUuid, serviceId.instanceId, isPrimary, bsaCharPreviousUuid, charIn.instanceId, bsaCharFilterUuid);

    memcpy(&charOut->uuid, &bsaCharUuid, sizeof(tBT_UUID));

    return status;
}

/*******************************************************************************
 **
 ** Function        LeGetFirstDescriptor
 **
 ** Description     Find the first descriptor of the characteristic on the
 **                 given server from cache.
 **
 ** Parameters      None
 **
 ** Returns         BluetoothLeStatus
 **
 *******************************************************************************/
BluetoothLeStatus BluetoothLeInterfaceC::LeGetFirstDescriptor(GattId *descrOut, uint32_t connId, GattId serviceId, bool isPrimary, GattId charIn, GattAttributeUuid descrFilter)
{
    tBT_UUID bsaServUuid;
    tBT_UUID bsaCharUuid;
    tBT_UUID bsaDescUuid;
    tBT_UUID bsaDescFilterUuid;
    BluetoothLeStatus status = BT_OK;

    BTHAL_IF_BLE_DEBUG("called");

    status = ConvertUuid(serviceId.uuid, &bsaServUuid);
    if (status != BT_OK)
    {
        return status;
    }
    status = ConvertUuid(charIn.uuid, &bsaCharUuid);
    if (status != BT_OK)
    {
        return status;
    }
    status = ConvertUuid(descrFilter, &bsaDescFilterUuid);
    if (status != BT_OK)
    {
        return status;
    }

    status = BtHalLeGetFirstDescriptor(&bsaDescUuid, connId, bsaServUuid, serviceId.instanceId, isPrimary, bsaCharUuid, charIn.instanceId, bsaDescFilterUuid);

    descrOut->instanceId = 0;
    memcpy(&descrOut->uuid, &bsaDescUuid, sizeof(tBT_UUID));

    return status;
}

/*******************************************************************************
 **
 ** Function        LeGetNextDescriptor
 **
 ** Description     Find the subsequent descriptor of the characteristic on the
 **                 given server from cache.
 **
 ** Parameters      None
 **
 ** Returns         BluetoothLeStatus
 **
 *******************************************************************************/
BluetoothLeStatus BluetoothLeInterfaceC::LeGetNextDescriptor(GattId *descrOut, uint32_t connId, GattId serviceId, bool isPrimary, GattId charIn, GattId descrIn, GattAttributeUuid descrFilter)
{
    tBT_UUID bsaServUuid;
    tBT_UUID bsaCharUuid;
    tBT_UUID bsaDescUuid;
    tBT_UUID bsaDescPreviousUuid;
    tBT_UUID bsaDescFilterUuid;
    BluetoothLeStatus status = BT_OK;

    BTHAL_IF_BLE_DEBUG("called");

    status = ConvertUuid(serviceId.uuid, &bsaServUuid);
    if (status != BT_OK)
    {
        return status;
    }
    status = ConvertUuid(charIn.uuid, &bsaCharUuid);
    if (status != BT_OK)
    {
        return status;
    }
    status = ConvertUuid(descrIn.uuid, &bsaDescPreviousUuid);
    if (status != BT_OK)
    {
        return status;
    }
    status = ConvertUuid(descrFilter, &bsaDescFilterUuid);
    if (status != BT_OK)
    {
        return status;
    }

    status = BtHalLeGetNextDescriptor(&bsaDescUuid, connId, bsaServUuid, serviceId.instanceId, isPrimary, bsaCharUuid, charIn.instanceId, bsaDescPreviousUuid, bsaDescFilterUuid);

    descrOut->instanceId = 0;
    memcpy(&descrOut->uuid, &bsaDescUuid, sizeof(tBT_UUID));

    return status;
}

/*******************************************************************************
 **
 ** Function         LeConvertBsaStatus
 **
 ** Description      Convert result from tBSA_STATUS to BluetoothLeStatus
 **
 ** Parameters       resultFromBsa: tBSA_STATUS to be converted
 **
 ** Returns          BluetoothLeStatus result
 **
 *******************************************************************************/
BluetoothLeStatus LeConvertBsaStatus(tBSA_STATUS resultFromBsa)
{
    BluetoothLeStatus result;

    if(resultFromBsa == BSA_SUCCESS)
    {
        return BT_OK;
    }

    switch(resultFromBsa)
    {
        /* GATT errors */
        case  GATT_INVALID_HANDLE:                // 0x01
            result = BT_GATT_ERR_INVALID_HANDLE;
        break;
        case  GATT_READ_NOT_PERMIT:               // 0x02
            result = BT_GATT_ERR_READ_PERMISSION;
        break;
        case  GATT_WRITE_NOT_PERMIT:              // 0x03
            result = BT_GATT_ERR_WRITE_PERMISSION;
        break;
        case  GATT_INVALID_PDU:                   // 0x04
            result = BT_GATT_ERR_INVALID_PDU;
        break;
        case  GATT_INSUF_AUTHENTICATION:          // 0x05
            result = BT_GATT_ERR_INVALID_AUTH;
        break;
        case  GATT_REQ_NOT_SUPPORTED:             // 0x06
            result = BT_GATT_ERR_REQ_NOT_SUPP;
        break;
        case  GATT_INVALID_OFFSET:                // 0x07
            result = BT_GATT_ERR_INVALID_OFFSET;
        break;
        case  GATT_INSUF_AUTHORIZATION:           // 0x08
        case  GATT_AUTH_FAIL:                     // 0x89
            result = BT_GATT_ERR_INVALID_AUTH;
        break;
        case  GATT_PREPARE_Q_FULL:                // 0x09
            result = BT_GATT_ERR_QUEUE_FULL;
        break;
        case  GATT_NOT_FOUND:                     // 0x0a
            result = BT_GATT_ERR_NOT_FOUND;
        break;
        case  GATT_NOT_LONG:                      // 0x0b
            result = BT_GATT_ERR_NOT_LONG;
        break;
        case  GATT_INSUF_KEY_SIZE:                // 0x0c
            result = BT_GATT_ERR_KEY_SIZE;
        break;
        case  GATT_INVALID_ATTR_LEN:              // 0x0d
            result = BT_GATT_ERR_INVALID_ATTR_LEN;
        break;
        case  GATT_ERR_UNLIKELY:                  // 0x0e
            result = BT_GATT_ERR_UNLIKELY;
        break;
        case  GATT_INSUF_ENCRYPTION:              // 0x0f
        case  GATT_NOT_ENCRYPTED:                 // 0x8e
            result = BT_GATT_ERR_INVALID_ENCRYPTION;
        break;
        case  GATT_UNSUPPORT_GRP_TYPE:            // 0x10
            result = BT_GATT_ERR_INVALID_GROUP_TYPE;
        break;
        case  GATT_INSUF_RESOURCE:                // 0x11
        case  GATT_NO_RESOURCES:                  // 0x80
            result = BT_GATT_ERR_NO_RESOURCES;
        break;
        case  GATT_ILLEGAL_PARAMETER:             // 0x87
            result = BT_GATT_ERR_INVALID_PARAM;
        break;
        case  GATT_INTERNAL_ERROR:                // 0x81
            result = BT_GATT_ERR_INTERNAL;
        break;
        case  GATT_WRONG_STATE:                   // 0x82
            result = BT_GATT_ERR_WRONG_STATE;
        break;
        case  GATT_DB_FULL:                       // 0x83
            result = BT_GATT_ERR_DB_FULL;
        break;
        case  GATT_BUSY:                          // 0x84
            result = BT_GATT_ERR_BUSY;
        break;
        case  GATT_ERROR:                         // 0x85
            result = BT_GATT_ERR;
        break;
        case  GATT_CMD_STARTED:                   // 0x86
        case  GATT_SERVICE_STARTED:               // 0x8c
            result = BT_GATT_ERR_STARTED;
        break;
        case  GATT_PENDING:                       // 0x88
            result = BT_GATT_ERR_PENDING;
        break;
        case  GATT_MORE:                          // 0x8a
            result = BT_GATT_ERR_MORE;
        break;
        case  GATT_INVALID_CFG:                   // 0x8b
            result = BT_GATT_ERR_INVALID_CONFIG;
        break;
        case  GATT_ENCRYPED_MITM:                 // GATT_SUCCESS
            result = BT_GATT_ERR_ENCRYPTED_MITM;
        break;
        case  GATT_ENCRYPED_NO_MITM:              // 0x8d
            result = BT_GATT_ERR_ENCRYPTED_NO_MITM;
        break;
        case  GATT_CONGESTED:                     // 0x8f
          result = BT_GATT_ERR_CONGESTED;
        break;

        /* Bad parameter errors */
        case BSA_ERROR_CLI_BAD_PARAM:                // 104
        case BSA_ERROR_CLI_NAME_TOO_LONG:            // 111
        case BSA_ERROR_SRV_BAD_PARAM:                // 200
        case BSA_ERROR_SRV_BAD_REQ_SIZE:             // 201
        case BSA_ERROR_SRV_DG_FIND_SERVICE_FAIL:     // 1502
            result = BT_ERR_BAD_PARAM;
        break;

        /* Bad message/response errors */
        case BSA_ERROR_CLI_BAD_RSP_SIZE:             // 100
        case BSA_ERROR_CLI_BAD_MSG:                  // 109
        case BSA_ERROR_CLI_BAD_PING_RSP:             // 110
        case BSA_ERROR_SRV_BAD_MSG_ID:               // 202
        case BSA_ERROR_SRV_HD_UNKNOWN_KEY_TYPE:      // 1702
        case BSA_ERROR_SRV_HD_WRONG_RPT_LEN:         // 1704
            result = BT_ERR_BAD_RESPONSE;
        break;

        /* Busy/not ready errors */
        case BSA_ERROR_CLI_DISC_BUSY:                // 102
        case BSA_ERROR_CLI_BUSY:                     // 106
        case BSA_ERROR_CLI_ALREADY_WAITING:          // 112
            result = BT_ERR_BUSY;
        break;

        /* Connection errors */
        case BSA_ERROR_CLI_NOT_CONNECTED:            // 105
        case BSA_ERROR_CLI_ALREADY_CONNECTED:        // 115
        case BSA_ERROR_SRV_ALREADY_ACTIVE:           // 203
        case BSA_ERROR_SRV_DG_CONNECTED:             // 1501
        case BSA_ERROR_SRV_HD_AREADY_CONNECTED:      // 1703
            result = BT_ERR_CONNECTION;
        break;

        /* Open/Enable errors */
        case BSA_ERROR_SRV_DG_OPEN_FAIL:             // 1500
        case BSA_ERROR_SRV_HD_FAIL:                  // 1700
        case BSA_ERROR_SRV_HD_NOT_ENABLED:           // 1701
            result = BT_ERR_OPEN;
        break;

        /* No resource/memory full errors */
        case BSA_ERROR_CLI_RES_ERROR:                // 103
        case BSA_ERROR_CLI_MEM_FULL:                 // 116
        case BSA_ERROR_SRV_MEM_FULL:                 // 206
            result = BT_ERR_NO_RES;
        break;

        /* General client errors */
        case BSA_ERROR_CLI_NYI:                      // 101
        case BSA_ERROR_CLI_TASK_ERROR:               // 113
        case BSA_ERROR_CLI_INTERNAL:                 // 114
            result = BT_ERR_CLIENT;
        break;

        /* General server errors */
        case BSA_ERROR_SRV_NYI:                      // 204
        case BSA_ERROR_SRV_BAD_CLIENT:               // 205
        case BSA_ERROR_SRV_HW_ERROR:                 // 207
        case BSA_ERROR_SRV_BLUETOOTH_DISABLE:        // 208
        case BSA_ERROR_SRV_INTERNAL:                 // 209
        case BSA_ERROR_SRV_NOT_COMPILED:             // 210
        case BSA_ERROR_SRV_WLAN_RESET:               // 211
            result = BT_ERR_SERVER;
        break;

        /* Security errors */
        case BSA_ERROR_SRV_SEC_BOND_ACTIVE:          // 300
        case BSA_ERROR_SRV_SEC_RM_DEV:               // 301
        case BSA_ERROR_SRV_SEC_BOND_CANCEL_ERROR:    // 302
            result = BT_ERR_SECURITY;
        break;

        /* General hid host errors */
        case BSA_ERROR_SRV_HH_DSCP_TOO_BIG:          // 400
        case BSA_ERROR_SRV_HH_DSCP_PENDING:          // 401
        case BSA_ERROR_SRV_HH_NOT_ENABLED:           // 402
        case BSA_ERROR_SRV_HH_NOT_SUPPORTED:         // 403
        case BSA_ERROR_SRV_HH_NOT_ALLOWED:           // 404
        case BSA_ERROR_SRV_HH_UNKNOWN_DEVICE:        // 405
        case BSA_ERROR_SRV_HH_ALREADY_ADDED:         // 406
        case BSA_ERROR_SRV_HH_ERR:                   // 413
            result = BT_HH_ERR;
        break;

        /* Specific hid host errors */
        case BSA_ERROR_SRV_HH_HS_HID_NOT_READY:      // 408
            result = BT_HS_HID_NOT_READY;
        break;
        case BSA_ERROR_SRV_HH_HS_INVALID_RPT_ID:     // 409
            result = BT_HS_INVALID_RPT_ID;
        break;
        case BSA_ERROR_SRV_HH_HS_TRANS_NOT_SPT:      // 410
            result = BT_HS_TRANS_NOT_SPT;
        break;
        case BSA_ERROR_SRV_HH_HS_INVALID_PARAM:      // 411
            result = BT_HS_INVALID_PARAM;
        break;
        case BSA_ERROR_SRV_HH_HS_ERROR:              // 412
            result = BT_HS_ERROR;
        break;
        case BSA_ERROR_SRV_HH_ERR_SDP:               // 414
            result = BT_ERR_SDP;
        break;
        case BSA_ERROR_SRV_HH_ERR_PROTO:             // 415
            result = BT_ERR_PROTO;
        break;
        case BSA_ERROR_SRV_HH_ERR_DB_FULL:           // 416
            result = BT_ERR_DB_FULL;
        break;
        case BSA_ERROR_SRV_HH_ERR_TOD_UNSPT:         // 417
            result = BT_ERR_TOD_UNSPT;
        break;
        case BSA_ERROR_SRV_HH_ERR_NO_RES:            // 418
            result = BT_ERR_NO_RES;
        break;
        case BSA_ERROR_SRV_HH_ERR_AUTH_FAILED:       // 419
            result = BT_ERR_AUTH_FAILED;
        break;
        case BSA_ERROR_SRV_HH_ERR_HDL:               // 420
            result = BT_ERR_HDL;
        break;

        /* File system errors */
        case BSA_ERROR_SRV_FS_PERMISSION:            // 600
        case BSA_ERROR_SRV_FS_NO_FILE:               // 601
            result = BT_ERR_FILE_SYS;
        break;

        /* Audio video errors */
        case BSA_ERROR_SRV_AV_NOT_ENABLED:           // 700
        case BSA_ERROR_SRV_AV_FEEDING_NOT_SUPPORTED: // 701
        case BSA_ERROR_SRV_AV_BUSY:                  // 702
        case BSA_ERROR_SRV_AV_NOT_OPENED:            // 703
        case BSA_ERROR_SRV_AV_NOT_STARTED:           // 704
        case BSA_ERROR_SRV_AV_NOT_STOPPED:           // 705
        case BSA_ERROR_SRV_AV_CP_NOT_SUPPORTED:      // 706
        case BSA_ERROR_SRV_AV_BCST_NOT_SUPPORTED:    // 707
        case BSA_ERROR_SRV_AV_BR_CT_NOT_SUPPORTED:   // 708
        case BSA_ERROR_SRV_AV_FAIL:                  // 709
        case BSA_ERROR_SRV_AV_FAIL_SDP:              // 710
        case BSA_ERROR_SRV_AV_FAIL_STREAM:           // 711
        case BSA_ERROR_SRV_AV_FAIL_RESOURCES:        // 712
        case BSA_ERROR_SRV_AV_FAIL_ROLE:             // 713
            result = BT_ERR_AUDIO_VIDEO;
        break;

        /* Discovery errors */
        case BSA_ERROR_SRV_DEV_DISC_BUSY:            // 900
        case BSA_ERROR_SRV_DEV_INFO_BUSY:            // 901
        case BSA_ERROR_SRV_DISC_ABORT_ERROR:         // 902
            result = BT_ERR_DISCOVERY;
        break;

        /* Test mode errors */
        case BSA_ERROR_SRV_TM_VSC_BUSY:              // 1100
        case BSA_ERROR_SRV_TM_HCI_ERROR:             // 1101
        case BSA_ERROR_SRV_TM_HCI_CMD_ERROR:         // 1102
            result = BT_ERR_TEST_MODE;
        break;

        /* DM errors */
        case BSA_ERROR_SRV_DM_3DTV_LOCK_FAIL:        // 1200
        case BSA_ERROR_SRV_DM_TBFC_UNSUPPORTED:      // 1201
        case BSA_ERROR_SRV_DM_TBFC_NOT_COMPILED:     // 1202
        case BSA_ERROR_SRV_DM_SWITCH_STACK_FAIL:     // 1203
            result = BT_ERR_DM;
        break;

        case BSA_ERROR_SRV_BLE_FAIL:                 // 1600
            result = BT_BLE_ERR;
        break;
        case BSA_ERROR_SRV_BLE_NOT_ENABLED:          // 1601
            result = BT_BLE_ERR_NOT_ENABLED;
        break;
        case BSA_ERROR_SRV_BLE_NO_RESOURCE:          // 1602
            result = BT_BLE_ERR_NO_RESOURCE;
        break;
        case BSA_ERROR_SRV_BLE_INVALID_PARAM:        // 1603
            result = BT_BLE_ERR_INVALID_PARAM;
        break;

        /* Robson errors */
        case BSA_ERROR_ROBSON_MEM_FULL:                       // 2100
        case BSA_ERROR_ROBSON_SET_TSI_NOT_ALLOWED:            // 2101
        case BSA_ERROR_ROBSON_EXIT_TSI_NOT_ALLOWED:           // 2102
        case BSA_ERROR_ROBSON_TSI_FAIL:                       // 2103
        case BSA_ERROR_ROBSON_SET_BURST_NOT_ALLOWED:          // 2104
        case BSA_ERROR_ROBSON_EXIT_BURST_NOT_ALLOWED:         // 2105
        case BSA_ERROR_ROBSON_BURST_FAIL:                     // 2106
        case BSA_ERROR_ROBSON_SWITCH_MODE_FAIL:               // 2107
        case BSA_ERROR_ROBSON_SET_PACKET_TYPE_FAIL:           // 2108
        case BSA_ERROR_ROBSON_READ_0RETRAN_FAIL:              // 2109
        case BSA_ERROR_ROBSON_SET_0RETRAN_MODE_FAIL:          // 2110
        case BSA_ERROR_ROBSON_SUPER_PICONET_FAIL:             // 2111
        case BSA_ERROR_ROBSON_SET_ENCRYPTION_FAIL:            // 2112
        case BSA_ERROR_ROBSON_LINK_DOWN:                      // 2113
        case BSA_ERROR_ROBSON_LLR_START_FAIL_OVER_MAX:        // 2114
        case BSA_ERROR_ROBSON_LLR_SET_TBFC_PARAM_FAIL:        // 2115
        case BSA_ERROR_ROBSON_LLR_SET_SLEEP_MODE_FAIL:        // 2116
        case BSA_ERROR_ROBSON_LLR_WRITE_STORED_LINK_KEY_FAIL: // 2117
        case BSA_ERROR_ROBSON_LLR_SET_REMOTE_ONOFF_FAIL:      // 2118
        case BSA_ERROR_ROBSON_SET_RADIO_FAIL:                 // 2119
        case BSA_ERROR_ROBSON_SET_FW_CONTROL_HOPPING_FAIL:    // 2120
        case BSA_ERROR_ROBSON_SET_MWS_CHANNEL_PARAMETER_FAIL: // 2121
        case BSA_ERROR_ROBSON_LLR_START_FAIL_CONNECTION_EXIST:  //2122
            result = BT_ERR_ROBSON;
        break;

        default:
            NN_SDK_LOG("[bluetooth] %s: Unknown BSA status returned: %d \n", __func__, resultFromBsa);

            if (resultFromBsa <= UINT8_MAX)
            {
                result = ConvertHciStatusToLeStatus(static_cast<uint8_t>(resultFromBsa));
            }
            else
            {
                result = BT_ERR;
            }
        break;
    }
    return result;
} // NOLINT(impl/function_size)

/*******************************************************************************
**
** Function        RegisterLeCoreDataPath
**
** Description     Register GATT operation data path for BLE Core
**
** Parameters      None
**
** Returns         BluetoothHhStatus
**
*******************************************************************************/
BluetoothLeStatus BluetoothLeInterfaceC::RegisterLeCoreDataPath(const GattAttributeUuid& uuid)
{
    return RegisterGattDataPath(uuid, HAL_QUEUE_BLE_CORE);
}

/*******************************************************************************
**
** Function        UnregisterLeCoreDataPath
**
** Description     Unregister GATT operation data path for BLE Core
**
** Parameters      None
**
** Returns         BluetoothHhStatus
**
*******************************************************************************/
BluetoothLeStatus BluetoothLeInterfaceC::UnregisterLeCoreDataPath(const GattAttributeUuid& uuid)
{
    return UnregisterGattDataPath(uuid, HAL_QUEUE_BLE_CORE);
}

/*******************************************************************************
**
** Function        RegisterLeHidDataPath
**
** Description     Register GATT operation data path for BLE HID
**
** Parameters      None
**
** Returns         BluetoothHhStatus
**
*******************************************************************************/
BluetoothLeStatus BluetoothLeInterfaceC::RegisterLeHidDataPath(const GattAttributeUuid& uuid)
{
    return RegisterGattDataPath(uuid, HAL_QUEUE_BLE_HID);
}

/*******************************************************************************
**
** Function        UnregisterLeHidDataPath
**
** Description     Unregister GATT operation data path for BLE HID
**
** Parameters      None
**
** Returns         BluetoothHhStatus
**
*******************************************************************************/
BluetoothLeStatus BluetoothLeInterfaceC::UnregisterLeHidDataPath(const GattAttributeUuid& uuid)
{
    return UnregisterGattDataPath(uuid, HAL_QUEUE_BLE_HID);
}

/*******************************************************************************
**
** Function        RegisterLeDataPath
**
** Description     Register GATT operation data path for BLE
**
** Parameters      None
**
** Returns         BluetoothHhStatus
**
*******************************************************************************/
BluetoothLeStatus BluetoothLeInterfaceC::RegisterLeDataPath(const GattAttributeUuid& uuid)
{
    return RegisterGattDataPath(uuid, HAL_QUEUE_BLE);
}

/*******************************************************************************
**
** Function        UnregisterLeDataPath
**
** Description     Unregister GATT operation data path for BLE
**
** Parameters      None
**
** Returns         BluetoothHhStatus
**
*******************************************************************************/
BluetoothLeStatus BluetoothLeInterfaceC::UnregisterLeDataPath(const GattAttributeUuid& uuid)
{
    BTHAL_IF_BLE_DEBUG("called");

    return UnregisterGattDataPath(uuid, HAL_QUEUE_BLE);
}

} // bluetooth
} // nn

