﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/nn_SdkLog.h>
#include <nn/os.h>

#include "bluetooth.h"
#include "bluetooth_queue.h"
#include "bluetooth_le_disc.h"
#include "bluetooth_le_client.h"
#include "bluetooth_le_interface.h"

/*******************************************************************************
**
** Function:        ParseBleAdvPacket
**
** Description:     Parse BLE Advertising packet.
**                  BLE advertising packet consists of at least one AD structure.
**                  The maximum size is 31 byte.
**                  An AD structure consists of {uint8_t length, uint8_t adType, uint8_t[length -1] value}
**                  The end point of the packet is delimited by a structure with length = 0.
**
** Parameters:      [out] BleAdStructure* pOutStructures : A pointer to AD structure array as results of parsing
** Parameters:      [out] uint8_t* pOutNum : Number of parsed AD structures
** Parameters:      [in] uint8_t inNum : Array size of acceptable pOutStructures.
** Parameters:      [in] uint8_t* advPacket : Raw bytes of received Advertising packet.
**
** Returns          0 if success, -1 otherwise
**
*******************************************************************************/
static int ParseBleAdvPacket(BleAdStructure* pOutStructures, uint8_t* pOutNum, uint8_t inNum, const uint8_t* advPacket)
{
    BTHAL_IF_BLE_DEBUG("called.");

    BleAdStructure structures[BleAdStructureCountMax];
    uint8_t num = 0;

    if (inNum == 0 || !pOutStructures || !advPacket)
    {
        BTHAL_IF_BLE_DEBUG("[bluetooth] Failed to parse BLE ADV packet. Invalid argument.\n");
        return -1;
    }

    for (size_t offset = 0; offset < BleAdvertisePacketSizeMax;)
    {
        uint8_t length = advPacket[offset];

        if(length == 0x00)
        {
            if (num == 0)
            {
                // An Advertising packet must include at least one AD structure
                BTHAL_IF_BLE_DEBUG("[bluetooth] Failed to parse BLE ADV packet. Invalid length field value (%d).\n", length);
                return -1;
            }
            else
            {
                // Parse complete
                break;
            }
        }
        else if (length == 0x01 || static_cast<size_t>(length) > BleAdvertisePacketSizeMax - 1 || offset + static_cast<size_t>(length) >= BleAdvertisePacketSizeMax)
        {
            BTHAL_IF_BLE_DEBUG("[bluetooth] Failed to parse BLE ADV packet. Invalid length filed value (%d) at offset %d.\n", length, offset);
            return -1;
        }

        structures[num].length = length;

        offset++;       // AD Type field

        uint8_t adType = advPacket[offset];
        structures[num].adType = (BleAdType)(adType);

        offset++;       // Payload
        memcpy(structures[num].data, &(advPacket[offset]), length - 1);
        num++;

        if (num == inNum || num == BleAdStructureCountMax)
        {
            break;
        }

        offset += (static_cast<size_t>(length) - 1);     // Next length field

        if (offset >= BleAdvertisePacketSizeMax)
        {
            // Parse complete
            break;
        }
    }

    *pOutNum = num;

    for (int i = 0; i < *pOutNum; ++i)
    {
        pOutStructures[i] = structures[i];
    }

    return 0;
}

/*******************************************************************************
**
** Function:        LeGenericScanCback
**
** Description:     BLE Scan callback
**
** Parameters:      [in] tBSA_DISC_EVT event : event type of callback
** Parameters:      [in] tBSA_DISC_MSG pData : callback data. Refer members based on the event type.
**
** Returns:         void
**
*******************************************************************************/
static void LeGenericScanCback(tBSA_DISC_EVT event, tBSA_DISC_MSG *pData)
{
    BTHAL_IF_BLE_DEBUG("called.");

    InfoFromLeScanStateChangedCallback cbData;

    switch (event)
    {
    /* a New Device has been discovered */
    case BSA_DISC_NEW_EVT:
        //BTHAL_IF_BLE_DEBUG("BSA_DISC_NEW_EVT");

        //BTHAL_IF_BLE_DEBUG("Device Found:");
        //BTHAL_IF_BLE_DEBUG("\tBD Address: %02X:%02X:%02X:%02X:%02X:%02X",
        //    pData->disc_new.bd_addr[0], pData->disc_new.bd_addr[1], pData->disc_new.bd_addr[2],
        //    pData->disc_new.bd_addr[3], pData->disc_new.bd_addr[4], pData->disc_new.bd_addr[5]);
        //BTHAL_IF_BLE_DEBUG("\tAddress Type: %d", pData->disc_new.ble_addr_type);
        //BTHAL_IF_BLE_DEBUG("\tRSSI: %d", pData->disc_new.rssi);
        //BTHAL_IF_BLE_DEBUG("\tADV Packet Raw: %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X "
        //                                     "%02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X",
        //    pData->disc_new.eir_data[0], pData->disc_new.eir_data[1], pData->disc_new.eir_data[2], pData->disc_new.eir_data[3],
        //    pData->disc_new.eir_data[4], pData->disc_new.eir_data[5], pData->disc_new.eir_data[6], pData->disc_new.eir_data[7],
        //    pData->disc_new.eir_data[8], pData->disc_new.eir_data[9], pData->disc_new.eir_data[10], pData->disc_new.eir_data[11],
        //    pData->disc_new.eir_data[12], pData->disc_new.eir_data[13], pData->disc_new.eir_data[14], pData->disc_new.eir_data[15],
        //    pData->disc_new.eir_data[16], pData->disc_new.eir_data[17], pData->disc_new.eir_data[18], pData->disc_new.eir_data[19],
        //    pData->disc_new.eir_data[20], pData->disc_new.eir_data[21], pData->disc_new.eir_data[22], pData->disc_new.eir_data[23],
        //    pData->disc_new.eir_data[24], pData->disc_new.eir_data[25], pData->disc_new.eir_data[26], pData->disc_new.eir_data[27],
        //    pData->disc_new.eir_data[28], pData->disc_new.eir_data[29], pData->disc_new.eir_data[30]);

        cbData.status = BT_OK;
        cbData.state = BLE_SCAN_STATE_FOUND_DEVICE;
        cbData.pduType = (BleAdvertisePduType)(pData->disc_new.ble_evt_type);

        cbData.addressType = pData->disc_new.ble_addr_type;
        copyBdAddr(cbData.address.address, pData->disc_new.bd_addr);
        cbData.rssi = pData->disc_new.rssi;

        if (ParseBleAdvPacket(cbData.adStructures, &(cbData.adStructureNum), BleAdStructureCountMax, pData->disc_new.eir_data) < 0)
        {
            BTHAL_IF_BLE_DEBUG("[bluetooth] Invalid BLE Advertise packet.\n");
            break;
        }

        BluetoothHalInsertToQueue(&cbData, sizeof(InfoFromLeScanStateChangedCallback), HAL_QUEUE_BLE_CORE, CALLBACK_TYPE_LE_SCAN_STATE_CHANGED);
    break;
    /* Discovery complete. */
    case BSA_DISC_CMPL_EVT:
        BTHAL_IF_BLE_DEBUG("BSA_DISC_CMPL_EVT");

        cbData.status = BT_OK;
        cbData.state = BLE_SCAN_STATE_COMPLETED;
        cbData.pduType = BleAdvertisePduType_Unknown;
        cbData.addressType = 0xFF;
        memset(cbData.address.address, 0x00, BD_ADDR_LEN);
        for (int i = 0; i < BleAdStructureCountMax; ++i)
        {
            memset(&cbData.adStructures[i], 0x00, sizeof(BleAdStructure));
        }
        cbData.adStructureNum = 0;
        cbData.rssi = 0;

        BluetoothHalInsertToQueue(&cbData, sizeof(InfoFromLeScanStateChangedCallback), HAL_QUEUE_BLE_CORE, CALLBACK_TYPE_LE_SCAN_STATE_CHANGED);
    break;
    default:
        break;
    }
}

/*******************************************************************************
**
** Function:        BtHalLeStartScan
**
** Description:     Start BLE Scan.
**
** Parameters:      [in] uint32_t duration : 0 means infinite scan, which stops when BtHalLeStopScan is called.
**
** Returns:         BluetoothLeStatus
**
** Callbacks:       InfoFromLeScanStateChangedCallback
**
*******************************************************************************/
BluetoothLeStatus BtHalLeStartScan(uint32_t duration)
{
    BTHAL_IF_BLE_DEBUG("called");

    tBSA_STATUS status;
    tBSA_DISC_START disc_start_param;
    InfoFromLeScanStateChangedCallback cbData;

    status = BSA_DiscStartInit(&disc_start_param);
    if(status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to start LE scan. status: %d\n", __func__, status);
        return LeConvertBsaStatus(status);
    }

    disc_start_param.cback = LeGenericScanCback;
    disc_start_param.nb_devices = 20;
    disc_start_param.duration = duration;
    disc_start_param.mode = BSA_BLE_GENERAL_INQUIRY;
    disc_start_param.update = BSA_DISC_UPDATE_ANY;
    disc_start_param.skip_name_request = true;      // Do not connect to the advertiser to get device name
    disc_start_param.services = 0;                  // Do not connect to the advertiser to get service

    status = BSA_DiscStart(&disc_start_param);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to start LE scan. status: %d\n", __func__, status);
        return LeConvertBsaStatus(status);
    }

    cbData.status = LeConvertBsaStatus(status);
    cbData.state = BLE_SCAN_STATE_SCANNING;
    cbData.pduType = BleAdvertisePduType_Unknown;
    cbData.addressType = 0xFF;
    memset(cbData.address.address, 0x00, BD_ADDR_LEN);
    for (int i = 0; i < BleAdStructureCountMax; ++i)
    {
        memset(&cbData.adStructures[i], 0x00, sizeof(BleAdStructure));
    }
    cbData.adStructureNum = 0;
    cbData.rssi = 0;

    BluetoothHalInsertToQueue(&cbData, sizeof(InfoFromLeScanStateChangedCallback), HAL_QUEUE_BLE_CORE, CALLBACK_TYPE_LE_SCAN_STATE_CHANGED);

    return LeConvertBsaStatus(status);
}

/*******************************************************************************
**
** Function:        BtHalLeStopScan
**
** Description:     Stop BLE Scan.
**
** Parameters:      None
**
** Returns:         BluetoothLeStatus
**
** Callbacks:       InfoFromLeScanStateChangedCallback
**
*******************************************************************************/
BluetoothLeStatus BtHalLeStopScan()
{
    BTHAL_IF_BLE_DEBUG("called.");

    tBSA_STATUS status;
    tBSA_DISC_ABORT disc_abort_param;

    status = BSA_DiscAbortInit(&disc_abort_param);
    if(status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to stop LE scan. status: %d\n", __func__, status);
        return LeConvertBsaStatus(status);
    }

    status = BSA_DiscAbort(&disc_abort_param);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to stop LE scan. status: %d\n", __func__, status);
        return LeConvertBsaStatus(status);
    }

    return LeConvertBsaStatus(status);
}

