﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/nn_SdkLog.h>
#include <nn/os.h>

#include <nn/bluetooth/bluetooth_Result.h>
#include <nn/bluetooth/bluetooth_ResultPrivate.h>
#include "bluetooth_le_client.h"
#include "bluetooth.h"
#include "bluetooth_queue.h"
#include "bluetooth_le_interface.h"

/*
 * Global Variables
 */
static bool g_IsServiceDiscovered = true;
static nn::os::MutexType g_MutexRegisteredNotification = NN_OS_MUTEX_INITIALIZER(false);

/*******************************************************************************
**
** Function:        FindClientFreeSpace
**
** Description:     find free space for BLE client application
**
** Parameters:      None
**
** Returns:         positive number(include 0) if successful, -1 otherwise
**
*******************************************************************************/
static int FindClientFreeSpace()
{
    for(int index = 0; index < BLE_CLIENT_MAX ; ++index)
    {
        if(!app_ble_cb.ble_client[index].enabled)
        {
            return index;
        }
    }
    return -1;
}

/*******************************************************************************
**
** Function:        DisplayClient
**
** Description:     Display BLE client
**
** Parameters:      None
**
** Returns:         void
**
*******************************************************************************/
static void DisplayClient()
{
    for(int i = 0; i < BLE_CLIENT_MAX; ++i)
    {
        if(app_ble_cb.ble_client[i].enabled)
        {
            if(app_ble_cb.ble_client[i].conn_id == BleInvalidConnectionHandle)
            {
                BTHAL_IF_BLE_DEBUG("Client #%d: client_if: %d", i, app_ble_cb.ble_client[i].client_if);
            }
            else
            {
                BTHAL_IF_BLE_DEBUG("Client #%d: client_if: %d bsa_conn_id: %d conn_id: %d server: %02X:%02X:%02X:%02X:%02X:%02X",
                                    i,
                                    app_ble_cb.ble_client[i].client_if,
                                    app_ble_cb.ble_client[i].bsa_conn_id,
                                    app_ble_cb.ble_client[i].conn_id,
                                    app_ble_cb.ble_client[i].server_addr[0], app_ble_cb.ble_client[i].server_addr[1],
                                    app_ble_cb.ble_client[i].server_addr[2], app_ble_cb.ble_client[i].server_addr[3],
                                    app_ble_cb.ble_client[i].server_addr[4], app_ble_cb.ble_client[i].server_addr[5]);
            }
        }
    }
}

/*******************************************************************************
**
** Function:        FindClientByClientInterface
**
** Description:     find BLE client index by identifier of client application
**
** Parameters:      [in] tBSA_BLE_IF clientIf: identifier of client application
**
** Returns:         index of client application database if successful, -1 otherwise
**
*******************************************************************************/
static int FindClientByClientInterface(tBSA_BLE_IF clientIf)
{
    for(int index = 0; index < BLE_CLIENT_MAX ; index++)
    {
        if(app_ble_cb.ble_client[index].enabled && app_ble_cb.ble_client[index].client_if == clientIf)
        {
            return index;
        }
    }
    return -1;
}

/*******************************************************************************
**
** Function:        FindClientByConnectionId
**
** Description:     find BLE client index by connection ID
**
** Parameters:      [in] uint16_t conn_id: connection ID of the target server
**
** Returns:         index of client application database if successful, -1 otherwise
**
*******************************************************************************/
static int FindClientBy32bitConnectionId(uint32_t connId)
{
    for(int index = 0; index < BLE_CLIENT_MAX ; index++)
    {
        if(app_ble_cb.ble_client[index].enabled &&
           app_ble_cb.ble_client[index].conn_id == connId &&
           app_ble_cb.ble_client[index].conn_id != BleInvalidConnectionHandle)
        {
            return index;
        }
    }
    return -1;
}

static uint32_t Find32bitConnectionId(uint16_t bsaConnId)
{
    for (int i = 0; i < BLE_CLIENT_MAX; ++i)
    {
        if (app_ble_cb.ble_client[i].enabled && app_ble_cb.ble_client[i].bsa_conn_id == bsaConnId)
        {
            return app_ble_cb.ble_client[i].conn_id;
        }
    }

    return BleInvalidConnectionHandle;
}

static uint32_t GetNew32bitConnId(uint16_t connId)
{
    if (connId == BSA_BLE_INVALID_CONN)
    {
        return BleInvalidConnectionHandle;
    }
    else
    {
        return static_cast<uint32_t>(connId);
    }
}

static BluetoothLeStatus RegisterNotificationImpl(
    tBSA_BLE_IF clientIf,
    const BD_ADDR& server_address,
    const tBT_UUID& serviceUuid, uint8_t serviceInstId, bool isPrimary,
    const tBT_UUID& characteristicUuid, uint8_t characteristicInstId)
{
    tBSA_BLE_CL_NOTIFREG ble_notifreg_param;

    int clientIndex = FindClientByClientInterface(clientIf);

    if (clientIndex < 0)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to find GATT client.\n", NN_CURRENT_FUNCTION_NAME);
        return BT_ERR_CLIENT;
    }

    tBSA_STATUS status = BSA_BleClNotifRegisterInit(&ble_notifreg_param);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to register notification. status: %d\n", __func__, status);
        return LeConvertBsaStatus(status);
    }

    ble_notifreg_param.notification_id.srvc_id.id.uuid      = serviceUuid;
    ble_notifreg_param.notification_id.srvc_id.id.inst_id   = serviceInstId;
    ble_notifreg_param.notification_id.srvc_id.is_primary   = isPrimary ? 1 : 0;

    ble_notifreg_param.notification_id.char_id.uuid     = characteristicUuid;
    ble_notifreg_param.notification_id.char_id.inst_id  = characteristicInstId;

    copyBdAddr(ble_notifreg_param.bd_addr, server_address);
    ble_notifreg_param.client_if = clientIf;

    status = BSA_BleClNotifRegister(&ble_notifreg_param);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to register notification. status: %d\n", __func__, status);
        return LeConvertBsaStatus(status);
    }

    nn::os::LockMutex(&g_MutexRegisteredNotification);

    for (auto& registeredNotification : app_ble_cb.ble_client[clientIndex].registered_notification)
    {
        if (!registeredNotification.is_registered)
        {
            registeredNotification.service_uuid = serviceUuid;
            registeredNotification.service_inst_id = serviceInstId;
            registeredNotification.is_primary_service = isPrimary;
            registeredNotification.char_uuid = characteristicUuid;
            registeredNotification.char_inst_id = characteristicInstId;

            registeredNotification.is_registered = true;

            nn::os::UnlockMutex(&g_MutexRegisteredNotification);

            return BT_OK;
        }
    }

    nn::os::UnlockMutex(&g_MutexRegisteredNotification);
    NN_ABORT_UNLESS_RESULT_SUCCESS(ResultDatabaseFull());
    return BT_GATT_ERR_NO_RESOURCES;
}

static BluetoothLeStatus DeregisterNotificationImpl(tBSA_BLE_IF clientIf,
                                                    const BD_ADDR& server_address,
                                                    const tBT_UUID& serviceUuid, uint8_t serviceInstId, bool isPrimary,
                                                    const tBT_UUID& characteristicUuid, uint8_t characteristicInstId)
{
    tBSA_BLE_CL_NOTIFREG ble_notifreg_param;
    int clientIndex = FindClientByClientInterface(clientIf);

    if (clientIndex < 0)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to find GATT client.\n", NN_CURRENT_FUNCTION_NAME);
        return BT_ERR_CLIENT;
    }

    tBSA_STATUS status = BSA_BleClNotifDeregisterInit(&ble_notifreg_param);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to deregister notification. status: %d\n", __func__, status);
        return LeConvertBsaStatus(status);
    }

    ble_notifreg_param.notification_id.srvc_id.id.uuid      = serviceUuid;
    ble_notifreg_param.notification_id.srvc_id.id.inst_id   = serviceInstId;
    ble_notifreg_param.notification_id.srvc_id.is_primary   = isPrimary ? 1 : 0;

    ble_notifreg_param.notification_id.char_id.uuid     = characteristicUuid;
    ble_notifreg_param.notification_id.char_id.inst_id  = characteristicInstId;

    copyBdAddr(ble_notifreg_param.bd_addr, server_address);
    ble_notifreg_param.client_if = clientIf;

    status = BSA_BleClNotifDeregister(&ble_notifreg_param);
    if (status != BSA_SUCCESS && status != BSA_ERROR_SRV_BLE_FAIL)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to deregister notification. status: %d\n", __func__, status);
        return LeConvertBsaStatus(status);
    }

    nn::os::LockMutex(&g_MutexRegisteredNotification);

    for (auto& registeredNotification : app_ble_cb.ble_client[clientIndex].registered_notification)
    {
        if (registeredNotification.is_registered &&
            memcmp(&registeredNotification.service_uuid, &serviceUuid, sizeof(tBT_UUID)) == 0 &&
            registeredNotification.service_inst_id == serviceInstId &&
            registeredNotification.is_primary_service == isPrimary &&
            memcmp(&registeredNotification.char_uuid, &characteristicUuid, sizeof(tBT_UUID)) == 0 &&
            registeredNotification.char_inst_id == characteristicInstId)
        {
            registeredNotification.is_registered = false;

            break;
        }
    }

    nn::os::UnlockMutex(&g_MutexRegisteredNotification);

    return BT_OK;
}

/*******************************************************************************
**
** Function:        LeClientStateChangedCallback
**
** Description:     Callback of client application state change
**
** Parameters:      [in] tBSA_BLE_EVT event     : event type
** Parameters:      [in] tBSA_BLE_MSG* pData    : event data
**
** Returns:         void
**
*******************************************************************************/
static void LeClientStateChangedCallback(tBSA_BLE_EVT event, tBSA_BLE_MSG *pData)
{
    BTHAL_IF_BLE_DEBUG("called.");

    InfoFromLeAppStateChangedCallback cbData;
    tBSA_BLE_IF clientIf = 0;
    int clientIdx = -1;

    switch (event)
    {
    case BSA_BLE_CL_DEREGISTER_EVT:
        BTHAL_IF_BLE_DEBUG("BSA_BLE_CL_DEREGISTER_EVT");
        clientIf = pData->cli_deregister.client_if;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    clientIdx = FindClientByClientInterface(clientIf);
    if (clientIdx < 0)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to deregister BLE client. Client not found.\n", __func__);

        cbData.status       = BT_ERR_CLIENT;
        cbData.clientIf     = BSA_BLE_INVALID_IF;
        cbData.isRegistered = false;
    }
    else
    {
        switch (event)
        {
        case BSA_BLE_CL_DEREGISTER_EVT:
            app_ble_cb.ble_client[clientIdx].enabled        = false;
            app_ble_cb.ble_client[clientIdx].client_if      = BSA_BLE_INVALID_IF;
            app_ble_cb.ble_client[clientIdx].bsa_conn_id    = BSA_BLE_INVALID_CONN;
            app_ble_cb.ble_client[clientIdx].conn_id        = BleInvalidConnectionHandle;

            NN_SDK_LOG("[bluetooth] %s: BLE client application deleted.\n", __func__);
            DisplayClient();

            cbData.clientIf = clientIf;
            cbData.status = LeConvertBsaStatus(pData->cli_deregister.status);
            cbData.isRegistered = false;
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
    BluetoothHalInsertToQueue(&cbData, sizeof(InfoFromLeAppStateChangedCallback), HAL_QUEUE_BLE_CORE, CALLBACK_TYPE_LE_CLIENT_STATE_CHANGED);
}

/*******************************************************************************
**
** Function:        LeClientConnectionStateChangedCallback
**
** Description:     Callback of client connection state change
**
** Parameters:      [in] tBSA_BLE_EVT event     : event type
** Parameters:      [in] tBSA_BLE_MSG* pData    : event data
**
** Returns:         void
**
*******************************************************************************/
static void LeClientConnectionStateChangedCallback(tBSA_BLE_EVT event, tBSA_BLE_MSG *pData)
{
    BTHAL_IF_BLE_DEBUG("called.");

    InfoFromLeConnStateChangedCallback cbData;
    tBSA_BLE_IF clientIf = 0;
    int clientIdx = -1;

    switch (event)
    {
    case BSA_BLE_CL_OPEN_EVT:
        BTHAL_IF_BLE_DEBUG("BSA_BLE_CL_OPEN_EVT");
        clientIf = pData->cli_open.client_if;
        break;
    case BSA_BLE_CL_CANCEL_OPEN_EVT:
        BTHAL_IF_BLE_DEBUG("BSA_BLE_CL_CANCEL_OPEN_EVT");
        clientIf = pData->cli_cancel_open.client_if;
        break;
    case BSA_BLE_CL_CLOSE_EVT:
        BTHAL_IF_BLE_DEBUG("BSA_BLE_CL_CLOSE_EVT");
        clientIf = pData->cli_close.client_if;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    clientIdx = FindClientByClientInterface(clientIf);
    if (clientIdx < 0)
    {
        NN_SDK_LOG("[bluetooth] %s: Error. Client not found. clientIf: %d\n", __func__, clientIf);

        cbData.status       = BT_ERR_CLIENT;
        cbData.clientIf     = BSA_BLE_INVALID_IF;
        cbData.connId       = BleInvalidConnectionHandle;
        cbData.connState    = BLE_CONN_STATE_DISCONNECTED;
        cbData.reason       = 0x00;
        memset(cbData.address.address, 0x00, BD_ADDR_LEN);
    }
    else
    {
        switch (event)
        {
        case BSA_BLE_CL_OPEN_EVT:
            if (pData->cli_open.status != BSA_SUCCESS)
            {
                NN_SDK_LOG("[bluetooth] %s: Failed to open connection. status: %d\n", __func__, pData->cli_open.status);
                app_ble_cb.ble_client[clientIdx].bsa_conn_id    = BSA_BLE_INVALID_CONN;
                app_ble_cb.ble_client[clientIdx].conn_id        = BleInvalidConnectionHandle;
                memset(app_ble_cb.ble_client[clientIdx].server_addr, 0x00, BD_ADDR_LEN);

                cbData.connId = BleInvalidConnectionHandle;
                memset(cbData.address.address,0x00, BD_ADDR_LEN);
            }
            else
            {
                app_ble_cb.ble_client[clientIdx].bsa_conn_id    = pData->cli_open.conn_id;
                app_ble_cb.ble_client[clientIdx].conn_id        = GetNew32bitConnId(pData->cli_open.conn_id);
                memcpy(app_ble_cb.ble_client[clientIdx].server_addr, pData->cli_open.bd_addr, BD_ADDR_LEN);

                cbData.connId = app_ble_cb.ble_client[clientIdx].conn_id;
                memcpy(cbData.address.address, pData->cli_open.bd_addr, BD_ADDR_LEN);
            }

            if (app_ble_cb.ble_client[clientIdx].conn_id != BleInvalidConnectionHandle)
            {
                cbData.connState = BLE_CONN_STATE_CONNECTED;
            }
            else
            {
                cbData.connState = BLE_CONN_STATE_DISCONNECTED;
            }

            cbData.status   = LeConvertBsaStatus(pData->cli_open.status);
            cbData.clientIf = pData->cli_open.client_if;
            cbData.reason   = 0x00;

            DisplayClient();
            break;
        case BSA_BLE_CL_CANCEL_OPEN_EVT:
            if (pData->cli_cancel_open.status != BSA_SUCCESS)
            {
                NN_SDK_LOG("[bluetooth] %s: Failed to cancle connection. status: %d\n", __func__, pData->cli_cancel_open.status);
            }

            app_ble_cb.ble_client[clientIdx].bsa_conn_id    = BSA_BLE_INVALID_CONN;
            app_ble_cb.ble_client[clientIdx].conn_id        = BleInvalidConnectionHandle;
            memset(app_ble_cb.ble_client[clientIdx].server_addr, 0x00, BD_ADDR_LEN);

            cbData.status       = LeConvertBsaStatus(pData->cli_cancel_open.status);
            cbData.clientIf     = pData->cli_cancel_open.client_if;
            cbData.connId       = BleInvalidConnectionHandle;
            cbData.connState    = BLE_CONN_STATE_CANCELED;
            cbData.reason       = 0x00;
            memset(cbData.address.address, 0x00, BD_ADDR_LEN);
            break;
        case BSA_BLE_CL_CLOSE_EVT:
            if (pData->cli_close.status != BSA_SUCCESS)
            {
                NN_SDK_LOG("[bluetooth] %s: Failed to close connection. status: %d\n", __func__, pData->cli_close.status);
            }

            // Get first
            cbData.connId       = Find32bitConnectionId(pData->cli_close.conn_id);
            memcpy(cbData.address.address, pData->cli_close.remote_bda, BD_ADDR_LEN);
            cbData.connState    = BLE_CONN_STATE_DISCONNECTED;
            cbData.status       = LeConvertBsaStatus(pData->cli_close.status);
            cbData.clientIf     = pData->cli_close.client_if;
            cbData.reason       = pData->cli_close.reason;

            // Then update
            app_ble_cb.ble_client[clientIdx].bsa_conn_id    = BSA_BLE_INVALID_CONN;
            app_ble_cb.ble_client[clientIdx].conn_id        = BleInvalidConnectionHandle;
            memset(app_ble_cb.ble_client[clientIdx].server_addr, 0x00, BD_ADDR_LEN);

            app_ble_cb.ble_client[clientIdx].read_pending   = false;
            app_ble_cb.ble_client[clientIdx].write_pending  = false;
            app_ble_cb.ble_client[clientIdx].congested      = false;

            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    BluetoothHalInsertToQueue(&cbData, sizeof(InfoFromLeConnStateChangedCallback), HAL_QUEUE_BLE_CORE, CALLBACK_TYPE_LE_CLIENT_CONN_STATE_CHANGED);
}// NOLINT(impl/function_size)

/*******************************************************************************
**
** Function:        LeClientGattOperationCallback
**
** Description:     Callback of client Gatt operation result
**
** Parameters:      [in] tBSA_BLE_EVT event     : event type
** Parameters:      [in] tBSA_BLE_MSG* pData    : event data
**
** Returns:         void
**
*******************************************************************************/
static void LeClientGattOperationCallback(tBSA_BLE_EVT event, tBSA_BLE_MSG *pData)
{
//    BTHAL_IF_BLE_DEBUG("called.");

    InfoFromLeGattOperationCallback cbData;
    InfoFromLeGattIndicationResponseNeededCallback responseData;

    tBT_UUID bsaServiceUuid, bsaCharacteristicUuid, bsaDescriptorUuid;
    tBSA_STATUS bsaStatus;
    uint16_t bsaConnId  = BSA_BLE_INVALID_CONN;
    uint32_t connId     = BleInvalidConnectionHandle;
    int clientIdx = -1;

    GattOperationType operation;

    switch (event)
    {
    case BSA_BLE_CL_READ_EVT:
        BTHAL_IF_BLE_DEBUG("BSA_BLE_CL_READ_EVT");
        bsaConnId               = pData->cli_read.conn_id;
        bsaStatus               = pData->cli_read.status;
        bsaServiceUuid          = pData->cli_read.srvc_id.id.uuid;
        bsaCharacteristicUuid   = pData->cli_read.char_id.uuid;
        bsaDescriptorUuid       = pData->cli_read.descr_type.uuid;

        if (bsaDescriptorUuid.len == 0)
        {
            operation = GattOperationType_ReadCharacteristic;
        }
        else
        {
            operation = GattOperationType_ReadDescriptor;
        }
        break;
    case BSA_BLE_CL_WRITE_EVT:
        BTHAL_IF_BLE_DEBUG("BSA_BLE_CL_WRITE_EVT");
        bsaConnId               = pData->cli_write.conn_id;
        bsaStatus               = pData->cli_write.status;
        bsaServiceUuid          = pData->cli_write.srvc_id.id.uuid;
        bsaCharacteristicUuid   = pData->cli_write.char_id.uuid;
        bsaDescriptorUuid       = pData->cli_write.descr_type.uuid;

        if (bsaDescriptorUuid.len == 0)
        {
            operation = GattOperationType_WriteCharacteristic;
        }
        else
        {
            operation = GattOperationType_WriteDescriptor;
        }
        break;
    case BSA_BLE_CL_NOTIF_EVT:
//        BTHAL_IF_BLE_DEBUG("BSA_BLE_CL_NOTIF_EVT");
        bsaConnId               = pData->cli_notif.conn_id;
        bsaStatus               = BSA_SUCCESS;
        bsaServiceUuid          = pData->cli_notif.char_id.srvc_id.id.uuid;
        bsaCharacteristicUuid   = pData->cli_notif.char_id.char_id.uuid;
        bsaDescriptorUuid       = pData->cli_notif.descr_type.uuid;

        if (pData->cli_notif.is_notify)
        {
            operation = GattOperationType_Notify;
        }
        else
        {
            operation = GattOperationType_Indicate;
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    cbData.operation = operation;
    memcpy(&(cbData.serviceUuid), &(bsaServiceUuid), sizeof(GattAttributeUuid));
    memcpy(&(cbData.charcteristicUuid), &(bsaCharacteristicUuid), sizeof(GattAttributeUuid));
    memcpy(&(cbData.descriptorUuid), &(bsaDescriptorUuid), sizeof(GattAttributeUuid));

    connId      = Find32bitConnectionId(bsaConnId);
    clientIdx   = FindClientBy32bitConnectionId(connId);

    if (clientIdx < 0)
    {
        NN_SDK_LOG("[bluetooth] %s: Error. Client not found. connId: %d\n", __func__, connId);

        cbData.status               = BT_ERR_CLIENT;
        cbData.connId               = BleInvalidConnectionHandle;
        cbData.length               = 0;
        memset(cbData.value, 0x00, NN_ARRAY_SIZE(cbData.value));
    }
    else
    {
        cbData.status = LeConvertBsaStatus(bsaStatus);
        cbData.connId = connId;

        if (bsaStatus != BSA_SUCCESS)
        {
            NN_SDK_LOG("[bluetooth] %s: Gatt operation failed. operation: %d, status: %d\n", NN_CURRENT_FUNCTION_NAME, operation, bsaStatus);
            NN_SDK_LOG("[bluetooth] %s: Connection ID: %d\n", NN_CURRENT_FUNCTION_NAME, connId);
            NN_SDK_LOG("[bluetooth] %s: Service UUID:        %s\n", NN_CURRENT_FUNCTION_NAME, toString((uint8_t*)(&bsaServiceUuid.uu), bsaServiceUuid.len));
            NN_SDK_LOG("[bluetooth] %s: Characteristic UUID: %s\n", NN_CURRENT_FUNCTION_NAME, toString((uint8_t*)(&bsaCharacteristicUuid.uu), bsaCharacteristicUuid.len));
            NN_SDK_LOG("[bluetooth] %s: Descriptor UUID:     %s\n", NN_CURRENT_FUNCTION_NAME, toString((uint8_t*)(&bsaDescriptorUuid.uu), bsaDescriptorUuid.len));
        }

        switch (event)
        {
        case BSA_BLE_CL_READ_EVT:
            if (pData->cli_read.len > NN_ARRAY_SIZE(cbData.value))
            {
                NN_SDK_LOG("[bluetooth] %s: Received too large data. %d Bytes for %d Bytes buffer.\n", NN_CURRENT_FUNCTION_NAME, pData->cli_read.len, NN_ARRAY_SIZE(cbData.value));
                cbData.status = BT_ERR_BAD_RESPONSE;
                cbData.length = 0;
                memset(cbData.value, 0x00, NN_ARRAY_SIZE(cbData.value));
            }
            else
            {
                cbData.length = pData->cli_read.len;
                memcpy(cbData.value, pData->cli_read.value, cbData.length);
            }

            app_ble_cb.ble_client[clientIdx].read_pending = false;

            break;
        case BSA_BLE_CL_WRITE_EVT:
            cbData.length = 0;
            memset(cbData.value, 0x00, NN_ARRAY_SIZE(cbData.value));

            app_ble_cb.ble_client[clientIdx].write_pending = false;

            break;
        case BSA_BLE_CL_NOTIF_EVT:
            if (pData->cli_notif.len > NN_ARRAY_SIZE(cbData.value))
            {
                NN_SDK_LOG("[bluetooth] %s: Received too large data. %d Bytes for %d Bytes buffer.\n", NN_CURRENT_FUNCTION_NAME, pData->cli_notif.len, NN_ARRAY_SIZE(cbData.value));
                cbData.status       = BT_ERR_BAD_RESPONSE;
                cbData.length       = 0;
                memset(cbData.value, 0x00, NN_ARRAY_SIZE(cbData.value));
            }
            else
            {
                cbData.length = pData->cli_notif.len;
                memcpy(cbData.value, pData->cli_notif.value, pData->cli_notif.len);

                if(operation == GattOperationType_Indicate)
                {
                    responseData.connId                         = cbData.connId;
                    responseData.serviceId.uuid                 = cbData.serviceUuid;
                    responseData.serviceId.instanceId           = pData->cli_notif.char_id.srvc_id.id.inst_id;
                    responseData.characteristicId.uuid          = cbData.charcteristicUuid;
                    responseData.characteristicId.instanceId    = pData->cli_notif.char_id.char_id.inst_id;
                    responseData.isPrimaryService               = pData->cli_notif.char_id.srvc_id.is_primary;
                }
            }
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    for (const auto& dataPath : g_GattDataPathFilter)
    {
        if (dataPath.inUse && dataPath.uuid == cbData.serviceUuid)
        {
            switch (dataPath.type)
            {
            case HAL_QUEUE_BLE:
//                BTHAL_IF_BLE_DEBUG("Gatt operation callback for HAL_QUEUE_BLE");
                if (BluetoothHalInsertToQueue(&cbData, sizeof(InfoFromLeGattOperationCallback), HAL_QUEUE_BLE, CALLBACK_TYPE_LE_CLIENT_GATT_OP) >= 0)
                {
                    if (cbData.operation == GattOperationType_Indicate)
                    {
                        BluetoothHalInsertToQueue(&responseData,
                            sizeof(InfoFromLeGattIndicationResponseNeededCallback),
                            HAL_QUEUE_BLE,
                            CALLBACK_TYPE_LE_CLIENT_GATT_INDICATION_RESPONSE_NEEDED);
                    }
                }
                return;
            case HAL_QUEUE_BLE_CORE:
//                BTHAL_IF_BLE_DEBUG("Gatt operation callback for HAL_QUEUE_BLE_CORE");
                if (BluetoothHalInsertToQueue(&cbData, sizeof(InfoFromLeGattOperationCallback), HAL_QUEUE_BLE_CORE, CALLBACK_TYPE_LE_CLIENT_GATT_OP) >= 0)
                {
                    if (cbData.operation == GattOperationType_Indicate)
                    {
                        BluetoothHalInsertToQueue(&responseData,
                            sizeof(InfoFromLeGattIndicationResponseNeededCallback),
                            HAL_QUEUE_BLE_CORE,
                            CALLBACK_TYPE_LE_CLIENT_GATT_INDICATION_RESPONSE_NEEDED);
                    }
                }
                return;
            case HAL_QUEUE_BLE_HID:
//                BTHAL_IF_BLE_DEBUG("Gatt operation callback for HAL_QUEUE_BLE_HID");
                if (BluetoothHalInsertToQueue(&cbData, sizeof(InfoFromLeGattOperationCallback), HAL_QUEUE_BLE_HID, CALLBACK_TYPE_LE_CLIENT_GATT_OP) >= 0)
                {
                    if (cbData.operation == GattOperationType_Indicate)
                    {
                        BluetoothHalInsertToQueue(&responseData,
                            sizeof(InfoFromLeGattIndicationResponseNeededCallback),
                            HAL_QUEUE_BLE_HID,
                            CALLBACK_TYPE_LE_CLIENT_GATT_INDICATION_RESPONSE_NEEDED);
                    }
                }
                return;
            default:
                NN_UNEXPECTED_DEFAULT;
            }
        }
    }

    NN_SDK_LOG("[bluetooth] Unhandled BLE GATT operation. Ignored.\n");
    NN_SDK_LOG("[bluetooth] Connection ID: %d   Operation: %d\n", cbData.connId, cbData.operation);
    NN_SDK_LOG("[bluetooth] Service UUID       : %s\n", toString((uint8_t*)(&cbData.serviceUuid.uu), cbData.serviceUuid.length));
    NN_SDK_LOG("[bluetooth] Characteristic UUID: %s\n", toString((uint8_t*)(&cbData.charcteristicUuid.uu), cbData.charcteristicUuid.length));
    NN_SDK_LOG("[bluetooth] Descriptor UUID    : %s\n", toString((uint8_t*)(&cbData.descriptorUuid.uu), cbData.descriptorUuid.length));

    return;
}// NOLINT(impl/function_size)

/*******************************************************************************
**
** Function:        LeClientServiceDiscoveryCallback
**
** Description:     Callback of client service discovery result
**
** Parameters:      [in] tBSA_BLE_EVT event     : event type
** Parameters:      [in] tBSA_BLE_MSG* pData    : event data
**
** Returns:         void
**
*******************************************************************************/
static void LeClientServiceDiscoveryCallback(tBSA_BLE_EVT event, tBSA_BLE_MSG *pData)
{
    BTHAL_IF_BLE_DEBUG("called.");
    InfoFromLeClientGattServiceDiscoveryCallback cbData;
    uint32_t connId = 0;
    int clientIdx = -1;

    switch (event)
    {
    case BSA_BLE_CL_SEARCH_RES_EVT:
        BTHAL_IF_BLE_DEBUG("BSA_BLE_CL_SEARCH_RES_EVT");
        connId = Find32bitConnectionId(pData->cli_search_res.conn_id);
        break;
    case BSA_BLE_CL_SEARCH_CMPL_EVT:
        BTHAL_IF_BLE_DEBUG("BSA_BLE_CL_SEARCH_CMPL_EVT");
        connId = Find32bitConnectionId(pData->cli_search_cmpl.conn_id);
        break;
    case BSA_BLE_CL_CACHE_LOAD_EVT:
        BTHAL_IF_BLE_DEBUG("BSA_BLE_CL_CACHE_LOAD_EVT");
        connId = Find32bitConnectionId(pData->cli_cache_load.conn_id);
        break;
    case BSA_BLE_CL_CACHE_SAVE_EVT:
        BTHAL_IF_BLE_DEBUG("BSA_BLE_CL_CACHE_SAVE_EVT");
        connId = Find32bitConnectionId(pData->cli_cache_save.conn_id);
        break;
    case BSA_BLE_CL_CACHE_SAVE_CMPL_EVT:
        BTHAL_IF_BLE_DEBUG("BSA_BLE_CL_CACHE_SAVE_CMPL_EVT");
        connId = Find32bitConnectionId(pData->cli_cache_save_cmpl.conn_id);
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    clientIdx = FindClientBy32bitConnectionId(connId);
    if (clientIdx < 0)
    {
        NN_SDK_LOG("[bluetooth] %s: Error. Client not found. connId: %d\n", __func__, connId);

        cbData.status           = BT_ERR_CLIENT;
        cbData.connId           = BSA_BLE_INVALID_CONN;
        cbData.numAttributes    = 0;
        memset(&cbData.attributes, 0, sizeof(cbData.attributes));
    }
    else
    {
        switch (event)
        {
        case BSA_BLE_CL_SEARCH_RES_EVT:
            g_IsServiceDiscovered = true;
            // Currently not informing the result to upper layer since it includes only Gatt Service
            break;
        case BSA_BLE_CL_SEARCH_CMPL_EVT:
            if (!g_IsServiceDiscovered)
            {
                NN_SDK_LOG("[bluetooth] %s: Failed to discover service. Service not found.\n", __func__);
                g_IsServiceDiscovered = true;
            }
            // Currently not informing the result to upper layer as BSA_BLE_CL_SEARCH_RES_EVT
            break;
        case BSA_BLE_CL_CACHE_LOAD_EVT:
        {
            Btbdaddr serverAddress;
            copyBdAddr(serverAddress.address, pData->cli_cache_load.bd_addr);
            BtHalLeClientLoadAttributeCache(app_ble_cb.ble_client[clientIdx].conn_id, &serverAddress);
            break;
        }
        case BSA_BLE_CL_CACHE_SAVE_EVT:
        {
            int attrIndex           = 0;
            cbData.status           = BT_OK;
            cbData.connId           = connId;
            cbData.numAttributes    = pData->cli_cache_save.num_attr;
            copyBdAddr(cbData.address.address, pData->cli_cache_save.bd_addr);

            for (attrIndex = 0; attrIndex < pData->cli_cache_save.num_attr; ++attrIndex)
            {
                memcpy(&(cbData.attributes[attrIndex].id.uuid), &(pData->cli_cache_save.attr[attrIndex].uuid), sizeof(GattAttributeUuid));
                cbData.attributes[attrIndex].id.uuid.length = (nn::bluetooth::GattAttributeUuidLength)pData->cli_cache_save.attr[attrIndex].uuid.len;
                cbData.attributes[attrIndex].id.instanceId = pData->cli_cache_save.attr[attrIndex].id;
                cbData.attributes[attrIndex].handle = pData->cli_cache_save.attr[attrIndex].s_handle;
                cbData.attributes[attrIndex].endGroupHandle = pData->cli_cache_save.attr[attrIndex].e_handle;
                cbData.attributes[attrIndex].type = (GattAttributeType)(pData->cli_cache_save.attr[attrIndex].attr_type);
                cbData.attributes[attrIndex].property = pData->cli_cache_save.attr[attrIndex].prop;
                cbData.attributes[attrIndex].isPrimaryService = pData->cli_cache_save.attr[attrIndex].is_primary;
            }

            BluetoothHalInsertToQueue(&cbData, sizeof(InfoFromLeClientGattServiceDiscoveryCallback), HAL_QUEUE_BLE_CORE, CALLBACK_TYPE_LE_CLIENT_GATT_SERVICE_DISCOVERY);

            break;
        }
        case BSA_BLE_CL_CACHE_SAVE_CMPL_EVT:
        {
            cbData.status           = BT_OK;
            cbData.connId           = connId;
            cbData.numAttributes    = 0;
            memset(cbData.attributes, 0, sizeof(GattSdpAttribute));
            copyBdAddr(cbData.address.address, pData->cli_cache_save_cmpl.bd_addr);
            BluetoothHalInsertToQueue(&cbData, sizeof(InfoFromLeClientGattServiceDiscoveryCallback), HAL_QUEUE_BLE_CORE, CALLBACK_TYPE_LE_CLIENT_GATT_SERVICE_DISCOVERY_STATE_CHANGED);
            break;
        }
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
}

/*******************************************************************************
**
** Function:        LeClientProfileCback
**
** Description:     BLE client general profile callback.
**
** Parameters:      tBSA_BLE_EVT event  : event type
** Parameters:      tBSA_BLE_MSG        : event data
**
** Returns:         void
**
*******************************************************************************/
static void LeClientProfileCback(tBSA_BLE_EVT event,  tBSA_BLE_MSG *pData)
{
//    BTHAL_IF_BLE_DEBUG("called.");

    int clientIdx = -1;

    switch (event)
    {
        case BSA_BLE_CL_DEREGISTER_EVT:         // BtHalLeDeregisterClient
            LeClientStateChangedCallback(event, pData);
            break;
        case BSA_BLE_CL_OPEN_EVT:               // BtHalLeClientOpenConnection
        case BSA_BLE_CL_CANCEL_OPEN_EVT:        // BtHalLeClientCancelOpenConnection
        case BSA_BLE_CL_CLOSE_EVT:              // BtHalLeClientCloseConnection
            LeClientConnectionStateChangedCallback(event, pData);
            break;
        case BSA_BLE_CL_READ_EVT:               // BtHalLeClientReadCharacteristic, BtHalLeClientReadDescriptor
        case BSA_BLE_CL_WRITE_EVT:              // BtHalLeClientWriteCharacteristic, BtHalLeClientWriteDescriptor
        case BSA_BLE_CL_NOTIF_EVT:              // Notification, Indication
            LeClientGattOperationCallback(event, pData);
            break;
        case BSA_BLE_CL_SEARCH_RES_EVT:         // BtHalLeClientDiscoverService (discover a specific service. When discovering all services, this event occurs for multiple times.)
        case BSA_BLE_CL_SEARCH_CMPL_EVT:        // BtHalLeClientDiscoverService (complete discovering all services the server provides. Notify only completion.)
        case BSA_BLE_CL_CACHE_SAVE_EVT:
        case BSA_BLE_CL_CACHE_LOAD_EVT:
        case BSA_BLE_CL_CACHE_SAVE_CMPL_EVT:
            LeClientServiceDiscoveryCallback(event, pData);
        break;
        case BSA_BLE_CL_CFG_MTU_EVT:
        {
            BTHAL_IF_BLE_DEBUG("BSA_BLE_CL_CFG_MTU_EVT");
            InfoFromLeClientMtuConfigurationCallback cbData;

            uint32_t connId = Find32bitConnectionId(pData->cli_cfg_mtu.conn_id);

            clientIdx = FindClientBy32bitConnectionId(connId);

            if (clientIdx < 0)
            {
                NN_SDK_LOG("[bluetooth] %s: Error. Client not found. connId: %d\n", __func__, connId);

                cbData.status   = BT_ERR_CLIENT;
                cbData.connId   = BleInvalidConnectionHandle;
                cbData.mtu      = 0;
            }
            else
            {
                cbData.status   = LeConvertBsaStatus(pData->cli_cfg_mtu.status);
                cbData.connId   = connId;
                cbData.mtu      = pData->cli_cfg_mtu.mtu;
            }

            BluetoothHalInsertToQueue(&cbData, sizeof(InfoFromLeClientMtuConfigurationCallback), HAL_QUEUE_BLE_CORE, CALLBACK_TYPE_LE_CLIENT_CONFIGURE_MTU);
        }
        break;
        case BSA_BLE_CL_APCF_ENABLE_EVT:
        {
            BTHAL_IF_BLE_DEBUG("BSA_BLE_CL_APCF_ENABLE_EVT");
            InfoFromLeScanFilterStateChangedCallback cbData;

            cbData.status = LeConvertBsaStatus(pData->cli_apcf_enable.status);

            cbData.operation = pData->cli_apcf_enable.enable ? BLE_SCAN_FILTER_OP_ENABLE : BLE_SCAN_FILTER_OP_DISABLE;

            BluetoothHalInsertToQueue(&cbData, sizeof(InfoFromLeScanFilterStateChangedCallback), HAL_QUEUE_BLE_CORE, CALLBACK_TYPE_LE_SCAN_FILTER_STATE_CHANGED);
        }
        break;
        case BSA_BLE_CL_APCF_CFG_EVT:
        {
            BTHAL_IF_BLE_DEBUG("BSA_BLE_CL_APCF_CFG_EVT");

            InfoFromLeScanFilterStateChangedCallback cbData;

            cbData.status = LeConvertBsaStatus(pData->cli_apcf_cfg.status);
            cbData.operation = (BleScanFilterOperation)(pData->cli_apcf_cfg.action);

            BluetoothHalInsertToQueue(&cbData, sizeof(InfoFromLeScanFilterStateChangedCallback), HAL_QUEUE_BLE_CORE, CALLBACK_TYPE_LE_SCAN_FILTER_STATE_CHANGED);
        }
        break;
        default:
            NN_SDK_LOG("[bluetooth] %s Unknown event: %d\n", __func__, event);
            break;
    }
} // NOLINT(impl/function_size)

/*******************************************************************************
**
** Function:        BtHalLeClientRegister
**
** Description:     Register BLE client application
**
** Parameters:      [in] tBT_UUID uuid: an identifier of client application
**
** Returns:         BluetoothLeStatus
**
** Callbacks:       InfoFromLeAppStateChangedCallback
**
*******************************************************************************/
BluetoothLeStatus BtHalLeClientRegister(tBT_UUID uuid)
{
    BTHAL_IF_BLE_DEBUG("called.");

    tBSA_STATUS status;
    tBSA_BLE_CL_REGISTER ble_appreg_param;
    InfoFromLeAppStateChangedCallback cbData;

    int clientIdx = FindClientFreeSpace();
    if(clientIdx < 0)
    {
        NN_SDK_LOG("[bluetooth] %s: BLE client is already registered upto maximum.\n", __func__);
        return BT_ERR_CLIENT;
    }

    status = BSA_BleClAppRegisterInit(&ble_appreg_param);
    if(status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to register BLE client. status: %d\n", __func__, status);
        return LeConvertBsaStatus(status);
    }

    ble_appreg_param.uuid = uuid;
    ble_appreg_param.p_cback = LeClientProfileCback;

    status = BSA_BleClAppRegister(&ble_appreg_param);
    if(status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to register BLE client. status: %d\n", __func__, status);
        return LeConvertBsaStatus(status);
    }

    app_ble_cb.ble_client[clientIdx].enabled = true;
    app_ble_cb.ble_client[clientIdx].client_if = ble_appreg_param.client_if;
    app_ble_cb.ble_client[clientIdx].bsa_conn_id = BSA_BLE_INVALID_CONN;
    app_ble_cb.ble_client[clientIdx].conn_id = BleInvalidConnectionHandle;

    DisplayClient();

    cbData.status = LeConvertBsaStatus(status);
    cbData.clientIf = ble_appreg_param.client_if;
    cbData.isRegistered = true;

    BluetoothHalInsertToQueue(&cbData, sizeof(InfoFromLeAppStateChangedCallback), HAL_QUEUE_BLE_CORE, CALLBACK_TYPE_LE_CLIENT_STATE_CHANGED);

    return LeConvertBsaStatus(status);
}

/*******************************************************************************
**
** Function:        BtHalLeClientDeregister
**
** Description:     Deregister BLE client application
**
** Parameters:      [in] tBSA_BLE_IF clientIf : identifier of client application
**
** Returns:         BluetoothLeStatus
**
** Callbacks:       InfoFromLeAppStateChangedCallback
**
*******************************************************************************/
BluetoothLeStatus BtHalLeClientDeregister(tBSA_BLE_IF clientIf)
{
    BTHAL_IF_BLE_DEBUG("called.");

    tBSA_STATUS status;
    tBSA_BLE_CL_DEREGISTER ble_appdereg_param;

    int clientIdx = FindClientByClientInterface(clientIf);
    if(clientIdx < 0)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to deregister BLE client. Client not found.\n", __func__);
        return BT_ERR_CLIENT;
    }

    status = BSA_BleClAppDeregisterInit(&ble_appdereg_param);
    if(status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to deregister BLE client. status: %d\n", __func__, status);
        return LeConvertBsaStatus(status);
    }

    ble_appdereg_param.client_if = app_ble_cb.ble_client[clientIdx].client_if;

    status = BSA_BleClAppDeregister(&ble_appdereg_param);
    if(status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to deregister BLE client. status: %d\n", __func__, status);
        return LeConvertBsaStatus(status);
    }

    return LeConvertBsaStatus(status);
}

/*******************************************************************************
**
** Function:        BtHalLeClientDeregisterAll
**
** Description:     Deregister all client applications.
**                  Internally call BtHalLeClientDeregister for all client applications.
**
** Parameters:      None
**
** Returns:         BluetoothLeStatus
**
** Callbacks:       InfoFromLeAppStateChangedCallback
**
*******************************************************************************/
BluetoothLeStatus BtHalLeClientDeregisterAll()
{
    BTHAL_IF_BLE_DEBUG("called.");

    for(int i = 0; i < BLE_CLIENT_MAX; ++i)
    {
        if(app_ble_cb.ble_client[i].enabled)
        {
            BluetoothLeStatus status = BtHalLeClientDeregister(app_ble_cb.ble_client[i].client_if);
            if(status != BT_OK)
            {
                return status;
            }
        }
    }

    return BT_OK;
}

/*******************************************************************************
**
** Function:        BtHalLeClientDiscoverService
**
** Description:     Discover services from profile cache.
**                  Currently, does not included Gatt characteristics neither descriptors.
**
** Parameters:      None
**
** Returns:         BluetoothLeStatus
**
** Callbacks:       Currently, none
**
*******************************************************************************/
BluetoothLeStatus BtHalLeClientDiscoverService(uint32_t connId, tBT_UUID filterUuid)
{
    BTHAL_IF_BLE_DEBUG("called.");

    tBSA_STATUS status;
    tBSA_BLE_CL_SEARCH ble_search_param;

    status = BSA_BleClSearchInit(&ble_search_param);
    if(status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to discover service. status: %d\n", __func__, status);
        return LeConvertBsaStatus(status);
    }

    int clientIdx = FindClientBy32bitConnectionId(connId);
    if(clientIdx < 0)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to discover service. Client not found.\n", __func__);
        return BT_ERR_CLIENT;
    }

    ble_search_param.conn_id = app_ble_cb.ble_client[clientIdx].bsa_conn_id;
    ble_search_param.uuid = filterUuid;

    status = BSA_BleClSearch(&ble_search_param);
    if(status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to discover service. status: %d\n", __func__, status);
        return LeConvertBsaStatus(status);
    }

    g_IsServiceDiscovered = false;

    return LeConvertBsaStatus(status);
}

/*******************************************************************************
**
** Function:        BtHalLeClientReadCharacteristic
**
** Description:     Read a characteristic of a connected BLE server
**
** Parameters:      [in] uint16_t connId                : connection ID of the target server
** Parameters:      [in] tBT_UUID srvcUuid              : attribute UUID of the service which the target characeteristic belongs to
** Parameters:      [in] uint8_t srvcInstId             : instance ID of the service
** Parameters:      [in] bool isPrimarySrvc             : flag indicating if the service is primary or not
** Parameters:      [in] tBT_UUID charUuid              : attribute UUID of the target characteristic
** Parameters:      [in] uint8_t charInstId             : instance ID of the characteristic
** Parameters:      [in] uint8_t authenticationType     : authentication type to read the characteristic
**
** Returns          BluetoothLeStatus
**
** Callbacks:       InfoFromLeGattOperationCallback
**
*******************************************************************************/
BluetoothLeStatus BtHalLeClientReadCharacteristic(uint32_t connId,
                                                  tBT_UUID srvcUuid, uint8_t srvcInstId, bool isPrimarySrvc,
                                                  tBT_UUID charUuid, uint8_t charInstId,
                                                  uint8_t authenticationType)
{
    BTHAL_IF_BLE_DEBUG("called.");

    tBSA_STATUS status;
    tBSA_BLE_CL_READ ble_read_param;

    int clientIdx = FindClientBy32bitConnectionId(connId);
    if(clientIdx < 0)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to read characteristic. Client not found.\n", __func__);
        return BT_ERR_CLIENT;
    }

    if (app_ble_cb.ble_client[clientIdx].read_pending || app_ble_cb.ble_client[clientIdx].write_pending || app_ble_cb.ble_client[clientIdx].congested)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to read characteristic. Client busy (Read Pending: %d, Write Pending: %d, Congested: %d).\n",
            __func__,
            app_ble_cb.ble_client[clientIdx].read_pending,
            app_ble_cb.ble_client[clientIdx].write_pending,
            app_ble_cb.ble_client[clientIdx].congested);
        return BT_ERR_BUSY;
    }

    status = BSA_BleClReadInit(&ble_read_param);
    if(status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to read characteristic. status: %d\n", __func__, status);
        return LeConvertBsaStatus(status);
    }

    ble_read_param.char_id.srvc_id.id.uuid = srvcUuid;
    ble_read_param.char_id.srvc_id.id.inst_id = srvcInstId;
    ble_read_param.char_id.srvc_id.is_primary = isPrimarySrvc ? 1 : 0;

    ble_read_param.char_id.char_id.uuid = charUuid;
    ble_read_param.char_id.char_id.inst_id = charInstId;

    ble_read_param.descr = false;
    ble_read_param.conn_id = app_ble_cb.ble_client[clientIdx].bsa_conn_id;
    ble_read_param.auth_req = authenticationType;

    status = BSA_BleClRead(&ble_read_param);
    if(status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to read characteristic. status: %d\n", __func__, status);
        return LeConvertBsaStatus(status);
    }

    app_ble_cb.ble_client[clientIdx].read_pending = true;

    return LeConvertBsaStatus(status);
}

/*******************************************************************************
**
** Function         BtHalLeClientReadDescriptor
**
** Description:     Read a descriptor of a connected BLE server
**
** Parameters:      [in] uint16_t connId                : connection ID of the target server
** Parameters:      [in] tBT_UUID srvcUuid              : attribute UUID of the service which the target descriptor belongs to
** Parameters:      [in] uint8_t srvcInstId             : instance ID of the service
** Parameters:      [in] bool isPrimarySrvc             : flag indicating if the service is primary or not
** Parameters:      [in] tBT_UUID charUuid              : attribute UUID of the characteristic which the target descriptor belongs to
** Parameters:      [in] uint8_t charInstId             : instance ID of the characteristic
** Parameters:      [in] tBT_UUID descrUuid             : attribute UUID of the target descriptor
** Parameters:      [in] uint8_t authenticationType     : authentication type to read the characteristic
**
** Returns          BluetoothLeStatus
**
** Callbacks:       InfoFromLeGattOperationCallback
**
*******************************************************************************/
BluetoothLeStatus BtHalLeClientReadDescriptor(uint32_t connId,
                                              tBT_UUID srvcUuid, bool isPrimarySrvc,
                                              tBT_UUID charUuid,
                                              tBT_UUID descrUuid,
                                              uint8_t authenticationType)
{
    BTHAL_IF_BLE_DEBUG("called.");

    tBSA_STATUS status;
    tBSA_BLE_CL_READ ble_read_param;

    int clientIdx = FindClientBy32bitConnectionId(connId);
    if(clientIdx < 0)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to read descriptor. Client not found.\n", __func__);
        return BT_ERR_CLIENT;
    }

    if (app_ble_cb.ble_client[clientIdx].read_pending || app_ble_cb.ble_client[clientIdx].write_pending || app_ble_cb.ble_client[clientIdx].congested)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to read descriptor. Client busy (Read Pending: %d, Write Pending: %d, Congested: %d).\n",
            __func__,
            app_ble_cb.ble_client[clientIdx].read_pending,
            app_ble_cb.ble_client[clientIdx].write_pending,
            app_ble_cb.ble_client[clientIdx].congested);
        return BT_ERR_BUSY;
    }

    status = BSA_BleClReadInit(&ble_read_param);
    if(status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to read descriptor. status: %d\n", __func__, status);
        return LeConvertBsaStatus(status);
    }

    ble_read_param.descr_id.char_id.srvc_id.id.uuid = srvcUuid;
    ble_read_param.descr_id.char_id.srvc_id.is_primary = isPrimarySrvc ? 1 : 0;

    ble_read_param.descr_id.char_id.char_id.uuid = charUuid;

    ble_read_param.descr_id.descr_id.uuid = descrUuid;
    ble_read_param.descr = true;

    ble_read_param.conn_id = app_ble_cb.ble_client[clientIdx].bsa_conn_id;
    ble_read_param.auth_req = authenticationType;

    status = BSA_BleClRead(&ble_read_param);
    if(status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to read descriptor. status: %d\n", __func__, status);
        return LeConvertBsaStatus(status);
    }

    app_ble_cb.ble_client[clientIdx].read_pending = true;

    return LeConvertBsaStatus(status);
}

/*******************************************************************************
**
** Function         BtHalLeClientWriteCharacteristic
**
** Description      Write to a characteristic of a connected BLE server
**
** Parameters:      [in] uint16_t connId                : connection ID of the target server
** Parameters:      [in] tBT_UUID srvcUuid              : attribute UUID of the service which the target characeteristic belongs to
** Parameters:      [in] uint8_t srvcInstId             : instance ID of the service
** Parameters:      [in] bool isPrimarySrvc             : flag indicating if the service is primary or not
** Parameters:      [in] tBT_UUID charUuid              : attribute UUID of the target characteristic
** Parameters:      [in] uint8_t charInstId             : instance ID of the characteristic
** Parameters:      [in] uint8_t* pData                 : data to write
** Parameters:      [in] uint16_t len                   : data length in byte
** Parameters:      [in] uint8_t authenticationType     : authentication type to read the characteristic
** Parameters:      [in] bool withResponse              : flag indicating if the operation waits a confirmation response from the server
**
** Returns          BluetoothLeStatus
**
** Callbacks:       InfoFromLeGattOperationCallback
**
*******************************************************************************/
BluetoothLeStatus BtHalLeClientWriteCharacteristic(uint32_t connId,
                                                   tBT_UUID srvcUuid, uint8_t srvcInstId, bool isPrimarySrvc,
                                                   tBT_UUID charUuid, uint8_t charInstId,
                                                   const uint8_t* pData, uint16_t len,
                                                   uint8_t authenticationType,
                                                   bool withResponse)
{
    BTHAL_IF_BLE_DEBUG("called.");

    tBSA_STATUS status;
    tBSA_BLE_CL_WRITE ble_write_param;

    if(BLE_WRITE_MAX < len)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to write characteristic. Data too large.\n", __func__);
        return BT_ERR_BAD_PARAM;
    }

    int clientIdx = FindClientBy32bitConnectionId(connId);
    if(clientIdx < 0)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to write characteristic. Client not found.\n", __func__);
        return BT_ERR_CLIENT;
    }

    if (app_ble_cb.ble_client[clientIdx].read_pending || app_ble_cb.ble_client[clientIdx].write_pending || app_ble_cb.ble_client[clientIdx].congested)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to write characteristic. Client busy (Read Pending: %d, Write Pending: %d, Congested: %d).\n",
            __func__,
            app_ble_cb.ble_client[clientIdx].read_pending,
            app_ble_cb.ble_client[clientIdx].write_pending,
            app_ble_cb.ble_client[clientIdx].congested);
        return BT_ERR_BUSY;
    }

    status = BSA_BleClWriteInit(&ble_write_param);
    if(status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to write characteristic. status: %d\n", __func__, status);
        return LeConvertBsaStatus(status);
    }

    ble_write_param.char_id.srvc_id.id.uuid = srvcUuid;
    ble_write_param.char_id.srvc_id.id.inst_id = srvcInstId;
    ble_write_param.char_id.srvc_id.is_primary = isPrimarySrvc ? 1 : 0;

    ble_write_param.char_id.char_id.uuid = charUuid;
    ble_write_param.char_id.char_id.inst_id = charInstId;

    ble_write_param.len = len;

    memcpy(ble_write_param.value, pData, ble_write_param.len);

    ble_write_param.write_type = withResponse ? BTA_GATTC_TYPE_WRITE : BTA_GATTC_TYPE_WRITE_NO_RSP;

    ble_write_param.descr = false;
    ble_write_param.conn_id = app_ble_cb.ble_client[clientIdx].bsa_conn_id;
    ble_write_param.auth_req = authenticationType;

    status = BSA_BleClWrite(&ble_write_param);
    if(status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to write characteristic. status: %d\n", __func__, status);
        return LeConvertBsaStatus(status);
    }

    app_ble_cb.ble_client[clientIdx].write_pending = true;

    return LeConvertBsaStatus(status);
}

/*******************************************************************************
**
** Function         BtHalLeClientWriteDescriptor
**
** Description:     Write to a descriptor of a connected BLE server
**
** Parameters:      [in] uint16_t connId                : connection ID of the target server
** Parameters:      [in] tBT_UUID srvcUuid              : attribute UUID of the service which the target descriptor belongs to
** Parameters:      [in] uint8_t srvcInstId             : instance ID of the service
** Parameters:      [in] bool isPrimarySrvc             : flag indicating if the service is primary or not
** Parameters:      [in] tBT_UUID charUuid              : attribute UUID of the characteristic which the target descriptor belongs to
** Parameters:      [in] uint8_t charInstId             : instance ID of the characteristic
** Parameters:      [in] tBT_UUID descrUuid             : attribute UUID of the target descriptor
** Parameters:      [in] uint8_t* pData                 : data to write
** Parameters:      [in] uint16_t len                   : data length in byte
** Parameters:      [in] uint8_t authenticationType     : authentication type to read the characteristic
**
** Returns          BluetoothLeStatus
**
** Callbacks:       InfoFromLeGattOperationCallback
**
*******************************************************************************/
BluetoothLeStatus BtHalLeClientWriteDescriptor(uint32_t connId,
                                               tBT_UUID srvcUuid, bool isPrimarySrvc,
                                               tBT_UUID charUuid,
                                               tBT_UUID descrUuid,
                                               const uint8_t* pData, uint16_t len,
                                               uint8_t authenticationType)
{
    BTHAL_IF_BLE_DEBUG("called.");

    tBSA_STATUS status;
    tBSA_BLE_CL_WRITE ble_write_param;

    if (BLE_WRITE_MAX < len)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to write descriptor. Data too large.\n", __func__);
        return BT_ERR_BAD_PARAM;
    }

    int clientIdx = FindClientBy32bitConnectionId(connId);
    if(clientIdx < 0)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to write descriptor. Client not found.\n", __func__);
        return BT_ERR_CLIENT;
    }

    if (app_ble_cb.ble_client[clientIdx].read_pending || app_ble_cb.ble_client[clientIdx].write_pending || app_ble_cb.ble_client[clientIdx].congested)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to write descriptor. Client busy (Read Pending: %d, Write Pending: %d, Congested: %d).\n",
            __func__,
            app_ble_cb.ble_client[clientIdx].read_pending,
            app_ble_cb.ble_client[clientIdx].write_pending,
            app_ble_cb.ble_client[clientIdx].congested);
        return BT_ERR_BUSY;
    }

    status = BSA_BleClWriteInit(&ble_write_param);
    if(status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to write descriptor. status: %d\n", __func__, status);
        return LeConvertBsaStatus(status);
    }

    ble_write_param.descr_id.char_id.srvc_id.id.uuid = srvcUuid;
    ble_write_param.descr_id.char_id.srvc_id.is_primary = isPrimarySrvc ? 1 : 0;

    ble_write_param.descr_id.char_id.char_id.uuid = charUuid;

    ble_write_param.descr_id.descr_id.uuid = descrUuid;

    ble_write_param.len = len;
    memcpy(ble_write_param.value, pData, ble_write_param.len);
    ble_write_param.descr = true;

    ble_write_param.write_type = BTA_GATTC_TYPE_WRITE;

    ble_write_param.conn_id = app_ble_cb.ble_client[clientIdx].bsa_conn_id;
    ble_write_param.auth_req = authenticationType;

    status = BSA_BleClWrite(&ble_write_param);
    if(status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to write characteristic. status: %d\n", __func__, status);
        return LeConvertBsaStatus(status);
    }

    app_ble_cb.ble_client[clientIdx].write_pending = true;

    return LeConvertBsaStatus(status);
}

/*******************************************************************************
**
** Function:        BtHalLeClientRegisterNotification
**
** Description:     Register notification to be received to BLE stack
**
** Parameters:      [in] tBSA_BLE_IF clientIf   : identifier of client application
** Parameters:      [in] tBT_UUID srvcUuid      : attribute UUID of the service which the target characteristic belongs to
** Parameters:      [in] uint8_t srvcInstId     : instance ID of the service
** Parameters:      [in] bool isPrimarySrvc     : flag indicating if the service is primary or not
** Parameters:      [in] tBT_UUID charUuid      : attribute UUID of the target characteristic
** Parameters:      [in] uint8_t charInstId     : instance ID of the characteristic
**
** Returns:         BluetoothLeStatus
**
** Callbacks:       None
**
*******************************************************************************/
BluetoothLeStatus BtHalLeClientRegisterNotification(uint32_t connId,
                                                    tBT_UUID srvcUuid, uint8_t srvcInstId, bool isPrimarySrvc,
                                                    tBT_UUID charUuid, uint8_t charInstId)
{
    BTHAL_IF_BLE_DEBUG("called.");

    BluetoothLeStatus ret = BT_OK;

    int clientIdx = FindClientBy32bitConnectionId(connId);
    if(clientIdx < 0)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to register notification. Client not found.\n", __func__);
        ret = BT_ERR_CLIENT;
    }
    else
    {
        ret = RegisterNotificationImpl(app_ble_cb.ble_client[clientIdx].client_if,
                                       app_ble_cb.ble_client[clientIdx].server_addr,
                                       srvcUuid, srvcInstId, isPrimarySrvc, charUuid, charInstId);
    }

    return ret;
}

/*******************************************************************************
**
** Function:        BtHalLeClientDeregisterNotification
**
** Description:     Deregister notification not to be received from BLE stack
**
** Parameters:      [in] tBSA_BLE_IF clientIf   : identifier of client application
** Parameters:      [in] tBT_UUID srvcUuid      : attribute UUID of the service which the target characteristic belongs to
** Parameters:      [in] uint8_t srvcInstId     : instance ID of the service
** Parameters:      [in] bool isPrimarySrvc     : flag indicating if the service is primary or not
** Parameters:      [in] tBT_UUID charUuid      : attribute UUID of the target characteristic
** Parameters:      [in] uint8_t charInstId     : instance ID of the characteristic
**
** Returns:         BluetoothLeStatus
**
** Callbacks:       None
**
*******************************************************************************/
BluetoothLeStatus BtHalLeClientDeregisterNotification(uint32_t connId,
                                                      tBT_UUID srvcUuid, uint8_t srvcInstId, bool isPrimarySrvc,
                                                      tBT_UUID charUuid, uint8_t charInstId)
{
    BTHAL_IF_BLE_DEBUG("called.");

    BluetoothLeStatus ret = BT_OK;

    int clientIdx = FindClientBy32bitConnectionId(connId);
    if(clientIdx < 0)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to deregister notification. Client not found.\n", __func__);
        ret = BT_ERR_CLIENT;
    }
    else
    {
        ret = DeregisterNotificationImpl(app_ble_cb.ble_client[clientIdx].client_if,
                                         app_ble_cb.ble_client[clientIdx].server_addr,
                                         srvcUuid, srvcInstId, isPrimarySrvc, charUuid, charInstId);
    }

    return ret;
}

/*******************************************************************************
**
** Function:        BtHalLeClientDeregisterNotificationAll
**
** Description:     Deregister all notifications not to be received from BLE stack
**
** Parameters:      [in] tBSA_BLE_IF clientIf   : identifier of client application
**
** Returns:         BluetoothLeStatus
**
** Callbacks:       None
**
*******************************************************************************/
BluetoothLeStatus BtHalLeClientDeregisterNotificationAll(tBSA_BLE_IF clientIf, const BluetoothAddress& serverBdAddr)
{
    BTHAL_IF_BLE_DEBUG("called.");

    BluetoothLeStatus status = BT_OK;

    int clientIdx = FindClientByClientInterface(clientIf);
    if(clientIdx < 0)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to deregister notification. Client not found.\n", __func__);
        status = BT_ERR_CLIENT;
    }
    else
    {
        tAPP_BLE_REGISTERED_NOTIFICATION registeredNotifications[BTA_GATTC_NOTIF_REG_MAX];

        nn::os::LockMutex(&g_MutexRegisteredNotification);
        memcpy(registeredNotifications, app_ble_cb.ble_client[clientIdx].registered_notification, sizeof(registeredNotifications));
        nn::os::UnlockMutex(&g_MutexRegisteredNotification);

        for (const auto& entry : registeredNotifications)
        {
            if (entry.is_registered)
            {
                BD_ADDR address;
                memcpy(&address, serverBdAddr.address, BD_ADDR_LEN);

                status = DeregisterNotificationImpl(clientIf,
                                                    address,
                                                    entry.service_uuid, entry.service_inst_id, entry.is_primary_service,
                                                    entry.char_uuid, entry.char_inst_id);

                if (status != BT_OK)
                {
                    break;
                }
            }
        }
    }

    return status;
}

/*******************************************************************************
**
** Function:        BtHalLeClientSendIndicationRsp
**
** Description:     Response to an indication from a BLE server
**
** Parameters:      [in] uint16_t connId    : connection ID of the target server
** Parameters:      [in] tBT_UUID srvcUuid  : attribute UUID of the service which the target characteristic belongs to
** Parameters:      [in] uint8_t srvcInstId : instance ID of the service
** Parameters:      [in] bool isPrimarySrvc : flag indicating if the service is primary or not
** Parameters:      [in] tBT_UUID charUuid  : attribute UUID of the target characteristic
** Parameters:      [in] uint8_t charInstId : instance ID of the characteristic
**
** Returns:         BluetoothLeStatus
**
** Callbacks:       None
**
*******************************************************************************/
BluetoothLeStatus BtHalLeClientSendIndicationRsp(uint32_t connId,
                                                 tBT_UUID srvcUuid, uint8_t srvcInstId, bool isPrimarySrvc,
                                                 tBT_UUID charUuid, uint8_t charInstId)
{
    BTHAL_IF_BLE_DEBUG("called.");

    tBSA_STATUS status;
    tBSA_BLE_CL_INDCONF ble_indresp_param;

    int clientIdx = FindClientBy32bitConnectionId(connId);
    if(clientIdx < 0)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to response to indication. Client not found.\n", __func__);
        return BT_ERR_CLIENT;
    }

    status = BSA_BleClIndConfInit(&ble_indresp_param);
    if(status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to response to indication. status: %d\n", __func__, status);
        return LeConvertBsaStatus(status);
    }

    ble_indresp_param.conn_id = app_ble_cb.ble_client[clientIdx].bsa_conn_id;

    ble_indresp_param.char_id.srvc_id.id.uuid = srvcUuid;
    ble_indresp_param.char_id.srvc_id.id.inst_id = srvcInstId;
    ble_indresp_param.char_id.srvc_id.is_primary = isPrimarySrvc;

    ble_indresp_param.char_id.char_id.uuid = charUuid;
    ble_indresp_param.char_id.char_id.inst_id = charInstId;

    status = BSA_BleClIndConf(&ble_indresp_param);
    if(status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to response to indication. status: %d\n", __func__, status);
        return LeConvertBsaStatus(status);
    }

    return LeConvertBsaStatus(status);
}

/*******************************************************************************
**
** Function:        BtHalLeClientOpenConnection
**
** Description:     Connect to a BLE server
**
** Parameters:      [in] tBSA_BLE_IF clientIf       : identifier of client application
** Parameters:      [in] Btbdaddr* pServerBdAddr    : BD Address of the target server
** Parameters:      [in] bool isDirectConnection    : flag indicating if direct or background connection
**
** Returns:         BluetoothLeStatus
**
** Callbacks:       InfoFromLeConnStateChangedCallback
**
*******************************************************************************/
BluetoothLeStatus BtHalLeClientOpenConnection(tBSA_BLE_IF clientIf,
                                              const Btbdaddr* pServerBdAddr,
                                              bool isDirectConnection)
{
    BTHAL_IF_BLE_DEBUG("called.");

    tBSA_STATUS status;
    tBSA_BLE_CL_OPEN ble_open_param;

    int clientIdx = FindClientByClientInterface(clientIf);
    if(clientIdx < 0)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to open connection. Client not found.\n", __func__);
        return BT_ERR_CLIENT;
    }

    if(app_ble_cb.ble_client[clientIdx].conn_id != BleInvalidConnectionHandle)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to open connection. Client already has a connection.", __func__);
        return BT_ERR_CLIENT;
    }

    status = BSA_BleClConnectInit(&ble_open_param);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to open connection. status: %d\n", __func__, status);
        return LeConvertBsaStatus(status);
    }

    ble_open_param.client_if = app_ble_cb.ble_client[clientIdx].client_if;
    ble_open_param.is_direct = isDirectConnection;
    copyBdAddr(app_ble_cb.ble_client[clientIdx].server_addr, pServerBdAddr->address);
    copyBdAddr(ble_open_param.bd_addr, pServerBdAddr->address);

    status = BSA_BleClConnect(&ble_open_param);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to open connection. status: %d\n", __func__, status);
        return LeConvertBsaStatus(status);
    }

    return LeConvertBsaStatus(status);
}

/*******************************************************************************
**
** Function:        BtHalLeClientCancelOpenConnection
**
** Description:     Cancel connection to a BLE server
**
** Parameters:      [in] tBSA_BLE_IF clientIf       : identifier of client application
** Parameters:      [in] Btbdaddr* pServerBdAddr    : BD Address of the target server
** Parameters:      [in] bool isDirectConnection    : flag indicating if direct or background connection
**
** Returns:         BluetoothLeStatus
**
** Callbacks:       InfoFromLeConnStateChangedCallback
**
*******************************************************************************/
BluetoothLeStatus BtHalLeClientCancelOpenConnection(tBSA_BLE_IF clientIf,
                                                    const Btbdaddr* pServerBdAddr,
                                                    bool isDirectConnection)
{
    BTHAL_IF_BLE_DEBUG("called.");

    tBSA_STATUS status;
    tBSA_BLE_CL_CANCEL_OPEN ble_cancel_open_param;

    int clientIdx = FindClientByClientInterface(clientIf);
    if(clientIdx < 0)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to cancel open connection. Client not found.\n", __func__);
        return BT_ERR_CLIENT;
    }

    if(app_ble_cb.ble_client[clientIdx].conn_id != BleInvalidConnectionHandle)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to cancel opne connection. Client already has a connection.", __func__);
        return BT_ERR_CLIENT;
    }

    status = BSA_BleClCancelConnectInit(&ble_cancel_open_param);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to cancle open connection. status: %d\n", __func__, status);
        return LeConvertBsaStatus(status);
    }

    ble_cancel_open_param.client_if = app_ble_cb.ble_client[clientIdx].client_if;
    ble_cancel_open_param.is_direct = isDirectConnection;
    copyBdAddr(app_ble_cb.ble_client[clientIdx].server_addr, pServerBdAddr->address);
    copyBdAddr(ble_cancel_open_param.bd_addr, pServerBdAddr->address);

    status = BSA_BleClCancelConnect(&ble_cancel_open_param);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to cancel open connection. status: %d\n", __func__, status);
        return LeConvertBsaStatus(status);
    }

    return LeConvertBsaStatus(status);
}

/*******************************************************************************
**
** Function:        BtHalLeClientCloseConnection
**
** Description:     Disconnect from a BLE server
**
** Parameters:      [in] uint16_t connId    : connection ID of the target server
**
** Returns:         BluetoothLeStatus
**
** Callbacks:       InfoFromLeConnStateChangedCallback
**
*******************************************************************************/
BluetoothLeStatus BtHalLeClientCloseConnection(uint32_t connId)
{
    BTHAL_IF_BLE_DEBUG("called.");

    tBSA_STATUS status;
    tBSA_BLE_CL_CLOSE ble_close_param;

    int clientIdx = FindClientBy32bitConnectionId(connId);
    if(clientIdx < 0)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to close connection. Client not found.\n", __func__);
        return BT_ERR_CLIENT;
    }

    if(app_ble_cb.ble_client[clientIdx].conn_id == BleInvalidConnectionHandle)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to close connection. Client has no connection.\n", __func__);
        return BT_ERR_CLIENT;
    }

    status = BSA_BleClCloseInit(&ble_close_param);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to close connection. status: %d\n", __func__, status);
        return LeConvertBsaStatus(status);
    }

    ble_close_param.conn_id = app_ble_cb.ble_client[clientIdx].bsa_conn_id;

    status = BSA_BleClClose(&ble_close_param);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to close connection. status: %d\n", __func__, status);
        return LeConvertBsaStatus(status);
    }

    return LeConvertBsaStatus(status);
}

/*******************************************************************************
**
** Function:        BtHalLeClientLoadAttributeCache
**
** Description:     Get GATT profile from a BLE server
**
** Parameters:      [in] uint16_t connId            : connection ID of the target server
** Parameters:      [in] Btbdaddr* pServerBdAddr    : BD Address of the target server
**
** Returns:         BluetoothLeStatus
**
** Callbacks:       InfoFromLeClientGattServiceDiscoveryCallback
**
*******************************************************************************/
BluetoothLeStatus BtHalLeClientLoadAttributeCache(uint32_t connId, const Btbdaddr* pServerBdAddr)
{
    BTHAL_IF_BLE_DEBUG("called.");

    tBSA_STATUS status;
    tBSA_BLE_CL_CACHE_LOAD cache_load_param;

    int clientIdx = FindClientBy32bitConnectionId(connId);
    if(clientIdx < 0)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to close connection. Client not found.\n", __func__);
        return BT_ERR_CLIENT;
    }

    if(app_ble_cb.ble_client[clientIdx].conn_id == BleInvalidConnectionHandle)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to close connection. Client has no connection.\n", __func__);
        return BT_ERR_CLIENT;
    }

    status = BSA_BleClCacheLoadInit(&cache_load_param);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to load attribute cache. status: %d\n", __func__, status);
        return LeConvertBsaStatus(status);
    }

    cache_load_param.conn_id = app_ble_cb.ble_client[clientIdx].bsa_conn_id;
    cache_load_param.num_attr = 0;
    cache_load_param.more = 0;
    copyBdAddr(cache_load_param.bd_addr, pServerBdAddr->address);

    status = BSA_BleClCacheLoad(&cache_load_param);
    if(status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to load attribute cache. status: %d\n", __func__, status);
        return LeConvertBsaStatus(status);
    }

    BTHAL_IF_BLE_DEBUG("[bluetooth] %s done\n", __func__);

    return LeConvertBsaStatus(status);
}

/*******************************************************************************
**
** Function:        BtHalLeClientConfigureMtu
**
** Description:     Start updating MTU
**
** Parameters:      [in] uint16_t connId    : connection ID of the target server
** Parameters:      [in] size_t mtu         : Prefferred MTU in Byte
**
** Returns:         BluetoothLeStatus
**
** Callbacks:       InfoFromLeClientMtuConfigurationCallback
**
*******************************************************************************/
BluetoothLeStatus BtHalLeClientConfigureMtu(uint32_t connId, const size_t mtu)
{
    BTHAL_IF_BLE_DEBUG("called.");

    tBSA_STATUS status;
    tBSA_BLE_CL_CFG_MTU ble_cfg_mtu_param;

    int clientIdx = FindClientBy32bitConnectionId(connId);
    if (clientIdx < 0)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to configure MTU. Client not found.\n", __func__);
        return BT_ERR_CLIENT;
    }

    if (app_ble_cb.ble_client[clientIdx].conn_id == BleInvalidConnectionHandle)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to configure MTU. Client has no connection.\n", __func__);
        return BT_ERR_CLIENT;
    }

    status = BSA_BleClCfgMTUInit(&ble_cfg_mtu_param);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] Failed to configure MTU. status: %d\n", status);
        return LeConvertBsaStatus(status);
    }

    ble_cfg_mtu_param.conn_id = app_ble_cb.ble_client[clientIdx].bsa_conn_id;;
    ble_cfg_mtu_param.mtu = mtu;

    status = BSA_BleClCfgMTU(&ble_cfg_mtu_param);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] Failed to configure MTU. status: %d\n", status);
        return LeConvertBsaStatus(status);
    }

    return LeConvertBsaStatus(status);
}

uint8_t activatedFilterConditions[16] = { 0x00 };
static BluetoothLeStatus BtHalLeClientActivateScanFilter(uint8_t filterIndex, BleAdType adType)
{
    BTHAL_IF_BLE_DEBUG("called.");

    tBSA_BLE_APCF_CFG bsa_ble_apcf_cfg;
    tBSA_STATUS status;
    uint8_t activateFilterCondition = activatedFilterConditions[filterIndex];

    status = BSA_BleApcfCfgInit(&bsa_ble_apcf_cfg);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] Failed to activate scan filter. status: %d\n", status);
        return LeConvertBsaStatus(status);
    }

    bsa_ble_apcf_cfg.filter_index                           = filterIndex;
    bsa_ble_apcf_cfg.cond_type                              = BSA_BLE_APCF_FILTER_SETTING;
    bsa_ble_apcf_cfg.action                                 = BSA_BLE_APCF_COND_ADD;
    bsa_ble_apcf_cfg.cond.filter_setting.rssi_high_thres    = 0x80;        // -128. Lowest Value.
    bsa_ble_apcf_cfg.cond.filter_setting.filter_logic_type  = BSA_BLE_APCF_FILT_LOGIC_AND;
    bsa_ble_apcf_cfg.cond.filter_setting.logic_type         = BSA_BLE_APCF_LIST_LOGIC_OR;

    switch (adType)
    {
    case BleAdType_ManufactureSpecificData:
        activateFilterCondition |= BSA_BLE_APCF_MANUF_NAME_CHECK;
        break;
    case BleAdType_IncompleteListOf16bitUuid:
    case BleAdType_CompleteListOf16bitUuid:
    case BleAdType_IncompleteListOf32bitUuid:
    case BleAdType_CompleteListOf32bitUuid:
    case BleAdType_IncompleteListOf128bitUuid:
    case BleAdType_CompleteListOf128bitUuid:
        activateFilterCondition |= BSA_BLE_APCF_SERV_UUID;
        break;
    default:
        NN_SDK_LOG("[bluetooth] Failed to activate scan filter. Unsupported AD Type (%d).\n", adType);
        return BT_BLE_ERR_INVALID_PARAM;
    }

    bsa_ble_apcf_cfg.cond.filter_setting.feat_seln |= activateFilterCondition;

    status = BSA_BleApcfCfg(&bsa_ble_apcf_cfg);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] Failed to activate scan filter. status: %d\n", status);
        return LeConvertBsaStatus(status);
    }

    activatedFilterConditions[filterIndex] = activateFilterCondition;

    return LeConvertBsaStatus(status);
}

/*******************************************************************************
**
** Function:        BtHalLeClientAddScanFilterCondition
**
** Description:     Add a scan filter condition to a scan filter
**
** Parameters:      [in] uint8_t filterIndex    : target filter index
** Parameters:      [in] BleAdType adType       : AD Type of filter condition
** Parameters:      [in] uint8_t* data          : an array of filter value
** Parameters:      [in] uint8_t dataLength     : length of filter value
** Parameters:      [in] uint8_t* mask          : mask for filter value
** Parameters:      [in] uint8_t maskLength     : length of mask
**
** Returns:         BluetoothLeStatus
**
** Callbacks:       InfoFromLeScanFilterStateChangedCallback
**
*******************************************************************************/
BluetoothLeStatus BtHalLeClientAddScanFilterCondition(uint8_t filterIndex, BleAdType adType, const uint8_t* pData, uint8_t dataLength, const uint8_t* pMask, uint8_t maskLength)
{
    tBSA_BLE_APCF_CFG bsa_ble_apcf_cfg;
    tBSA_STATUS status;

    if (dataLength > BSA_BLE_APCF_COND_MAX_LEN)
    {
        NN_SDK_LOG("[bluetooth] Failed to add scan filter. Data must be smaller than %d bytes\n", BSA_BLE_APCF_COND_MAX_LEN);
        return BT_BLE_ERR_INVALID_PARAM;
    }
    if (dataLength < maskLength)
    {
        NN_SDK_LOG("[bluetooth] Failed to add scan filter. Filter mask (%d bytes) must be smaller than data (%d bytes).\n", maskLength, dataLength);
        return BT_BLE_ERR_INVALID_PARAM;
    }

    status = BSA_BleApcfCfgInit(&bsa_ble_apcf_cfg);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] Failed to add scan filter. status: %d\n", status);
        return LeConvertBsaStatus(status);
    }

    bsa_ble_apcf_cfg.filter_index                           = filterIndex;
    bsa_ble_apcf_cfg.action                                 = BSA_BLE_APCF_COND_ADD;
    bsa_ble_apcf_cfg.cond.filter_setting.filter_logic_type  = BSA_BLE_APCF_FILT_LOGIC_AND;
    bsa_ble_apcf_cfg.cond.filter_setting.logic_type         = BSA_BLE_APCF_LIST_LOGIC_OR;

    switch (adType)
    {
    case BleAdType_ManufactureSpecificData:
        bsa_ble_apcf_cfg.cond_type = BSA_BLE_APCF_MANU_DATA;
        bsa_ble_apcf_cfg.cond.manu_data.company_id = (pData[0] << 0) + (pData[1] << 8);
        bsa_ble_apcf_cfg.cond.manu_data.company_id_mask = (pMask[0] << 0) + (pMask[1] << 8);

        bsa_ble_apcf_cfg.cond.manu_data.data_len = dataLength - 2;
        memcpy(bsa_ble_apcf_cfg.cond.manu_data.pattern, pData + sizeof(bsa_ble_apcf_cfg.cond.manu_data.company_id), bsa_ble_apcf_cfg.cond.manu_data.data_len);
        memcpy(bsa_ble_apcf_cfg.cond.manu_data.pattern_mask, pMask + sizeof(bsa_ble_apcf_cfg.cond.manu_data.company_id_mask), maskLength - sizeof(bsa_ble_apcf_cfg.cond.manu_data.company_id_mask));
        break;
    case BleAdType_IncompleteListOf16bitUuid:
    case BleAdType_CompleteListOf16bitUuid:
        bsa_ble_apcf_cfg.cond_type = BSA_BLE_APCF_SRVC_UUID;
        bsa_ble_apcf_cfg.cond.srvc_uuid.uuid.len = GattAttributeUuidLength_16;
        bsa_ble_apcf_cfg.cond.srvc_uuid.uuid.uu.uuid16 = (pData[0] << 8) + (pData[1] << 0);
        bsa_ble_apcf_cfg.cond.srvc_uuid.uuid_mask.uuid16_mask = 0xFFFF;
        break;
    case BleAdType_IncompleteListOf32bitUuid:
    case BleAdType_CompleteListOf32bitUuid:
        bsa_ble_apcf_cfg.cond_type = BSA_BLE_APCF_SRVC_UUID;
        bsa_ble_apcf_cfg.cond.srvc_uuid.uuid.len = GattAttributeUuidLength_32;
        bsa_ble_apcf_cfg.cond.srvc_uuid.uuid.uu.uuid32 = (pData[0] << 24) + (pData[1] << 16) + (pData[2] << 8) + (pData[3] << 0);
        bsa_ble_apcf_cfg.cond.srvc_uuid.uuid_mask.uuid32_mask = 0xFFFFFFFF;
        break;
    case BleAdType_IncompleteListOf128bitUuid:
    case BleAdType_CompleteListOf128bitUuid:
        bsa_ble_apcf_cfg.cond_type = BSA_BLE_APCF_SRVC_UUID;
        bsa_ble_apcf_cfg.cond.srvc_uuid.uuid.len = GattAttributeUuidLength_128;
        memcpy(bsa_ble_apcf_cfg.cond.srvc_uuid.uuid.uu.uuid128, pData, GattAttributeUuidLength_128);
        memset(bsa_ble_apcf_cfg.cond.srvc_uuid.uuid_mask.uuid128_mask, 0xFF, GattAttributeUuidLength_128);
        break;
    default:
        NN_SDK_LOG("[bluetooth] Failed to add scan filter. Unsupported AD Type (%d).\n", adType);
        return BT_BLE_ERR_INVALID_PARAM;
    }

    status = BSA_BleApcfCfg(&bsa_ble_apcf_cfg);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] Failed to add scan filter. status:%d\n", status);
        return LeConvertBsaStatus(status);
    }

    return BtHalLeClientActivateScanFilter(filterIndex, adType);
}

/*******************************************************************************
**
** Function:        BtHalLeClientDeleteScanFilterCondition
**
** Description:     Delete a scan filter condition from a scan filter
**
** Parameters:      [in] uint8_t filterIndex    : target filter index
** Parameters:      [in] BleAdType adType       : AD Type of filter condition
** Parameters:      [in] uint8_t* data          : an array of filter value
** Parameters:      [in] uint8_t dataLength     : length of filter value
** Parameters:      [in] uint8_t* mask          : mask for filter value
** Parameters:      [in] uint8_t maskLength     : length of mask
**
** Returns:         BluetoothLeStatus
**
** Callbacks:       InfoFromLeScanFilterStateChangedCallback
**
*******************************************************************************/
BluetoothLeStatus BtHalLeClientDeleteScanFilterCondition(uint8_t filterIndex, BleAdType adType, const uint8_t* pData, uint8_t dataLength, const uint8_t* pMask, uint8_t maskLength)
{
    BTHAL_IF_BLE_DEBUG("called.");

    tBSA_STATUS status;
    tBSA_BLE_APCF_CFG bsa_ble_apcf_cfg;

    if (dataLength > BSA_BLE_APCF_COND_MAX_LEN)
    {
        NN_SDK_LOG("[bluetooth] Failed to delete scan filter. Data must be smaller than %d bytes\n", BSA_BLE_APCF_COND_MAX_LEN);
        return BT_BLE_ERR_INVALID_PARAM;
    }
    if (dataLength < maskLength)
    {
        NN_SDK_LOG("[bluetooth] Failed to delete scan filter. Filter mask (%d bytes) must be smaller than data (%d bytes).\n", maskLength, dataLength);
        return BT_BLE_ERR_INVALID_PARAM;
    }

    status = BSA_BleApcfCfgInit(&bsa_ble_apcf_cfg);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] Failed to delete scan filter. status: %d\n", status);
        return LeConvertBsaStatus(status);
    }

    bsa_ble_apcf_cfg.filter_index = filterIndex;
    bsa_ble_apcf_cfg.action = BSA_BLE_APCF_COND_DELETE;

    switch (adType)
    {
    case BleAdType_ManufactureSpecificData:
        bsa_ble_apcf_cfg.cond_type = BSA_BLE_APCF_MANU_DATA;
        bsa_ble_apcf_cfg.cond.manu_data.company_id = (pData[0] << 0) + (pData[1] << 8);
        bsa_ble_apcf_cfg.cond.manu_data.company_id_mask = (pMask[0] << 0) + (pMask[1] << 8);

        bsa_ble_apcf_cfg.cond.manu_data.data_len = dataLength - sizeof(bsa_ble_apcf_cfg.cond.manu_data.company_id);
        memcpy(bsa_ble_apcf_cfg.cond.manu_data.pattern, pData + sizeof(uint8_t) * sizeof(bsa_ble_apcf_cfg.cond.manu_data.company_id), bsa_ble_apcf_cfg.cond.manu_data.data_len);
        memcpy(bsa_ble_apcf_cfg.cond.manu_data.pattern_mask, pMask + sizeof(uint8_t) * sizeof(bsa_ble_apcf_cfg.cond.manu_data.company_id_mask), maskLength - sizeof(bsa_ble_apcf_cfg.cond.manu_data.company_id_mask));
        break;
    case BleAdType_IncompleteListOf16bitUuid:
    case BleAdType_CompleteListOf16bitUuid:
        bsa_ble_apcf_cfg.cond_type = BSA_BLE_APCF_SRVC_UUID;
        bsa_ble_apcf_cfg.cond.srvc_uuid.uuid.len = GattAttributeUuidLength_16;
        bsa_ble_apcf_cfg.cond.srvc_uuid.uuid.uu.uuid16 = (pData[0] << 8) + (pData[1] << 0);
        bsa_ble_apcf_cfg.cond.srvc_uuid.uuid_mask.uuid16_mask = 0xFFFF;
        break;
    case BleAdType_IncompleteListOf32bitUuid:
    case BleAdType_CompleteListOf32bitUuid:
        bsa_ble_apcf_cfg.cond_type = BSA_BLE_APCF_SRVC_UUID;
        bsa_ble_apcf_cfg.cond.srvc_uuid.uuid.len = GattAttributeUuidLength_32;
        bsa_ble_apcf_cfg.cond.srvc_uuid.uuid.uu.uuid32 = (pData[0] << 24) + (pData[1] << 16) + (pData[2] << 8) + (pData[3] << 0);
        bsa_ble_apcf_cfg.cond.srvc_uuid.uuid_mask.uuid32_mask = 0xFFFFFFFF;
        break;
    case BleAdType_IncompleteListOf128bitUuid:
    case BleAdType_CompleteListOf128bitUuid:
        bsa_ble_apcf_cfg.cond_type = BSA_BLE_APCF_SRVC_UUID;
        bsa_ble_apcf_cfg.cond.srvc_uuid.uuid.len = GattAttributeUuidLength_128;
        memcpy(bsa_ble_apcf_cfg.cond.srvc_uuid.uuid.uu.uuid128, pData, GattAttributeUuidLength_128);
        memset(bsa_ble_apcf_cfg.cond.srvc_uuid.uuid_mask.uuid128_mask, 0xFF, GattAttributeUuidLength_128);
        break;
    default:
        NN_SDK_LOG("[bluetooth] Failed to delete scan filter. Unsupported AD Type.\n");
        return BT_BLE_ERR_INVALID_PARAM;
    }

    status = BSA_BleApcfCfg(&bsa_ble_apcf_cfg);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] Failed to delete scan filter. status:%d\n", status);
        return LeConvertBsaStatus(status);
    }

    return LeConvertBsaStatus(status);
}

/*******************************************************************************
**
** Function:        BtHalLeClientDeleteScanFilter
**
** Description:     Delete a scan filter. Until disabling scan filter, any advertising packets will be filtered out.
**
** Parameters:      [in] uint8_t filterIndex    : target filter index
**
** Returns:         BluetoothLeStatus
**
** Callbacks:       InfoFromLeScanFilterStateChangedCallback
**
*******************************************************************************/
BluetoothLeStatus BtHalLeClientDeleteScanFilter(uint8_t filterIndex)
{
    BTHAL_IF_BLE_DEBUG("called.");

    tBSA_STATUS status;
    tBSA_BLE_APCF_CFG bsa_ble_apcf_cfg;

    status = BSA_BleApcfCfgInit(&bsa_ble_apcf_cfg);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] Failed to delete a scan filter (Index: %d). status: %d\n", filterIndex, status);
        return LeConvertBsaStatus(status);
    }

    bsa_ble_apcf_cfg.filter_index = filterIndex;
    bsa_ble_apcf_cfg.cond_type = BSA_BLE_APCF_FILTER_SETTING;
    bsa_ble_apcf_cfg.action = BSA_BLE_APCF_COND_DELETE;
    bsa_ble_apcf_cfg.cond.filter_setting.rssi_high_thres = 0x80;
    bsa_ble_apcf_cfg.cond.filter_setting.feat_seln = 0;
    bsa_ble_apcf_cfg.cond.filter_setting.filter_logic_type = BSA_BLE_APCF_FILT_LOGIC_AND;

    status = BSA_BleApcfCfg(&bsa_ble_apcf_cfg);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] Failed to delete a scan filter (Index: %d). status: %d\n", filterIndex, status);
        return LeConvertBsaStatus(status);
    }

    activatedFilterConditions[filterIndex] = 0x00;

    return LeConvertBsaStatus(status);
}

/*******************************************************************************
**
** Function:        BtHalLeClientClearScanFilters
**
** Description:     Clear all the scan filters. Until disabling scan filter, any advertising packets will be filtered out.
**
** Parameters:      None
**
** Returns:         BluetoothLeStatus
**
** Callbacks:       InfoFromLeScanFilterStateChangedCallback
**
*******************************************************************************/
BluetoothLeStatus BtHalLeClientClearScanFilters()
{
    BTHAL_IF_BLE_DEBUG("called.");

    tBSA_STATUS status;
    tBSA_BLE_APCF_CFG bsa_ble_apcf_cfg;

    status = BSA_BleApcfCfgInit(&bsa_ble_apcf_cfg);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] Failed to clear scan filters. status: %d\n", status);
        return LeConvertBsaStatus(status);
    }

    bsa_ble_apcf_cfg.filter_index = 0;
    bsa_ble_apcf_cfg.cond_type = BSA_BLE_APCF_FILTER_SETTING;
    bsa_ble_apcf_cfg.action = BSA_BLE_APCF_COND_CLEAR;
    bsa_ble_apcf_cfg.cond.filter_setting.rssi_high_thres = 0;
    bsa_ble_apcf_cfg.cond.filter_setting.feat_seln = 0;
    bsa_ble_apcf_cfg.cond.filter_setting.filter_logic_type = BSA_BLE_APCF_FILT_LOGIC_AND;

    status = BSA_BleApcfCfg(&bsa_ble_apcf_cfg);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] Failed to clear scan filters. status: %d\n", status);
        return LeConvertBsaStatus(status);
    }

    for (int i = 0; i < 16; ++i)
    {
        activatedFilterConditions[i] = 0x00;
    }

    return LeConvertBsaStatus(status);
}

/*******************************************************************************
**
** Function:        BtHalLeClientEnableScanFilter
**
** Description:     Enable/Disable scan filters.
**                  Enabling scan filter with no conditions will filter outs any advertising packets
**                  Disabling scan filter let any advertising be passed.
**
** Parameters:      [in] bool enable    : flag indicating if enable/disable scan filter
**
** Returns:         BluetoothLeStatus
**
** Callbacks:       InfoFromLeScanFilterStateChangedCallback
**
*******************************************************************************/
BluetoothLeStatus BtHalLeClientEnableScanFilter(bool enable)
{
    BTHAL_IF_BLE_DEBUG("called.");

    tBSA_STATUS status;
    tBSA_BLE_APCF_ENABLE bsa_ble_apcf_enable;


    status = BSA_BleApcfEnableInit(&bsa_ble_apcf_enable);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] Failed to enable scan filter. status: %d\n", status);
        return LeConvertBsaStatus(status);
    }

    bsa_ble_apcf_enable.enable = enable;

    status = BSA_BleApcfEnable(&bsa_ble_apcf_enable);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] Failed to enable scan filter. status: %d\n", status);
        return LeConvertBsaStatus(status);
    }

    return LeConvertBsaStatus(status);
}

/*******************************************************************************
**
** Function:        BtHalLeGetFirstCharacteristic
**
** Description:     Get the first characteristic from a service
**
** Parameters:      [in] tBSA_BLE_CL_GET_FIRST_CHAR bsaFirstChar
**
** Returns:         BluetoothLeStatus
**
** Callbacks:       None
**
*******************************************************************************/
BluetoothLeStatus BtHalLeGetFirstCharacteristic(tBT_UUID* pOutFirstCharUuid, uint8_t* pOutCharInstId, uint8_t* pOutProperty,
                                                uint32_t connId,
                                                tBT_UUID srvcUuid, uint8_t srvcInstId, bool isPrimary,
                                                tBT_UUID filterCharUuid)
{
    BTHAL_IF_BLE_DEBUG("called.");

    tBSA_STATUS status;
    tBSA_BLE_CL_GET_FIRST_CHAR firstChar;

    status =  BSA_BleClGetFirstCharInit(&firstChar);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] Failed to init first characteristic. status: %d\n", status);
        return LeConvertBsaStatus(status);
    }

    int clientIdx = FindClientBy32bitConnectionId(connId);
    if (clientIdx < 0)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to first characteristic. Client not found.\n", __func__);
        return BT_ERR_CLIENT;
    }

    firstChar.conn_id = app_ble_cb.ble_client[clientIdx].bsa_conn_id;

    memcpy(&firstChar.service_uuid.id.uuid, &srvcUuid, sizeof(tBT_UUID));
    firstChar.service_uuid.id.inst_id = srvcInstId;
    firstChar.service_uuid.is_primary = isPrimary;

    memcpy(&firstChar.char_uuid_cond, &filterCharUuid, sizeof(tBT_UUID));

    status = BSA_BleClGetFirstChar(&firstChar);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] Failed to get first characteristic. status: %d\n", status);
        return LeConvertBsaStatus(status);
    }

    memcpy(pOutFirstCharUuid, &firstChar.char_result.char_id.uuid, sizeof(tBT_UUID));
    *pOutCharInstId = firstChar.char_result.char_id.inst_id;
    *pOutProperty = firstChar.property;

    return LeConvertBsaStatus(status);
}

/*******************************************************************************
**
** Function:        BtHalLeGetNextCharacteristic
**
** Description:     Get the next characteristic from a service
**
** Parameters:      [in] tBSA_BLE_CL_GET_NEXT_CHAR bsaNextChar
**
** Returns:         BluetoothLeStatus
**
** Callbacks:       None
**
*******************************************************************************/
BluetoothLeStatus BtHalLeGetNextCharacteristic(tBT_UUID* pOutFirstCharUuid, uint8_t* pOutCharInstId, uint8_t* pOutProperty,
                                               uint32_t connId,
                                               tBT_UUID srvcUuid, uint8_t srvcInstId, bool isPrimary,
                                               tBT_UUID previousCharUuid, uint8_t previousCharInstId, tBT_UUID filterCharUuid)
{
    BTHAL_IF_BLE_DEBUG("called.");

    tBSA_STATUS status;
    tBSA_BLE_CL_GET_NEXT_CHAR nextChar;

    status =  BSA_BleClGetNextCharInit(&nextChar);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] Failed to init next characteristic. status: %d\n", status);
        return LeConvertBsaStatus(status);
    }

    int clientIdx = FindClientBy32bitConnectionId(connId);
    if (clientIdx < 0)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to next characteristic. Client not found.\n", __func__);
        return BT_ERR_CLIENT;
    }

    nextChar.conn_id = app_ble_cb.ble_client[clientIdx].bsa_conn_id;

    memcpy(&nextChar.start_char_id.srvc_id.id.uuid, &srvcUuid, sizeof(tBT_UUID));
    nextChar.start_char_id.srvc_id.id.inst_id = srvcInstId;
    nextChar.start_char_id.srvc_id.is_primary = isPrimary;

    memcpy(&nextChar.start_char_id.char_id.uuid, &previousCharUuid, sizeof(tBT_UUID));
    nextChar.start_char_id.char_id.inst_id = previousCharInstId;

    memcpy(&nextChar.char_uuid_cond, &filterCharUuid, sizeof(tBT_UUID));

    status = BSA_BleClGetNextChar(&nextChar);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] Failed to get next characteristic. status: %d\n", status);
        return LeConvertBsaStatus(status);
    }

    memcpy(pOutFirstCharUuid, &nextChar.char_result.char_id.uuid, sizeof(tBT_UUID));
    *pOutCharInstId = nextChar.char_result.char_id.inst_id;
    *pOutProperty = nextChar.property;

    return LeConvertBsaStatus(status);
}

/*******************************************************************************
**
** Function:        BtHalLeGetFirstDescriptor
**
** Description:     Find the first descriptor of the
 **                 characteristic on the given server from cache.
**
** Parameters:      [in] tBSA_BLE_CL_GET_FIRST_CHAR_DESCR bsaFirstDescr
**
** Returns:         BluetoothLeStatus
**
** Callbacks:       None
**
*******************************************************************************/
BluetoothLeStatus BtHalLeGetFirstDescriptor(tBT_UUID* pOutDescUuid,
                                            uint32_t connId,
                                            tBT_UUID srvcUuid, uint8_t srvcInstId, bool isPrimary,
                                            tBT_UUID charUuid, uint8_t charInstId,
                                            tBT_UUID filterDescUuid)
{
    BTHAL_IF_BLE_DEBUG("called.");

    tBSA_STATUS status;
    tBSA_BLE_CL_GET_FIRST_CHAR_DESCR firstDescr;

    status =  BSA_BleClGetFirstCharDescrInit(&firstDescr);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] Failed to init first descriptor. status: %d\n", status);
        return LeConvertBsaStatus(status);
    }

    int clientIdx = FindClientBy32bitConnectionId(connId);
    if (clientIdx < 0)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to get first descriptor. Client not found.\n", __func__);
        return BT_ERR_CLIENT;
    }

    firstDescr.conn_id = app_ble_cb.ble_client[clientIdx].bsa_conn_id;

    memcpy(&firstDescr.char_id.srvc_id.id.uuid, &srvcUuid, sizeof(tBT_UUID));
    firstDescr.char_id.srvc_id.id.inst_id = srvcInstId;
    firstDescr.char_id.srvc_id.is_primary = isPrimary;

    memcpy(&firstDescr.char_id.char_id.uuid, &charUuid, sizeof(tBT_UUID));
    firstDescr.char_id.char_id.inst_id = charInstId;

    memcpy(&firstDescr.descr_uuid_cond, &filterDescUuid, sizeof(tBT_UUID));

    status = BSA_BleClGetFirstCharDescr(&firstDescr);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] Failed to get first descriptor. status: %d\n", status);
        return LeConvertBsaStatus(status);
    }

    memcpy(&pOutDescUuid, &firstDescr.descr_result.descr_id.uuid, sizeof(tBT_UUID));

    return LeConvertBsaStatus(status);
}

/*******************************************************************************
**
** Function:        BtHalLeGetNextDescriptor
**
** Description:     Find the next descriptor of the
 **                 characteristic on the given server from cache.
**
** Parameters:      [in] tBSA_BLE_CL_GET_NEXT_CHAR_DESCR bsaNextDescr
**
** Returns:         BluetoothLeStatus
**
** Callbacks:       None
**
*******************************************************************************/
BluetoothLeStatus BtHalLeGetNextDescriptor(tBT_UUID* pOutDescUuid,
                                           uint32_t connId,
                                           tBT_UUID srvcUuid, uint8_t srvcInstId, bool isPrimary,
                                           tBT_UUID charUuid, uint8_t charInstId,
                                           tBT_UUID previousDescUuid, tBT_UUID filterDescUuid)
{
    BTHAL_IF_BLE_DEBUG("called.");

    tBSA_STATUS status;
    tBSA_BLE_CL_GET_NEXT_CHAR_DESCR nextDescr;

    status =  BSA_BleClGetNextCharDescrInit(&nextDescr);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] Failed to init next descriptor. status: %d\n", status);
        return LeConvertBsaStatus(status);
    }

    int clientIdx = FindClientBy32bitConnectionId(connId);
    if (clientIdx < 0)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to get next descriptor. Client not found.\n", __func__);
        return BT_ERR_CLIENT;
    }

    nextDescr.conn_id = app_ble_cb.ble_client[clientIdx].bsa_conn_id;

    memcpy(&nextDescr.start_descr_id.char_id.srvc_id.id.uuid, &srvcUuid, sizeof(tBT_UUID));
    nextDescr.start_descr_id.char_id.srvc_id.id.inst_id = srvcInstId;
    nextDescr.start_descr_id.char_id.srvc_id.is_primary = isPrimary;

    memcpy(&nextDescr.start_descr_id.char_id.char_id.uuid, &charUuid, sizeof(tBT_UUID));
    nextDescr.start_descr_id.char_id.char_id.inst_id = charInstId;

    memcpy(&nextDescr.start_descr_id.descr_id.uuid, &previousDescUuid, sizeof(tBT_UUID));

    memcpy(&nextDescr.descr_uuid_cond, &filterDescUuid, sizeof(tBT_UUID));

    status = BSA_BleClGetNextCharDescr(&nextDescr);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] Failed to get next descriptor. status: %d\n", status);
        return LeConvertBsaStatus(status);
    }

    memcpy(&pOutDescUuid, &nextDescr.descr_result.descr_id.uuid, sizeof(tBT_UUID));

    return LeConvertBsaStatus(status);
}
