﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "bluetooth_InternalTypes.h"




namespace nn {
namespace bluetooth {

/**
 * @brief Represents the standard Bluetooth interface.
 */
class BluetoothHidInterfaceC
{
    NN_DISALLOW_COPY(BluetoothHidInterfaceC);
    NN_DISALLOW_MOVE(BluetoothHidInterfaceC);

public:
    BluetoothHidInterfaceC()                NN_NOEXCEPT : m_BluetoothHidInterface() {}

    BluetoothHhInterface& GetBluetoothHidInterface() NN_NOEXCEPT
    {
        return m_BluetoothHidInterface;
    };

    /* Register the BtHh callbacks and interface version */
    static BluetoothHhStatus InitializeHid(BluetoothHhCallbacks* callbacks, uint16_t interfaceVersion);

    /* connect to hid device */
    static BluetoothHhStatus connect(Btbdaddr *pBdAddr);

    /* Disconnect from hid device */
    static BluetoothHhStatus disconnect(Btbdaddr *pBdAddr);

    /* Virtual UnPlug (VUP) the specified HID device */
    static BluetoothHhStatus VirtualUnplug(Btbdaddr *pBdAddr);

    /* Set the HID device descriptor for the specified HID device. */
    static BluetoothHhStatus SetInfo(Btbdaddr *pBdAddr, BluetoothHhHidInfo HidInfo);

    /* Get the HID proto mode. */
    static BluetoothHhStatus GetProtocol(Btbdaddr *pBdAddr, BluetoothHhProtocolMode protocolMode);

    /* Set the HID proto mode. */
    static BluetoothHhStatus SetProtocol(Btbdaddr *pBdAddr, BluetoothHhProtocolMode protocolMode);

    /* Send a GET_REPORT to HID device. */
    static BluetoothHhStatus GetReport(Btbdaddr *pBdAddr, BluetoothHhReportType reportType, uint8_t reportId, int bufferSize);

    /* Send a SET_REPORT to HID device. */
    static BluetoothHhStatus SetReport(Btbdaddr *pBdAddr, BluetoothHhReportType reportType, char* report);

    /* Send data to HID device. */
    static BluetoothHhStatus SendData(Btbdaddr *pBdAddr, char* data);

    /* Send wake command to HID device. */
    static BluetoothHhStatus WakeController(Btbdaddr *pBdAddr);

    /* Add paired device settings */
    static BluetoothHhStatus AddPairedDevice(nn::settings::system::BluetoothDevicesSettings *pDevice);

    /* Get paired device settings */
    static BluetoothHhStatus GetPairedDevice(Btbdaddr *pBdAddr, nn::settings::system::BluetoothDevicesSettings *pDeviceOut);

    /* Closes the interface. */
    static void CleanupHid(void);

    /* Get the latest packet loss rate (for debugging, for WIT) */
    static void GetLatestPlr(nn::bluetooth::PlrStatistics *pOut);

    /* Get AFH channel map list */
    static void GetChannelMap(ChannelMap* pOutChannelMap);

private:
    BluetoothHhInterface m_BluetoothHidInterface;
};
#ifdef HID_INPUT_STATS
void BtUpdateTsiForStats(const tBSA_ROBSON_TSI_MODE tsiMode, const Btbdaddr bdAddr);
#endif
} // bluetooth
} // nn

int               BtHalBsaExitHh(void);
void              BtHalBsaHidClientInit(uint16_t interfaceVersion);
void              BtHalInitializeHid(void);

// Bluetooth HAL HID API is currently defined in bluetooth_hal_hh.cpp
BluetoothHhStatus BtHalConnect(const Btbdaddr *pBdAddr);
BluetoothHhStatus BtHalDisconnect(const Btbdaddr *pBdAddr);
BluetoothHhStatus BtHalSetInfo(void);
BluetoothHhStatus BtHalSetReport(const Btbdaddr *pBdAddr, BluetoothHhReportType reportType, const char* report);
BluetoothHhStatus BtHalGetReport(const Btbdaddr *pBdAddr, BluetoothHhReportType reportType, uint8_t reportId, int bufferSize);
BluetoothHhStatus BtHalSendData(const Btbdaddr *pBdAddr, const char* data);
BluetoothHhStatus BtHalWakeController(const Btbdaddr *pBdAddr);
BluetoothHhStatus BtHalAddPairedDevice(const nn::settings::system::BluetoothDevicesSettings *pDevice);
BluetoothHhStatus BtHalGetPairedDevice(const Btbdaddr *pBdAddr, nn::settings::system::BluetoothDevicesSettings* pDeviceOut);
BluetoothHhStatus BtHalGetAfhChannelMap(const Btbdaddr *pBdAddr, ChannelInfo* pOutChannelInfo);


