﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkLog.h>
#include <nn/os.h>
#include "bluetooth.h"
#include "bluetooth_hid.h"
#include "bluetooth_InternalTypes.h"
#include "bluetooth_queue.h"

#ifdef HID_INPUT_STATS
#include <nn/nn_Assert.h>
#include <nn/nn_SystemThreadDefinition.h>

namespace {
BluetoothHhconnectionStateFuncType ConnectionStateCallback = NULL;
BluetoothHhgetReportFuncType       GetReportCallback = NULL;

const int RfMonitorIntervalMs = 5000;
const size_t RfMonitorThreadStackSize = 8192;
NN_OS_ALIGNAS_THREAD_STACK char g_RfMonitorThreadStack[RfMonitorThreadStackSize];
nn::os::ThreadType g_RfMonitorThread;
nn::os::EventType  g_RfMonitorFinalizeThreadEvent;

nn::os::MutexType  g_RfMonitorMutex;

const int PlrAvgSampleNum = 5;

const int HistBins = 5;
const int HistBinWidthMs = 10;

enum RfCounterState
{
    RfCounterState_Invalid,
    RfCounterState_Registered,
    RfCounterState_Counting,
    RfCounterState_Stopped,
};

struct RfCounter
{
    uint32_t count              = 0;
    Btbdaddr bdAddr             = {{0}};
    ChannelInfo channelInfo     = {{0}, 0};
    RfCounterState state        = RfCounterState_Invalid;
    uint32_t sniffInterval      = 0;
    uint32_t receivedPackets    = 0;
    uint32_t sendPackets        = 0;
    double plr[PlrAvgSampleNum] = {0};
    double plrMovingAvg         = 0.0;
    uint16_t hist[HistBins]     = {0};
    nn::os::Tick lastPacketTick = nn::os::Tick(0);
};

RfCounter g_RfCounter[nn::bluetooth::RfListSizeMax];
uint32_t g_SampleNumber = 0;

void RegisterController(const Btbdaddr &bdAddr)
{
    nn::os::LockMutex(&g_RfMonitorMutex);
    for (int i = 0; i < nn::bluetooth::RfListSizeMax; ++i)
    {
        if (RfCounterState_Invalid == g_RfCounter[i].state)
        {
            g_RfCounter[i].state = RfCounterState_Registered;
            memcpy(&g_RfCounter[i].bdAddr, &bdAddr, sizeof(Btbdaddr));
            // Init PLR stats
            g_RfCounter[i].sniffInterval = 0;
            g_RfCounter[i].count = 0;
            g_RfCounter[i].receivedPackets = 0;
            g_RfCounter[i].sendPackets = 0;
            memset(g_RfCounter[i].plr, 0, PlrAvgSampleNum);
            g_RfCounter[i].plrMovingAvg = 0.0;
            // Init channel map
            memset(g_RfCounter[i].channelInfo.channels, 0, ARRAY_SIZE_OF_CHANNEL_INFO);
            g_RfCounter[i].channelInfo.count = 0;

            g_RfCounter[i].lastPacketTick = nn::os::GetSystemTick();
            memset(g_RfCounter[i].hist, 0, sizeof g_RfCounter[i].hist);

            nn::os::UnlockMutex(&g_RfMonitorMutex);
            return;
        }
    }
    nn::os::UnlockMutex(&g_RfMonitorMutex);
    NN_ASSERT(0, "Could not register a new controller. List is full.");
}

int FindControllerCounter(const Btbdaddr &bdAddr)
{
    for (int i = 0; i < nn::bluetooth::RfListSizeMax; ++i)
    {
        if (RfCounterState_Invalid != g_RfCounter[i].state && memcmp(&g_RfCounter[i].bdAddr, &bdAddr, sizeof(Btbdaddr)) == 0)
        {
            return i;
        }
    }
    return -1;
}

void UnregisterController(const Btbdaddr &bdAddr)
{
    nn::os::LockMutex(&g_RfMonitorMutex);
    int id = FindControllerCounter(bdAddr);
    if (id >= 0)
    {
        g_RfCounter[id].state = RfCounterState_Invalid;
        memset(&g_RfCounter[id].bdAddr, 0, sizeof(Btbdaddr));
        // Init PLR stats
        g_RfCounter[id].sniffInterval = 0;
        g_RfCounter[id].count = 0;
        g_RfCounter[id].receivedPackets = 0;
        g_RfCounter[id].sendPackets = 0;
        memset(g_RfCounter[id].plr, 0, PlrAvgSampleNum);
        g_RfCounter[id].plrMovingAvg = 0.0;
        // Init channel map
        memset(g_RfCounter[id].channelInfo.channels, 0, ARRAY_SIZE_OF_CHANNEL_INFO);
        g_RfCounter[id].channelInfo.count = 0;
    }
    nn::os::UnlockMutex(&g_RfMonitorMutex);
}

double CalcPlr(const uint32_t sendPacketsNum, const uint32_t receivedPacketsNum)
{
    double plr = 100.0 - ((double)receivedPacketsNum / (double)sendPacketsNum) * 100.0;

    plr = (plr <= 0) ? 0 : plr;
    plr = (100 <= plr) ? 100 : plr;

    return plr;
}

void UpdatePlrMovingAverage(const double currentPlr, const int counterIndex)
{
    double sum = 0;

    NN_ASSERT(counterIndex < nn::bluetooth::PlrListSizeMax);
    NN_ASSERT(0 <= counterIndex);

    if (g_RfCounter[counterIndex].state == RfCounterState_Counting)
    {
        for(int i = 0; i < PlrAvgSampleNum; ++i)
        {
            if(i < PlrAvgSampleNum - 1)
            {
                g_RfCounter[counterIndex].plr[i] = g_RfCounter[counterIndex].plr[i + 1];
            }
            else
            {
                g_RfCounter[counterIndex].plr[i] = currentPlr;
            }

            sum += g_RfCounter[counterIndex].plr[i];
        }

        g_RfCounter[counterIndex].plrMovingAvg = sum / (double)PlrAvgSampleNum;
    }
}

void RfMonitorThreadFunc(void *arg)
{
    NN_UNUSED(arg);

    while (true)
    {
        if (nn::os::TimedWaitEvent(&g_RfMonitorFinalizeThreadEvent, nn::TimeSpan::FromMilliSeconds(RfMonitorIntervalMs)))
        {
            NN_SDK_LOG("[bluetooth] Finalize event for bluetooth RF monitor\n");
            break;
        }
        else
        {
            nn::os::LockMutex(&g_RfMonitorMutex);

            for (int i = 0; i < nn::bluetooth::PlrListSizeMax; ++i)
            {
                if (RfCounterState_Registered == g_RfCounter[i].state)
                {
                    g_RfCounter[i].state = RfCounterState_Counting;
                }
                else if (RfCounterState_Counting == g_RfCounter[i].state && g_RfCounter[i].sniffInterval != 0)
                {
                    g_RfCounter[i].sendPackets = (double)RfMonitorIntervalMs / (double)g_RfCounter[i].sniffInterval;
                    g_RfCounter[i].receivedPackets = g_RfCounter[i].count;
                    double currentPlr = CalcPlr(g_RfCounter[i].sendPackets, g_RfCounter[i].receivedPackets);

                    UpdatePlrMovingAverage(currentPlr, i);

                    ChannelInfo tempChannelInfo = {{0}, 0};
                    BluetoothHhStatus mapStatus = BtHalGetAfhChannelMap(&(g_RfCounter[i].bdAddr), &tempChannelInfo);

                    if(mapStatus == BTHH_OK)
                    {
                        // Check if AFH channel map is updated
                        if(memcmp(g_RfCounter[i].channelInfo.channels, tempChannelInfo.channels, ARRAY_SIZE_OF_CHANNEL_INFO) != 0)
                        {
                            memcpy(g_RfCounter[i].channelInfo.channels, tempChannelInfo.channels, ARRAY_SIZE_OF_CHANNEL_INFO);
                            g_RfCounter[i].channelInfo.count = tempChannelInfo.count;
                        }
                    }

                    NN_SDK_LOG("%d\t%s    Rcv/Snd pckt: %4d/%4d    PLR curr/avg: %5.2f/%5.2f    Hist(%dms): %4d/%2d/%2d/%2d/%2d",
                            g_SampleNumber, toString(g_RfCounter[i].bdAddr.address, BD_ADDR_LEN),
                            g_RfCounter[i].receivedPackets, g_RfCounter[i].sendPackets, g_RfCounter[i].plr[PlrAvgSampleNum - 1], g_RfCounter[i].plrMovingAvg,
                            HistBinWidthMs, g_RfCounter[i].hist[0], g_RfCounter[i].hist[1], g_RfCounter[i].hist[2], g_RfCounter[i].hist[3], g_RfCounter[i].hist[4]);

                    if(mapStatus == BTHH_OK)
                    {
                        NN_SDK_LOG("    Ch: %02X%02X%02X%02X%02X%02X%02X%02X%02X%02X    Ch cnt: %d\n",
                                    g_RfCounter[i].channelInfo.channels[9], g_RfCounter[i].channelInfo.channels[8], g_RfCounter[i].channelInfo.channels[7], g_RfCounter[i].channelInfo.channels[6], g_RfCounter[i].channelInfo.channels[5],
                                    g_RfCounter[i].channelInfo.channels[4], g_RfCounter[i].channelInfo.channels[3], g_RfCounter[i].channelInfo.channels[2], g_RfCounter[i].channelInfo.channels[1], g_RfCounter[i].channelInfo.channels[0],
                                    g_RfCounter[i].channelInfo.count);
                    }
                    else
                    {
                        NN_SDK_LOG("    Unable to get AFH channel map\n");
                    }

                    g_RfCounter[i].count = 0;
                    memset(g_RfCounter[i].hist, 0, sizeof g_RfCounter[i].hist);
                }
            }

            ++g_SampleNumber;
            nn::os::UnlockMutex(&g_RfMonitorMutex);
        }
    }
}


/*---------------------------------------------------------------------------*
*
*    File scope functions/symbols defined for this file
*    -- Function Prototypes, File Scope Data --
*
*---------------------------------------------------------------------------*/
static void RegisterMonitorCallback(Btbdaddr *pBdAddr, BluetoothHhConnectionState state)
{
    if (state == BTHH_CONN_STATE_CONNECTED)
    {
        RegisterController(*pBdAddr);
    }
    else if (state == BTHH_CONN_STATE_DISCONNECTED)
    {
        UnregisterController(*pBdAddr);
    }
    ConnectionStateCallback(pBdAddr, state);
}

static void SniffReportCallback(Btbdaddr *pBdAddr, BluetoothHhStatus hhStatus, uint8_t* RptData, int RptSize)
{
    if (RptData != NULL)
    {
        tBSA_HH_UIPC_REPORT *p_uipc_report = (tBSA_HH_UIPC_REPORT *)RptData;

        nn::os::LockMutex(&g_RfMonitorMutex);
        int id = FindControllerCounter(*pBdAddr);
        if (id >= 0)
        {
            g_RfCounter[id].count++;

            // find duration since last packet
            nn::os::Tick prev = g_RfCounter[id].lastPacketTick;
            g_RfCounter[id].lastPacketTick = nn::os::GetSystemTick();
            int i = ( g_RfCounter[id].lastPacketTick - prev ).ToTimeSpan().GetMilliSeconds() / HistBinWidthMs;
            if (i < HistBins)
                g_RfCounter[id].hist[i]++;
            else
                g_RfCounter[id].hist[HistBins - 1]++;
        }
        nn::os::UnlockMutex(&g_RfMonitorMutex);

        if (GetReportCallback != NULL)
        {
            GetReportCallback(pBdAddr,BTHH_OK,(uint8_t *)p_uipc_report,sizeof(tBSA_HH_UIPC_REPORT));
        }
    }
}

}   // namespace

#endif  // HID_INPUT_STATS

namespace nn {
namespace bluetooth {
/*---------------------------------------------------------------------------*
*
*    Globals/Externs
*    -- Variables/Functions --
*
*---------------------------------------------------------------------------*/
uint16_t g_BtInterfaceVersion = 0;

/*******************************************************************************
 **
 ** Function        BluetoothHhStatus
 **
 ** Description     Initializes the HID client
 **
 ** Parameters      callbacks - IMPL registered callbacks
 **
 ** Returns         BluetoothHhStatus
 **
 *******************************************************************************/
BluetoothHhStatus BluetoothHidInterfaceC::InitializeHid(BluetoothHhCallbacks* callbacks, uint16_t interfaceVersion)
{
    g_BtInterfaceVersion = interfaceVersion;
    BTHAL_IF_DEBUG("called");

    if (callbacks==NULL)
    {
        NN_SDK_LOG("[bluetooth] %s: no HID callbacks registered\n", __func__);
        return BTHH_ERR_BAD_PARAM;
    }
#ifdef HID_INPUT_STATS
    nn::os::InitializeMutex(&g_RfMonitorMutex, false, 0);
    nn::os::InitializeEvent(&g_RfMonitorFinalizeThreadEvent, false, nn::os::EventClearMode_AutoClear);

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&g_RfMonitorThread, RfMonitorThreadFunc,
            NULL, g_RfMonitorThreadStack, RfMonitorThreadStackSize, NN_SYSTEM_THREAD_PRIORITY(bluetooth, RfMonitorThread)));

    nn::os::StartThread(&g_RfMonitorThread);

    ConnectionStateCallback = callbacks->ConnectionStateCallback;
    callbacks->ConnectionStateCallback = RegisterMonitorCallback;

    GetReportCallback = callbacks->GetReportCallback;
    callbacks->GetReportCallback = SniffReportCallback;
#endif  // HID_INPUT_STATS

    ::BtHalCallbackHidRegister(callbacks);
    BtHalBsaHidClientInit(interfaceVersion);
    return BTHH_OK;
}

/***********************************************************************************************
 **
 ** Function         connect
 **
 ** Description      Connect a device after bonding
 **
 ** Parameters       pBdAddr - BD_ADDR of the device to which we are opening a connection with
 **
 ** Returns          BluetoothHhStatus
 **
 ***********************************************************************************************/
BluetoothHhStatus BluetoothHidInterfaceC::connect(Btbdaddr *pBdAddr)
{
    BluetoothHhStatus status = BTHH_OK;
    BTHAL_IF_DEBUG("called");
    NN_SDK_LOG("[bluetooth] %s: bd_addr = %02x:%02x:%02x:%02x:%02x:%02x\n", __func__,
               pBdAddr->address[0],pBdAddr->address[1],pBdAddr->address[2],
               pBdAddr->address[3],pBdAddr->address[4],pBdAddr->address[5]);
    status = BtHalConnect(pBdAddr);
    return status;
}

/****************************************************************************************
 **
 ** Function        disconnect
 **
 ** Description     Disconnect an open connection with the device specified in pBdAddr
 **
 ** Parameters      pBdAddr - BD_ADDR of the device being disconnected
 **
 ** Returns         BluetoothHhStatus
 **
 ****************************************************************************************/
BluetoothHhStatus BluetoothHidInterfaceC::disconnect(Btbdaddr *pBdAddr)
{
    BluetoothHhStatus status = BTHH_OK;
    BTHAL_IF_DEBUG("called");
    NN_SDK_LOG("[bluetooth] %s: bd_addr = %02x:%02x:%02x:%02x:%02x:%02x\n", __func__,
               pBdAddr->address[0],pBdAddr->address[1],pBdAddr->address[2],
               pBdAddr->address[3],pBdAddr->address[4],pBdAddr->address[5]);
    status = BtHalDisconnect(pBdAddr);
    return status;
}

/*******************************************************************************
 **
 ** Function        VirtualUnplug
 **
 ** Description
 **
 ** Parameters      pBdAddr - Bluetooth device address of remote device
 **
 ** Returns         BluetoothHhStatus
 **
 *******************************************************************************/
BluetoothHhStatus BluetoothHidInterfaceC::VirtualUnplug(Btbdaddr *pBdAddr)
{
    BluetoothHhStatus status = BTHH_OK;
    NN_SDK_LOG("[bluetooth] %s: NOT SUPPORTED\n", __FUNCTION__);
    return status;
}

/*******************************************************************************
 **
 ** Function        SetInfo
 **
 ** Description     Sets device info (e.g. vendor_id)
 **
 ** Parameters      pBdAddr - Bluetooth device address of remote device
 **                 HidInfo - HID device info (e.g. vendor_id/product_id)
 **
 ** Returns         BluetoothHhStatus
 **
 *******************************************************************************/
BluetoothHhStatus BluetoothHidInterfaceC::SetInfo(Btbdaddr *pBdAddr, BluetoothHhHidInfo HidInfo)
{
    BluetoothHhStatus status = BTHH_OK;
    NN_SDK_LOG("[bluetooth] %s: NOT YET SUPPORTED\n", __FUNCTION__);
    BtHalSetInfo();
    return status;
}

/*******************************************************************************
 **
 ** Function        GetProtocol
 **
 ** Description     Get protocol mode. Currently only Report mode is supported
 **
 ** Parameters      pBdAddr      - Bluetooth device address of remote device
 **                 protocolMode - Protocol mode (report/boot)
 **
 ** Returns         BluetoothHhStatus
 **
 *******************************************************************************/
BluetoothHhStatus BluetoothHidInterfaceC::GetProtocol(Btbdaddr *pBdAddr, BluetoothHhProtocolMode protocolMode)
{
    BluetoothHhStatus status = BTHH_OK;
    BTHAL_IF_DEBUG("called");
    protocolMode = BTHH_REPORT_MODE;
    return status;
}

/*******************************************************************************
 **
 ** Function        SetProtocol
 **
 ** Description     Sets the protocol mode (report/boot). Currently only Report mode is supported
 **
 ** Parameters      pBdAddr      - Bluetooth device address of remote device
 **                 protocolMode - Protocol mode (report/boot)
 **
 ** Returns         BluetoothHhStatus
 **
 *******************************************************************************/
BluetoothHhStatus BluetoothHidInterfaceC::SetProtocol(Btbdaddr *pBdAddr, BluetoothHhProtocolMode protocolMode)
{
    BluetoothHhStatus status = BTHH_OK;
    NN_SDK_LOG("[bluetooth] %s: only BTHH_REPORT_MODE is currently supported \n", __FUNCTION__);
    return status;
}

/*******************************************************************************
 **
 ** Function        GetReport
 **
 ** Description     Get the report type
 **
 ** Parameters      pBdAddr     - Bluetooth device address of remote device
 **                 reportType  - report type (input/output/feature)
 **                 reportId    - report ID
 **                 bufferSize  - size of the report
 **
 ** Returns         BluetoothHhStatus
 **
 *******************************************************************************/
BluetoothHhStatus BluetoothHidInterfaceC::GetReport(Btbdaddr *pBdAddr,
                                                    BluetoothHhReportType reportType,
                                                    uint8_t reportId,
                                                    int bufferSize)
{
    BluetoothHhStatus status = BTHH_OK;
    BTHAL_IF_DEBUG("%s: bd_addr = %02x:%02x:%02x:%02x:%02x:%02x", __func__,
               pBdAddr->address[0],pBdAddr->address[1],pBdAddr->address[2],
               pBdAddr->address[3],pBdAddr->address[4],pBdAddr->address[5]);
    status = (BluetoothHhStatus)BtHalGetReport(pBdAddr,reportType,reportId,bufferSize);
    return status;
}

/*******************************************************************************
 **
 ** Function        SetReport
 **
 ** Description     Sets a report type
 **
 ** Parameters      pBdAddr     - Bluetooth device address of remote device
 **                 reportType  - report type (input/output/feature)
 **                 report      - pointer to the report data
 **
 ** Returns         BluetoothHhStatus
 **
 *******************************************************************************/
BluetoothHhStatus BluetoothHidInterfaceC::SetReport(Btbdaddr *pBdAddr, BluetoothHhReportType reportType, char* report)
{
    BluetoothHhStatus status = BTHH_OK;
    BTHAL_IF_DEBUG("%s: bd_addr = %02x:%02x:%02x:%02x:%02x:%02x", __func__,
               pBdAddr->address[0],pBdAddr->address[1],pBdAddr->address[2],
               pBdAddr->address[3],pBdAddr->address[4],pBdAddr->address[5]);
    status = BtHalSetReport(pBdAddr,reportType,report);
    return status;
}

/*******************************************************************************
 **
 ** Function        SendData
 **
 ** Description     Send HID data to the device specified in pBdAddr
 **
 ** Parameters      pBdAddr - Target device
 **                 data    - pointer to the data being sent to the target device
 **
 ** Returns         BluetoothHhStatus
 **
 *******************************************************************************/
BluetoothHhStatus BluetoothHidInterfaceC::SendData(Btbdaddr *pBdAddr, char* data)
{
    BluetoothHhStatus status = BTHH_OK;
//    BTHAL_IF_DEBUG("%s: bd_addr = %02x:%02x:%02x:%02x:%02x:%02x", __func__,
//               pBdAddr->address[0],pBdAddr->address[1],pBdAddr->address[2],
//               pBdAddr->address[3],pBdAddr->address[4],pBdAddr->address[5]);
    status = BtHalSendData(pBdAddr,data);
    return status;
}

/*******************************************************************************
 **
 ** Function        WakeController
 **
 ** Description     Send wake command to the remote device
 **
 ** Parameters      pBdAddr - Bluetooth device address of remote device
 **
 ** Returns         BluetoothHhStatus
 **
 *******************************************************************************/
BluetoothHhStatus BluetoothHidInterfaceC::WakeController(Btbdaddr *pBdAddr)
{
    BluetoothHhStatus status = BTHH_OK;
    BTHAL_IF_DEBUG("%s: bd_addr = %02x:%02x:%02x:%02x:%02x:%02x", __func__,
               pBdAddr->address[0],pBdAddr->address[1],pBdAddr->address[2],
               pBdAddr->address[3],pBdAddr->address[4],pBdAddr->address[5]);
    status = BtHalWakeController(pBdAddr);
    return status;
}

/*******************************************************************************
 **
 ** Function        AddPairedDevice
 **
 ** Description     Add paired device information
 **
 ** Parameters      pDevice - Paired bluetooth device settings
 **
 ** Returns         BluetoothHhStatus
 **
 *******************************************************************************/
BluetoothHhStatus BluetoothHidInterfaceC::AddPairedDevice(nn::settings::system::BluetoothDevicesSettings *pDevice)
{
    BluetoothHhStatus status = BTHH_OK;
    BTHAL_IF_DEBUG("%s: bd_addr = %02x:%02x:%02x:%02x:%02x:%02x", __func__,
               pDevice->bd_addr[0],pDevice->bd_addr[1],pDevice->bd_addr[2],
               pDevice->bd_addr[3],pDevice->bd_addr[4],pDevice->bd_addr[5]);
    status = BtHalAddPairedDevice(pDevice);
    return status;
}

/*******************************************************************************
 **
 ** Function        GetPairedDevice
 **
 ** Description     Get paired device settings
 **
 ** Parameters      pBdAddr - Target device
 **
 ** Returns         nn::settings::system::BluetoothDevicesSettings
 **
 *******************************************************************************/
BluetoothHhStatus BluetoothHidInterfaceC::GetPairedDevice(Btbdaddr *pBdAddr, nn::settings::system::BluetoothDevicesSettings *pDeviceOut)
{
    BluetoothHhStatus status = BTHH_OK;
    BTHAL_IF_DEBUG("%s: bd_addr = %02x:%02x:%02x:%02x:%02x:%02x", __func__,
               pBdAddr->address[0],pBdAddr->address[1],pBdAddr->address[2],
               pBdAddr->address[3],pBdAddr->address[4],pBdAddr->address[5]);
    status = BtHalGetPairedDevice(pBdAddr, pDeviceOut);
    return status;
}

/*******************************************************************************
 **
 ** Function        CleanupHid
 **
 ** Description     Disable the HID client interface and callbacks
 **
 ** Parameters      None
 **
 ** Returns         None
 **
 *******************************************************************************/
void BluetoothHidInterfaceC::CleanupHid()
{
    BTHAL_IF_DEBUG("called");

    BluetoothHhCallbacks invalidateHidCallbacks;
    BluetoothExtCallbacks invalidateExtensionCallbacks;

    memset(&invalidateHidCallbacks, 0, sizeof(BluetoothHhCallbacks));
    invalidateHidCallbacks.size = sizeof(BluetoothHhCallbacks);
    BtHalCallbackHidRegister(&invalidateHidCallbacks);

    memset(&invalidateExtensionCallbacks, 0, sizeof(BluetoothExtCallbacks));
    invalidateExtensionCallbacks.size = sizeof(BluetoothExtCallbacks);
    BtHalCallbackRegisterExtension(&invalidateExtensionCallbacks);
#ifdef HID_INPUT_STATS
    nn::os::SignalEvent(&g_RfMonitorFinalizeThreadEvent);
    nn::os::WaitThread(&g_RfMonitorThread);
    nn::os::DestroyThread(&g_RfMonitorThread);
    nn::os::FinalizeEvent(&g_RfMonitorFinalizeThreadEvent);
    nn::os::FinalizeMutex(&g_RfMonitorMutex);
#endif
}

void BluetoothHidInterfaceC::GetLatestPlr(nn::bluetooth::PlrStatistics *pOut)
{
#ifdef HID_INPUT_STATS
    nn::os::LockMutex(&g_RfMonitorMutex);
    pOut->sampleNumber = g_SampleNumber;
    for (int i = 0; i < nn::bluetooth::PlrListSizeMax; ++i)
    {
        memcpy(&pOut->plrList[i].bluetoothAddress.address[0], &g_RfCounter[i].bdAddr.address[0], sizeof(Btbdaddr::address));
        pOut->plrList[i].plr = g_RfCounter[i].plr[PlrAvgSampleNum - 1];
        pOut->plrList[i].received = g_RfCounter[i].receivedPackets;
    }
    nn::os::UnlockMutex(&g_RfMonitorMutex);
#else
    pOut->sampleNumber = 0;
    for (int i = 0; i < nn::bluetooth::PlrListSizeMax; ++i)
    {
        memset(&pOut->plrList[i].bluetoothAddress, 0, sizeof(nn::bluetooth::BluetoothAddress));
        pOut->plrList[i].plr = 0;
        pOut->plrList[i].received = 0;
    }
#endif
}

/*******************************************************************************
 **
 ** Function        GetAfhChannelMap
 **
 ** Description     Function to get current AFH channel map
 **
 ** Parameters      pChannelInfo - out structure for AFH channel map and count
 **
 ** Returns         int
 **
 *******************************************************************************/
void BluetoothHidInterfaceC::GetChannelMap(ChannelMap* pOutChannelMap)
{
    BTHAL_IF_DEBUG("called");

#ifdef HID_INPUT_STATS
    nn::os::LockMutex(&g_RfMonitorMutex);
    for(int i = 0; i < channelMapListSizeMax; ++i)
    {
        memcpy(&pOutChannelMap->chList[i].bluetoothAddress.address[0], &g_RfCounter[i].bdAddr.address[0], sizeof(Btbdaddr::address));
        memcpy(pOutChannelMap->chList[i].channelInfo.channels, g_RfCounter[i].channelInfo.channels, ARRAY_SIZE_OF_CHANNEL_INFO);
        pOutChannelMap->chList[i].channelInfo.count = g_RfCounter[i].channelInfo.count;
    }
    nn::os::UnlockMutex(&g_RfMonitorMutex);
#else
    for(int i = 0; i < channelMapListSizeMax; ++i)
    {
        pOutChannelMap->chList[i].channelInfo.count = 0;
        memset(pOutChannelMap->chList[i].channelInfo.channels, 0, ARRAY_SIZE_OF_CHANNEL_INFO);
    }
#endif
}

#ifdef HID_INPUT_STATS
void BtUpdateTsiForStats(const tBSA_ROBSON_TSI_MODE tsiMode, const Btbdaddr bdAddr)
{
    NN_SDK_LOG("[bluetooth] %s\n", __FUNCTION__);
    int id = FindControllerCounter(bdAddr);

    if(id >= 0)
    {
        switch(tsiMode)
        {
            case TSI_MODE_1:
                g_RfCounter[id].sniffInterval = 5;

                if(g_RfCounter[id].state == RfCounterState_Stopped)
                {
                    g_RfCounter[id].state = RfCounterState_Registered;
                }
                break;
            case TSI_MODE_0:
            case TSI_MODE_2:
            case TSI_MODE_4:
            case TSI_MODE_9:
                g_RfCounter[id].sniffInterval = 10;

                if(g_RfCounter[id].state == RfCounterState_Stopped)
                {
                    g_RfCounter[id].state = RfCounterState_Registered;
                }
                break;
            case TSI_MODE_3:
            case TSI_MODE_5:
            case TSI_MODE_6:
            case TSI_MODE_7:
            case TSI_MODE_8:
            case TSI_MODE_10:
                g_RfCounter[id].sniffInterval = 15;

                if(g_RfCounter[id].state == RfCounterState_Stopped)
                {
                    g_RfCounter[id].state = RfCounterState_Registered;
                }
                break;
            case UNDEFINED_TSI_MODE:
                g_RfCounter[id].state = RfCounterState_Stopped;
                g_RfCounter[id].sniffInterval = 0;
                g_RfCounter[id].count = 0;
                g_RfCounter[id].receivedPackets = 0;
                for(int j = 0; j < PlrAvgSampleNum; ++j)
                {
                    g_RfCounter[id].plr[j] = 0.0;
                }
                g_RfCounter[id].plrMovingAvg = 0.0;
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
        }
    }
}
#endif // HID_INPUT_STATS


} // bluetooth
} // nn

