﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SystemThreadDefinition.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/bgtc/bgtc_Api.h>
#include <nn/fs/fs_ApiPrivate.h>
#include <nn/fs/fs_SystemSaveData.h> // DisableAutoSaveDataCreation 用
#include <nn/nifm/nifm_ApiForSystem.h>
#include <nn/socket.h>
#include <nn/socket/socket_SystemConfig.h>
#include <nn/time.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/npns/npns_ApiSystem.h>
#include <nn/bcat/service/bcat_Server.h>
#include <nn/news/service/news_Server.h>
#include <nn/prepo/service/prepo_Server.h>

#include "bcat_EventWatcherThread.h"
#include "bcat_NintendoAccountCacheUpdateThread.h"
#include "bcat_PowerStateEventManager.h"

namespace
{
    // TORIAEZU:
    nn::Bit8 g_Heap[640 * 1024];
}

namespace
{
    // ソケットの同時利用数。（bcat:1, news:1, prepo:1）
    const int SocketCount = 3;
    const int SocketConcurrencyCount = SocketCount;

    nn::socket::SystemConfigLightDefaultWithMemory<SocketCount, 0> g_SocketConfigWithMemory(SocketConcurrencyCount);
}

void Initialize() NN_NOEXCEPT;
void Finalize() NN_NOEXCEPT;

extern "C" void nninitStartup()
{
    nn::init::InitializeAllocator(g_Heap, sizeof (g_Heap));
}

extern "C" void nnMain()
{
    nn::os::SetThreadNamePointer(nn::os::GetCurrentThread(), NN_SYSTEM_THREAD_NAME(bcat, Main));

    Initialize();

    nn::prepo::service::StartServer();
    nn::bcat::service::StartServer();
    nn::news::service::StartServer();

    NintendoAccountCacheUpdateThread::Start();
    EventWatcherThread::Start();

    PowerStateEventManager::LoopAuto();

    // シャットダウンイベントの通知を受けても LoopAuto は返らないため、これ以降の処理には到達しない。

    EventWatcherThread::Stop();
    NintendoAccountCacheUpdateThread::Stop();

    nn::news::service::StopServer();
    nn::bcat::service::StopServer();
    nn::prepo::service::StopServer();

    Finalize();
}

void Initialize() NN_NOEXCEPT
{
    nn::socket::Initialize(g_SocketConfigWithMemory);
    nn::nifm::InitializeSystem();
    nn::time::Initialize();
    nn::account::InitializeForSystemService();
    nn::npns::InitializeForSystem();

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::bgtc::Initialize());
    nn::bgtc::SetTaskClientName("bcat.MT");

    nn::fs::InitializeWithMultiSessionForSystem();
    nn::fs::DisableAutoSaveDataCreation();
}

void Finalize() NN_NOEXCEPT
{
    nn::bgtc::Finalize();
    nn::npns::FinalizeForSystem();
    nn::time::Finalize();
    nn::socket::Finalize();
}

// 以下は例外関連コードがリンクされてプロセスが肥大化するのを防ぐためのコード。

void* operator new(size_t size)
{
    return std::malloc(size);
}

void* operator new(size_t size, const std::nothrow_t&) NN_NOEXCEPT
{
    return std::malloc(size);
}

void operator delete(void* ptr) NN_NOEXCEPT
{
    std::free(ptr);
}

void operator delete(void* ptr, const std::nothrow_t&) NN_NOEXCEPT
{
    std::free(ptr);
}

void* operator new[](size_t size)
{
    return std::malloc(size);
}

void* operator new[](size_t size, const std::nothrow_t&) NN_NOEXCEPT
{
    return std::malloc(size);
}

void operator delete[](void* ptr) NN_NOEXCEPT
{
    std::free(ptr);
}

void operator delete[](void* ptr, const std::nothrow_t&) NN_NOEXCEPT
{
    std::free(ptr);
}
