﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/account/account_ServiceResource.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/os/os_TimerEvent.h>
#include <nn/psc/psc_Types.h>

#include "account_AccountDaemonSettings.h"
#include "account_AccountDaemonTaskScheduler.h"
#include "account_Util.h"

namespace nn {

class AccountDaemon
{
private:
    void ThreadFunctionImpl() NN_NOEXCEPT;
    void ControlRequestHandlerImpl(account::BackgroundDaemonControlRequest request) NN_NOEXCEPT;
    void ReplyToControlRequest() NN_NOEXCEPT;

    CancelPoint m_CancelPoint;

    os::Event m_Terminator;
    os::Event m_SuspendRequest;
    std::atomic<account::BackgroundDaemonControlRequest> m_SuspendRequestBody;
    os::Event m_SuspendResponse;
    std::atomic_bool m_IsSuspended {false};

    AccountDaemonSettings m_Settings;
    class RunnableState
    {
    private:
        bool m_IsRunnable {true};
        mutable os::SdkMutex m_Lock;
        os::Event m_Notifier;
    public:
        RunnableState() NN_NOEXCEPT;
        void Update(bool value) NN_NOEXCEPT;
        bool Test() const NN_NOEXCEPT;

        void Clear() NN_NOEXCEPT
        {
            m_Notifier.Clear();
        }
        os::EventType* GetEventBase() NN_NOEXCEPT
        {
            return m_Notifier.GetBase();
        }
    } m_RunnableState;
    AccountDaemonTaskScheduler m_Scheduler;

    bool IsRunning() const NN_NOEXCEPT
    {
        return m_RunnableState.Test() && !m_IsSuspended;
    }

public:
    static void ThreadFunction(void* ptr) NN_NOEXCEPT
    {
        reinterpret_cast<AccountDaemon*>(ptr)->ThreadFunctionImpl();
    }
    static void ControlRequestHandler(account::BackgroundDaemonControlRequest request, void* ptr) NN_NOEXCEPT
    {
        reinterpret_cast<AccountDaemon*>(ptr)->ControlRequestHandlerImpl(request);
    }

    AccountDaemon() NN_NOEXCEPT;

    TimeSpan GetNextWakeupRequestInUptime() const NN_NOEXCEPT;

    void NotifyEnteredHalfAwake() NN_NOEXCEPT;
    void OnEnteringFullAwake(nn::psc::PmState currentState) NN_NOEXCEPT;
    void OnEnteringMinimumAwake(nn::psc::PmState currentState) NN_NOEXCEPT;
    void OnEnteringSleepReady(nn::psc::PmState currentState) NN_NOEXCEPT;
    void OnEnteringShutdownReady(nn::psc::PmState currentState) NN_NOEXCEPT;
    void RequestStop();
};

} // ~namespace nn
