﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_Mutex.h>

namespace nn { namespace dmnt { namespace io {

const size_t HtcsReceiveBufferSize = 1024;

class HtcsReceiveBuffer
{
public:
    HtcsReceiveBuffer() :
        m_ReadableEvent(nn::os::EventClearMode_ManualClear),
        m_WritableEvent(nn::os::EventClearMode_ManualClear),
        m_Mutex(false),
        m_DataSize(0),
        m_HeadIndex(0),
        m_Valid(true)
    {
    };

    // バッファに書かれたデータを読む
    // 返り値は失敗したら -1 で、成功したら読み取ったバイト数
    int Read(void* dst, size_t size) NN_NOEXCEPT;

    // バッファにデータを書く
    // 返り値は失敗したら -1 で、成功したら書き込んだバイト数
    int Write(void* src, size_t size) NN_NOEXCEPT;

    // バッファが空でなければ読み取り可能
    inline bool IsReadable() NN_NOEXCEPT
    {
        return m_DataSize != 0;
    }

    // バッファが空なら書き込み可能
    inline bool IsWritable() NN_NOEXCEPT
    {
        return m_DataSize == 0;
    }

    // Invalidate() が呼ばれていなければ true を返す
    inline bool IsValid() NN_NOEXCEPT
    {
        return m_Valid;
    }

    // 読み取り可能になるまでブロックします。スレッドセーフではありません。
    // 返り値 -1: Invalidate() が呼ばれた
    //        1: 読み取り可能
    int WaitToBeReadable() NN_NOEXCEPT;

    // 読み取り可能になるまでブロックします。スレッドセーフではありません。
    // 返り値 -1: Invalidate() が呼ばれた
    //        0: タイムアウト
    //        1: 読み取り可能
    int WaitToBeReadable(nn::TimeSpan timeout) NN_NOEXCEPT;

    // 書き込み可能になるまでブロックします。スレッドセーフではありません。
    // 返り値 -1: Invalidate() が呼ばれた
    //        1: 読み取り可能
    int WaitToBeWritable() NN_NOEXCEPT;

    // バッファの使用を停止します
    // Write は不可能になりますが、受信済のデータは Read できます
    // WaitXXX のウェイトは解除されます
    void Invalidate() NN_NOEXCEPT;
private:
    uint8_t m_Buffer[HtcsReceiveBufferSize];

    nn::os::Event m_ReadableEvent;
    nn::os::Event m_WritableEvent;
    nn::os::Mutex m_Mutex;

    // 有効なデータサイズ
    size_t m_DataSize;

    // バッファ上のデータの先頭を示すインデックス
    size_t m_HeadIndex;

    // バッファが使用可能か
    bool m_Valid;
};

}}}
