﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SystemThreadDefinition.h>

#include <nn/sasbus/sasbus_ShimInternal.h>
#include <nn/sasbus/driver/sasbus_Lib.h>
#include <nn/sasbus/server/sasbus_Server.h>

#include "../detail/Bus_Log.h"
#include "Bus_SasbusServer.h"

namespace nn { namespace bus { namespace sasbus {

namespace
{
    const size_t ThreadStackSize = 4 * 1024;
    NN_ALIGNAS(4096) uint8_t g_ThreadStack[ThreadStackSize];
    nn::os::ThreadType g_Thread;

    //! サーバのループスレッド
    void ServerThread(void *)
    {
        // 追加のモジュール初期化処理（他モジュールへの依存のためにメインスレッド内では呼ばないほうが良いもの）
        nn::sasbus::driver::Initialize();
        nn::sasbus::server::StartHipcServer();

        NN_DETAIL_BUS_INFO("[Bus] Entering Sasbus Server Loop\n");
        nn::sasbus::server::LoopHipcServer();

        // モジュール終了処理
        nn::sasbus::driver::Finalize();
    };
}

//! プロセスメインスレッド内で早い段階で同期的に済ませておく必要がある処理（プロセス内他スレッドが起動直後に API を叩いて来たら競合しうるもの）
void InitializeServer() NN_NOEXCEPT
{
    // sm への HIPC サーバの登録
    nn::sasbus::server::RegisterHipcServer();

    // プロセス内シムへのサービスオブジェクト登録
    nn::sasbus::InitializeWith(nn::sasbus::server::GetServiceObject());
}

void StartServer() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::os::CreateThread(&g_Thread, ServerThread, nullptr, g_ThreadStack, ThreadStackSize, NN_SYSTEM_THREAD_PRIORITY(sasbus, IpcServer))
    );
    nn::os::SetThreadNamePointer(&g_Thread, NN_SYSTEM_THREAD_NAME(sasbus, IpcServer));
    nn::os::StartThread(&g_Thread);
}

void StopServer() NN_NOEXCEPT
{
    // 処理ループの停止リクエストを送り、LoopAuto 関数を返す
    nn::sasbus::server::StopHipcServer();
}

void WaitAndFinalizeServer() NN_NOEXCEPT
{
    nn::os::WaitThread(&g_Thread);
    nn::os::DestroyThread(&g_Thread);
}

}}}
