﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <cstdlib>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/xcd/xcd_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include "xcd_VibratorAgent.h"

namespace nn { namespace xcd {

namespace {

const int AmFmCodeNoAction = 0x18;  //!< 何もしないという意味の AMFM 符号

const int AmpDiffCoeff = 32;        //!< 振幅の差を評価する際の重み付け
const int FreqDiffCoeff = 32;       //!< 周波数の差を評価する際の重み付け

//!< 5bit 符号を使うか 7bit 符号を使うか判断する基準となる情報を集めた構造体
struct AmFmCodeDicisionInfo
{
    int32_t amfm5bitCode;
    int16_t am7bitCode;
    int16_t fm7bitCode;
    uint32_t _logAmpCurrent;
    uint32_t _logFreqCurrent;
    uint32_t _logAmp5bit;
    uint32_t _logFreq5bit;
    uint32_t _logAmp7bit;
    uint32_t _logFreq7bit;
    uint16_t _relAmpCountCurrent;
    uint16_t _relFreqCountCurrent;
    uint16_t _relAmpCount5bit;
    uint16_t _relFreqCount5bit;

    //!< 試しにエンコードして判断基準となる情報を計算します
    static AmFmCodeDicisionInfo CalcInfo(float amplitude, float frequency, const VibrationAmFmEncoderContext* pContext, bool is5bitUsed) NN_NOEXCEPT
    {
        AmFmCodeDicisionInfo info;

        info._logAmpCurrent = pContext->_decoderContext._logAmp;
        info._logFreqCurrent = pContext->_decoderContext._logFreq;
        info._relAmpCountCurrent = pContext->_relAmpCount;
        info._relFreqCountCurrent = pContext->_relFreqCount;

        //!< 試しに 5bit エンコード
        if(is5bitUsed)
        {
            VibrationAmFmEncoderContext ctx5bit = *pContext;
            info.amfm5bitCode = static_cast<int32_t>(EncodeVibrationToAmFmCode(amplitude, frequency, &ctx5bit));
            info._logAmp5bit = ctx5bit._decoderContext._logAmp;
            info._logFreq5bit = ctx5bit._decoderContext._logFreq;
            info._relAmpCount5bit = ctx5bit._relAmpCount;
            info._relFreqCount5bit = ctx5bit._relFreqCount;
        }

        //!< 試しに 7bit エンコード
        VibrationAmFmEncoderContext ctx7bit = *pContext;
        info.am7bitCode = static_cast<int16_t>(EncodeVibrationToAm7bitCode(amplitude, &ctx7bit));
        info.fm7bitCode = static_cast<int16_t>(EncodeVibrationToFm7bitCode(frequency, &ctx7bit));
        info._logAmp7bit = ctx7bit._decoderContext._logAmp;
        info._logFreq7bit = ctx7bit._decoderContext._logFreq;

        return info;
    }

    //!< 値の更新を省略可能な状況か取得します
    bool IsUpdateSkippable(const VibrationAmFmEncoderContext* pContext) const NN_NOEXCEPT
    {
        return (
            amfm5bitCode == static_cast<int32_t>(AmFmCodeNoAction) &&
            pContext->_relAmpCount < pContext->_relAmpCountMax &&
            pContext->_relFreqCount < pContext->_relFreqCountMax);
    }

    //!< 更新を行わなかった場合の理想値との差を取得します
    int GetDiffOnNoAction() const NN_NOEXCEPT
    {
        return std::abs(static_cast<int>(_logAmpCurrent) - static_cast<int>(_logAmp7bit)) * AmpDiffCoeff
            + std::abs(static_cast<int>(_logFreqCurrent) - static_cast<int>(_logFreq7bit)) * FreqDiffCoeff
            + _relAmpCountCurrent + _relFreqCountCurrent;
    }

    //!< 5bit AMFM 符号を採用した場合の理想値との差を取得します
    int GetDiffOnAmFm5bit() const NN_NOEXCEPT
    {
        return std::abs(static_cast<int>(_logAmp5bit) - static_cast<int>(_logAmp7bit)) * AmpDiffCoeff
            + std::abs(static_cast<int>(_logFreq5bit) - static_cast<int>(_logFreq7bit)) * FreqDiffCoeff
            + _relAmpCount5bit + _relFreqCount5bit;
    }

    //!< 7bit AM 符号だけ採用した場合 (7bit FM 符号は採用しなかった場合) の理想値との差を取得します
    int GetDiffOnAm7bit() const NN_NOEXCEPT
    {
        return std::abs(static_cast<int>(_logFreqCurrent) - static_cast<int>(_logFreq7bit)) * FreqDiffCoeff
            + _relFreqCountCurrent;
    }

    //!< 7bit FM 符号だけ採用した場合 (7bit AM 符号は採用しなかった場合) の理想値との差を取得します
    int GetDiffOnFm7bit() const NN_NOEXCEPT
    {
        return std::abs(static_cast<int>(_logAmpCurrent) - static_cast<int>(_logAmp7bit)) * AmpDiffCoeff
            + _relAmpCountCurrent;
    }

    //!< 5bit AMFM 符号を採用します
    int ApplyAmFm5bitCode(VibrationAmFmEncoderContext* pContext) const NN_NOEXCEPT
    {
        pContext->_relAmpCount = _relAmpCount5bit;
        pContext->_relFreqCount = _relFreqCount5bit;
        pContext->_decoderContext._logAmp = _logAmp5bit;
        pContext->_decoderContext._logFreq = _logFreq5bit;
        return static_cast<int>(amfm5bitCode);
    }

    //!< 7bit AM 符号を採用します
    int ApplyAm7bitCode(VibrationAmFmEncoderContext* pContext) const NN_NOEXCEPT
    {
        pContext->_relAmpCount = 0;
        pContext->_decoderContext._logAmp = _logAmp7bit;
        return static_cast<int>(am7bitCode);
    }

    //!< 7bit FM 符号を採用します
    int ApplyFm7bitCode(VibrationAmFmEncoderContext* pContext) const NN_NOEXCEPT
    {
        pContext->_relFreqCount = 0;
        pContext->_decoderContext._logFreq = _logFreq7bit;
        return static_cast<int>(fm7bitCode);
    }
};

const int RelCountMaxOffset = 100;

// エンコード時の選択の幅が狭まることが無いよう、一時的に relCountMax 値を拡張します
void ExtendRelCountMax(VibrationAmFmEncoderContext* pContext)
{
    pContext->_relAmpCountMax += RelCountMaxOffset;
    pContext->_relFreqCountMax += RelCountMaxOffset;
}

// 一時的に拡張された relCountMax 値を元に戻します
void RestoreRelCountMax(VibrationAmFmEncoderContext* pContext)
{
    pContext->_relAmpCountMax -= RelCountMaxOffset;
    pContext->_relFreqCountMax -= RelCountMaxOffset;
}

bool IsVibrationValuesSame(const VibrationValue* vibArray, int vibCount) NN_NOEXCEPT
{
    for(int i = 1; i < vibCount; i++)
    {
        if(vibArray[0].amplitudeLow != vibArray[i].amplitudeLow
            || vibArray[0].frequencyLow != vibArray[i].frequencyLow
            || vibArray[0].amplitudeHigh != vibArray[i].amplitudeHigh
            || vibArray[0].frequencyHigh != vibArray[i].frequencyHigh)
        {
            return false;
        }
    }

    return true;
}

VibrationAmFmPack GeneratePackedAmFmCodes1(
    VibrationAmFmEncoderContext* pContextLow,
    VibrationAmFmEncoderContext* pContextHigh,
    AmFmCodeDicisionInfo* pInfoLow,
    AmFmCodeDicisionInfo* pInfoHigh) NN_NOEXCEPT
{
    // 更新不要ならばフォーマット Zero を返します
    if(pInfoLow->IsUpdateSkippable(pContextLow) && pInfoHigh->IsUpdateSkippable(pContextHigh))
    {
        return VibrationAmFmPackFormat::Zero::Make();
    }

    int am7bitLow = pInfoLow->ApplyAm7bitCode(pContextLow);
    int fm7bitLow = pInfoLow->ApplyFm7bitCode(pContextLow);
    int am7bitHigh = pInfoHigh->ApplyAm7bitCode(pContextHigh);
    int fm7bitHigh = pInfoHigh->ApplyFm7bitCode(pContextHigh);

    return VibrationAmFmPackFormat::One28bit::Make(am7bitLow, fm7bitLow, am7bitHigh, fm7bitHigh);
}

VibrationAmFmPack GeneratePackedAmFmCodes2(
    VibrationValue* vibArray,
    VibrationAmFmEncoderContext* pContextLow,
    VibrationAmFmEncoderContext* pContextHigh,
    AmFmCodeDicisionInfo* pInfoLow,
    AmFmCodeDicisionInfo* pInfoHigh) NN_NOEXCEPT
{
    // 振動値がすべて同一ならば One28bit を利用
    if(IsVibrationValuesSame(vibArray, 2))
    {
        return GeneratePackedAmFmCodes1(pContextLow, pContextHigh, pInfoLow, pInfoHigh);
    }

    int codesLow[2];
    int codesHigh[2];

    // 7bit 符号を低周波帯に利用するか高周波帯に利用するか決定します
    int diffOn7bitForLow = pInfoHigh->GetDiffOnAmFm5bit();
    int diffOn7bitForHigh = pInfoLow->GetDiffOnAmFm5bit();
    bool is7bitCodeForLow = (diffOn7bitForLow <= diffOn7bitForHigh);

    // 1 番目の振動値のエンコード
    ExtendRelCountMax(pContextLow);
    ExtendRelCountMax(pContextHigh);
    int am7bit0 = 0;
    int fm7bit0 = 0;
    if(is7bitCodeForLow)
    {
        am7bit0 = pInfoLow->ApplyAm7bitCode(pContextLow);
        fm7bit0 = pInfoLow->ApplyFm7bitCode(pContextLow);
        codesLow[0] = AmFmCodeNoAction;
        codesHigh[0] = EncodeVibrationToAmFmCode(vibArray[0].amplitudeHigh, vibArray[0].frequencyHigh, pContextHigh);
    }
    else
    {
        am7bit0 = pInfoHigh->ApplyAm7bitCode(pContextHigh);
        fm7bit0 = pInfoHigh->ApplyFm7bitCode(pContextHigh);
        codesLow[0] = EncodeVibrationToAmFmCode(vibArray[0].amplitudeLow, vibArray[0].frequencyLow, pContextLow);
        codesHigh[0] = AmFmCodeNoAction;
    }

    bool isSkippable = pInfoLow->IsUpdateSkippable(pContextLow) && pInfoHigh->IsUpdateSkippable(pContextHigh);

    // 2 番目の振動値は両方とも 5bit 符号でエンコード
    codesLow[1] = EncodeVibrationToAmFmCode(vibArray[1].amplitudeLow, vibArray[1].frequencyLow, pContextLow);
    codesHigh[1] = EncodeVibrationToAmFmCode(vibArray[1].amplitudeHigh, vibArray[1].frequencyHigh, pContextHigh);
    RestoreRelCountMax(pContextLow);
    RestoreRelCountMax(pContextHigh);

    // 2 番目の振動値も含めて更新不要ならばフォーマット Zero を返します
    isSkippable &= (codesLow[1] == AmFmCodeNoAction);
    isSkippable &= (codesHigh[1] == AmFmCodeNoAction);
    if(isSkippable)
    {
        return VibrationAmFmPackFormat::Zero::Make();
    }

    // パック
    if(is7bitCodeForLow)
    {
        return VibrationAmFmPackFormat::Two14bitLow::Make(am7bit0, fm7bit0, codesLow, codesHigh);
    }
    else
    {
        return VibrationAmFmPackFormat::Two14bitHigh::Make(am7bit0, fm7bit0, codesLow, codesHigh);
    }
}

VibrationAmFmPack GenerateThree7bitFormatPack(
    VibrationValue* vibArray,
    VibrationAmFmEncoderContext* pContextLow,
    VibrationAmFmEncoderContext* pContextHigh,
    AmFmCodeDicisionInfo* pInfoLow,
    AmFmCodeDicisionInfo* pInfoHigh,
    bool isFrequency, bool isHigh) NN_NOEXCEPT
{
    int amfm7bit0 = 0;
    int codesLow[3];
    int codesHigh[3];
    codesLow[0] = AmFmCodeNoAction;
    codesHigh[0] = AmFmCodeNoAction;

    // 1 番目の振動値のエンコード
    if(isFrequency)
    {
        if(isHigh)
        {
            amfm7bit0 = pInfoHigh->ApplyFm7bitCode(pContextHigh);
        }
        else
        {
            amfm7bit0 = pInfoLow->ApplyFm7bitCode(pContextLow);
        }
    }
    else
    {
        if(isHigh)
        {
            amfm7bit0 = pInfoHigh->ApplyAm7bitCode(pContextHigh);
        }
        else
        {
            amfm7bit0 = pInfoLow->ApplyAm7bitCode(pContextLow);
        }
    }

    bool isSkippable = pInfoLow->IsUpdateSkippable(pContextLow) && pInfoHigh->IsUpdateSkippable(pContextHigh);

    // 2 番目と 3 番目の振動値のエンコード
    ExtendRelCountMax(pContextLow);
    ExtendRelCountMax(pContextHigh);
    for(int i = 1; i < 3; i++)
    {
        codesLow[i] = EncodeVibrationToAmFmCode(vibArray[i].amplitudeLow, vibArray[i].frequencyLow, pContextLow);
        codesHigh[i] = EncodeVibrationToAmFmCode(vibArray[i].amplitudeHigh, vibArray[i].frequencyHigh, pContextHigh);
        isSkippable &= (codesLow[i] == AmFmCodeNoAction);
        isSkippable &= (codesHigh[i] == AmFmCodeNoAction);
    }
    RestoreRelCountMax(pContextLow);
    RestoreRelCountMax(pContextHigh);
    // 2 番目と 3 番目の振動値も含めて更新不要ならばフォーマット Zero を返します
    if(isSkippable)
    {
        return VibrationAmFmPackFormat::Zero::Make();
    }

    return VibrationAmFmPackFormat::Three7bit::Make(amfm7bit0, codesLow, codesHigh, isFrequency, isHigh);
}

VibrationAmFmPack GeneratePackedAmFmCodes3(
    VibrationValue* vibArray,
    VibrationAmFmEncoderContext* pContextLow,
    VibrationAmFmEncoderContext* pContextHigh,
    AmFmCodeDicisionInfo* pInfoLow,
    AmFmCodeDicisionInfo* pInfoHigh) NN_NOEXCEPT
{
    // 振動値がすべて同一ならば One28bit を利用
    if(IsVibrationValuesSame(vibArray, 3))
    {
        return GeneratePackedAmFmCodes1(pContextLow, pContextHigh, pInfoLow, pInfoHigh);
    }

    // Three7bit{Amplitude,Frequency}{Low,High} ==> Three14bit{Low,High} ==> Three
    // の順で最適なフォーマットを探します

    int diffOnThree = 0;
    VibrationAmFmEncoderContext ctxLowOnThree = *pContextLow;
    VibrationAmFmEncoderContext ctxHighOnThree = *pContextHigh;
    int codesLow[3];
    int codesHigh[3];

    // Three フォーマットを選択した場合の誤差を算出
    {
        // 1 番目の振動値のエンコード
        diffOnThree += pInfoLow->GetDiffOnAmFm5bit() + pInfoHigh->GetDiffOnAmFm5bit();
        codesLow[0] = pInfoLow->ApplyAmFm5bitCode(&ctxLowOnThree);
        codesHigh[0] = pInfoHigh->ApplyAmFm5bitCode(&ctxHighOnThree);

        // 2 番目と 3 番目の振動値のエンコード
        for(int i = 1; i < 3; i++)
        {
            AmFmCodeDicisionInfo infoLow = AmFmCodeDicisionInfo::CalcInfo(
                vibArray[i].amplitudeLow, vibArray[i].frequencyLow, &ctxLowOnThree, true);
            AmFmCodeDicisionInfo infoHigh = AmFmCodeDicisionInfo::CalcInfo(
                vibArray[i].amplitudeHigh, vibArray[i].frequencyHigh, &ctxHighOnThree, true);
            diffOnThree += infoLow.GetDiffOnAmFm5bit() + infoHigh.GetDiffOnAmFm5bit();
            codesLow[i] = infoLow.ApplyAmFm5bitCode(&ctxLowOnThree);
            codesHigh[i] = infoHigh.ApplyAmFm5bitCode(&ctxHighOnThree);
        }
    }

    // Three7bit{Amplitude,Frequency}{Low,High}
    {
        // 4 つの候補の中で最も誤差が少なくなるものを選択
        int diffOnAmpLow = pInfoLow->GetDiffOnAm7bit() + pInfoHigh->GetDiffOnNoAction();
        int diffOnFreqLow = pInfoLow->GetDiffOnFm7bit() + pInfoHigh->GetDiffOnNoAction();
        int diffOnAmpHigh = pInfoLow->GetDiffOnNoAction() + pInfoHigh->GetDiffOnAm7bit();
        int diffOnFreqHigh = pInfoLow->GetDiffOnNoAction() + pInfoHigh->GetDiffOnFm7bit();
        int minDiff= diffOnAmpLow;
        if(diffOnFreqLow < minDiff) minDiff = diffOnFreqLow;
        if(diffOnAmpHigh < minDiff) minDiff = diffOnAmpHigh;
        if(diffOnFreqHigh < minDiff) minDiff = diffOnFreqHigh;

        // 誤差が十分小さければ Three7bit{Amplitude,Frequency}{Low,High} を採用
        if(minDiff <= diffOnThree)
        {
            bool isFrequency = false;
            bool isHigh = false;
            if(minDiff == diffOnAmpLow)
            {
                isFrequency = false;
                isHigh = false;
            }
            else if(minDiff == diffOnFreqLow)
            {
                isFrequency = true;
                isHigh = false;
            }
            else if(minDiff == diffOnAmpHigh)
            {
                isFrequency = false;
                isHigh = true;
            }
            else
            {
                isFrequency = true;
                isHigh = true;
            }
            return GenerateThree7bitFormatPack(vibArray, pContextLow, pContextHigh, pInfoLow, pInfoHigh, isFrequency, isHigh);
        }
    }

    // Three14bit{Low,High} (または One28bit / Zero)
    {
        // エンコード処理は 2 個の場合の処理を流用
        VibrationAmFmEncoderContext ctxLow = *pContextLow;
        VibrationAmFmEncoderContext ctxHigh = *pContextHigh;
        VibrationAmFmPack pack = GeneratePackedAmFmCodes2(vibArray, &ctxLow, &ctxHigh, pInfoLow, pInfoHigh);

        // Two14bit{Low,High} であれば Three14bit{Low,High} に変換
        if(pack.Get<VibrationAmFmPackFormat::Two14bit::Count>() == VibrationAmFmPackFormat::Two14bit::CountValueForTwo)
        {
            pack.Set<VibrationAmFmPackFormat::Two14bit::Count>(VibrationAmFmPackFormat::Two14bit::CountValueForThree);
        }

        // 3 番目の振動値の誤差を計測
        AmFmCodeDicisionInfo infoLow = AmFmCodeDicisionInfo::CalcInfo(
            vibArray[2].amplitudeLow, vibArray[2].frequencyLow, &ctxLow, false);
        AmFmCodeDicisionInfo infoHigh = AmFmCodeDicisionInfo::CalcInfo(
            vibArray[2].amplitudeHigh, vibArray[2].frequencyHigh, &ctxHigh, false);
        int diff = infoLow.GetDiffOnNoAction() + infoHigh.GetDiffOnNoAction();

        // 誤差が十分小さければ Three14bit{Low,High} を採用
        if(diff <= diffOnThree)
        {
            *pContextLow = ctxLow;
            *pContextHigh = ctxHigh;
            return pack;
        }
    }

    // Three を採用
    {
        *pContextLow = ctxLowOnThree;
        *pContextHigh = ctxHighOnThree;

        return VibrationAmFmPackFormat::Three::Make(codesLow, codesHigh);
    }
}

VibrationAmFmPack GeneratePackedAmFmCodesWith5bitCodes(
    VibrationValue* vibArray,
    VibrationAmFmEncoderContext* pContextLow,
    VibrationAmFmEncoderContext* pContextHigh,
    AmFmCodeDicisionInfo* pInfoLow,
    AmFmCodeDicisionInfo* pInfoHigh,
    int codePairCount) NN_NOEXCEPT
{
    int codesLow[3];
    int codesHigh[3];

    // 1 番目の振動値のエンコード
    codesLow[0] = pInfoLow->ApplyAmFm5bitCode(pContextLow);
    codesHigh[0] = pInfoHigh->ApplyAmFm5bitCode(pContextHigh);

    // 2 番目以降の振動値のエンコード
    for(int i = 1; i < codePairCount; i++)
    {
        AmFmCodeDicisionInfo infoLow = AmFmCodeDicisionInfo::CalcInfo(
            vibArray[i].amplitudeLow, vibArray[i].frequencyLow, pContextLow, true);
        AmFmCodeDicisionInfo infoHigh = AmFmCodeDicisionInfo::CalcInfo(
            vibArray[i].amplitudeHigh, vibArray[i].frequencyHigh, pContextHigh, true);
        codesLow[i] = infoLow.ApplyAmFm5bitCode(pContextLow);
        codesHigh[i] = infoHigh.ApplyAmFm5bitCode(pContextHigh);
    }

    switch(codePairCount)
    {
    case 1:
        return VibrationAmFmPackFormat::One::Make(codesLow[0], codesHigh[0]);
    case 2:
        return VibrationAmFmPackFormat::Two::Make(codesLow, codesHigh);
    case 3:
        return VibrationAmFmPackFormat::Three::Make(codesLow, codesHigh);
    default:
        return VibrationAmFmPackFormat::Zero::Make();
    }
}

} // anonymous-namespace

/**
 * AmFmCodePairCountDecider
 */
const VibrationCountDecider::Config VibrationCountDecider::s_Config5ms = {1, 1, 2};
const VibrationCountDecider::Config VibrationCountDecider::s_Config10ms = {2, 1, 4};
const VibrationCountDecider::Config VibrationCountDecider::s_Config15ms = {3, 0, 5};

void VibrationCountDecider::Reset() NN_NOEXCEPT
{
    m_PrevCount = 0;
    m_BufferPosition = 0;

    nn::os::Tick now = nn::os::GetSystemTick();
    for (int i = 0; i < BufferLength; i++)
    {
        m_TickBuffer[i] = now;
    }
}

int VibrationCountDecider::DecideNextCount(const VibratorBufferStatus& bufferStatus) NN_NOEXCEPT
{
    const Config& config = GetConfig();
    int nextCount = config.defaultCount;

    // バッファ残量レベルが params で指定された範囲の外にある場合に個数を調整する
    // ただし連続して増減が繰り返されるパターンに陥らないようにする
    if(bufferStatus.remainingDataLevel < config.dataLevelMin && m_PrevCount <= config.defaultCount)
    {
        nextCount++;
    }
    else if(bufferStatus.remainingDataLevel > config.dataLevelMax && m_PrevCount >= config.defaultCount)
    {
        nextCount--;
    }

    if(nextCount > CountMax) nextCount = CountMax;
    if(nextCount < CountMin) nextCount = CountMin;

    return nextCount;
}

void VibrationCountDecider::SetPrevCount(int prevCount) NN_NOEXCEPT
{
    m_PrevCount = prevCount;
}

const VibrationCountDecider::Config& VibrationCountDecider::GetConfig() NN_NOEXCEPT
{
    // 5ms, 10ms, 15ms の設定を切り替える閾値 (単位はマイクロ秒)
    const int IntervalThreshold1 = 7500;
    const int IntervalThreshold2 = 12500;

    // チック値を更新して時間間隔 (の移動平均) を取得
    nn::os::Tick now = nn::os::GetSystemTick();
    int interval = static_cast<int>((now - m_TickBuffer[m_BufferPosition]).ToTimeSpan().GetMicroSeconds()) / BufferLength;
    m_TickBuffer[m_BufferPosition] = now;
    m_BufferPosition = (m_BufferPosition + 1) % BufferLength;

    // 時間間隔に応じて Config を返す
    if (interval < IntervalThreshold1)
    {
        return s_Config5ms;
    }
    else if (interval < IntervalThreshold2)
    {
        return s_Config10ms;
    }
    else
    {
        return s_Config15ms;
    }
}


/**
 * VibrationQueue
 */
void VibrationQueue::Clear() NN_NOEXCEPT
{
    m_EnqueueIdx = 0;
    m_DequeueIdx = 0;
    for(int i = 0; i < QueueLength; i++)
    {
        m_Queue[i].amplitudeLow = 0.0f;
        m_Queue[i].frequencyLow = VibratorAgent::BaseFrequencyLow;
        m_Queue[i].amplitudeHigh = 0.0f;
        m_Queue[i].frequencyHigh = VibratorAgent::BaseFrequencyHigh;
    }
}

bool VibrationQueue::IsFull() const NN_NOEXCEPT
{
    int nextEnqueueIdx = (m_EnqueueIdx + 1) % QueueLength;
    return (nextEnqueueIdx == m_DequeueIdx);
}

bool VibrationQueue::IsEmpty() const NN_NOEXCEPT
{
    return (m_EnqueueIdx == m_DequeueIdx);
}

int VibrationQueue::GetCount() const NN_NOEXCEPT
{
    return (m_EnqueueIdx + QueueLength - m_DequeueIdx) % QueueLength;
}

void VibrationQueue::Enqueue(const VibrationValue& value) NN_NOEXCEPT
{
    if(IsFull())
    {
        int prevIdx = (m_EnqueueIdx + QueueLength - 1) % QueueLength;
        m_Queue[prevIdx] = value;
    }
    else
    {
        m_Queue[m_EnqueueIdx] = value;
        m_EnqueueIdx = (m_EnqueueIdx + 1) % QueueLength;
    }
}

const VibrationValue& VibrationQueue::Dequeue() NN_NOEXCEPT
{
    if(IsEmpty())
    {
        int prevIdx = (m_DequeueIdx + QueueLength - 1) % QueueLength;
        return m_Queue[prevIdx];
    }
    else
    {
        const VibrationValue& v = m_Queue[m_DequeueIdx];
        m_DequeueIdx = (m_DequeueIdx + 1) % QueueLength;
        return v;
    }
}


/**
 * VibratorAgent
 */
const float VibratorAgent::BaseFrequencyLow = 160.0f;
const float VibratorAgent::BaseFrequencyHigh = 320.0f;
const float VibratorAgent::AmplitudeLimitCoeff = 0.5556f;

void VibratorAgent::Activate(DeviceType type) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    Reset();
    m_Type = type;
    m_IsActivated = true;
}

void VibratorAgent::Deactivate() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    m_IsActivated = false;
    Reset();
}

void VibratorAgent::Reset() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    m_CountDecider.Reset();
    m_Queue.Clear();

    ResetVibrationAmFmEncoderContext(&m_EncoderContextLow, BaseFrequencyLow);
    ResetVibrationAmFmEncoderContext(&m_EncoderContextHigh, BaseFrequencyHigh);

    // 7bit 符号を使えない場合は旧フォーマットに合わせてリセットする
    if(!m_IsAmFm7bitCodeAvailable)
    {
        const int RelAmCodeCountMaxFor5bit = 100;
        const int RelFmCodeCountMaxFor5bit = 100;

        m_EncoderContextLow._relAmpCountMax = RelAmCodeCountMaxFor5bit;
        m_EncoderContextLow._relFreqCountMax = RelFmCodeCountMaxFor5bit;
        m_EncoderContextHigh._relAmpCountMax = RelAmCodeCountMaxFor5bit;
        m_EncoderContextHigh._relFreqCountMax = RelFmCodeCountMaxFor5bit;
    }

    m_BufferStatus.isBufferUnderrun = false;
    m_BufferStatus.remainingDataLevel = 0;
}

void VibratorAgent::ParseInputReport(uint8_t report) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    if(m_IsActivated)
    {
        m_BufferStatus.isBufferUnderrun = ((report & BufferUnderrunMask) != 0);
        m_BufferStatus.remainingDataLevel = (report & RemainingDataLevelMask);
    }
}

int VibratorAgent::GetOutputReport(uint8_t* pOutValue) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    NN_SDK_ASSERT_NOT_NULL(pOutValue);
    if(m_IsActivated)
    {
        int codePairCount = m_CountDecider.DecideNextCount(m_BufferStatus);
        uint32_t report = GeneratePackedAmFmCodes(codePairCount).storage;

        if(report != 0)
        {
            for(size_t i = 0; i < sizeof(uint32_t); i++)
            {
                // Little Endian
                pOutValue[i] = static_cast<uint8_t>((report >> (i * 8)) & 0xFF);
            }
            return sizeof(uint32_t);
        }
    }

    memset(pOutValue, 0, sizeof(uint32_t));
    return 0;
}

void VibratorAgent::GetBufferStatus(VibratorBufferStatus* pOutValue) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    NN_SDK_ASSERT_NOT_NULL(pOutValue);
    if(m_IsActivated)
    {
        *pOutValue = m_BufferStatus;
    }
}

void VibratorAgent::GetRequiredVibrationValueCount(int* pOutValue) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    NN_SDK_ASSERT_NOT_NULL(pOutValue);
    if(m_IsActivated)
    {
        int required = m_CountDecider.DecideNextCount(m_BufferStatus) - m_Queue.GetCount();
        if(required < 0) required = 0;
        *pOutValue = required;
    }
    else
    {
        *pOutValue = 0;
    }
}

Result VibratorAgent::SendVibrationValue(const VibrationValue& value) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    NN_RESULT_THROW_UNLESS(m_IsActivated, ResultVibratorNotEnabled());

    // 接続時振動を再生中はエンキューせず処理を終了する
    if (m_pVibrationOnConnect != nullptr && m_pVibrationOnConnect->IsPlaying())
    {
        NN_RESULT_SUCCESS;
    }

    bool wasFull = m_Queue.IsFull();
    m_Queue.Enqueue(value);
    NN_RESULT_THROW_UNLESS(!wasFull, ResultVibrationDataQueueFull());
    NN_RESULT_SUCCESS;
}

void VibratorAgent::GetActualVibrationValue(VibrationValue* pOutValue) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    NN_SDK_ASSERT_NOT_NULL(pOutValue);
    if(m_IsActivated)
    {
        pOutValue->amplitudeLow = GetVibrationAmplitudeFromAmFmDecoder(&m_EncoderContextLow._decoderContext);
        pOutValue->frequencyLow = GetVibrationFrequencyFromAmFmDecoder(&m_EncoderContextLow._decoderContext);
        pOutValue->amplitudeHigh = GetVibrationAmplitudeFromAmFmDecoder(&m_EncoderContextHigh._decoderContext);
        pOutValue->frequencyHigh = GetVibrationFrequencyFromAmFmDecoder(&m_EncoderContextHigh._decoderContext);

        RevertEqualizer(
            &(pOutValue->amplitudeLow),
            &(pOutValue->amplitudeHigh),
            pOutValue->frequencyLow,
            pOutValue->frequencyHigh);

        if(m_IsAmplitudeLimitEnabled)
        {
            pOutValue->amplitudeLow /= AmplitudeLimitCoeff;
            pOutValue->amplitudeHigh /= AmplitudeLimitCoeff;
        }
    }
}

void VibratorAgent::SetCalibrationValue(const VibratorCalibrationValue& value) NN_NOEXCEPT
{
    // 未実装
    NN_UNUSED(value);
}

void VibratorAgent::GetCalibrationValue(VibratorCalibrationValue* pOutValue) NN_NOEXCEPT
{
    // 未実装
    NN_UNUSED(pOutValue);
}

void VibratorAgent::SetAmplitudeLimitEnabled(bool isEnabled) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    m_IsAmplitudeLimitEnabled = isEnabled;
}

bool VibratorAgent::IsAmplitudeLimitEnabled() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    return m_IsAmplitudeLimitEnabled;
}

void VibratorAgent::SetEqualizerEnabled(bool isEnabled) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    m_IsEqualizerEnabled = isEnabled;
}

bool VibratorAgent::IsEqualizerEnabled() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    return m_IsEqualizerEnabled;
}

void VibratorAgent::SetAmFm7bitCodeAvailable(bool isAvailable) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    if(m_IsAmFm7bitCodeAvailable != isAvailable)
    {
        m_IsAmFm7bitCodeAvailable = isAvailable;
        Reset();
    }
}

bool VibratorAgent::IsAmFm7bitCodeAvailable() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    return m_IsAmFm7bitCodeAvailable;
}

VibrationAmFmPack VibratorAgent::GeneratePackedAmFmCodes(int codePairCount) NN_NOEXCEPT
{
    if(codePairCount == 0)
    {
        m_CountDecider.SetPrevCount(0);
        return VibrationAmFmPackFormat::Zero::Make();
    }

    // マスターボリュームを取得
    float masterVolume = 0.0f;
    (void)GetVibrationMasterVolume(&masterVolume);

    // キューから振動値を取り出す
    const int MaxVibrationCount = 3;
    VibrationValue vibArray[MaxVibrationCount];

    // スタックオーバーフローを防ぐ
    if (codePairCount > MaxVibrationCount)
    {
        codePairCount = MaxVibrationCount;
    }

    for(int i = 0; i < codePairCount; i++)
    {
        if (m_pVibrationOnConnect != nullptr && m_pVibrationOnConnect->IsPlaying())
        {
            vibArray[i] = m_pVibrationOnConnect->GetNextVibration();
        }
        else
        {
            vibArray[i] = m_Queue.Dequeue();
        }

        // 振幅値の調整
        LimitAmplitude(&vibArray[i].amplitudeLow, &vibArray[i].amplitudeHigh);
        ApplyEqualizer(&vibArray[i].amplitudeLow, &vibArray[i].amplitudeHigh, vibArray[i].frequencyLow, vibArray[i].frequencyHigh);
        ApplyMasterVolume(&vibArray[i].amplitudeLow, &vibArray[i].amplitudeHigh, masterVolume);
    }

    // エンコード結果がフォーマット Zero だったときのために元のコンテクストを保持
    VibrationAmFmEncoderContext savedCtxLow = m_EncoderContextLow;
    VibrationAmFmEncoderContext savedCtxHigh = m_EncoderContextHigh;

    // 最初の振動値を試しにエンコード
    AmFmCodeDicisionInfo infoLow = AmFmCodeDicisionInfo::CalcInfo(
        vibArray[0].amplitudeLow, vibArray[0].frequencyLow, &m_EncoderContextLow, true);
    AmFmCodeDicisionInfo infoHigh = AmFmCodeDicisionInfo::CalcInfo(
        vibArray[0].amplitudeHigh, vibArray[0].frequencyHigh, &m_EncoderContextHigh, true);

    // 詰め込む振動値の個数に応じてフォーマットを決定
    VibrationAmFmPack pack = VibrationAmFmPackFormat::Zero::Make();
    if(m_IsAmFm7bitCodeAvailable)
    {
        switch(codePairCount)
        {
        case 1:
            pack = GeneratePackedAmFmCodes1(&m_EncoderContextLow, &m_EncoderContextHigh, &infoLow, &infoHigh);
            break;
        case 2:
            pack = GeneratePackedAmFmCodes2(vibArray, &m_EncoderContextLow, &m_EncoderContextHigh, &infoLow, &infoHigh);
            break;
        case 3:
            pack = GeneratePackedAmFmCodes3(vibArray, &m_EncoderContextLow, &m_EncoderContextHigh, &infoLow, &infoHigh);
            break;
        default:
            break;
        }
    }
    else
    {
        // 7bit 符号を使えないので 5bit 符号のみを利用
        pack = GeneratePackedAmFmCodesWith5bitCodes(vibArray,
            &m_EncoderContextLow, &m_EncoderContextHigh,
            &infoLow, &infoHigh, codePairCount);
    }

    // CountDecider を更新
    if(pack.storage == 0)
    {
        // データ送信を省略するので元のコンテクストに戻す (relCount だけ増加させておく)
        uint16_t relCountOffset = static_cast<uint16_t>(codePairCount);
        savedCtxLow._relAmpCount += relCountOffset;
        savedCtxLow._relFreqCount += relCountOffset;
        savedCtxHigh._relAmpCount += relCountOffset;
        savedCtxHigh._relFreqCount += relCountOffset;
        m_EncoderContextLow = savedCtxLow;
        m_EncoderContextHigh = savedCtxHigh;

        m_CountDecider.SetPrevCount(0);
    }
    else
    {
        m_CountDecider.SetPrevCount(codePairCount);
    }

    return pack;
}

void VibratorAgent::LimitAmplitude(float* pAmplitudeLow, float* pAmplitudeHigh) NN_NOEXCEPT
{
    if(m_IsAmplitudeLimitEnabled)
    {
        float coeff = AmplitudeLimitCoeff;
        float sum = *pAmplitudeLow + *pAmplitudeHigh;
        if(sum > 1.0f)
        {
            coeff /= sum;
        }
        *pAmplitudeLow *= coeff;
        *pAmplitudeHigh *= coeff;
    }
}

float VibratorAgent::GetEqualizerCoeff(float frequency) NN_NOEXCEPT
{
    struct CoeffTableElement
    {
        float frequency;
        float coeff;
    };

    // ハードウェア検証から得られた係数をハードコード for JoyCon
    const CoeffTableElement CoeffTableForJoyCon[] = {
        { 170.0f, 1.00f }, { 178.0f, 0.7344f }, { 190.0f, 0.7344f }, { 200.0f, 1.00f },
        { 310.0f, 1.00f }, { 322.0f, 0.6156f }, { 344.0f, 0.6156f }, { 356.0f, 1.00f } };

    // ハードウェア検証から得られた係数をハードコード for FullCon
    const CoeffTableElement CoeffTableForFullCon[] = {
        { 322.0f, 1.00f   }, { 344.0f, 0.6156f }, { 356.0f, 1.00f } };

    //FullConかJoyConを判別してリミッターの値を設定する
    const CoeffTableElement *CoeffTable = (m_Type == DeviceType_FullKey) ? CoeffTableForFullCon : CoeffTableForJoyCon;

    //ポインタ変数からstd::extentで要素数を求める方法がわからなかったため、直接配列を参照している。
    const int CoeffTableLength = (m_Type == DeviceType_FullKey) ? std::extent<decltype(CoeffTableForFullCon)>::value :
        std::extent<decltype(CoeffTableForJoyCon)>::value;

    for(int i = 0; i < CoeffTableLength; i++)
    {
        if(frequency < CoeffTable[i].frequency)
        {
            if(i == 0)
            {
                return CoeffTable[0].coeff;
            }
            else
            {
                // 線形補間で係数を算出
                return CoeffTable[i - 1].coeff
                    + (frequency - CoeffTable[i - 1].frequency)
                    * (CoeffTable[i].coeff - CoeffTable[i - 1].coeff)
                    / (CoeffTable[i].frequency - CoeffTable[i - 1].frequency);
            }
        }
    }
    return CoeffTable[CoeffTableLength - 1].coeff;
}

void VibratorAgent::ApplyEqualizer(float* pAmplitudeLow, float* pAmplitudeHigh, float frequencyLow, float frequencyHigh) NN_NOEXCEPT
{
    if(m_IsEqualizerEnabled)
    {
        *pAmplitudeLow *= GetEqualizerCoeff(frequencyLow);
        *pAmplitudeHigh *= GetEqualizerCoeff(frequencyHigh);
    }
}

void VibratorAgent::RevertEqualizer(float* pAmplitudeLow, float* pAmplitudeHigh, float frequencyLow, float frequencyHigh) NN_NOEXCEPT
{
    if(m_IsEqualizerEnabled)
    {
        *pAmplitudeLow /= GetEqualizerCoeff(frequencyLow);
        *pAmplitudeHigh /= GetEqualizerCoeff(frequencyHigh);
    }
}

void VibratorAgent::ApplyMasterVolume(float* pAmplitudeLow, float* pAmplitudeHigh, float masterVolume) NN_NOEXCEPT
{
    // とりあえず単純に乗算する
    *pAmplitudeLow *= masterVolume;
    *pAmplitudeHigh *= masterVolume;
}

}} // namespace nn::xcd
