﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   AMFM 符号コーデック内で用いる型と変換関数の定義
 */

#pragma once

#include <cstdint>
#include <nn/nn_Macro.h>

namespace nn { namespace xcd {

/**
 * @brief   固定小数点数を提供する構造体です。
 * @tparam  ScaleBit    小数点以下のビット幅数。(0 < ScaleBit < 32)
 * @tparam  Offset      ゼロオフセット値。(Offset >= 0)
 * @details
 *  Offset に 0 を指定すると通常の符号なし固定小数点型となります。
 *  Offset に正の値を指定すると内部値は常に Offset だけ加算された値で保持されることになり、
 *  結果として -Offset 以上の負の値を表現できる型になります。
 */
template<int TScaleBit, int TOffset = 0>
struct VibrationFixedPointT
{
    static const int ScaleBit = TScaleBit;                                              //!< 小数点以下のビット幅
    static const int Offset = TOffset;                                                  //!< ゼロオフセット値
    static const uint32_t Scale = static_cast<uint32_t>(1) << ScaleBit;                 //!< float 値と相互に変換する際に用いるスケール
    static const uint32_t ScaledOffset = static_cast<uint32_t>(Offset) << ScaleBit;     //!< 内部値に用いるゼロオフセット値
    static const uint32_t IntPartMask = static_cast<uint32_t>(0xffffffff) << ScaleBit;  //!< 整数部分を取り出すためのマスク
    static const uint32_t FracPartMask = ~IntPartMask;                                  //!< 小数部分を取り出すためのマスク

    typedef uint32_t Type;  //!< 値の実体を保持する型

    //!< float に変換します。
    static inline float GetFloat(const Type& arg) NN_NOEXCEPT
    {
        return (arg - ScaledOffset) / static_cast<float>(Scale);
    }

    //!< float から生成します。
    static inline Type FromFloat(const float& f) NN_NOEXCEPT
    {
        return static_cast<Type>((f + Offset) * Scale);
    }

    //!< 2 つの値を足した値を返します。
    static inline Type GetSum(const Type& arg1, const Type& arg2) NN_NOEXCEPT
    {
        return (arg1 + arg2) - ScaledOffset;
    }

    //!< 指定された範囲に収まるようクランプした値を返します。
    static inline Type GetClamped(const Type& arg, const Type& min, const Type& max) NN_NOEXCEPT
    {
        if(arg < min) return min;
        if(arg > max) return max;
        return arg;
    }

    //!< 指定された値以下の最大の整数を int で返します。
    static inline int GetFloor(const Type& arg) NN_NOEXCEPT
    {
        return static_cast<int>(arg >> ScaleBit) - Offset;
    }

    //!< 指定された値の小数部分をそのまま uint32_t で返します。
    static inline uint32_t GetFracPartIdx(const Type& arg) NN_NOEXCEPT
    {
        return arg & FracPartMask;
    }

    //!< 2 つの値の間の距離を uint32_t で返します。
    static inline uint32_t GetDistance(const Type& arg1, const Type& arg2) NN_NOEXCEPT
    {
        return (arg1 >= arg2) ? (arg1 - arg2) : (arg2 - arg1);
    }
};

typedef VibrationFixedPointT<20> VibrationAmpFreqT; //!< 振幅値や周波数値を扱うための固定小数点型 (符号無し)
typedef VibrationFixedPointT<5, 256> VibrationLogT; //!< 対数値を扱うための固定小数点型 (符号付き)

//!< べき乗計算を提供する構造体
struct VibrationPowTable
{
    typedef VibrationFixedPointT<31> PowDataT;  //!< べき乗計算に必要なデータを扱うための固定小数点型 (符号無し)
    static const PowDataT::Type s_Table[];      //!< べき乗計算が内部的に参照するテーブル

    //!< 2 のべき乗計算をします。
    static inline VibrationAmpFreqT::Type CalcPow(const VibrationLogT::Type& exponent) NN_NOEXCEPT
    {
        PowDataT::Type powValue = s_Table[VibrationLogT::GetFracPartIdx(exponent)];
        int shiftWidth = PowDataT::ScaleBit - VibrationAmpFreqT::ScaleBit - VibrationLogT::GetFloor(exponent);
        return powValue >> shiftWidth;
    }
};

}} // namespace nn::xcd
