﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   AMFM 符号のエンコーダー (内部実装)
 */

#pragma once

#include "xcd_VibrationAmFmDecoder.h"

namespace nn { namespace xcd {

/**
 * @brief       AMFM エンコーダを表すクラスです。
 * @details
 *  公開ヘッダのエンコード関連 API が振幅や周波数を float で扱うのに対し、
 *  このクラスでは高速化のため固定小数点型で扱います。
 *
 *  静的メンバ関数はエンコーダコンテクストを引数にとり、
 *  エンコーダのインスタンスを生成しなくてもエンコード処理を実行できる機能を提供します。@n
 *  静的ではないメンバ関数を実行すると、エンコーダのインスタンスが
 *  内部で保持しているエンコーダコンテクストに対して処理が実行されます。
 */
class VibrationAmFmEncoder
{
    NN_DISALLOW_COPY(VibrationAmFmEncoder);
    NN_DISALLOW_MOVE(VibrationAmFmEncoder);

public:
    VibrationAmFmEncoder() NN_NOEXCEPT {}
    ~VibrationAmFmEncoder() NN_NOEXCEPT {}

    //!< このエンコーダインスタンスを初期化します。
    void Reset(VibrationAmpFreqT::Type baseFrequency) NN_NOEXCEPT
    {
        Reset(&m_Context, baseFrequency);
    }

    //!< 振幅値と周波数値をエンコードして内部状態を更新し、AMFM 符号を取得します。
    int Encode(VibrationAmpFreqT::Type amplitude, VibrationAmpFreqT::Type frequency) NN_NOEXCEPT
    {
        return Encode(amplitude, frequency, &m_Context);
    }

    //!< 振幅値をエンコードして内部状態を更新し、7bit AM 符号を取得します。
    int EncodeToAm7bitCode(VibrationAmpFreqT::Type amplitude) NN_NOEXCEPT
    {
        return EncodeToAm7bitCode(amplitude, &m_Context);
    }

    //!< 周波数値をエンコードして内部状態を更新し、7bit FM 符号を取得します。
    int EncodeToFm7bitCode(VibrationAmpFreqT::Type frequency) NN_NOEXCEPT
    {
        return EncodeToFm7bitCode(frequency, &m_Context);
    }

public:
    //!< コンテクストをリセットします。
    static inline void Reset(
        VibrationAmFmEncoderContext* pContext,
        VibrationAmpFreqT::Type baseFrequency) NN_NOEXCEPT
    {
        VibrationAmFmDecoder::Reset(
            &pContext->_decoderContext,
            VibrationAmFmFormat_5BitVer2,   // エンコーダが対応するのは 5Bit AMFM 符号テーブル Ver.2.0 のみ
            baseFrequency);
        pContext->_relAmpCountMax = DefaultRelAmCodeCountMax;   // 振幅相対指定型コマンドはここで指定した値を超えて連続しない
        pContext->_relFreqCountMax = DefaultRelFmCodeCountMax;  // 周波数相対指定型コマンドはここで指定した値を超えて連続しない
        pContext->_relAmpCount = pContext->_relAmpCountMax;
        pContext->_relFreqCount = pContext->_relFreqCountMax;
    }

    //!< 振幅値と周波数値をエンコードしてコンテクストを更新し、AMFM 符号を取得します。
    static inline int Encode(
        VibrationAmpFreqT::Type amplitude,
        VibrationAmpFreqT::Type frequency,
        VibrationAmFmEncoderContext* pContext) NN_NOEXCEPT
    {
        // エンコーダが対応するのは 5Bit AMFM 符号テーブル Ver.2.0 のみ
        static const VibrationAmFmCommand::Config* pConfig = VibrationAmFmCommand::GetConfig(VibrationAmFmFormat_5BitVer2);
        pContext->_decoderContext._pConfig = pConfig;

        // 周波数値は baseFreq との比率として扱う
        frequency /= (pContext->_decoderContext._baseFreq >> VibrationAmpFreqT::ScaleBit);

        // クランプ処理
        amplitude = VibrationAmpFreqT::GetClamped(amplitude, MinAmp, MaxAmp);
        frequency = VibrationAmpFreqT::GetClamped(frequency, MinFreq, MaxFreq);

        // AMFM 符号の候補を取得
        int candidates[CandidateCodesCountMax];
        int candidatesCount = GetCandidateCodes(candidates, amplitude, frequency, pContext);

        // 候補の中から最も理想に近いものを採用
        int argminCode = candidates[0];
        uint32_t minDistance = CalcDistance(candidates[0], amplitude, frequency, &pContext->_decoderContext);
        for(int i = 1; i < candidatesCount; i++)
        {
            uint32_t distance = CalcDistance(candidates[i], amplitude, frequency, &pContext->_decoderContext);
            if(distance < minDistance)
            {
                argminCode = candidates[i];
                minDistance = distance;
            }
        }

        // 計算結果をコンテクストに反映
        VibrationAmFmDecoder::Decode(argminCode, &pContext->_decoderContext);
        UpdateRelativeCodeCount(argminCode, pContext);

        return argminCode;
    }

    //!< 振幅値をエンコードしてコンテクストを更新し、7bit AM 符号を取得します。
    static inline int EncodeToAm7bitCode(
        VibrationAmpFreqT::Type amplitude,
        VibrationAmFmEncoderContext* pContext) NN_NOEXCEPT
    {
        // 7bit AM 符号の中で最適な候補を選出
        int am7bitCode = GetRangeIdx(amplitude, Am7bitThresholdArray, Am7bitIdxArray, Am7bitIdxCount);

        // 計算結果をコンテクストに反映
        VibrationAmFmDecoder::DecodeAm7bitCode(am7bitCode, &pContext->_decoderContext);
        pContext->_relAmpCount = 0;

        return am7bitCode;
    }

    //!< 周波数値をエンコードしてコンテクストを更新し、7bit FM 符号を取得します。
    static inline int EncodeToFm7bitCode(
        VibrationAmpFreqT::Type frequency,
        VibrationAmFmEncoderContext* pContext) NN_NOEXCEPT
    {
        // 周波数値は baseFreq との比率として扱う
        frequency /= (pContext->_decoderContext._baseFreq >> VibrationAmpFreqT::ScaleBit);

        // 7bit AM 符号の中で最適な候補を選出
        int fm7bitCode = GetRangeIdx(frequency, Fm7bitThresholdArray, Fm7bitIdxArray, Fm7bitIdxCount);

        // 計算結果をコンテクストに反映
        VibrationAmFmDecoder::DecodeFm7bitCode(fm7bitCode, &pContext->_decoderContext);
        pContext->_relFreqCount = 0;

        return fm7bitCode;
    }

private:
    // 相対指定型コマンドの連続上限数のデフォルト値
    static const int DefaultRelAmCodeCountMax = 20;
    static const int DefaultRelFmCodeCountMax = 20;

    // 内部のクランプ処理に利用する最小値・最大値
    static const VibrationAmpFreqT::Type MinAmp;
    static const VibrationAmpFreqT::Type MaxAmp;
    static const VibrationAmpFreqT::Type MinFreq;
    static const VibrationAmpFreqT::Type MaxFreq;

    // AMFM 符号の候補の最大数
    static const int CandidateCodesCountMax = 4;

    // AMFM 符号が絶対指定型か判別する際に利用する定数
    static const int AbsAmCodeMin = 1;
    static const int AbsAmCodeMax = 11;
    static const int AbsFmCodeMin = 12;
    static const int AbsFmCodeMax = 16;

    // 振幅値絶対指定型コマンド (Set 型コマンド) の選出に利用する定数
    static const int AbsAmIdxCount = 11;
    static const int AbsAmIdxArray[AbsAmIdxCount];
    static const VibrationAmpFreqT::Type AbsAmThresholdArray[AbsAmIdxCount - 1];

    // 周波数値絶対指定型コマンド (Set 型コマンド) の選出に利用する定数
    static const int AbsFmIdxCount = 5;
    static const int AbsFmIdxArray[AbsFmIdxCount];
    static const VibrationAmpFreqT::Type AbsFmThresholdArray[AbsFmIdxCount - 1];

    // 相対値指定型コマンド (Multiply 型コマンド) の選出に利用する定数
    static const int RelAmFmBaseCode = 0x18;
    static const int RelAmFmIdxCountA = 5;
    static const int RelAmFmOffsetArrayA[RelAmFmIdxCountA];
    static const int RelAmFmIdxCountF = 3;
    static const int RelAmFmOffsetArrayF[RelAmFmIdxCountF];

    // 7bit AM 符号の選出に利用する定数
    static const int Am7bitIdxCount = 128;
    static const int Am7bitIdxArray[Am7bitIdxCount];
    static const VibrationAmpFreqT::Type Am7bitThresholdArray[Am7bitIdxCount - 1];

    // 7bit FM 符号の選出に利用する定数
    static const int Fm7bitIdxCount = 127;
    static const int Fm7bitIdxArray[Fm7bitIdxCount];
    static const VibrationAmpFreqT::Type Fm7bitThresholdArray[Fm7bitIdxCount - 1];

    // 距離計算時に利用する定数
    static const int DistanceCoeffA = 10;
    static const int DistanceCoeffF = 1;

private:
    VibrationAmFmEncoderContext m_Context;  //!< インスタンスが内部に保持するコンテクスト

private:
    //!< 指定された値が属している範囲を特定して、その範囲に対応した番号を返します。
    static inline int GetRangeIdx(
        const VibrationAmpFreqT::Type& target,          //!< 処理対象の値
        const VibrationAmpFreqT::Type thresholdArray[], //!< 範囲の境界を表す閾値の配列 (長さは rangeCount - 1)
        const int idxArray[],                           //!< 範囲に対応する番号の配列 (長さは rangeCount)
        int rangeCount                                  //!< 範囲の個数
        ) NN_NOEXCEPT
    {
        // 処理対象の値と閾値との大小関係を二分探索で判定
        int startIdx = 0;
        int endIdx = rangeCount - 1;
        while(startIdx < endIdx)
        {
            int midIdx = (startIdx + endIdx) / 2;
            if(target < thresholdArray[midIdx])
            {
                endIdx = midIdx;
            }
            else
            {
                startIdx = midIdx + 1;
            }
        }
        return idxArray[startIdx];
    }

    //!< AMFM 符号の候補を生成して、生成した候補の個数を返します。
    static inline int GetCandidateCodes(
        int* pOutArray,
        const VibrationAmpFreqT::Type& amplitude,
        const VibrationAmpFreqT::Type& frequency,
        const VibrationAmFmEncoderContext* pContext) NN_NOEXCEPT
    {
        int cnt = 0;
        bool isRelAbsAllowed = (pContext->_relAmpCount < pContext->_relAmpCountMax);
        bool isRelFreqAllowed = (pContext->_relFreqCount < pContext->_relFreqCountMax);

        // 相対値指定型コマンド
        if(isRelAbsAllowed && isRelFreqAllowed)
        {
            pOutArray[cnt] = GetRelativeAmFmCode(amplitude, frequency, &pContext->_decoderContext);
            cnt++;
        }

        // Reset コマンド
        pOutArray[cnt] = 0;
        cnt++;

        // 振幅値絶対指定型コマンド（＝周波数相対指定コマンド）
        if(isRelFreqAllowed)
        {
            pOutArray[cnt] = GetAbsoluteAmCode(amplitude);
            cnt++;
        }

        // 周波数値絶対指定型コマンド（＝振幅相対指定コマンド）
        if(isRelAbsAllowed)
        {
            pOutArray[cnt] = GetAbsoluteFmCode(frequency);
            cnt++;
        }

        return cnt;
    }

    //!< AMFM 符号が絶対指定型か相対指定型か判別します。
    static inline bool IsAbsoluteAmCode(int code) NN_NOEXCEPT
    {
        return (code == 0) || (AbsAmCodeMin <= code && code <= AbsAmCodeMax);
    }

    //!< AMFM 符号が絶対指定型か相対指定型か判別します。
    static inline bool IsAbsoluteFmCode(int code) NN_NOEXCEPT
    {
        return (code == 0) || (AbsFmCodeMin <= code && code <= AbsFmCodeMax);
    }

    //!< エンコーダコンテクスト内の相対指定型コマンドの連続数を更新します。
    static inline void UpdateRelativeCodeCount(int code, VibrationAmFmEncoderContext* pContext) NN_NOEXCEPT
    {
        // 振幅相対指定型コマンドの連続数を更新
        if(IsAbsoluteAmCode(code))
        {
            pContext->_relAmpCount = 0;
        }
        else
        {
            pContext->_relAmpCount += 1;
        }

        // 周波数相対指定型コマンドの連続数を更新
        if(IsAbsoluteFmCode(code))
        {
            pContext->_relFreqCount = 0;
        }
        else
        {
            pContext->_relFreqCount += 1;
        }
    }

    //!< 振幅値絶対指定型コマンド (Set 型コマンド) の中で最適な候補を選出します。
    static inline int GetAbsoluteAmCode(const VibrationAmpFreqT::Type& amplitude) NN_NOEXCEPT
    {
        return GetRangeIdx(amplitude, AbsAmThresholdArray, AbsAmIdxArray, AbsAmIdxCount);
    }

    //!< 周波数値絶対指定型コマンド (Set 型コマンド) の中で最適な候補を選出します。
    static inline int GetAbsoluteFmCode(const VibrationAmpFreqT::Type& frequency) NN_NOEXCEPT
    {
        return GetRangeIdx(frequency, AbsFmThresholdArray, AbsFmIdxArray, AbsFmIdxCount);
    }

    // 相対値指定型コマンド (Multiply 型コマンド) の中で最適な候補を選出します。
    static inline int GetRelativeAmFmCode(
        const VibrationAmpFreqT::Type& amplitude,
        const VibrationAmpFreqT::Type& frequency,
        const VibrationAmFmDecoderContext* pContext) NN_NOEXCEPT
    {
        VibrationAmpFreqT::Type ampArray[RelAmFmIdxCountA];
        VibrationAmpFreqT::Type freqArray[RelAmFmIdxCountF];
        VibrationAmpFreqT::Type ampThresholdArray[RelAmFmIdxCountA - 1];
        VibrationAmpFreqT::Type freqThresholdArray[RelAmFmIdxCountF - 1];

        // 閾値を生成
        for(int i = 0; i < RelAmFmIdxCountA; i++)
        {
            VibrationAmFmDecoderContext dummyContext = *pContext;
            int amFmCode = RelAmFmBaseCode + RelAmFmOffsetArrayA[i];
            VibrationAmFmDecoder::Decode(amFmCode, &dummyContext);
            ampArray[i] = VibrationPowTable::CalcPow(dummyContext._logAmp);
        }
        for(int i = 0; i < RelAmFmIdxCountF; i++)
        {
            VibrationAmFmDecoderContext dummyContext = *pContext;
            int amFmCode = RelAmFmBaseCode + RelAmFmOffsetArrayF[i];
            VibrationAmFmDecoder::Decode(amFmCode, &dummyContext);
            freqArray[i] = VibrationPowTable::CalcPow(dummyContext._logFreq);
        }
        for(int i = 0; i < RelAmFmIdxCountA - 1; i++)
        {
            ampThresholdArray[i] = (ampArray[i] + ampArray[i + 1]) / 2;
            if(i >= RelAmFmIdxCountA / 2)
            {
                // 真ん中の範囲を少しだけ広めにとるための措置
                ampThresholdArray[i] += 1;
            }
        }
        for(int i = 0; i < RelAmFmIdxCountF - 1; i++)
        {
            freqThresholdArray[i] = (freqArray[i] + freqArray[i + 1]) / 2;
            if(i >= RelAmFmIdxCountF / 2)
            {
                // 真ん中の範囲を少しだけ広めにとるための措置
                freqThresholdArray[i] += 1;
            }
        }

        int ampOffset = GetRangeIdx(amplitude, ampThresholdArray, RelAmFmOffsetArrayA, RelAmFmIdxCountA);
        int freqOffset = GetRangeIdx(frequency, freqThresholdArray, RelAmFmOffsetArrayF, RelAmFmIdxCountF);
        return RelAmFmBaseCode + ampOffset + freqOffset;
    }

    //!< 指定された AMFM 符号を採用した場合に、目標値とどれくらいかけ離れた値になるか取得します。
    static inline uint32_t CalcDistance(
        int amFmCode,
        const VibrationAmpFreqT::Type& targetAmplitude,
        const VibrationAmpFreqT::Type& targetFrequency,
        const VibrationAmFmDecoderContext* pContext) NN_NOEXCEPT
    {
        VibrationAmFmDecoderContext dummyContext = *pContext;
        VibrationAmFmDecoder::Decode(amFmCode, &dummyContext);
        VibrationAmpFreqT::Type a = VibrationPowTable::CalcPow(dummyContext._logAmp);
        VibrationAmpFreqT::Type f = VibrationPowTable::CalcPow(dummyContext._logFreq);
        uint32_t distanceA = VibrationAmpFreqT::GetDistance(a, targetAmplitude);
        uint32_t distanceF = VibrationAmpFreqT::GetDistance(f, targetFrequency);
        return distanceA * DistanceCoeffA + distanceF * DistanceCoeffF;
    }
};

}} // namespace nn::xcd
