﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   AMFM コマンドに関する宣言
 */

#pragma once

#include <nn/xcd/xcd_VibrationAmFmCodec.h>
#include "xcd_VibrationAmFmTypes.h"

namespace nn { namespace xcd {

struct VibrationAmFmCommand
{
    //!< 振幅値や周波数値に対して行う更新処理を表す列挙型
    enum class AmFmAction : uint8_t
    {
        None,       //!< 現状の値を維持する
        Reset,      //!< 初期値にリセットする
        Set,        //!< 指定された値にセットする
        Multiply,   //!< 指定された値を乗算して更新する
    };

    //!< AMFM 符号に対応する AMFM コマンドを表す構造体
    struct AmFmCommand
    {
        AmFmAction amAction;        //!< 振幅値に対する更新処理
        AmFmAction fmAction;        //!< 周波数値に対する更新処理
        VibrationLogT::Type amArg;  //!< 振幅値を更新する際に用いる値 (Set or Multiply の場合に参照される)
        VibrationLogT::Type fmArg;  //!< 周波数値を更新する際に用いる値 (Set or Multiply の場合に参照される)
    };

    //!< デコード時の設定値セットを表す構造体
    struct Config
    {
        VibrationLogT::Type logResetAmp;    //!< 振幅の初期値の対数
        VibrationLogT::Type logMinAmp;      //!< 振幅の最小値の対数
        VibrationLogT::Type logMaxAmp;      //!< 振幅の最大値の対数
        VibrationLogT::Type logZeroAmp;     //!< 振幅のゼロ閾値の対数 (内部値がこの値を下回ると振幅としてはゼロが出力される)
        VibrationLogT::Type logResetFreq;   //!< 周波数の初期値の対数
        VibrationLogT::Type logMinFreq;     //!< 周波数の最小値の対数
        VibrationLogT::Type logMaxFreq;     //!< 周波数の最大値の対数
        const AmFmCommand* pTable;          //!< 符号テーブル
    };

    //!< 指定されたフォーマットに対応する Config 構造体を返します。
    static inline const Config* GetConfig(VibrationAmFmFormat format) NN_NOEXCEPT
    {
        switch(format)
        {
        case VibrationAmFmFormat_5BitVer1:
            return &s_Config5bitVer1;
        case VibrationAmFmFormat_5BitVer2:
            return &s_Config5bitVer2;
        default:
            return nullptr;
        }
    }

    static const AmFmCommand s_Table5bitVer1[]; //!< 5bit AMFM 符号テーブル ver.1.0 本体
    static const AmFmCommand s_Table5bitVer2[]; //!< 5bit AMFM 符号テーブル ver.2.0 本体
    static const Config s_Config5bitVer1;       //!< 5bit AMFM 符号テーブル ver.1.0 使用時の設定値セット
    static const Config s_Config5bitVer2;       //!< 5bit AMFM 符号テーブル ver.2.0 使用時の設定値セット
};

}} // namespace nn::xcd
