﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_Macro.h>  // type_traits 対応判定のため、先に必要
#if defined(NN_BUILD_CONFIG_COMPILER_SUPPORTS_STD_TYPE_TRAITS)
    #include <type_traits>
#endif
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_StaticAssert.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/xcd/xcd_Result.h>
#include "xcd_TeraStateMachine.h"
#include "xcd_ReportTypes.h"
#include "detail/xcd_TeraCommon.h"

namespace nn { namespace xcd {

namespace
{

// ステート遷移のタイムアウト時間
// この時間が経過してもステートが変化しない場合はリトライする。値は暫定。
const auto StateSetTimeout = nn::TimeSpan::FromMilliSeconds(2000);

// MCU リセット後に Idle 状態になるまでのタイムアウト
const auto ResetTimeout = nn::TimeSpan::FromSeconds(30);

// 直接遷移が許可されていない要求を弾く
nn::Result CheckDirectTransition(
    nn::xcd::detail::InternalMcuState currentInternalState,
    nn::xcd::McuState destinationState) NN_NOEXCEPT
{
    // 前回の要求が完了状態で呼ばれるはずなので、 Initializing などの遷移途中状態になっていることはない。
    NN_RESULT_THROW_UNLESS(
        (currentInternalState == detail::InternalMcuState_Standby)
            || (destinationState == McuState_Standby),
        nn::xcd::ResultInvalidMcuState());

    NN_RESULT_SUCCESS;
}

}  // anonymous

void TeraStateMachine::Activate(
    DeviceType type,
    CommandHandler* pCommandHandler,
    StateTransitionCallbackType callback,
    ICommandListener* pCallbackListener) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pCommandHandler);
    NN_SDK_REQUIRES_NOT_NULL(callback);
    NN_SDK_REQUIRES_NOT_NULL(pCallbackListener);

    m_Type = type;

    // コマンド発行用のハンドラを TeraBase から取得する
    m_pCommandHandler = pCommandHandler;

    // 状態遷移完了の通知が来たときに即座に各モジュールの制御を行うため、
    // TeraBase のインスタンスを持っておく
    m_StateTransitionCallbackFunction  = callback;
    m_pStateTransitionCallbackListener = pCallbackListener;
}

void TeraStateMachine::Deactivate() NN_NOEXCEPT
{
    m_Type                             = DeviceType_Unknown;
    m_StateTransitionCallbackFunction  = nullptr;
    m_pStateTransitionCallbackListener = nullptr;
    m_pCommandHandler                  = nullptr;
}

void TeraStateMachine::NotifyAck(Result result, uint8_t id) NN_NOEXCEPT
{
    if (result.IsFailure())
    {
        return;
    }

    if (id == Command_McuResume.Id)
    {
        CheckMcuResumeNotify();
    }
}

void TeraStateMachine::CheckMcuResumeNotify() NN_NOEXCEPT
{
    bool isCallbackNeeded;
    UpdateMcuPowerStatusOnMcuResume(&isCallbackNeeded);

    if (isCallbackNeeded)
    {
        NotifyStateTransition();
    }
}

void TeraStateMachine::UpdateMcuPowerStatusOnMcuResume(bool* pOutIsCallbackNeeded) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutIsCallbackNeeded);

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    *pOutIsCallbackNeeded = false;
    switch (m_McuPowerStatus)
    {
    case McuPowerStatus::EnableRequested:
        {
            // Tera 起動
            NN_XCD_TERA_LOG("%s: Resumed\n", NN_CURRENT_FUNCTION_NAME);
            m_McuPowerStatus  = McuPowerStatus::Enabled;
            m_CurrentMcuState = detail::InternalMcuState_Initializing;

            if (m_IsTransitionWithoutMcuInput)
            {
                // MCU Write によるステート遷移が予約されている場合は、引き続きステート遷移を行う
                SendStateSetCommand(m_DestinationMcuState);
            }
        }
        return;
    case McuPowerStatus::DisableRequested:
        {
            // Standby コマンドの Ack が返ったことを確認して、電源状態を Disable にする。
            NN_XCD_TERA_LOG("%s: Disabled\n", NN_CURRENT_FUNCTION_NAME);
            m_DestinationMcuState = detail::InternalMcuState_Nop;
            m_CurrentMcuState     = detail::InternalMcuState_Standby;
            m_McuPowerStatus      = McuPowerStatus::Disabled;
            *pOutIsCallbackNeeded = true;
        }
        return;
    case McuPowerStatus::SystemBootRequested:
        {
            // システムブート
            NN_XCD_TERA_LOG("%s: System booted\n", NN_CURRENT_FUNCTION_NAME);
            m_McuPowerStatus      = McuPowerStatus::SystemBooted;
            m_CurrentMcuState     = detail::InternalMcuState_SystemBoot;
            *pOutIsCallbackNeeded = true;
        }
        return;
    default:
        {
            // 既に resume / suspend した状態で来た通知は無視する
            // (McuResume が複数回打たれた場合への対処)
            NN_XCD_TERA_LOG("Invalid power status: %d\n", m_McuPowerStatus);
        }
        return;
    }
}

void TeraStateMachine::NotifyMcuRead(const uint8_t* pBuffer, size_t size) NN_NOEXCEPT
{
    NN_UNUSED(pBuffer);
    NN_UNUSED(size);

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    // ペイロードの解析は上位で行っているので、ここでは見ない

    if (!m_WaitStateTransitTimer.IsStarted())
    {
        // 他モードへの遷移要求が完了したことを示す McuRead かどうか判定
        bool isStateChangeRequested =
            m_DestinationMcuState != detail::InternalMcuState_Nop &&
            m_McuPowerStatus != McuPowerStatus::DisableRequested;

        if (isStateChangeRequested)
        {
            m_WaitStateTransitTimer.Start(StateSetTimeout);
            NN_XCD_TERA_LOG("Sent state set command %d\n", m_DestinationMcuState);
        }
    }

    if (IsTransitionWithoutMcuInput() && IsProcessing())
    {
        SendStateSetCommand(m_DestinationMcuState);
    }
}

void TeraStateMachine::UpdateState(detail::InternalMcuState currentState) NN_NOEXCEPT
{
    // 内部状態を更新する
    UpdateCurrentMcuState(currentState);
    CheckStateTransitionComplete(currentState);
    ProcessStateTransition();
}

void TeraStateMachine::UpdateStateByMcuRead(
    const uint8_t* pBuffer,
    size_t size,
    detail::InternalMcuState currentState) NN_NOEXCEPT
{
    m_IsMcuWriteSent = false;

    // MCU Read の処理
    NotifyMcuRead(pBuffer, size);

    // 内部状態を更新する
    UpdateCurrentMcuState(currentState);
    CheckStateTransitionComplete(currentState);
    ProcessStateTransition();
}

void TeraStateMachine::UpdateCurrentMcuState(detail::InternalMcuState state) NN_NOEXCEPT
{
    // ステートの更新は MCU 起動中、かつ Standby 以外の状態への遷移のみ実施
    // (Standby への遷移完了は定常パケットで送信される保証がないため、McuSuspend のAck 到着をもって状態変更を行う)
    if (IsMcuReady() && state != detail::InternalMcuState_Standby)
    {
        m_CurrentMcuState = state;
    }
}

void TeraStateMachine::CheckStateTransitionComplete(detail::InternalMcuState currentState) NN_NOEXCEPT
{
    if (m_DestinationMcuState == detail::InternalMcuState_Nop)
    {
        // ステート遷移要求が出ていない
        return;
    }

    if (IsMcuInitializing())
    {
        CheckResetTimeout();
        return;
    }

    if (m_DestinationMcuState != currentState)
    {
        // ステート遷移が完了していない
        CheckStateTransitionTimeout();
        return;
    }

    {
        std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

        // ステート遷移完了
        NN_XCD_TERA_LOG("Succeeded to change state: %d\n", m_DestinationMcuState);
        m_DestinationMcuState = detail::InternalMcuState_Nop;
        StopStateTransitionTimeout();
    }

    // コールバックを呼び出す際にはmutexを解放しておく
    NotifyStateTransition();
}

void TeraStateMachine::CheckResetTimeout() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    // Idle への遷移要求が出ている場合はリセットなので、タイムアウト判定する
    if (m_DestinationMcuState == detail::InternalMcuState_Idle &&
        m_WaitStateTransitTimer.IsExpired())
    {
        m_WaitStateTransitTimer.Stop();

        NN_SDK_REQUIRES_NOT_NULL(m_ResetTimeoutCallbackFunction);
        NN_XCD_TERA_LOG("[StateMachine] Reset timed out\n");

        // コールバック呼び出し時は一旦 Unlock する
        m_Mutex.Unlock();
        m_ResetTimeoutCallbackFunction(m_pResetTimeoutCallbackListener);
        m_Mutex.Lock();
    }
}

void TeraStateMachine::CheckStateTransitionTimeout() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    if (m_WaitStateTransitTimer.IsExpired())
    {
        // 規定時間経過後もステートが遷移しない場合はリトライする
        //  MCU Write は Bluetooth レイヤーで再送されるため、通常はリトライは発生しない。
        //  ステート遷移コマンドが届いたものの、Tera MCU がステート遷移を受け付けない状態だった場合に発生し得る。
        m_WaitStateTransitTimer.Stop();
        m_IsStateSetRequested = false;
        NN_XCD_TERA_LOG("State is not changed yet. Retry.\n");
    }
}

void TeraStateMachine::ProcessStateTransition() NN_NOEXCEPT
{
    if (!IsMcuReady() || IsMcuInitializing())
    {
        return;
    }

    if (m_DestinationMcuState == detail::InternalMcuState_Nop)
    {
        // ステート遷移要求が出ていない
        return;
    }

    if (m_DestinationMcuState == detail::InternalMcuState_Standby)
    {
        // Standby 遷移は McuResume で行うので、個別にコマンドは送らない
        return;
    }

    if (m_IsStateSetRequested)
    {
        // 遷移要求済み
        return;
    }

    {
        // ステート遷移要求を送信。
        // 送信完了後、ステート遷移タイムアウト判定を開始する (NotifyMcuRead を参照)。
        std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

        SendStateSetCommand(m_DestinationMcuState);
        NN_XCD_TERA_LOG("[StateMachine] Send state set command: %d\n", m_DestinationMcuState);
    }
}

void TeraStateMachine::RequestMcuResume(McuResumeValueType resumeType) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    switch (resumeType)
    {
    case McuResumeValueType_Resume:
        {
            // 通常起動中は何もしない
            if (m_McuPowerStatus != McuPowerStatus::Disabled && !IsSystemBoot())
            {
                return;
            }

            // Tera MCU を起動
            m_McuPowerStatus = McuPowerStatus::EnableRequested;
        }
        break;
    case McuResumeValueType_Suspend:
        {
            // Tera MCU を停止
            m_McuPowerStatus = McuPowerStatus::DisableRequested;

            if (m_Type == DeviceType_FullKey)
            {
                // FullKey の場合は Tera を止めずにリセットする
                m_pCommandHandler->McuResume(McuResumeValueType_Resume, this);
                return;
            }
        }
        break;
    case McuResumeValueType_ResumeForUpdate:
        {
            // Tera MCU を System boot
            m_McuPowerStatus = McuPowerStatus::SystemBootRequested;
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    m_pCommandHandler->McuResume(resumeType, this);
}

void TeraStateMachine::SendStateSetCommand(detail::InternalMcuState state) NN_NOEXCEPT
{
    // TERA_CONTROL によるステート遷移コマンドを MCU Write で送る
    uint8_t commandBuffer[McuWriteSize_Data] = {};
    commandBuffer[detail::McuCommonOutputOffsetByte_ResultCode] =
        CreateMcuPacketModeByte(
            detail::BluetoothMode_McuWrite,
            detail::CommandProtocol_Common);
    commandBuffer[detail::McuCommonOutputOffsetByte_Command] = detail::CommonCommandId_TeraControl;
    commandBuffer[detail::McuCommonOutputOffsetByte_State]   = static_cast<uint8_t>(state);  // 実質使用する値は 255 以下のため問題のないキャスト
    commandBuffer[McuWriteSize_Data - 1] =
        detail::CalcCrc8(commandBuffer + 1, McuWriteSize_Data - 2);

    // MCU Input を使用しないステート遷移の場合は、コマンドリスナーを (コールバック引数として渡されている) TeraBase にする
    auto pListener = (m_IsTransitionWithoutMcuInput && m_pStateTransitionCallbackListener != nullptr) ?
        m_pStateTransitionCallbackListener :
        this;
    m_pCommandHandler->McuWrite(commandBuffer, McuWriteSize_Data, pListener);

    m_IsStateSetRequested = true;
    m_IsMcuWriteSent      = true;
}

nn::Result TeraStateMachine::CheckMcuState(McuState expectedState) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    // ステート遷移要求が出ている場合はエラーを返す
    NN_RESULT_THROW_UNLESS(
        m_DestinationMcuState == detail::InternalMcuState_Nop,
        nn::xcd::ResultInvalidMcuState());

    NN_RESULT_THROW_UNLESS(
        GetCurrentState() == expectedState,
        nn::xcd::ResultInvalidMcuState());
    NN_RESULT_SUCCESS;
}

void TeraStateMachine::ClearInternalState() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    StopStateTransitionTimeout();
    m_McuPowerStatus      = McuPowerStatus::Disabled;
    m_CurrentMcuState     = detail::InternalMcuState_Standby;
    m_DestinationMcuState = detail::InternalMcuState_Nop;
    m_IsMcuWriteSent      = false;
}

void TeraStateMachine::StopStateTransitionTimeout() NN_NOEXCEPT
{
    m_IsStateSetRequested = false;
    m_WaitStateTransitTimer.Stop();
}

bool TeraStateMachine::IsMcuInitializing() const NN_NOEXCEPT
{
    return m_CurrentMcuState == detail::InternalMcuState_Initializing;
}

bool TeraStateMachine::IsMcuReady() const NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    return m_McuPowerStatus != McuPowerStatus::Disabled &&
           m_McuPowerStatus != McuPowerStatus::EnableRequested;
}

bool TeraStateMachine::IsSystemBoot() const NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    return m_McuPowerStatus == McuPowerStatus::SystemBooted;
}

bool TeraStateMachine::IsTransitionWithoutMcuInput() const NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    return m_IsTransitionWithoutMcuInput;
}

bool TeraStateMachine::IsProcessing() const NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    return m_DestinationMcuState != detail::InternalMcuState_Nop ||
        m_McuPowerStatus == McuPowerStatus::SystemBootRequested;
}

bool TeraStateMachine::IsMcuWriteSent() const NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    return m_IsMcuWriteSent;
}

nn::Result TeraStateMachine::SetDestinationState(nn::xcd::McuState mcuState) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    // 通常のステート遷移要求の場合は MCU Input 不使用状態を解除する。失敗時は元に戻す
    bool needsRollback = true;
    bool prevIsTransitionWithoutMcuInput = m_IsTransitionWithoutMcuInput;
    m_IsTransitionWithoutMcuInput = false;
    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            m_IsTransitionWithoutMcuInput = prevIsTransitionWithoutMcuInput;
        }
    };

    // Standby への遷移時は前回の要求をキャンセルする
    if (mcuState == nn::xcd::McuState_Standby)
    {
        StopStateTransitionTimeout();
        m_DestinationMcuState = detail::InternalMcuState_Nop;
    }

    // 前回の要求が未完了の場合はエラーを返す
    NN_RESULT_THROW_UNLESS(
        !IsProcessing(),
        nn::xcd::ResultMcuBusy());

    // 既に目的のステートになっている場合は何もしない
    if (m_CurrentMcuState == detail::ConvertMcuStateToInternal(mcuState))
    {
        needsRollback = false;
        NN_RESULT_SUCCESS;
    }

    // 直接遷移が許可されていない要求を弾く
    NN_RESULT_DO(CheckDirectTransition(m_CurrentMcuState, mcuState));

    auto result = SetDestinationStateImpl(mcuState);
    if (result.IsSuccess())
    {
        needsRollback = false;
    }

    return result;
}

nn::Result TeraStateMachine::SetDestinationStateWithoutMcuInput(McuState mcuState) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    m_IsTransitionWithoutMcuInput = true;

    // Standby への遷移時は前回の要求をキャンセルする
    if (mcuState == nn::xcd::McuState_Standby)
    {
        StopStateTransitionTimeout();
        m_DestinationMcuState = detail::InternalMcuState_Nop;
    }

    // 前回の要求が未完了の場合はエラーを返す
    NN_RESULT_THROW_UNLESS(
        !IsProcessing(),
        nn::xcd::ResultMcuBusy());

    // 既に目的のステートになっている場合は何もしない
    if (m_CurrentMcuState == detail::ConvertMcuStateToInternal(mcuState))
    {
        m_IsTransitionWithoutMcuInput = false;
        NN_RESULT_SUCCESS;
    }

    // 直接遷移が許可されていない要求を弾く
    NN_RESULT_DO(CheckDirectTransition(m_CurrentMcuState, mcuState));

    auto result = SetDestinationStateImpl(mcuState);
    if (result.IsFailure())
    {
        m_IsTransitionWithoutMcuInput = false;
    }

    return result;
}

nn::Result TeraStateMachine::SetDestinationStateForFirmwareUpdate(nn::xcd::McuState mcuState) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    // MCU Input 不使用状態を解除する。失敗時は元に戻す
    bool needsRollback = true;
    bool prevIsTransitionWithoutMcuInput = m_IsTransitionWithoutMcuInput;
    m_IsTransitionWithoutMcuInput = false;
    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            m_IsTransitionWithoutMcuInput = prevIsTransitionWithoutMcuInput;
        }
    };

    // システムブートまたは Standby への遷移時は前回の要求をキャンセルする
    if (mcuState == nn::xcd::McuState_Standby ||
        mcuState == nn::xcd::McuState_FirmwareUpdate)
    {
        StopStateTransitionTimeout();
        m_DestinationMcuState = detail::InternalMcuState_Nop;
    }

    // 前回の要求が未完了の場合はエラーを返す
    NN_RESULT_THROW_UNLESS(
        !IsProcessing(),
        nn::xcd::ResultMcuBusy());

    // 既に目的のステートになっている場合は何もしない
    if (m_CurrentMcuState == detail::ConvertMcuStateToInternal(mcuState))
    {
        needsRollback = false;
        NN_RESULT_SUCCESS;
    }

    if (!IsSystemBoot())
    {
        // 通常起動中はシステムブートのみ許可する
        NN_RESULT_THROW_UNLESS(
            mcuState == McuState_FirmwareUpdate,
            nn::xcd::ResultInvalidMcuState());

        RequestMcuResume(McuResumeValueType_ResumeForUpdate);
        StopStateTransitionTimeout();
        needsRollback = false;
        NN_RESULT_SUCCESS;
    }

    // システムブート中は Idle への遷移のみ許可する
    NN_RESULT_THROW_UNLESS(
        mcuState == McuState_Idle,
        nn::xcd::ResultInvalidMcuState());

    auto result = SetDestinationStateImpl(mcuState);
    if (result.IsSuccess())
    {
        needsRollback = false;
    }

    return result;
}

nn::Result TeraStateMachine::SetDestinationStateImpl(nn::xcd::McuState mcuState) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    m_DestinationMcuState = detail::ConvertMcuStateToInternal(mcuState);
    NN_RESULT_THROW_UNLESS(
        m_DestinationMcuState <= detail::InternalMcuState_Ir,
        nn::xcd::ResultInvalidMcuState());

    RequestMcuResume(
        mcuState == McuState_Standby ?
            McuResumeValueType_Suspend :
            McuResumeValueType_Resume);

    // ステート遷移要求を出していない状態にする
    StopStateTransitionTimeout();

    NN_RESULT_SUCCESS;
}

void TeraStateMachine::AbortStateTransition() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    if (m_CurrentMcuState != detail::InternalMcuState_Standby)
    {
        // 進行中のステート遷移要求は破棄して Standby にする
        m_IsTransitionWithoutMcuInput = false;
        SetDestinationStateImpl(nn::xcd::McuState_Standby);
    }
}

nn::Result TeraStateMachine::Reset(ResetTimeoutCallbackType callback, ICommandListener* pCallbackListener) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(callback);
    NN_SDK_REQUIRES_NOT_NULL(pCallbackListener);

    if (m_McuPowerStatus == McuPowerStatus::EnableRequested)
    {
        // XXX: 既に起動処理中の場合は無視 (仮)
        NN_RESULT_SUCCESS;
    }

    // XXX: リセット後の戻り先ステートは Idle
    m_DestinationMcuState = detail::InternalMcuState_Idle;
    m_CurrentMcuState     = detail::InternalMcuState_Standby;
    m_McuPowerStatus      = McuPowerStatus::Disabled;

    m_ResetTimeoutCallbackFunction  = callback;
    m_pResetTimeoutCallbackListener = pCallbackListener;

    NN_XCD_TERA_LOG("[StateMachine] Start reset\n");
    RequestMcuResume(McuResumeValueType_Resume);

    m_IsStateSetRequested = true;
    m_WaitStateTransitTimer.Start(ResetTimeout);

    NN_RESULT_SUCCESS;
}

McuState TeraStateMachine::GetCurrentState() NN_NOEXCEPT
{
    if (m_McuPowerStatus == McuPowerStatus::SystemBooted)
    {
        return McuState_FirmwareUpdate;
    }

    return detail::ConvertInternalMcuStateToExternal(m_CurrentMcuState);
}

void TeraStateMachine::NotifyStateTransition() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_StateTransitionCallbackFunction);

    m_StateTransitionCallbackFunction(m_CurrentMcuState, m_pStateTransitionCallbackListener);
}

}} // namespace nn::xcd
