﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/xcd/xcd_Input.h>
#include <nn/xcd/xcd_Result.h>

#include "xcd_SerialFlashAccessor.h"
#include "xcd_SerialFlashMap.h"

namespace nn { namespace xcd {

SerialFlashAccessor::SerialFlashAccessor() NN_NOEXCEPT :
    m_pCommand(nullptr),
    m_State(SerialFlashAccessState_Idle),
    m_pSystemEvent(nullptr),
    m_Address(0),
    m_Size(0),
    m_AddressOffset(0),
    m_SizeUnit(0),
    m_pReadBuffer(nullptr),
    m_pWriteBuffer(nullptr),
    m_Result(ResultSuccess())
{
    // 何もしない
}

SerialFlashAccessor::~SerialFlashAccessor() NN_NOEXCEPT
{
    // 何もしない
}

//!< デバイスが接続されたときに呼ばれる関数
void SerialFlashAccessor::Activate() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pCommand);

    m_State = SerialFlashAccessState_Idle;
    m_Address = 0;
    m_Size = 0;
    m_AddressOffset = 0;
    m_SizeUnit = 0;
    m_pReadBuffer = nullptr;
    m_pWriteBuffer = nullptr;
    m_Result = ResultSuccess();
}

void SerialFlashAccessor::Deactivate() NN_NOEXCEPT
{
    m_State = SerialFlashAccessState_Idle;
}

void SerialFlashAccessor::SetCommandHandler(CommandHandler* pCommand) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pCommand);
    m_pCommand = pCommand;
};


Result SerialFlashAccessor::Read(
    const uint32_t address,
    uint8_t* pOutBuffer,
    int size,
    nn::os::SystemEventType* m_pEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutBuffer);
    NN_SDK_REQUIRES_NOT_NULL(m_pEvent);
    NN_SDK_REQUIRES_GREATER(size, 0);
    NN_RESULT_THROW_UNLESS(m_State != SerialFlashAccessState_OnRead, ResultSerialFlashOnRead());
    NN_RESULT_THROW_UNLESS(m_State != SerialFlashAccessState_OnWrite, ResultSerialFlashOnWrite());

    m_Address = address;
    m_AddressOffset = 0;
    m_pReadBuffer = pOutBuffer;
    m_Size = size;
    m_pSystemEvent = m_pEvent;
    m_State = SerialFlashAccessState_OnRead;

    this->ProceedRead();

    NN_RESULT_SUCCESS;
}


Result SerialFlashAccessor::Write(
    const uint32_t address,
    const uint8_t* pBuffer,
    int size,
    nn::os::SystemEventType* m_pEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);
    NN_SDK_REQUIRES_NOT_NULL(m_pEvent);
    NN_SDK_REQUIRES_GREATER(size, 0);
    NN_RESULT_THROW_UNLESS(m_State != SerialFlashAccessState_OnRead, ResultSerialFlashOnRead());
    NN_RESULT_THROW_UNLESS(m_State != SerialFlashAccessState_OnWrite, ResultSerialFlashOnWrite());

    m_Address = address;
    m_AddressOffset = 0;
    m_pWriteBuffer = pBuffer;
    m_Size = size;
    m_pSystemEvent = m_pEvent;
    m_State = SerialFlashAccessState_OnWrite;

    this->ProceedWrite();

    NN_RESULT_SUCCESS;
}

Result SerialFlashAccessor::GetResult() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_State == SerialFlashAccessState_Completed, ResultSerialFlashOnWrite());
    m_State = SerialFlashAccessState_Idle;
    auto result = m_Result;
    m_Result = ResultSuccess();
    NN_RESULT_DO(result);
    NN_RESULT_SUCCESS;
}

void SerialFlashAccessor::NotifySerialFlashRead(const uint8_t* pBuffer, int size) NN_NOEXCEPT
{
    NN_UNUSED(size);
    std::memcpy(&m_pReadBuffer[m_AddressOffset], pBuffer, m_SizeUnit);
    m_AddressOffset += m_SizeUnit;
    if (m_AddressOffset == m_Size)
    {
        OperationComplete(ResultSuccess());
        return;
    }

    this->ProceedRead();
}

void SerialFlashAccessor::NotifyAck(Result result, uint8_t id) NN_NOEXCEPT
{
    NN_UNUSED(id);
    if (result.IsFailure())
    {
        OperationComplete(result);
        return;
    }

    m_AddressOffset += m_SizeUnit;
    if (m_AddressOffset == m_Size)
    {
        OperationComplete(ResultSuccess());
        return;
    }

    this->ProceedWrite();
}

void SerialFlashAccessor::ProceedRead() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_EQUAL(m_State, SerialFlashAccessState_OnRead);

    m_SizeUnit = (m_Size - m_AddressOffset) > m_pCommand->GetSerialFlashReadUnitSize() ?
        m_pCommand->GetSerialFlashReadUnitSize() :
        m_Size - m_AddressOffset;
    m_pCommand->ReadSerialFlash(
        m_Address + m_AddressOffset,
        static_cast<uint8_t>(m_SizeUnit),
        this);
}

void SerialFlashAccessor::ProceedWrite() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_EQUAL(m_State, SerialFlashAccessState_OnWrite);

    m_SizeUnit = (m_Size - m_AddressOffset) > m_pCommand->GetSerialFlashWriteUnitSize() ?
        m_pCommand->GetSerialFlashWriteUnitSize() :
        m_Size - m_AddressOffset;
    auto result = m_pCommand->WriteSerialFlash(
        m_Address + m_AddressOffset,
        &m_pWriteBuffer[m_AddressOffset],
        static_cast<uint8_t>(m_SizeUnit),
        this
    );

    if (result.IsFailure())
    {
        OperationComplete(result);
    }
}

void SerialFlashAccessor::OperationComplete(Result result) NN_NOEXCEPT
{
    m_State = SerialFlashAccessState_Completed;
    m_Result = result;
    nn::os::SignalSystemEvent(m_pSystemEvent);
    m_pSystemEvent = nullptr;
}



}} // namespace nn::xcd
