﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

namespace nn { namespace xcd {

//!< HID Reportの種類
enum ReportType
{
    ReportType_Output,
    ReportType_Input,
    ReportType_Feature,
};

//!< InputReportに含まれるPad情報の種類
enum PadType
{
    PadType_None,              // Pad情報を含まない
    PadType_Basic,             // モーター, ボタン, スティック
    PadType_Full,              // モーター, ボタン, スティック, 6軸センサー
};

//!< Reportの特徴
template <uint8_t TId,
          size_t TSize,
          ReportType TType,
          PadType TPadType,
          bool TVibrator>
struct ReportDescryptor
{
    static const uint8_t    Id       = TId;                     //!< ReportId
    static const size_t     Size     = TSize;                   //!< Reportのサイズ
    static const ReportType Type     = TType;                   //!< Reportの種類
    static const PadType    padType  = TPadType;                //!< InputReportに含まれるPad情報の種類
    static const bool       Vibrator = TVibrator;               //!< OutputReportにVibratorを含むかどうか
};

// XCDでサポートされているHID Report一覧
typedef ReportDescryptor<0x01, 49,  ReportType_Output, PadType_None, true> Report_CommandOut;
typedef ReportDescryptor<0x03, 318, ReportType_Output, PadType_None, true> Report_McuUpdateOut;
typedef ReportDescryptor<0x10, 10,  ReportType_Output, PadType_None, true> Report_BasicOut;
typedef ReportDescryptor<0x11, 48,  ReportType_Output, PadType_None, true> Report_McuOut;
typedef ReportDescryptor<0x12, 48,  ReportType_Output, PadType_None, true> Report_AttachmentOut;
typedef ReportDescryptor<0x13, 362, ReportType_Output, PadType_None, true> Report_AudioOut;
typedef ReportDescryptor<0x14, 318, ReportType_Output, PadType_None, true> Report_BurstOut;

typedef ReportDescryptor<0x21, 49,  ReportType_Output, PadType_Basic, false> Report_CommandIn;
typedef ReportDescryptor<0x23, 49,  ReportType_Output, PadType_Basic, false> Report_McuUpdateIn;
typedef ReportDescryptor<0x30, 48,  ReportType_Output, PadType_Full, false> Report_BasicIn;
typedef ReportDescryptor<0x31, 361, ReportType_Output, PadType_Full, false> Report_McuIn;
typedef ReportDescryptor<0x32, 361, ReportType_Output, PadType_Full, false> Report_AttachmentIn;
typedef ReportDescryptor<0x33, 361, ReportType_Output, PadType_Full, false> Report_AudioIn;
typedef ReportDescryptor<0x3f, 48,  ReportType_Output, PadType_None, false> Report_GenericIn;

typedef ReportDescryptor<0x70, 1,   ReportType_Feature, PadType_None, false> Report_OtaEnableFwu;
typedef ReportDescryptor<0x71, 8,   ReportType_Feature, PadType_None, false> Report_OtaSetupRead;
typedef ReportDescryptor<0x72, 361, ReportType_Feature, PadType_None, false> Report_OtaRead;
typedef ReportDescryptor<0x73, 8,   ReportType_Feature, PadType_None, false> Report_OtaErase;
typedef ReportDescryptor<0x74, 361, ReportType_Feature, PadType_None, false> Report_OtaWrite;
typedef ReportDescryptor<0x75, 8,   ReportType_Feature, PadType_None, false> Report_OtaLaunch;

typedef ReportDescryptor<0x80, 2,   ReportType_Output, PadType_None, false> Report_ExtGripOut;
typedef ReportDescryptor<0x81, 8,   ReportType_Input,  PadType_None, false> Report_ExtGripIn;
typedef ReportDescryptor<0x82, 2,   ReportType_Output, PadType_None, false> Report_ExtGripSys;

const uint16_t ReportByte_Base                           = 0;                              //!< 基準となるバイト
const uint16_t ReportByte_ReportId                       = ReportByte_Base;                //!< ReportId
const uint16_t ReportByte_Payload                        = ReportByte_Base + 1;            //!< ペイロード

// Output Report のデータフォーマットに関する定義
const uint16_t OutputReportByte_SampleNumber             = ReportByte_Payload;             //!< サンプル番号 (下位 4bit のみ)
const uint16_t OutputReportByte_Vibrator                 = ReportByte_Payload + 1;         //!< 振動モーター

const uint16_t OutputReportSize_Vibrator                 = 8;                              //!< 振動モーターのデータサイズ

// Input Report のデータフォーマットに関する定義
const uint16_t InputReportByte_SampleNumber              = ReportByte_Payload;             //!< サンプル番号
const uint16_t InputReportByte_Status                    = ReportByte_Payload + 1;          //!< ステータス
const uint16_t InputReportByte_Button                    = ReportByte_Payload + 2;         //!< ボタン
const uint16_t InputReportByte_AnalogStickMain           = ReportByte_Payload + 5;         //!< アナログスティック Main
const uint16_t InputReportByte_AnalogStickSub            = ReportByte_Payload + 8;         //!< アナログスティック Sub
const uint16_t InputReportByte_Vibrator                  = ReportByte_Payload + 11;        //!< 振動モーター
const uint16_t InputReportByte_Accelerometer             = ReportByte_Payload + 12;        //!< 加速度センサー
const uint16_t InputReportByte_Gyroscope                 = ReportByte_Payload + 18;        //!< ジャイロセンサー

const uint16_t InputReportSize_Vibrator                  = 1;                              //!< 振動モーターのデータサイズ
const uint16_t InputReportSize_SixAxisSensor             = 12;                             //!< 6軸センサーのデータサイズ
const uint16_t InputReportSize_PadBasic                  = 11;                             //!< センサーを含まないPadのデータサイズ
const uint16_t InputReportSize_PadFull                   = InputReportSize_PadBasic
                                                           + InputReportSize_SixAxisSensor * 3; //!< センサーを含まないPadのデータサイズ

const uint16_t InputReportBit_BatteryLeast               = 5;                              //!< Status用のビット: バッテリー
const uint16_t InputReportBit_Charge                     = 4;                              //!< Status用のビット: 充電状態
const uint16_t InputReportBit_AttachmentLeast            = 1;                              //!< Status用のビット: アタッチメント
const uint16_t InputReportBit_AttachmentExtra            = 0;                              //!< Status用のビット: アタッチメント Extra

const uint16_t InputReportByte_Powered                   = InputReportByte_Button + 1;     //!< Powered の情報が拡張されたバイト
const uint16_t InputReportBit_Powered                    = 7;                              //!< Status用のビット: 電源状態

enum AttachmentType
{
    AttachmentType_Nwcp                                  = 0,                              //!< UART 通信型域 (本体 or 電気拡張グリップ)
    AttachmentType_ExtDevice                             = 6,                              //!< 拡張デバイス
    AttachmentType_NotAttached                           = 7,                              //!< Attach されていない
};

// CommandOut Report
const uint16_t CommandOutReportByte_Payload              = ReportByte_Payload + 9;         //!< CommandOutのペイロード
const uint16_t CommandOutReportSize_Payload              = 39;                             //!< CommandOutのサイズ

// McuUpdateOut Report
const uint16_t McuUpdateOutReportByte_Payload            = ReportByte_Payload + 9;         //!< McuUpdateOut のペイロード
const uint16_t McuUpdateOutReportSize_Payload            = 308;                            //!< McuUpdateOut のサイズ

// McuOut Report
const uint16_t McuOutReportByte_Payload                  = ReportByte_Payload + 9;         //!< McuOutのペイロード
const uint16_t McuOutReportSize_Payload                  = 38;                             //!< McuOutのサイズ

// AttachmentOut Report
const uint16_t AttachmentOutReportByte_Payload           = ReportByte_Payload + 9;         //!< AttachmentOutのペイロード
const uint16_t AttachmentOutReportSize_Payload           = 38;                             //!< AttachmentOutのサイズ

// AudioOut Report
const uint16_t AudioOutReportByte_Payload                = ReportByte_Payload + 9;         //!< AudioOutのペイロード
const uint16_t AudioOutReportSize_Payload                = 352;                            //!< AudioOutのサイズ

// BurstOut Report
const uint16_t BurstOutReportByte_Payload                = ReportByte_Payload + 9;         //!< BurstOutのペイロード
const uint16_t BurstOutReportSize_Payload                = 308;                            //!< BurstOutのサイズ

// CommandIn Report
const uint16_t CommandInReportByte_Payload               = ReportByte_Payload + 12;        //!< CommandInのペイロード
const uint16_t CommandInReportSize_Payload               = 37;                             //!< CommandInのサイズ

// McuUpdateIn Report
const uint16_t McuUpdateInReportByte_Payload             = ReportByte_Payload + 12;        //!< McuUpdateのペイロード
const uint16_t McuUpdateInReportSize_Payload             = 37;                              //!< McuUpdateのサイズ

// McuIn Report
const uint16_t McuInReportByte_Payload                   = ReportByte_Payload + 48;        //!< Mcuのペイロード
const uint16_t McuInReportSize_Payload                   = 313;                            //!< Mcuのサイズ

// AttachmentIn Report
const uint16_t AttachmentInReportByte_Payload = ReportByte_Payload + 36;                   //!< Attachmentのペイロード
const uint16_t AttachmentInReportSize_Payload = 12;                                        //!< Attachmentのサイズ

// AudioIn Report
const uint16_t AudioInReportByte_Payload                 = ReportByte_Payload + 24;        //!< Audioのペイロード
const uint16_t AudioInReportSize_Payload                 = 313;                            //!< Audioのサイズ

// Ota Report
const uint16_t OtaReportByte_ReportId                    = ReportByte_Base;                //!< Ota の ReportId
const uint16_t OtaReportByte_Address                     = ReportByte_Base + 1;            //!< Ota の Address
const uint16_t OtaReportByte_Length                      = ReportByte_Base + 5;            //!< Ota の Length
const uint16_t OtaReportByte_Data                        = ReportByte_Base + 7;            //!< Ota の Data

// ExtGripOut Report
const uint16_t ExtGripOutReportByte_Status               = ReportByte_Payload;             //!< ExtGripOut の Status
enum ExtGripStatus : uint8_t
{
    ExtGripStatus_Inquiry                                = 0x01,                           //!< NxController のデバイス情報取得
    ExtGripStatus_CreateHidConnection                    = 0x02,                           //!< NxController との Hid Connection を確立する
    ExtGripStatus_DeleteHidConnection                    = 0x03,                           //!< NxController との Hid Connection を切断する
    ExtGripStatus_StartHidData                           = 0x04,                           //!< NxController との Hid 通信を開始する
    ExtGripStatus_StopHidData                            = 0x05,                           //!< NxController との Hid 通信を終了する
    ExtGripStatus_SendResetPageAfterReboot               = 0x06,                           //!< NxController を再起動して再接続させる（PageAfterReboot）
    ExtGripStatus_GetVersion                             = 0x07,                           //!< ExtGrip の FW バージョンを返す
    ExtGripStatus_GetLeftAndRight                        = 0x08,                           //!< 接続先が右コン0x02か左コン0x01を返す（Proコンの場合は左）

    ExtGripStatus_DeviceCommandPassThru                  = 0x91,
    ExtGripStatus_StandardHidCommunicationPassThru       = 0x92,
};

enum ExtGripSysStatus : uint8_t
{
    ExtGripSysStatus_GotoFwUpdateMode                    = 0x01,                           //!< ExtGrip を FW 更新モード（ DFU モード）に遷移させる
    ExtGripSysStatus_Reboot                              = 0x02,                           //!< ExtGrip を再起動させる
};

// ExtGripIn Report
const uint16_t ExtGripInReportByte_Status               = ReportByte_Payload;             //!< ExtGripIn の Status
const uint16_t ExtGripInReportByte_InquiryResult        = ReportByte_Payload + 1;         //!< ExtGripIn の Inquiry の Result
const uint16_t ExtGripInReportByte_InquiryDeviceType    = ReportByte_Payload + 2;         //!< ExtGripIn の Inquiry の DeviceType
const uint16_t ExtGripInReportByte_InquiryBdAddr        = ReportByte_Payload + 3;         //!< ExtGripIn の Inquiry の BD_ADDR

}} // namespace nn::xcd
