﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Result.h>
#include <nn/nn_Macro.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/xcd/xcd_IrsensorTypes.h>

namespace nn { namespace xcd {

//!< Irsensor Processor のための基底クラス
class IrsensorProcessorBase
{
public:
    IrsensorProcessorBase() NN_NOEXCEPT :
        m_Mutex(true),
        m_IsProcessorReady(false),
        m_ResultCode(0),
        m_BrightnessThreshold(0),
        m_TimeStamp(0),
        m_DiffTimeStamp(0),
        m_LatestAckCount(0)
    {
        for (int i = 0; i < InputReportSamplingCountMax; i++)
        {
            std::memset(&m_IrCommonData[i], 0, sizeof(IrCommonData));
        }
    }

    virtual ~IrsensorProcessorBase() NN_NOEXCEPT;

protected:
    //!< IRセンサのイメージヘッダをパースします。プロセッサ起動時には必ず取得できます。
    Result ParseIrsensorHeader(const uint8_t* pBuffer, size_t size, uint8_t sampleNumber) NN_NOEXCEPT;
    //!< プロセッサが使用可能かどうかを判定します。
    bool IsProcessorReady() NN_NOEXCEPT;

protected:
    static const size_t InputReportHeaderSize             = 10; //!< イメージヘッダのサイズ
    static const int InputReportHeaderOffset              = 0;  //!< レポートデータ内でのヘッダのオフセット
    static const int InputReportTimeStampOffset           = 310; //!< レポートデータ内でのTimeStampのオフセット
    static const int InputReportLatestAckCountOffset      = 311; //!< レポートデータ内での LatestAckCount のオフセット
    static const int InputReportSamplingCountMax          = 3;  //!< 保存するヘッダパケット数の上限

protected:
    nn::os::Mutex m_Mutex;                                     //!< IRセンサプロセッサ内の排他制御用
    bool m_IsProcessorReady;                                   //!< プロセッサにバッファが渡され、使用可能状態かどうかを表すフラグ
    IrCommonData m_IrCommonData[InputReportSamplingCountMax];  //!< ヘッダ内の共通データ
    int m_ResultCode;                                          //!< 結果
    int m_BrightnessThreshold;                                 //!< LightPixelOn/Off の輝度閾値設定
    int m_TimeStamp;                                           //!< 最新のTimeStamp値
    int m_DiffTimeStamp;                                       //!< 1フレーム前のTimeStamp値との差分
    int m_LatestAckCount;                                      //!< WriteRegister の最新 Ack 番号
};

}} // namespace nn::xcd
