﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <mutex>
#include <nn/nn_Result.h>
#include <nn/nn_Macro.h>
#include <nn/fs.h>
#include <nn/os/os_Mutex.h>
#include <nn/xcd/xcd_Device.h>
#include <nn/xcd/xcd_Tera.h>
#include <nn/xcd/xcd_TeraFirmware.h>
#include <nn/xcd/xcd_Result.h>
#include "detail/xcd_TeraUpdaterTypes.h"
#include "detail/xcd_TeraUpdaterStream.h"
#include "detail/xcd_TeraCommon.h"
#include "detail/xcd_TimeCounter.h"

namespace nn { namespace xcd {

/*
 * 引数に指定された文字列を TeraUpdater のログとして出力します。
 */
#ifdef NN_BUILD_CONFIG_COMPILER_VC
#define NN_XCD_TERA_UPDATER_LOG(...)            NN_SDK_LOG("[xcd:TeraUpdater] " ##__VA_ARGS__)
#else
#define NN_XCD_TERA_UPDATER_LOG(format, ...)    NN_SDK_LOG("[xcd:TeraUpdater] " format, ##__VA_ARGS__)
#endif

/*
 * VERBOSE モード時のみ、引数に指定された文字列を TeraUpdater のログとして出力します。
 */
#ifdef VERBOSE
#define NN_XCD_TERA_UPDATER_LOG_VERBOSE(...)    NN_XCD_TERA_UPDATER_LOG(__VA_ARGS__)
#else
#define NN_XCD_TERA_UPDATER_LOG_VERBOSE(...)    static_cast<void>(0)
#endif

/**
 * @brief   Tera MCU の FW アップデート処理を行うためのインターフェースクラス
 */
class ITeraUpdater
{
    NN_DISALLOW_COPY(ITeraUpdater);
    NN_DISALLOW_MOVE(ITeraUpdater);

public:
    /**
     * @brief   アップデートモード
     */
    enum class UpdateMode
    {
        Normal,     //!< Customer code のみ
        Full        //!< 全域 (IAP を含む)
    };

    /**
     * @brief   アップデートの段階
     */
    enum class UpdatePhase
    {
        Stopped,                //!< アップデートを行っていない
        BootPrepare,            //!< システムブート準備
        Boot,                   //!< システムブート中
        BootFinish,             //!< システムブート完了
        Invalidate,             //!< 起動フラグ無効化
        Erase,                  //!< 消去
        Erasing,                //!< 消去中
        EraseFinish,            //!< 消去完了
        Write,                  //!< 書き込み
        Finalize,               //!< ファイナライズ
        Finalizing,             //!< ファイナライズ中
        FinalizeFinish,         //!< ファイナライズ完了
        Error                   //!< エラー発生
    };

    /**
     * @brief   アップデート段階の更新時に呼び出すコールバック関数型
     */
    typedef void(*PhaseChangeCallbackType)(UpdatePhase, void*);

public:
    /**
     * @brief   FW 更新失敗エミュレーションの有効状態を設定します。
     */
    static void SetUpdateFailureEmulationEnabled(bool isEnabled) NN_NOEXCEPT
    {
        g_DebugOptions.Set<DebugOption::UpdateFailureEmulation>(isEnabled);
    }

    ITeraUpdater() NN_NOEXCEPT {}
    virtual ~ITeraUpdater() NN_NOEXCEPT {}

    virtual void Activate(
        DeviceType deviceType,
        PhaseChangeCallbackType callback,
        void* pCallbackArgument) NN_NOEXCEPT = 0;
    virtual void Deactivate() NN_NOEXCEPT = 0;

    /**
     * @brief   書き込む FW イメージを設定
     *
     * @param[in]   image       FW イメージ
     * @param[in]   mode        アップデートモード
     */
    virtual nn::Result SetFirmwareImage(
        const FirmwareImage& image,
        UpdateMode mode) NN_NOEXCEPT = 0;

    /**
     * @brief   アップデート処理の中断要求
     */
    virtual nn::Result Abort() NN_NOEXCEPT = 0;

    /**
     * @brief   アップデート状況の取得
     *
     * @param[out]  pOutStateInfo   状態の取得先
     */
    virtual nn::Result GetState(
        McuUpdateStateInfo* pOutStateInfo) const NN_NOEXCEPT = 0;

    /**
     * @brief   Firmware Update 中に受け取ったデータをパースします
     */
    virtual void ParseMcuUpdateInputReport(
        const uint8_t* pBuffer,
        size_t size,
        uint8_t sampleNumber) NN_NOEXCEPT = 0;

    /**
     * @brief   Firmware Update でコントローラーに送信するデータを取得します。
     */
    virtual size_t GetMcuUpdateOutputReport(
        uint8_t* pOutValue,
        size_t size) NN_NOEXCEPT = 0;

    /**
     * @brief   システムブートローダとの同期を開始
     */
    virtual void StartSyncWithBootLoader() NN_NOEXCEPT = 0;

    /**
     * @brief   ROM の消去を開始
     */
    virtual void StartEraseRom() NN_NOEXCEPT = 0;

    /**
     * @brief   ROM の書き込みを開始
     */
    virtual void StartWriteRom() NN_NOEXCEPT = 0;

    /**
     * @brief   アップデートの終了
     */
    virtual void Finish() NN_NOEXCEPT = 0;

protected:
    // デバッグ用の設定
    struct DebugOption
    {
        typedef ::nn::util::BitFlagSet<32, DebugOption>::Flag<0> UpdateFailureEmulation;
    };

    // DebugOption の集合を扱う型
    typedef ::nn::util::BitFlagSet<32, DebugOption> DebugOptionSet;

protected:
    static DebugOptionSet g_DebugOptions;   //!< デバッグ用の設定
};

}} // namespace nn::xcd
