﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/os/os_Config.h>
#include <nn/os/os_LightEvent.h>
#include <nn/os/os_TimerEvent.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/xcd/xcd_Result.h>
#include <nn/xcd/xcd_ResultForPrivate.h>
#include "xcd_Bluetooth.h"
#include "xcd_DeviceListManager.h"
#include "xcd_TaskManager.h"
#include "xcd_DeviceHandleGenerator.h"
#include "xcd_DeviceHandler.h"

#if defined(NN_BUILD_CONFIG_OS_WIN)
#include "detail/xcd_AardvarkAccessor-os.win32.h"
#endif  // defined(NN_BUILD_CONFIG_OS_WIN)

namespace nn { namespace xcd {

namespace
{
    //!< vid
    const int XcdVendorId = 0x57E;
    //!< pid
    const int XcdProductIds[] = {0x2006, 0x2007, 0x2009};

    //!< Bluetooth 接続が Busy だった場合にリトライをトリガーする時間
    const auto BluetoothConnectionRetrySpan = ::nn::TimeSpan::FromMilliSeconds(500);

    //!< Bluetooth 接続に Timeout するまでの時間
    const auto BluetoothConnectionTimeoutSpan = ::nn::TimeSpan::FromSeconds(6);

    bool CheckPid(int pid)
    {
        for(auto& xcdPid : XcdProductIds)
        {
            if(pid == xcdPid)
            {
                return true;
            }
        }
        return false;
    }
}

DeviceListManager::DeviceListManager() NN_NOEXCEPT
    : m_Activated(false),
      m_WiredDeviceActivationEnabled(true)
{
    // 何もしない
}

DeviceListManager::~DeviceListManager() NN_NOEXCEPT
{
    // 何もしない
}

void DeviceListManager::Activate() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_EQUAL(m_Activated, false);

    // デバイスハンドルの初期化
    DeviceHandleGenerator::Get().Init();

    // NAND から振動マスターボリュームをロード
    float vibrationMasterVolume = 0.0f;
    (void)LoadAndGetVibrationMasterVolume(&vibrationMasterVolume);

    // NAND から Pro Controller 有線 USB 通信の設定をロード
    m_FullKeyUsbEnabled = ::nn::settings::system::IsUsbFullKeyEnabled();
    detail::GetLinkMonitor().SetFullKeyUsbEnabled(m_FullKeyUsbEnabled);

    ::nn::os::InitializeTimerEvent(&m_BluetoothConnectionRetryEvent, nn::os::EventClearMode_ManualClear);
    ::nn::os::InitializeEvent(&m_BluetoothConnectionCompleteEvent, false, nn::os::EventClearMode_ManualClear);
    ::nn::os::InitializeLightEvent(&m_DeviceUpdateEvent, false, nn::os::EventClearMode_ManualClear);

    detail::GetLinkMonitor().StartMonitoring(&m_DeviceUpdateEvent);

    ::nn::os::InitializeLightEvent(&m_SuspendCompleteEvent, false, ::nn::os::EventClearMode_ManualClear);

    m_PairingManager.Activate();
    for (auto& handler : m_Handlers)
    {
        handler.SetPairingManager(&m_PairingManager);
    }

    ::nn::os::InitializeLightEvent(&m_PairingDatabaseClearedEvent, false, nn::os::EventClearMode_ManualClear);
    m_PairingManager.SetDatabaseClearedEvent(&m_PairingDatabaseClearedEvent);

    GetTaskManager().RegisterPeriodicTask(this);

    m_Activated = true;
}

void DeviceListManager::Deactivate() NN_NOEXCEPT
{
    GetTaskManager().UnregisterPeriodicTask(this);

    ::nn::os::FinalizeTimerEvent(&m_BluetoothConnectionRetryEvent);
    ::nn::os::FinalizeLightEvent(&m_DeviceUpdateEvent);
    ::nn::os::FinalizeLightEvent(&m_SuspendCompleteEvent);
    ::nn::os::FinalizeLightEvent(&m_PairingDatabaseClearedEvent);

    m_Activated = false;
}

void DeviceListManager::Suspend() NN_NOEXCEPT
{
    detail::GetLinkMonitor().Suspend();
}

void DeviceListManager::Resume() NN_NOEXCEPT
{
    detail::GetLinkMonitor().Resume();

    // デバイスリストの更新を行う
    ::nn::os::SignalLightEvent(&m_DeviceUpdateEvent);
    ::nn::os::ClearLightEvent(&m_SuspendCompleteEvent);
    ::nn::os::WaitLightEvent(&m_SuspendCompleteEvent);
}

void DeviceListManager::EventFunction(const ::nn::os::MultiWaitHolderType* pMultiWaitHolder) NN_NOEXCEPT
{
    NN_UNUSED(pMultiWaitHolder);
}

void DeviceListManager::PeriodicEventFunction() NN_NOEXCEPT
{
    if (::nn::os::TryWaitLightEvent(&m_DeviceUpdateEvent))
    {
        ::nn::os::ClearLightEvent(&m_DeviceUpdateEvent);
        HandleDeviceListUpdateEvent();
        HandleBleDeviceListUpdateEvent();
    }
    else if (::nn::os::TryWaitTimerEvent(&m_BluetoothConnectionRetryEvent))
    {
        ::nn::os::ClearTimerEvent(&m_BluetoothConnectionRetryEvent);
        TrySendConnectionTriggerImpl();
    }
    else if (::nn::os::TryWaitEvent(&m_BluetoothConnectionCompleteEvent))
    {
        ::nn::os::ClearEvent(&m_BluetoothConnectionCompleteEvent);

        // 接続に失敗したのでタイムアウトを通知
        ::nn::os::SignalSystemEvent(m_pBluetoothConnectionTimeoutEvent);
    }
    else if (::nn::os::TryWaitLightEvent(&m_PairingDatabaseClearedEvent))
    {
        ::nn::os::ClearLightEvent(&m_PairingDatabaseClearedEvent);
        TriggerPairingOnAllDevice();
    }
}

void DeviceListManager::BindLinkUpdateEvent(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemEvent);

    m_pStatusUpdateEvent = pSystemEvent;

    for(auto& handler : m_Handlers)
    {
        handler.SetStatusUpdateEvent(pSystemEvent);
    }
}

void DeviceListManager::UnbindLinkUpdateEvent(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemEvent);
    NN_UNUSED(pSystemEvent);

    for(auto& handler : m_Handlers)
    {
        handler.ClearStatusUpdateEvent(pSystemEvent);
    }
    m_pStatusUpdateEvent = nullptr;
}

Result DeviceListManager::ListDevices(DeviceList* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    pOutValue->deviceCount = 0;

    for(auto& handler : m_Handlers)
    {
        if(handler.IsConnected() == true)
        {
            pOutValue->handleList[pOutValue->deviceCount] = handler.GetDeviceHandle();
            pOutValue->deviceCount++;
        }
    }

    NN_RESULT_THROW_UNLESS(pOutValue->deviceCount > 0, nn::xcd::ResultNotFound());
    NN_RESULT_SUCCESS;
}

Result DeviceListManager::ListDevices(BleDeviceList* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    pOutValue->deviceCount = 0;

    for(auto& handler : m_BleHandlers)
    {
        if(handler.IsActivated() == true)
        {
            pOutValue->handleList[pOutValue->deviceCount] = handler.GetDeviceHandle();
            pOutValue->deviceCount++;
        }
    }

    NN_RESULT_THROW_UNLESS(pOutValue->deviceCount > 0, nn::xcd::ResultNotFound());
    NN_RESULT_SUCCESS;
}

Result DeviceListManager::GetDeviceHandler(DeviceHandler** pOutValue, DeviceHandle handle) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(handle.IsValid(), nn::xcd::ResultNotConnected());

    if (m_Handlers[handle._lower].IsConnected() == true &&
        m_Handlers[handle._lower].GetDeviceHandle() == handle)
    {
        *pOutValue = &m_Handlers[handle._lower];
        NN_RESULT_SUCCESS;
    }

    NN_RESULT_THROW(nn::xcd::ResultNotConnected());
}

Result DeviceListManager::GetBleDeviceHandler(BleDeviceHandler** pOutValue, BleConnectionHandle handle) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(handle != BleInvalidConnectionHandle, nn::xcd::ResultNotConnected());

    for (auto& handler : m_BleHandlers)
    {
        if (handler.GetDeviceHandle() == handle &&
            handler.IsActivated() == true &&
            handler.IsSamplingStarted() == true)
        {
            *pOutValue = &handler;
            NN_RESULT_SUCCESS;
        }
    }

    NN_RESULT_THROW(nn::xcd::ResultNotConnected());
}

void DeviceListManager::BindPairingCompletedEvent(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemEvent);

    m_PairingManager.BindPairingCompletedEvent(pSystemEvent);
}

void DeviceListManager::UnbindPairingCompletedEvent(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemEvent);

    m_PairingManager.UnbindPairingCompletedEvent(pSystemEvent);
}

size_t DeviceListManager::GetRegisteredDevices(RegisteredDeviceInfo* pOutValue, size_t length) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    return m_PairingManager.GetRegisteredDevices(pOutValue, length);
}

void DeviceListManager::AddRegisteredDeviceInfo(const RegisteredDeviceInfo& info) NN_NOEXCEPT
{
    m_PairingManager.RegisterDevice(info);
}

void DeviceListManager::SetNwcpEnabled(bool enabled) NN_NOEXCEPT
{
    detail::GetLinkMonitor().SetNwcpEnabled(enabled);
}

void DeviceListManager::SetRailUpdateEvent(nn::os::SystemEventType* pEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEvent);
    detail::GetLinkMonitor().SetRailUpdateEvent(pEvent);
}

RailUpdateEventInfo DeviceListManager::GetRailUpdateEventInfo() NN_NOEXCEPT
{
    RailUpdateEventInfo outEventInfo;
    ::nn::bluetooth::Address address;

    detail::GetLinkMonitor().GetRailUpdateEventType(&outEventInfo.eventType, &address);
    if (outEventInfo.eventType != RailUpdateEventType_None)
    {
        RegisteredDeviceInfo controllerInfo;
        if (m_PairingManager.GetRegisteredDeviceInfo(&controllerInfo, address) == true)
        {
            outEventInfo.colorAvailable = true;
            outEventInfo.mainColor = controllerInfo.color.color[0];
            outEventInfo.subColor = controllerInfo.color.color[1];
        }
        else
        {
            outEventInfo.colorAvailable = false;
        }
    }

    return outEventInfo;
}

AwakeTriggerReason DeviceListManager::GetAwakeTriggerReasonForLeftRail() NN_NOEXCEPT
{
    return detail::GetLinkMonitor().GetAwakeTriggerReasonForLeftRail();
}

AwakeTriggerReason DeviceListManager::GetAwakeTriggerReasonForRightRail() NN_NOEXCEPT
{
    return detail::GetLinkMonitor().GetAwakeTriggerReasonForRightRail();
}

bool DeviceListManager::IsDeviceAttachedOnRail(RailType type) NN_NOEXCEPT
{
    return (type == RailType_Left) ?
        detail::GetLinkMonitor().IsLeftNwcpAttached() :
        detail::GetLinkMonitor().IsRightNwcpAttached();
}

bool DeviceListManager::IsUsbHidSupported(UsbHidDeviceInfo deviceInfo) NN_NOEXCEPT
{
    return detail::GetLinkMonitor().IsUsbHidSupported(deviceInfo);
}

Result DeviceListManager::AddUsbHidDevice(int index, UsbHidDeviceInfo deviceInfo) NN_NOEXCEPT
{
    NN_RESULT_DO(
        detail::GetLinkMonitor().AddUsbHidDevice(index, deviceInfo));
    NN_RESULT_SUCCESS;
}

Result DeviceListManager::RemoveUsbHidDevice(int index) NN_NOEXCEPT
{
    NN_RESULT_DO(
        detail::GetLinkMonitor().RemoveUsbHidDevice(index));
    NN_RESULT_SUCCESS;
}

Result DeviceListManager::SetUsbHidInputReport(int index, uint8_t *pBuffer, size_t length) NN_NOEXCEPT
{
    NN_RESULT_DO(
        detail::GetLinkMonitor().SetUsbHidInputReport(index, pBuffer, length));
    NN_RESULT_SUCCESS;
}

size_t DeviceListManager::GetUsbHidOutputReport(int index, uint8_t *pOutBuffer, size_t length) NN_NOEXCEPT
{
    return detail::GetLinkMonitor().GetUsbHidOutputReport(index, pOutBuffer, length);
}

void DeviceListManager::SetWiredDeviceActivationEnabled(bool enabled) NN_NOEXCEPT
{
    m_WiredDeviceActivationEnabled = enabled;

    // デバイスリストの更新をトリガする
    nn::os::SignalLightEvent(&m_DeviceUpdateEvent);
}

void DeviceListManager::SetFullKeyUsbEnabled(bool enabled) NN_NOEXCEPT
{
    detail::GetLinkMonitor().SetFullKeyUsbEnabled(enabled);
    (void)::nn::settings::system::SetUsbFullKeyEnabled(enabled);
    m_FullKeyUsbEnabled = enabled;
    detail::GetLinkMonitor().UpdateUsbHidDeviceLists();
}

void DeviceListManager::GetFullKeyUsbEnabled(bool* pOutEnabled) NN_NOEXCEPT
{
    *pOutEnabled = m_FullKeyUsbEnabled;
}

void DeviceListManager::GetUsbConnected(bool* pOutValue, DeviceHandle handle) NN_NOEXCEPT
{
    *pOutValue = detail::GetLinkMonitor().IsUsbConnected(handle);
}

::nn::Result DeviceListManager::SendConnectionTrigger(const ::nn::bluetooth::Address address) NN_NOEXCEPT
{
    detail::HidDeviceInfo devices[MaxXcdDevices];

    // 接続済みデバイスの取得
    auto deviceCount = detail::GetLinkMonitor().GetDevices(devices, MaxXcdDevices);

    for (size_t i = 0; i < deviceCount; i++)
    {
        if (devices[i].address == address)
        {
            NN_RESULT_THROW(ResultBluetoothAlreadyConnected());
        }
    }

    m_Address = address;
    NN_RESULT_DO(TrySendConnectionTriggerImpl());
    NN_RESULT_SUCCESS;
}

void DeviceListManager::SetConnectionTriggerTimeoutEvent(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemEvent);
    m_pBluetoothConnectionTimeoutEvent = pSystemEvent;
}

int DeviceListManager::GetMaxBluetoothLinks() NN_NOEXCEPT
{
    return detail::GetLinkMonitor().GetMaxBluetoothLinks();
}

Result DeviceListManager::SetFirmwareUpdatingDevice(DeviceHandle handle) NN_NOEXCEPT
{
    return detail::GetLinkMonitor().SetFirmwareUpdatingDevice(handle);
}

void DeviceListManager::UnsetFirmwareUpdatingDevice() NN_NOEXCEPT
{
    detail::GetLinkMonitor().UnsetFirmwareUpdatingDevice();
}

void DeviceListManager::DisableBle() NN_NOEXCEPT
{
    detail::GetLinkMonitor().DisableBle();
}

void DeviceListManager::SetIsPalmaAllConnectable(bool connectable) NN_NOEXCEPT
{
    detail::GetLinkMonitor().SetIsPalmaAllConnectable(connectable);
}

void DeviceListManager::SetIsPalmaPairedConnectable(bool connectable) NN_NOEXCEPT
{
    detail::GetLinkMonitor().SetIsPalmaPairedConnectable(connectable);
}

Result DeviceListManager::PairPalma(const nn::bluetooth::Address& address) NN_NOEXCEPT
{
    NN_RESULT_DO(detail::GetLinkMonitor().PairPalma(address));
    NN_RESULT_SUCCESS;
}

bool DeviceListManager::IsFiftyConnected() NN_NOEXCEPT
{
    return detail::GetLinkMonitor().IsFiftyConnected();
}

PadState DeviceListManager::GetFiftyPadState() NN_NOEXCEPT
{
    return detail::GetLinkMonitor().GetFiftyPadState();
}

void DeviceListManager::HandleDeviceListUpdateEvent() NN_NOEXCEPT
{
    // 接続状態の更新
    auto deviceCount = detail::GetLinkMonitor().GetDevices(m_LinkedDevices, MaxXcdDevices);

    // 切断されたデバイスの探索
    for(auto& handler : m_Handlers)
    {
        if(handler.isActivated() == false)
        {
            continue;
        }

        bool detached = true;

        for(size_t i = 0; i < deviceCount; i++)
        {
            if(m_LinkedDevices[i].vid == XcdVendorId && CheckPid(m_LinkedDevices[i].pid) == true)
            {
                // ジョイコン or フルコンが見つかった
                if(handler.GetDeviceHandle() == m_LinkedDevices[i].deviceHandle)
                {
                    // 接続中
                    detached = false;
                    break;
                }
            }
        }

        // 切断されたデバイスはディアクティベート
        if(detached == true)
        {
            DeviceStatus status;
            handler.GetDeviceStatus(&status);
            if (status.interfaceType != InterfaceType_Bluetooth)
            {
                DeviceInfo info;
                handler.GetDeviceInfo(&info);
                ProtectBluetoothDeviceInfo(info.address, false);
            }
            handler.Deactivate();
        }
    }

    // 新たに接続されたデバイスの探索
    for(size_t i = 0; i < deviceCount; i++)
    {
#if defined(NN_BUILD_CONFIG_OS_WIN)
        if ((m_LinkedDevices[i].vid == XcdVendorId && CheckPid(m_LinkedDevices[i].pid) == true) ||
            m_LinkedDevices[i].vid == detail::AardvarkVendorId)
#else
        if (m_LinkedDevices[i].vid == XcdVendorId && CheckPid(m_LinkedDevices[i].pid) == true)
#endif  // defined(NN_BUILD_CONFIG_OS_WIN)
        {
            // ジョイコン or フルコンが見つかった

            // 新規のデバイスかどうかをチェックする
            bool newDevice = true;
            for(auto& handler : m_Handlers)
            {
                if(handler.GetDeviceHandle() == m_LinkedDevices[i].deviceHandle)
                {
                    // 既に登録済み
                    newDevice = false;
                }
            }

            // 新規に接続されたデバイスはアクティベート
            if(newDevice == true &&
               (m_LinkedDevices[i].interfaceType == InterfaceType_Bluetooth || m_WiredDeviceActivationEnabled))
            {
                // 使われていないDeviceHandlerをActivateする
                int handlerIndex = 0;
                for(handlerIndex = 0; handlerIndex < NN_ARRAY_SIZE(m_Handlers); ++handlerIndex)
                {
                    if(m_Handlers[handlerIndex].isActivated() == false)
                    {
                        if (m_LinkedDevices[i].interfaceType != InterfaceType_Bluetooth)
                        {
                            IncreaseBluetoothDeviceInfoOrder(m_LinkedDevices[i].address);
                            ProtectBluetoothDeviceInfo(m_LinkedDevices[i].address, true);
                        }
                        auto deviceHandle = m_LinkedDevices[i].deviceHandle;
                        deviceHandle._lower = handlerIndex;
                        m_Handlers[handlerIndex].Activate(deviceHandle, m_LinkedDevices[i].interfaceType, detail::GetLinkMonitor().GetHidAccessor(m_LinkedDevices[i].deviceHandle));
                        break;
                    }
                }
            }

#if defined(NN_BUILD_CONFIG_OS_WIN)
            if (m_LinkedDevices[i].vid == detail::AardvarkVendorId)
            {
                // Aardvark が見つかった
                NN_SDK_LOG("[xcd] Found aardvark\n");
            }
#endif  // defined(NN_BUILD_CONFIG_OS_WIN)
        }
    }

    // Suspend/Resume 処理中の場合はイベントの完了を通知
    ::nn::os::SignalLightEvent(&m_SuspendCompleteEvent);
}

void DeviceListManager::HandleBleDeviceListUpdateEvent() NN_NOEXCEPT
{
    // 接続状態の更新
    auto deviceCount = detail::GetLinkMonitor().GetBleDevices(m_LinkedBleDevices, MaxXcdBleDevices);

    // 切断されたデバイスの探索
    for (auto& handler : m_BleHandlers)
    {
        if (handler.IsActivated() == false)
        {
            continue;
        }

        bool detached = true;

        for (size_t i = 0; i < deviceCount; i++)
        {
            // Ble デバイスが見つかった
            if (handler.GetDeviceHandle() == m_LinkedBleDevices[i])
            {
                // 接続中
                detached = false;
                break;
            }
        }

        // 切断されたデバイスはディアクティベート
        if (detached == true)
        {
            handler.Deactivate();
        }
    }

    // 新たに接続されたデバイスの探索
    for (size_t i = 0; i < deviceCount; i++)
    {
        // 新規のデバイスかどうかをチェックする
        bool newDevice = true;
        for (auto& handler : m_BleHandlers)
        {
            if (handler.IsActivated() && handler.GetDeviceHandle() == m_LinkedBleDevices[i])
            {
                // 既に登録済み
                newDevice = false;
                break;
            }
        }

        // 新規に接続されたデバイスはアクティベート
        if (newDevice == true )
        {
            // 使われていないDeviceHandlerをActivateする
            int handlerIndex = 0;
            for (handlerIndex = 0; handlerIndex < NN_ARRAY_SIZE(m_BleHandlers); ++handlerIndex)
            {
                if (m_BleHandlers[handlerIndex].IsActivated() == false)
                {
                    m_BleHandlers[handlerIndex].Activate(m_LinkedBleDevices[i], detail::GetLinkMonitor().GetBleHidAccessor(m_LinkedBleDevices[i]));
                    break;
                }
            }
        }
    }
    // Suspend/Resume 処理中の場合はイベントの完了を通知
    ::nn::os::SignalLightEvent(&m_SuspendCompleteEvent);
}

::nn::Result DeviceListManager::TrySendConnectionTriggerImpl() NN_NOEXCEPT
{
    auto result = TriggerBluetoothConnection(m_Address, &m_BluetoothConnectionCompleteEvent);
    if (ResultBusy().Includes(result))
    {
        ::nn::os::StartOneShotTimerEvent(&m_BluetoothConnectionRetryEvent, BluetoothConnectionRetrySpan);
        NN_RESULT_SUCCESS;
    }

    NN_RESULT_DO(result);
    NN_RESULT_SUCCESS;
}

void DeviceListManager::TriggerPairingOnAllDevice() NN_NOEXCEPT
{
    // アクティベートされた Handler に対してペアリングを指示する
    for (auto& handler : m_Handlers)
    {
        if (handler.isActivated() == true)
        {
            handler.TriggerPairing();
        }
    }

    // USB Hid 状態で待機済みのものは再度接続処理を行う
    detail::GetLinkMonitor().ReInitializePendingUsbDevices();
}

Result DeviceListManager::GetKuinaVersion(KuinaVersionData* pOutMcuVersionData, int index) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutMcuVersionData);
    NN_RESULT_DO(
        detail::GetLinkMonitor().GetKuinaVersion(pOutMcuVersionData,index));
    NN_RESULT_SUCCESS;
}

Result DeviceListManager::RequestKuinaVersion(int index) NN_NOEXCEPT
{
    NN_RESULT_DO(
        detail::GetLinkMonitor().RequestKuinaVersion(index));
    NN_RESULT_SUCCESS;
}

Result DeviceListManager::SetKuinaToFirmwareUpdateMode(int index) NN_NOEXCEPT
{
    NN_RESULT_DO(
        detail::GetLinkMonitor().SetKuinaToFirmwareUpdateMode(index));
    NN_RESULT_SUCCESS;
}


}} // namespace nn::xcd
