﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/os.h>
#include <nn/os/os_SystemEventApi.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/xcd/xcd_Pairing.h>
#include <nn/xcd/xcd_TeraFirmware.h>
#include <nn/xcd/detail/xcd_Log.h>

#include "xcd_Bluetooth.h"
#include "xcd_ColorUtil.h"
#include "xcd_CommandTypes.h"
#include "xcd_DeviceHandler.h"
#include "xcd_ReportTypes.h"
#include "xcd_SerialFlashMap.h"
#include "xcd_TaskManager.h"
#include "detail/xcd_HidAccessor.h"
#include "detail/xcd_LinkMonitor.h"

#define ENABLE_WIRED_PAIRING

namespace nn { namespace xcd {

namespace {

bool IsParsableReportId(uint8_t reportId)
{
    return reportId == Report_CommandIn::Id ||
           reportId == Report_McuUpdateIn::Id ||
           reportId == Report_BasicIn::Id ||
           reportId == Report_McuIn::Id ||
           reportId == Report_AttachmentIn::Id ||
           reportId == Report_AudioIn::Id;
}

}


DeviceHandler::DeviceHandler() NN_NOEXCEPT
    : m_Sequence(DeviceSequence_Nolink)
    , m_pPairingManager(nullptr)
    , m_IndicatorLedPattern(0x00)
{
    m_Flags.Reset();
    m_PadInput.SetCommandHandler(&m_Command);
    m_VibratorBase.SetCommandHandler(&m_Command);
    m_TeraBase.SetCommandHandler(&m_Command);
    m_AttachmentBase.SetCommandHandler(&m_Command);
    m_Output.SetPeripherals(&m_Command, &m_TeraBase, &m_VibratorBase);
    m_SerialFlashAccessor.SetCommandHandler(&m_Command);
}

DeviceHandler::~DeviceHandler() NN_NOEXCEPT
{
    // 何もしない
}

void DeviceHandler::Activate(DeviceHandle deviceHandle, InterfaceType interfaceType, detail::HidAccessor* pAccessor) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_EQUAL(m_Flags.Test<DeviceHandlerFlag::IsActivated>(), false);
    NN_SDK_REQUIRES_NOT_NULL(pAccessor);

    m_Handle = deviceHandle;
    m_pAccessor = pAccessor;
    m_RegisteredInfo = RegisteredDeviceInfo();
    m_FirmwareVersion = FirmwareVersionImpl();

    // SlotMode 変更通知の登録
    nn::os::InitializeLightEvent(&m_SlotSizeEvent, false, nn::os::EventClearMode_ManualClear);
    detail::GetLinkMonitor().SetSlotSizeUpdateEvent(deviceHandle, &m_SlotSizeEvent);

    nn::os::InitializeTimerEvent(&m_ChargerTimer, nn::os::EventClearMode_AutoClear);

    m_pAccessor->StartSampling(InputReportHandler, this);
    m_Sequence = DeviceSequence_Linkup;

    m_Status.interfaceType = interfaceType;

    // データフォーマットの初期化
    m_TargetDataFormat = PeriodicDataFormat_Basic;
    m_DataFormat = PeriodicDataFormat_Basic;
    m_Output.SetPeriodicDataFormat(m_DataFormat);

    m_Flags.Reset();
    m_Flags.Set<DeviceHandlerFlag::IsActivated>();
    m_Flags.Set<DeviceHandlerFlag::NotifyOnConnectionComplete>();

    GetTaskManager().RegisterPeriodicTask(this);
}

void DeviceHandler::Deactivate() NN_NOEXCEPT
{
    GetTaskManager().UnregisterPeriodicTask(this);

    m_Sequence = DeviceSequence_Nolink;
    m_PadInput.Deactivate();
    m_VibratorBase.Deactivate();
    m_TeraBase.Deactivate();
    m_Command.Deactivate();
    m_BtFirmwareUpdater.Deactivate();
    m_SerialFlashAccessor.Deactivate();

    // Attachment の処理
    auto attachmentStatus = m_AttachmentBase.GetAttachmentStatus();
    if (attachmentStatus == AttachmentStatus_CommunicationAvailable ||
        attachmentStatus == AttachmentStatus_Ready)
    {
        AttachmentPowerEnable(false);
    }

    // ハンドルを初期化
    m_Handle = DeviceHandle::GetInvalidHandle();

    m_pSamplingEvent = nullptr;

    nn::os::FinalizeTimerEvent(&m_ChargerTimer);
    nn::os::FinalizeLightEvent(&m_SlotSizeEvent);

    m_Flags.Reset();

    if (m_pStatusUpdateEvent != nullptr)
    {
        nn::os::SignalSystemEvent(m_pStatusUpdateEvent);
    }
}

void DeviceHandler::SetStatusUpdateEvent(nn::os::SystemEventType* pEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEvent);

    m_pStatusUpdateEvent = pEvent;
}

void DeviceHandler::ClearStatusUpdateEvent(nn::os::SystemEventType* pEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEvent);

    if (m_pStatusUpdateEvent == pEvent)
    {
        m_pStatusUpdateEvent = nullptr;
    }
}

Result DeviceHandler::SetSamplingEvent(nn::os::SystemEventType* pEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEvent);

    m_pSamplingEvent = pEvent;

    NN_RESULT_SUCCESS;
}

bool DeviceHandler::IsConnected() NN_NOEXCEPT
{
    return m_Sequence == DeviceSequence_Active;
}

void DeviceHandler::SetPairingManager(PairingManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);

    m_pPairingManager = pManager;
}

void DeviceHandler::Detach() NN_NOEXCEPT
{
    m_pAccessor->DetachDevice();
}

void DeviceHandler::Reboot(bool reconnect, bool isFlash) NN_NOEXCEPT
{
    m_Command.Reset(reconnect);

    if (m_BtFirmwareUpdater.IsInsideFirmwareUpdateSequence() == true
        || isFlash == true )
    {
        // ファームウェア更新中は、Output Report の生成/送信を行う
        SendOutputReport();
    }
}

void DeviceHandler::Reboot(bool reconnect) NN_NOEXCEPT
{
    DeviceHandler::Reboot(reconnect, false);
}

void DeviceHandler::McuResume(bool isResume) NN_NOEXCEPT
{
    if (isResume)
    {
        m_Command.McuResume(McuResumeValueType_Resume, nullptr);
    }
    else
    {
        m_Command.McuResume(McuResumeValueType_Suspend, nullptr);
    }
}

BtFirmwareVersion DeviceHandler::GetBtFirmwareVersion() NN_NOEXCEPT
{
    BtFirmwareVersion version;
    version.major = m_FirmwareVersion.bluetooth.major;
    version.minor = m_FirmwareVersion.bluetooth.minor;

    return version;
}

void DeviceHandler::EventFunction(const ::nn::os::MultiWaitHolderType* pMultiWaitHolder) NN_NOEXCEPT
{
    NN_UNUSED(pMultiWaitHolder);
}

void DeviceHandler::PeriodicEventFunction() NN_NOEXCEPT
{
    if (nn::os::TryWaitLightEvent(&m_SlotSizeEvent))
    {
        nn::os::ClearLightEvent(&m_SlotSizeEvent);
        OnTsiChanged();
    }
}

void DeviceHandler::SetDataFormat(PeriodicDataFormat format) NN_NOEXCEPT
{
    // 変更中は何もしない
    if (m_TargetDataFormat == format)
    {
        return;
    }

    m_TargetDataFormat = format;

    m_Flags.Set<DeviceHandlerFlag::NeedsChangeSlotSizeRequest>();
    TryUpdateDataFormat();
}

bool DeviceHandler::SetDataFormatAndPollingTimedWait(PeriodicDataFormat format, nn::TimeSpan pollingTime, nn::TimeSpan timeOut) NN_NOEXCEPT
{
    SetDataFormat(format);
    nn::os::Tick startTick = nn::os::GetSystemTick();
    while (GetDataFormat() != format)
    {
        nn::os::SleepThread(pollingTime);
        if ((nn::os::GetSystemTick() - startTick).ToTimeSpan() > timeOut)
        {
            return false;
        }
    }
    NN_DETAIL_XCD_INFO("SetDataFormat   Waiting Time = %4d ms\n", (nn::os::GetSystemTick() - startTick).ToTimeSpan().GetMilliSeconds());
    return true;
}

PeriodicDataFormat DeviceHandler::GetDataFormat() NN_NOEXCEPT
{
    return m_DataFormat;
}

DeviceColor DeviceHandler::GetColor() NN_NOEXCEPT
{
    return ConvertColor(m_DeviceColor);
}

void DeviceHandler::SetColor(const ::nn::util::Color4u8Type& mainColor, const ::nn::util::Color4u8Type& subColor) NN_NOEXCEPT
{
    // Debug 機能なので現状、Result はハンドリングしない
    m_Command.UpdateControllerColor(mainColor, subColor, this);
}

void DeviceHandler::SetDesignInfo(const DeviceColor& color, uint8_t variation) NN_NOEXCEPT
{
    // バージョンが量産以前のものを更新すると、デザイン情報以外がおかしくなるので更新しない
    if (m_FormatVersion >= 1)
    {
        // Debug 機能なので現状、Result はハンドリングしない
        m_Command.UpdateDesignInfo(color, variation, this);
    }
}

Result DeviceHandler::GetIndicatorLedPattern(uint8_t* pOutPattern) const NN_NOEXCEPT
{
    *pOutPattern = m_IndicatorLedPattern;

    NN_RESULT_SUCCESS;
}

Result DeviceHandler::ControlLed(uint8_t pattern) NN_NOEXCEPT
{
    m_Command.SetIndicatorLed(pattern, nullptr);
    m_IndicatorLedPattern = pattern;

    NN_RESULT_SUCCESS;
}

Result DeviceHandler::AttachmentPowerEnable(bool isEnabled) NN_NOEXCEPT
{
    // ジョイコンの NFC が壊れていた場合、起動に 5.1 s かかるため、 6s を Timeout にする
    const nn::TimeSpan Timeout = nn::TimeSpan::FromSeconds(6);
    const nn::TimeSpan PollingInterval = nn::TimeSpan::FromMilliSeconds(5);

    // 接続されてない状態で enable はすぐにエラーを返すが、 disable にする方は処理を進める
    if (isEnabled && !m_AttachmentBase.IsExtDevAttached())
    {
        return nn::xcd::ResultAttachmentDeviceNotAttached();
    }

    // 拡張デバイス自体に電源があり、すでに UART 通信可能な状態なら、電源制御は行わない
    if (!isEnabled || m_AttachmentBase.GetAttachmentStatus() == AttachmentStatus_Attached)
    {
        switch (m_Info.deviceType)
        {
        case nn::xcd::DeviceType_Right:
        {
            if (isEnabled)
            {
                // Tera を使う場合はこの流れで Version Check をする
                nn::xcd::McuVersionData    mcuVersion;

                // Tera の Vout を使うために Background 状態へ遷移させる
                NN_RESULT_DO(m_TeraBase.SetMcuStateImmediate(McuState_Background));
                nn::os::Tick startTick = nn::os::GetSystemTick();
                while (m_TeraBase.GetMcuState() != nn::xcd::McuState_Background)
                {
                    nn::os::SleepThread(PollingInterval);
                    if (nn::xcd::GetMcuVersion(&mcuVersion, m_Handle).IsSuccess())
                    {
                        // バージョンが取れる状態になっていたら、バージョンチェックを行う
                        if (mcuVersion.isCorrupted || !m_AttachmentBase.McuVersionCheck(mcuVersion))
                        {
                            return nn::xcd::ResultAttachmentDeviceNeedUpdate();
                        }
                    }

                    if ((nn::os::GetSystemTick() - startTick).ToTimeSpan() > Timeout)
                    {
                        NN_DETAIL_XCD_INFO("SetMcuState Timeout\n");
                        return nn::xcd::ResultAttachmentDeviceAttachmentReadyTimeout();
                    }
                }
                NN_DETAIL_XCD_INFO("SetMcuState (To Background)  Waiting Time = %4d ms\n", (nn::os::GetSystemTick() - startTick).ToTimeSpan().GetMilliSeconds());

                // ループをすぐに抜けてきたときのためにここでもチェック
                nn::xcd::GetMcuVersion(&mcuVersion, m_Handle);
                if (mcuVersion.isCorrupted || !m_AttachmentBase.McuVersionCheck(mcuVersion))
                {
                    return nn::xcd::ResultAttachmentDeviceNeedUpdate();
                }

                NN_RESULT_DO(m_TeraBase.ExtControl(true, &m_AttachmentBase));
                m_AttachmentBase.WaitExtControlAck();
            }
            else
            {
                // Background state の時のみ、以下は実行し、そうではない時はすでに Disable 済みとして、処理をスキップする
                if (m_TeraBase.GetMcuState() == McuState_Background)
                {
                    // Tera へ電源 OFF の命令を送るが、 Tera 側で実際に落ちたかどうかは保証できないため、
                    // できる範囲で対応し、最悪、 Tera リセットのみを行う。(電源制御的には問題ない)
                    //m_TeraBase.ExtControl(false, &m_AttachmentBase);
                    m_TeraBase.ExtControl(false, &m_AttachmentBase);
                    m_AttachmentBase.WaitExtControlAck();

                    // 単純に Tera をリセット
                    NN_RESULT_DO(m_TeraBase.SetMcuStateImmediate(McuState_Standby));
                    nn::os::Tick startTick = nn::os::GetSystemTick();
                    while (m_TeraBase.GetMcuState() != nn::xcd::McuState_Standby)
                    {
                        nn::os::SleepThread(PollingInterval);
                        if ((nn::os::GetSystemTick() - startTick).ToTimeSpan() > Timeout)
                        {
                            return nn::xcd::ResultAttachmentDeviceAttachmentReadyTimeout();
                        }
                    }
                    NN_DETAIL_XCD_INFO("SetMcuState     Waiting Time = %4d ms\n", (nn::os::GetSystemTick() - startTick).ToTimeSpan().GetMilliSeconds());
                }
                else
                {
                    NN_DETAIL_XCD_INFO("Already Set Stanby State, Skip SetMcuState()\n");
                }
            }
        }
        break;

        case nn::xcd::DeviceType_Left:
            // 電源の ON
            m_Command.AttachmentEnable(isEnabled, nullptr);
            break;

        case nn::xcd::DeviceType_Unknown:
        case nn::xcd::DeviceType_FullKey:
        case nn::xcd::DeviceType_MiyabiLeft:
        case nn::xcd::DeviceType_MiyabiRight:
        case nn::xcd::DeviceType_Tarragon:
            // 拡張デバイスが接続できないものに対しては何もせずに返る。
            return nn::xcd::ResultAttachmentDeviceNotSupported();
        default: NN_UNEXPECTED_DEFAULT;
        }
    }
    else
    {
        NN_DETAIL_XCD_INFO("Skip Power Control To Attachment Device\n");
    }

    NN_RESULT_SUCCESS;
}


Result DeviceHandler::StartBtFirmwareUpdate(const FirmwareImage& image, nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemEvent);

    // Firmware Updater の初期化
    NN_RESULT_DO(m_BtFirmwareUpdater.InitializeFirmwareUpdate(image, pSystemEvent));

    if (m_Status.interfaceType == InterfaceType_Bluetooth)
    {
        // データフォーマットの変更
        // 実際は TSI の変更のみ
        SetDataFormat(PeriodicDataFormat_BtUpdate);
    }
    else
    {
        m_BtFirmwareUpdater.StartFirmwareUpdate();
    }

    // データフォーマットはペアリング終了時に Reset することで初期化される

    NN_RESULT_SUCCESS;
}

Result DeviceHandler::AbortBtFirmwareUpdate() NN_NOEXCEPT
{
    bool isRebootRequired;
    auto result = m_BtFirmwareUpdater.AbortFirmwareUpdate(&isRebootRequired);
    if (result.IsSuccess() && isRebootRequired)
    {
        // BT FW 更新中は Reboot コマンドが通らないため切断する
        Detach();
    }

    return result;
}

bool DeviceHandler::IsBtFirmwareUpdateThroughBtRequired() NN_NOEXCEPT
{
    return (m_FirmwareVersion.bluetooth < FirmwareVersionBt_0346 && m_Status.interfaceType != InterfaceType_Bluetooth);
}

nn::hid::detail::RxPacketHistory DeviceHandler::GetRxPacketHistory() NN_NOEXCEPT
{
    return m_TransactionMonitor.GetRxPacketHistory();
}

void DeviceHandler::TriggerPairing() NN_NOEXCEPT
{
    if (m_pPairingManager->IsWiredPairingRequired(m_Status.interfaceType, m_Info.deviceType))
    {
        TriggerPairingImpl();
    }
}

IdentificationCode DeviceHandler::GetIdentificationCode() NN_NOEXCEPT
{
    return m_RegisteredInfo.identificationCode;
}

void DeviceHandler::NotifyAck(Result result, uint8_t id) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(id == Command_Shipment.Id || id == Command_SetDataFormat.Id || id == Command_WriteChargeSetting.Id);

    if (id == Command_Shipment.Id)
    {
        m_Flags.Reset<DeviceHandlerFlag::IsOnShipmentClear>();
        if (result.IsSuccess() == false)
        {
            m_Flags.Set<DeviceHandlerFlag::IsErrorOccurred>();
        }
    }
}

void DeviceHandler::NotifySetDataFormat(Result result, PeriodicDataFormat dataFormat) NN_NOEXCEPT
{
    if (result.IsSuccess())
    {
        m_DataFormat = dataFormat;
        m_Output.SetPeriodicDataFormat(m_DataFormat);
        TryUpdateDataFormat();
    }
    // 現状 HidCommand は Reuslt.IsFailure() を返さない
}

void DeviceHandler::NotifyDeviceInfo(const DeviceInfo& deviceInfo,
    FirmwareVersionUnit firmwareVersion,
    int serialFlashFormatVersion) NN_NOEXCEPT
{

    m_Info = deviceInfo;
    m_RegisteredInfo.deviceType = deviceInfo.deviceType;
    m_RegisteredInfo.bluetooth.address = deviceInfo.address;
    m_FirmwareVersion.bluetooth = firmwareVersion;
    m_FormatVersion = serialFlashFormatVersion;

    NN_DETAIL_XCD_INFO("Device Connected -> %02x:%02x:%02x:%02x:%02x:%02x firmwareVersion:%02x.%02x Handle:%08x%08x\n", m_Info.address.address[0],
                                                                                                        m_Info.address.address[1],
                                                                                                        m_Info.address.address[2],
                                                                                                        m_Info.address.address[3],
                                                                                                        m_Info.address.address[4],
                                                                                                        m_Info.address.address[5],
                                                                                                        m_FirmwareVersion.bluetooth.major,
                                                                                                        m_FirmwareVersion.bluetooth.minor,
                                                                                                        m_Handle._lower, m_Handle._upper);

    m_Flags.Reset<DeviceHandlerFlag::IsDeviceInfoReading>();
}

void DeviceHandler::NotifyIdentificationCode(const IdentificationCode& code) NN_NOEXCEPT
{
    if (std::memcmp(m_RegisteredInfo.identificationCode._code, code._code, IdentificationCodeLength) != 0)
    {
        //!< SerialNumber が更新されている場合は再登録
        m_RegisteredInfo.identificationCode = code;
        m_Flags.Set<DeviceHandlerFlag::IsRegisterationRequired>();
    }
    m_Flags.Reset<DeviceHandlerFlag::IsIdentificationCodeReading>();
}

void DeviceHandler::NotifyPairingDatabaseUpdate(const BluetoothDeviceInfo& deviceInfo) NN_NOEXCEPT
{
    m_RegisteredInfo.bluetooth = deviceInfo;
    // ペアリングに用いられた Interface の種別を設定
    m_RegisteredInfo.interfaceType = m_Status.interfaceType;
    // ペアリングデータベースを更新する
    m_pPairingManager->RegisterDevice(m_RegisteredInfo);
    m_Flags.Reset<DeviceHandlerFlag::IsRegisterationRequired>();
}

void DeviceHandler::NotifyPairingComplete(Result result) NN_NOEXCEPT
{
    NN_UNUSED(result);
    m_Flags.Reset<DeviceHandlerFlag::IsOnPairing>();
}

/**
 * @brief      HidCommandに対して Design 領域の読み出しが完了した際にに通知を受けるための関数です
 *
 * @param[in]  color                   色情報
 * @param[in]  variation               デザインバリエーション
 *
 */
void DeviceHandler::NotifyDesign(const DeviceColor& color,
                              uint8_t variation) NN_NOEXCEPT
{
    if (m_FormatVersion >= SerialFlashVersion_AnalogStickValid)
    {
        if ((std::memcmp(&m_RegisteredInfo.color.color[0], &color.color[0], sizeof(m_RegisteredInfo.color.color[0])) ||
             std::memcmp(&m_RegisteredInfo.color.color[1], &color.color[1], sizeof(m_RegisteredInfo.color.color[1]))) ||
            (m_FormatVersion >= SerialFlashVersion_DesignExt &&
             std::memcmp(&m_RegisteredInfo.color, &color, sizeof(m_RegisteredInfo.color))))
        {
            // 色情報が更新されている場合は再登録
            m_RegisteredInfo.color = color;
            m_Flags.Set<DeviceHandlerFlag::IsRegisterationRequired>();
        }
    }
    else
    {
        switch (m_Info.deviceType)
        {
        case nn::xcd::DeviceType_Left:
        case nn::xcd::DeviceType_Right:
        case nn::xcd::DeviceType_MiyabiLeft:
        case nn::xcd::DeviceType_MiyabiRight:
            m_RegisteredInfo.color = nn::hid::detail::ColorJoyConDefault;
            break;

        case nn::xcd::DeviceType_Unknown:
        case nn::xcd::DeviceType_FullKey:
        case nn::xcd::DeviceType_Tarragon:
            m_RegisteredInfo.color = nn::hid::detail::ColorSwitchProControllerDefault;
            break;
        default: NN_UNEXPECTED_DEFAULT;
        }
    }

    // Tarragon の場合はデフォルトカラー上書き
    if (m_Info.deviceType == nn::xcd::DeviceType_Tarragon)
    {
        m_RegisteredInfo.color = nn::hid::detail::ColorSwitchProControllerDefault;
    }

    if (m_FormatVersion >= SerialFlashVersion_DesignExt)
    {
        m_RegisteredInfo.designVariation = variation;
    }
    else
    {
        m_RegisteredInfo.designVariation = 0;
    }

    m_DeviceColor = m_RegisteredInfo.color;
    m_Flags.Reset<DeviceHandlerFlag::IsColorReading>();
}

void DeviceHandler::NotifyUpdateControllerColor(Result result) NN_NOEXCEPT
{
    NN_UNUSED(result);
    // 何もしない
}

void DeviceHandler::NotifyUpdateDesignInfo(Result result) NN_NOEXCEPT
{
    if(result.IsSuccess() && m_FormatVersion == 1)
    {
        // バージョンを 2に挙げる
        m_Command.UpdateFormatVersion(2, this);
    }
}

void DeviceHandler::TryUpdateDataFormat() NN_NOEXCEPT
{
    auto currentSlotSize = detail::GetLinkMonitor().GetSlotSize(m_Handle);

    if (currentSlotSize < 0)
    {
        // SlotSize 変更中は何もしない
        return;
    }

    // スロットサイズが0のときは制約なし
    if (currentSlotSize != 0)
    {
        int targetSlotSize = GetTargetSlotSize(m_TargetDataFormat);
        if (targetSlotSize > 0 && m_Flags.Test<DeviceHandlerFlag::NeedsChangeSlotSizeRequest>())
        {
            m_Flags.Reset<DeviceHandlerFlag::NeedsChangeSlotSizeRequest>();

            detail::GetLinkMonitor().SetSlotSize(m_Handle, targetSlotSize);

            if (detail::GetLinkMonitor().IsChangingSlotSize(m_Handle))
            {
                // スロットサイズ変更待ちが必要な場合は、変更が完了してからデータフォーマットを変える
                return;
            }
        }
    }

    // BtUpdate は Hid のデータフォーマット変更は発生しない
    if (m_TargetDataFormat != m_DataFormat &&
        m_TargetDataFormat != PeriodicDataFormat_BtUpdate)
    {
        // SlotSize が一致しているもしくは0ならば、フォーマット変更
        m_Command.SetDataFormat(m_TargetDataFormat, this);
    }
}

int DeviceHandler::GetTargetSlotSize(PeriodicDataFormat dataFormat) NN_NOEXCEPT
{
    int currentTargetSlotSize = detail::GetLinkMonitor().GetTargetSlotSize(m_Handle);

    // 0xFF は Active
    // 高速 MCU FW 更新対応なら McuUpdate も Active を使用
    if (dataFormat == PeriodicDataFormat_BtUpdate ||
        (dataFormat == PeriodicDataFormat_McuUpdate &&
            m_TeraBase.IsAvailableFastFirmwareUpdate()))
    {
        return 0xFF;
    }

    // 4 slot フォーマットに変更する場合で現在2 Slot ならば先にSlotSize を変更
    // 高速 MCU FW 更新非対応なら McuUpdate は 4 slot
    else if ((dataFormat == PeriodicDataFormat_MCU ||
        dataFormat == PeriodicDataFormat_McuUpdate) &&
        currentTargetSlotSize != 4)
    {
        return 0x04;
    }

    // 2 Slot フォーマットに変更する場合で DataFormat 変更が完了していれば、SlotSize を変更
    else if (dataFormat == PeriodicDataFormat_Basic &&
        dataFormat == m_DataFormat &&
        currentTargetSlotSize != 2)
    {
        return 0x02;
    }

    return 0x00;
}

void DeviceHandler::ParseInputReport(const uint8_t* buffer, size_t size) NN_NOEXCEPT
{
    NN_UNUSED(size);

    if (IsParsableReportId(buffer[ReportByte_ReportId]) == true)
    {
        // パケット受信履歴の更新
        int sampleSinceLast = 0;
        if (m_TransactionMonitor.IsActivated())
        {
            ::nn::TimeSpan interval;
            if (m_pAccessor->GetInterval(&interval).IsFailure())
            {
                interval = ::nn::TimeSpan(0);
            }
            m_TransactionMonitor.UpdateRxMonitoringResult(&sampleSinceLast, buffer[InputReportByte_SampleNumber], interval);
        }

        // ボタン・アナログスティック・センサーの解釈
        SensorSleepValueType sensor = SensorSleepValueType_Sleep;
        auto validSensorCount = m_AttachmentBase.GetValidSensorCount();
        if (buffer[ReportByte_ReportId] == Report_CommandIn::Id ||
            (validSensorCount == 0))
        {
            sensor = SensorSleepValueType_Sleep;
        }
        else
        {
            if (m_FirmwareVersion.bluetooth > FirmwareVersionUnit({ 0, 0 }))
            {
                sensor = SensorSleepValueType_Active;
                FirmwareVersionUnit const dscaleMinVersion = { 0x3, 0x8F }; // TORIAEZU: Preliminary version for internal development
                FirmwareVersionUnit const currentVersion = m_FirmwareVersion.bluetooth;
                if (currentVersion >= dscaleMinVersion)
                {
                    if (validSensorCount == 2) // TORIAEZU
                    {
                        sensor = SensorSleepValueType_Dscale1Payload6;
                        validSensorCount = 3;
                    }
                    else
                    {
                        sensor = SensorSleepValueType_Dscale1;
                    }
                }
            }
            else
            {
                sensor = SensorSleepValueType_Sleep;
            }
        }

        auto padSize = (sensor != SensorSleepValueType_Sleep) ? InputReportSize_PadFull : InputReportSize_PadBasic;

        m_PadInput.ParseInputReport(buffer, padSize, sampleSinceLast, sensor, validSensorCount);
        if (m_pSamplingEvent != nullptr)
        {
            nn::os::SignalSystemEvent(m_pSamplingEvent);
        }

        // コントローラの状態のパース
        ParseControllerStatus(buffer[InputReportByte_Status], buffer[InputReportByte_Powered]);

        // 振動モーターのパース
        m_VibratorBase.ParseInputReport(&buffer[InputReportByte_Vibrator], InputReportSize_Vibrator, buffer[InputReportByte_SampleNumber]);

        // ReportId毎の処理
        if (buffer[ReportByte_ReportId] == Report_CommandIn::Id)
        {
            // HidCommandの処理
            m_Command.ParseInputReport(&buffer[CommandInReportByte_Payload], CommandInReportSize_Payload);
        }
        else if(buffer[ReportByte_ReportId] == Report_McuUpdateIn::Id)
        {
            // McuUpdateInの処理
            m_TeraBase.ParseMcuUpdateInputReport(&buffer[McuUpdateInReportByte_Payload], McuUpdateInReportSize_Payload, buffer[InputReportByte_SampleNumber]);
        }
        else if (buffer[ReportByte_ReportId] == Report_McuIn::Id)
        {
            // McuInの処理
            m_TeraBase.ParseInputReport(&buffer[McuInReportByte_Payload], McuInReportSize_Payload, buffer[InputReportByte_SampleNumber]);
        }
        else if (buffer[ReportByte_ReportId] == Report_AttachmentIn::Id)
        {
            // AttachmentInの処理
            // 現時点では Attachment In は使用せず、 BasicIn を Config して使用する
        }
        else if (buffer[ReportByte_ReportId] == Report_AudioIn::Id)
        {
            // TODO: AudioInの処理
        }

        // AttachmentInの処理
        if (buffer[ReportByte_ReportId] == Report_BasicIn::Id)
        {
            m_AttachmentBase.ParseInputReport(&buffer[ReportByte_ReportId], Report_BasicIn::Size, buffer[InputReportByte_SampleNumber]);
        }

    }
}

void DeviceHandler::SendOutputReport() NN_NOEXCEPT
{
    uint8_t buffer[HidBufferSize];

    auto size = m_Output.GetHidReport(buffer, sizeof(buffer), m_BtFirmwareUpdater.IsInsideFirmwareUpdateSequence());
    // サイズチェックは Accessor 下で行う
    m_pAccessor->SetOutputReport(buffer, size);
}

void DeviceHandler::ParseControllerStatus(const uint8_t status, const uint8_t statusForPower) NN_NOEXCEPT
{
    // CONDET の安定時間
    const nn::TimeSpan CondetStableTime = nn::TimeSpan::FromMilliSeconds(30);

    m_Status.batteryLevel = static_cast<BatteryLevel>((status >> InputReportBit_BatteryLeast) & 0x07);
//    NN_ABORT_UNLESS_NOT_EQUAL(m_Status.batteryLevel, BatteryLevel_None);
    auto previousCharged = m_Status.charged;
    m_Status.charged = (status & (1 << InputReportBit_Charge)) != 0;
    bool isChargeStatusUpdated = (m_Status.charged != previousCharged);
    m_Status.powered = (statusForPower & (1 << InputReportBit_Powered)) != 0;
    bool attachmentReadyCurrentStatus = (status & (1 << InputReportBit_AttachmentExtra)) != 0;
    auto previousAttachmentExBit = m_Status.isAttachmentReady;
    uint8_t attachmentType = (status >> InputReportBit_AttachmentLeast) & 0x07;

    // バッテリーの充電状態をチェック
    CheckChargeStatus(isChargeStatusUpdated);

    if (previousAttachmentExBit != attachmentReadyCurrentStatus)
    {
        //NN_DETAIL_XCD_ERROR("Attachment Ex Bit is Changed\n");
    }

    auto previousConnector = m_Status.connector;

    // attachmentType (CONDET) の値は即値が返ってくるため、 30ms の待ちをカウントした後に、実際の connecor に反映する
    if (m_Status._AttachmentTypeTemp == attachmentType)
    {
        // 現在の tick を取得して、CondetStableTime 経っていたら、反映
        if ((nn::os::GetSystemTick() - m_Status._connectorTempGetTime).ToTimeSpan() > CondetStableTime)
        {
            switch (attachmentType)
            {
            case AttachmentType_Nwcp:
                // 拡張グリップ検出は暫定的に無効
                m_Status.connector = Connector_Detached;
                // Uart デバイスかつ給電状態の場合はケーブル接続状態とする
                m_Status.cablePlugged = m_Status.powered;
                break;
            case AttachmentType_ExtDevice:
                m_Status.connector = Connector_AttachedToEXDevice;
                break;
            case AttachmentType_NotAttached:
                m_Status.cablePlugged = false;
                m_Status.connector = Connector_Detached;
                break;
            default:
                m_Status.cablePlugged = false;
                m_Status.connector = Connector_Detached;
                break;
            }
        }
        else
        {
            //NN_DETAIL_XCD_ERROR("CONDET is Changed\n");
        }
        // 経ってなかったら、反映しない
    }
    else
    {
        // CONDET の変化が起きた可能性があるので、temp を更新して、取得時間も更新
        m_Status._AttachmentTypeTemp = attachmentType;
        m_Status._connectorTempGetTime = nn::os::GetSystemTick();
        //NN_DETAIL_XCD_ERROR("CONDET is Changed\n");
    }

    // AttachmentExBit の安定後の値を取り出す。 (安定していなかったら古い値が取り出される)
    m_Status.isAttachmentReady = m_AttachmentBase.GetStabledAttachmentExBitValue(attachmentReadyCurrentStatus,
                                                                                 nn::os::GetSystemTick().ToTimeSpan());

    // 通信インタフェースが Nwcp の場合は、ケーブルは常時 True, connector は常時 AttachedToConsole
    if (m_Status.interfaceType == InterfaceType_Uart)
    {
        // 本体ジョイント状態については、本体の給電状態をそのままケーブル接続状態に反映する
        m_Status.cablePlugged = detail::GetLinkMonitor().IsConsolePowered();
        // 拡張グリップ検出は暫定的に無効
        m_Status.connector = (m_Status.interfaceType == InterfaceType_Uart) ? Connector_AttachedToConsole : Connector_Detached;
    }

    // フルコンの場合は、電源状態 = ケーブル状態
    if (m_Info.deviceType == DeviceType_FullKey ||
        m_Info.deviceType == DeviceType_MiyabiLeft ||
        m_Info.deviceType == DeviceType_MiyabiRight)
    {
        m_Status.cablePlugged = m_Status.powered;
    }

    if (previousConnector != m_Status.connector)
    {
        // コネクタの状態に変更があったら通知
        if (m_pStatusUpdateEvent != nullptr)
        {
            ::nn::os::SignalSystemEvent(m_pStatusUpdateEvent);
        }
    }

    // アタッチメント周りに更新があったらステータスを変更する
    if (previousConnector != m_Status.connector || previousAttachmentExBit != m_Status.isAttachmentReady)
    {
        m_AttachmentBase.UpdateAttachmentStatus(m_Status);
    }
}

void DeviceHandler::InputReportHandler(void* pParser, const uint8_t* buffer, size_t size) NN_NOEXCEPT
{
    reinterpret_cast<DeviceHandler*>(pParser)->InputReportHandlerImpl(buffer, size);
}

void DeviceHandler::InitializeDevice() NN_NOEXCEPT
{
    m_Command.SetDataFormat(m_TargetDataFormat, nullptr);

    m_VibratorBase.SetInterfaceType(m_Status.interfaceType);

    m_TransactionMonitor.Activate(m_Info, m_FirmwareVersion);
    m_BtFirmwareUpdater.Activate(m_Info.deviceType, m_pAccessor);
    m_TeraBase.Activate(m_Info.deviceType, m_FirmwareVersion);
    m_TeraBase.SetHidAccessor(m_pAccessor);
    m_VibratorBase.Activate(m_Info.deviceType, m_FirmwareVersion);
    m_PadInput.Activate(m_Info.deviceType, m_FirmwareVersion);
    m_PadInput.SetSerialFlashFormatVersion(m_FormatVersion);
    m_AttachmentBase.Activate(m_Info.deviceType, m_FirmwareVersion);
    m_SerialFlashAccessor.Activate();
}

void DeviceHandler::UpdateSequence() NN_NOEXCEPT
{
    auto nextSequence = m_Sequence;

    if (m_Flags.Test<DeviceHandlerFlag::IsErrorOccurred>())
    {
        // 何か想定されないエラーが起きた場合は切断シーケンスに入る
        nextSequence = DeviceSequence_Detach;
    }
    else
    {
        switch (m_Sequence)
        {
        case DeviceSequence_Linkup:
            nextSequence = DeviceSequence_ReadDeviceInfo;
            break;

        case DeviceSequence_ReadDeviceInfo:
            // DeviceInfo の読み出し完了を待つ
            if (m_Flags.Test<DeviceHandlerFlag::IsDeviceInfoReading>() == false)
            {
                nextSequence = DeviceSequence_ShipmentClear;
            }
            break;

        case DeviceSequence_ShipmentClear:
            // Shipment 処理の完了を待つ
            if (m_Flags.Test<DeviceHandlerFlag::IsOnShipmentClear>() == false)
            {
                // シリアルの読み出し
                nextSequence = DeviceSequence_ReadIdCode;
            }
            break;

        case DeviceSequence_ReadIdCode:
            // S/N の読み出し完了を待つ
            if (m_Flags.Test<DeviceHandlerFlag::IsIdentificationCodeReading>() == false)
            {
                nextSequence = DeviceSequence_ReadColor;
            }
            break;

        case DeviceSequence_ReadColor:
            // 色情報の読み出し完了を待つ
            if (m_Flags.Test<DeviceHandlerFlag::IsColorReading>() == false)
            {
                nextSequence = m_pPairingManager->IsWiredPairingRequired(m_Status.interfaceType, m_Info.deviceType) ? DeviceSequence_Pairing : DeviceSequence_Initialize;
            }
            break;

        case DeviceSequence_Pairing:
            // ペアリング処理の完了を待つ
            if (m_Flags.Test<DeviceHandlerFlag::IsOnPairing>() == false)
            {
                nextSequence = DeviceSequence_Initialize;
            }
            break;

        case DeviceSequence_Initialize:
            // ペリフェラル毎の初期化完了待ち
            if (IsPeripheralInitializationCompleted())
            {
                nextSequence = DeviceSequence_Active;
            }
            break;

        case DeviceSequence_Active:
        case DeviceSequence_Detach:
        case DeviceSequence_Nolink:
            // 何もしない
            break;

        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    if (nextSequence != m_Sequence)
    {
        m_Sequence = nextSequence;
        HandleNextSequence();
    }
}

void DeviceHandler::HandleNextSequence() NN_NOEXCEPT
{
    switch (m_Sequence)
    {
    case DeviceSequence_Linkup:
        // 何もしない
        break;

    case DeviceSequence_ReadDeviceInfo:
        m_Command.Activate();
        m_Command.GetDeviceInfo(this);
        m_Flags.Set<DeviceHandlerFlag::IsDeviceInfoReading>();
        break;

    case DeviceSequence_ShipmentClear:
        m_Command.SetShipment(false, this);
        m_Flags.Set<DeviceHandlerFlag::IsOnShipmentClear>();
        break;

    case DeviceSequence_ReadIdCode:
        m_pPairingManager->GetRegisteredDeviceInfo(&m_RegisteredInfo, m_Info.address);
        m_Command.ReadIdentificationCode(this);
        m_Flags.Set<DeviceHandlerFlag::IsIdentificationCodeReading>();
        break;

    case DeviceSequence_ReadColor:
        m_Command.ReadDesign(this);
        m_Flags.Set<DeviceHandlerFlag::IsColorReading>();
        break;

    case DeviceSequence_Pairing:
        TriggerPairingImpl();
        break;

    case DeviceSequence_Initialize:
        if (m_Flags.Test<DeviceHandlerFlag::IsRegisterationRequired>())
        {
            m_pPairingManager->RegisterDevice(m_RegisteredInfo);
            m_Flags.Reset<DeviceHandlerFlag::IsRegisterationRequired>();
        }

        InitializeDevice();
        break;

    case DeviceSequence_Detach:
        if (m_Flags.Test<DeviceHandlerFlag::IsRegisterationRequired>())
        {
            m_pPairingManager->RegisterDevice(m_RegisteredInfo);
            m_Flags.Reset<DeviceHandlerFlag::IsRegisterationRequired>();
        }

        m_pAccessor->DetachDevice();
        break;

    case DeviceSequence_Active:
        if (m_Flags.Test<DeviceHandlerFlag::NotifyOnConnectionComplete>())
        {
            if (m_pStatusUpdateEvent != nullptr)
            {
                nn::os::SignalSystemEvent(m_pStatusUpdateEvent);
            }
            m_Flags.Reset<DeviceHandlerFlag::NotifyOnConnectionComplete>();
        }
        break;

    case DeviceSequence_Nolink:
        // 何もしない
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

bool DeviceHandler::IsPeripheralInitializationCompleted() NN_NOEXCEPT
{
    return (m_PadInput.IsActivated() == true &&
            m_TeraBase.IsActivated() == true &&
            m_VibratorBase.IsActivated() == true &&
            m_BtFirmwareUpdater.IsActivated() == true &&
            m_AttachmentBase.IsActivated()
        );
}

void DeviceHandler::InputReportHandlerImpl(const uint8_t* buffer, size_t size) NN_NOEXCEPT
{
    // 受信した Report の Parse
    ParseInputReport(buffer, size);

    // Seqence の処理
    UpdateSequence();

    // Output Report の生成/送信
    SendOutputReport();
}

void DeviceHandler::OnTsiChanged() NN_NOEXCEPT
{
    if (m_TargetDataFormat != m_DataFormat)
    {
        auto slotSize = detail::GetLinkMonitor().GetSlotSize(m_Handle);
        if (m_TargetDataFormat == PeriodicDataFormat_BtUpdate &&
            (slotSize == 0x04 || slotSize == 0xFF))
        {
            // TSI の変更完了
            m_DataFormat = PeriodicDataFormat_BtUpdate;
            // Firmware の更新開始
            m_BtFirmwareUpdater.StartFirmwareUpdate();
        }
        TryUpdateDataFormat();
    }
}

void DeviceHandler::TriggerPairingImpl() NN_NOEXCEPT
{
    BluetoothDeviceInfo deviceInfo;

    deviceInfo = GetHostBluetoothDeviceInfo();
    m_Command.Pairing(deviceInfo,
        this,
        m_pPairingManager->IsDevicePaired(m_Info.address) && m_FirmwareVersion.bluetooth >= FirmwareVersionBt_0304
    );
    m_Flags.Set<DeviceHandlerFlag::IsOnPairing>();
}

void DeviceHandler::CheckChargeStatus(bool isChargeStatusUpdated) NN_NOEXCEPT
{
    // (WorkAround:IAAA-4066) 充電が開始/停止されたタイミングでタイマーを制御
    if (nn::os::TryWaitTimerEvent(&m_ChargerTimer))
    {
        // IR 使用中の場合のみリセット
        if (m_Status.charged && m_TeraBase.GetMcuState() == McuState_Ir)
        {
            ResetChargeStatus();
        }
        // リセット直後も、何もなかった場合でもタイマーをリスタート
        nn::os::StartOneShotTimerEvent(&m_ChargerTimer, ContinuousChargeIntervalMax);
    }
    if (isChargeStatusUpdated)
    {
        if (m_Status.charged)
        {
            // タイマーを開始
            nn::os::StartOneShotTimerEvent(&m_ChargerTimer, ContinuousChargeIntervalMax);
        }
        else
        {
            // タイマーを停止
            nn::os::StopTimerEvent(&m_ChargerTimer);
        }
    }
}

void DeviceHandler::ResetChargeStatus() NN_NOEXCEPT
{
    NN_DETAIL_XCD_INFO("Reset controller charge status\n");
    m_Command.WriteChargeSetting(false, this);
    m_Command.WriteChargeSetting(true, this);
}

void DeviceHandler::ResetBatteryChargerTimer() NN_NOEXCEPT
{
    // 次回のパケット到着時にハード側の電池充電タイマーをリセット
    nn::os::SignalTimerEvent(&m_ChargerTimer);
}

}} // namespace nn::xcd
