﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <vector>
#include <nn/nn_Result.h>
#include <nn/nn_Macro.h>
#include <nn/fs.h>
#include <nn/xcd/xcd_Firmware.h>
#include "xcd_Peripheral.h"
#include "xcd_BtFirmwareUpdaterTypes.h"
#include "xcd_OutputGenerator.h"
#include "detail/xcd_FirmwareUpdaterStream.h"
#include "detail/xcd_HidAccessor.h"

namespace nn { namespace xcd {

//!< ボタン, アナログスティック, 6軸センサーを扱うクラス
class BtFirmwareUpdater final : detail::IHidListener
{

private:
    //!< Firmware イメージのストリーム
    detail::FirmwareUpdaterStream m_Stream;

    //!< ファームウェア更新終了時に通知する SystemEvent
    nn::os::SystemEventType* m_pSystemEvent;

    //!< 書き込み/読み込みを行う最小単位
    static const size_t OtaUnitSize = 249;

    //!< 更新失敗エミュレーションの有効状態
    static bool g_IsUpdateFailureEmulationEnabled;

    //!< HidAccessor
    detail::HidAccessor* m_pHidAccessor;

    //!< Firmware 更新処理のステージ
    enum InternalUpdateStage
    {
        InternalUpdateStage_NotStarted,                          //!< Firmware 更新開始前
        InternalUpdateStage_EnableOtaFirmwareUpdate,             //!< Firmware Update 機能の有効化
        InternalUpdateStage_CheckSignature,                      //!< Signature の確認
        InternalUpdateStage_EraseDynamicSection,                 //!< Dynamic Section の消去
        InternalUpdateStage_Download,                            //!< Firmware のダウンロード
        InternalUpdateStage_Verify,                              //!< Firmware のベリファイ
        InternalUpdateStage_EraseFailsafeForDs1,                 //!< Failsafe Section の消去 (DS1 更新時)
        InternalUpdateStage_EraseFailsafeForDs2,                 //!< Failsafe Section の消去 (DS2 更新時)
        InternalUpdateStage_UpdateDs2Offset,                     //!< DS2 Offset　の更新
        InternalUpdateStage_UpdateSignature,                     //!< Signature の更新
        InternalUpdateStage_Launch,                              //!< Firmware の再起動 (使用しない)
        InternalUpdateStage_Completed,                           //!< Firmware 更新完了
        InternalUpdateStage_Error,                               //!< エラー状態
    };
    InternalUpdateStage m_InternalStage;

    //!< Firmware 更新する Dynamic Section の Index
    OtaDynamicSectionIndex m_DynamicSectionIndex;

    //!< SetReport した ReportID
    uint8_t m_SetReportId;

    //!< エラー時のReason を保持するための Result
    Result m_ErrorResult;

    //!< Serial Flsah に対する操作の種類
    enum MemoryControlType
    {
        MemoryControlType_None,   //!< 何も操作を行っていない
        MemoryControlType_Read,   //!< Read
        MemoryControlType_Write,  //!< Write
        MemoryControlType_Erase,  //!< Erase;
    };

    //!< 連続的に Serial Flash を操作するためのパラメーター
    struct MemoryControlParameters
    {
        MemoryControlType type;   //!< 操作の種類
        uint32_t address;        //!< 開始アドレス
        uint32_t length;         //!< 読み出す大きさ
        uint32_t unitLength;     //!< 一回当たりの操作単位
        uint32_t index;          //!< 現在の操作位置
        uint32_t previousIndex;  //!< 前回の操作位置
        uint8_t* pBuffer;        //!< 【Write のみ】 書き込むデータのバッファ
    };
    MemoryControlParameters m_MemoryControlParameters;

public:
    //!< 更新失敗エミュレーションの有効状態を設定する関数
    static void SetFirmwareUpdateFailureEmulationEnabled(bool isEnabled) NN_NOEXCEPT
    {
        g_IsUpdateFailureEmulationEnabled = isEnabled;
    }

    BtFirmwareUpdater() NN_NOEXCEPT;
    virtual ~BtFirmwareUpdater() NN_NOEXCEPT NN_OVERRIDE;

    //!< デバイスが接続されたときに呼ばれる関数
    void Activate(DeviceType type, detail::HidAccessor *pHidAccessor) NN_NOEXCEPT;
    //!< 接続直後の初期化処理が完了したかどうかを確認する関数。初期化が完了したら必ずtrueを返してください
    bool IsActivated() NN_NOEXCEPT;
    //!< デバイスが切断された時に呼ばれる関数
    void Deactivate() NN_NOEXCEPT;

    Result InitializeFirmwareUpdate(const FirmwareImage& fileHandle, nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT;

    Result StartFirmwareUpdate() NN_NOEXCEPT;

    Result AbortFirmwareUpdate(bool* pOutIsRebootRequired) NN_NOEXCEPT;

    Result GetFirmwareUpdateProgress(FirmwareUpdateStage* pOutStage, int* pOutProgress) NN_NOEXCEPT;

    bool IsInsideFirmwareUpdateSequence() NN_NOEXCEPT;

    virtual void GetReportComplete(uint8_t* pBuffer, size_t size) NN_NOEXCEPT NN_OVERRIDE;
    virtual void GetReportFailed(int status) NN_NOEXCEPT NN_OVERRIDE;
    virtual void SetReportComplete(Result result) NN_NOEXCEPT NN_OVERRIDE;
    virtual void FirmwareUpdateModeEnableComplete() NN_NOEXCEPT NN_OVERRIDE;

private:
    void ProceedNextStage() NN_NOEXCEPT;

    void EnableOtaFirmwareUpdate() NN_NOEXCEPT;
    void CheckSignature() NN_NOEXCEPT;
    void EraseDynamicSection() NN_NOEXCEPT;
    void DownloadFirmware() NN_NOEXCEPT;
    void VerifyFirmware() NN_NOEXCEPT;
    void UpdateFailsafeSection() NN_NOEXCEPT;
    void UpdateDs2Offset() NN_NOEXCEPT;
    void UpdateSignature() NN_NOEXCEPT;
    void CompleteUpdate() NN_NOEXCEPT;
    void Launch() NN_NOEXCEPT;
    void ErrorOccurred(Result result) NN_NOEXCEPT;

    Result ContinueDownload() NN_NOEXCEPT;
    Result ContinueVerify() NN_NOEXCEPT;
    Result VerifyFirmwareImpl(uint8_t* pBuffer, size_t size) NN_NOEXCEPT;

    void ReadMemory(uint32_t address, uint16_t length) NN_NOEXCEPT;
    void WriteMemory(uint32_t address, uint32_t length, uint8_t* pBuffer) NN_NOEXCEPT;
    void EraseMemory(uint32_t address, uint32_t length) NN_NOEXCEPT;
    void StartMemoryControl(uint32_t address, uint32_t length, uint8_t* pBuffer, MemoryControlType type) NN_NOEXCEPT;
    Result ProceedMemoryControl() NN_NOEXCEPT;

    void SetReportImpl(ReportType reportType, uint8_t* pBuffer, size_t size) NN_NOEXCEPT;
    void EnableFwuImpl() NN_NOEXCEPT;
    void SetupReadImpl(uint32_t address, uint16_t length) NN_NOEXCEPT;
    void ReadImpl() NN_NOEXCEPT;
    void EraseImpl(uint32_t address, uint16_t length) NN_NOEXCEPT;
    void WriteImpl(uint32_t address, uint16_t length, uint8_t* pBuffer) NN_NOEXCEPT;
    void LaunchImpl(uint32_t address) NN_NOEXCEPT;
};

}} // namespace nn::xcd
