﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/os/os_Thread.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_BitPack.h>
#include <nn/xcd/xcd_Firmware.h>
#include <nn/xcd/xcd_Result.h>
#include <nn/xcd/xcd_ResultForPrivate.h>
#include "xcd_BtFirmwareUpdater.h"
#include "xcd_BtFirmwareUpdaterTypes.h"
#include "xcd_ReportTypes.h"

#if 0
#define OTA_LOG(...) NN_SDK_LOG(__VA_ARGS__)
#else
#define OTA_LOG(...)
#endif

namespace nn { namespace xcd {

namespace {

void SetUint32ToBuffer(uint8_t* pBuffer, uint32_t value)
{
    pBuffer[0] = static_cast<uint8_t>(value & 0xff);
    pBuffer[1] = static_cast<uint8_t>(value >>  8 & 0xff);
    pBuffer[2] = static_cast<uint8_t>(value >> 16 & 0xff);
    pBuffer[3] = static_cast<uint8_t>(value >> 24 & 0xff);
}

void SetUint16ToBuffer(uint8_t* pBuffer, uint16_t value)
{
    pBuffer[0] = static_cast<uint8_t>(value & 0xff);
    pBuffer[1] = static_cast<uint8_t>(value >>  8 & 0xff);
}

//!< Checksum を計算する
uint8_t CalculateChecksum(uint8_t* pBuffer, size_t size) NN_NOEXCEPT
{
    uint8_t summary = 0;

    for (size_t i  = 0; i < size; i++)
    {
        summary += pBuffer[i];
    }

    return ~(summary - 1);
}

//!< OTA のレポートフォーマットに合わせてバッファを構成する
void SetupOtaReport(uint8_t* pReport,
                    uint8_t reportId,
                    uint32_t address,
                    uint16_t length,
                    uint8_t* pData,
                    size_t dataSize)
{
    pReport[OtaReportByte_ReportId] =  reportId;
    SetUint32ToBuffer(&pReport[OtaReportByte_Address], address);
    SetUint16ToBuffer(&pReport[OtaReportByte_Length], length);
    if (dataSize > 0)
    {
        NN_SDK_REQUIRES_NOT_NULL(pData);
        std::memcpy(&pReport[OtaReportByte_Data], pData, dataSize);
    }
    int payloadSize = static_cast<int>(dataSize + 7);
    pReport[payloadSize] = CalculateChecksum(pReport, payloadSize);
}

}

// 更新失敗エミュレーションのデフォルトは OFF
bool BtFirmwareUpdater::g_IsUpdateFailureEmulationEnabled = false;

BtFirmwareUpdater::BtFirmwareUpdater() NN_NOEXCEPT :
            m_Stream(),
            m_pSystemEvent(nullptr),
            m_pHidAccessor(nullptr),
            m_InternalStage(InternalUpdateStage_NotStarted),
            m_SetReportId(0)
{
    m_MemoryControlParameters.type = MemoryControlType_None;
}

BtFirmwareUpdater::~BtFirmwareUpdater() NN_NOEXCEPT
{
    // 何もしない
}

//!< デバイスが接続されたときに呼ばれる関数
void BtFirmwareUpdater::Activate(DeviceType type, detail::HidAccessor* pHidAccessor) NN_NOEXCEPT
{
    NN_UNUSED(type);

    m_InternalStage = InternalUpdateStage_NotStarted;

    m_pHidAccessor = pHidAccessor;
}

//!< 接続直後の初期化処理が完了したかどうかを確認する関数。初期化が完了したら必ずtrueを返してください
bool BtFirmwareUpdater::IsActivated() NN_NOEXCEPT
{
    return true;
};

void BtFirmwareUpdater::Deactivate() NN_NOEXCEPT
{
    m_InternalStage = InternalUpdateStage_NotStarted;
}

void BtFirmwareUpdater::GetReportComplete(uint8_t* pBuffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);
    NN_UNUSED(size);

    if (m_InternalStage == InternalUpdateStage_Error)
    {
        // エラー発生状態では何もしない
        return;
    }

    uint16_t length = (((uint16_t)pBuffer[6] << 8) & 0xff00) |
                       ((uint16_t)pBuffer[5] & 0x00ff);
    uint8_t* pData =  &pBuffer[7];

    auto result = ProceedMemoryControl();
    // Read が完了したかどうか
    if (ResultBtFirmwareUpdateMemoryControlCompleted().Includes(result))
    {
        // 読み込みが完了
        if (m_MemoryControlParameters.address == OtaAddress_Signature)
        {
            // Signature の Read が完了
            // Siganture が一致する場合は、DS1 を更新
            if (std::memcmp(pData, OtaSignature, sizeof(OtaSignature)) == 0)
            {
                OTA_LOG("[xcd] Update Dynamic Section 1\n");
                m_DynamicSectionIndex = OtaDynamicSectionIndex_1;
            }
            else
            {
                OTA_LOG("[xcd] Update Dynamic Section 2\n");
                m_DynamicSectionIndex = OtaDynamicSectionIndex_2;
            }
            ProceedNextStage();
        }
        else if (m_MemoryControlParameters.address >= OtaAddress_DynamicSection[m_DynamicSectionIndex] &&
                 m_MemoryControlParameters.address <= OtaAddress_DynamicSection[m_DynamicSectionIndex] + OtaSize_DynamicSection)
        {
            auto verifyResult = VerifyFirmwareImpl(pData, length);
            if(verifyResult.IsFailure())
            {
                ErrorOccurred(verifyResult);
                return;
            }
            else
            {
                verifyResult = ContinueVerify();
                if (ResultBtFirmwareUpdateVerifyCompleted().Includes(verifyResult))
                {
                    // Verify 終了
                    ProceedNextStage();
                }
                else
                {
                    NN_ABORT_UNLESS_RESULT_SUCCESS(verifyResult);
                }
            }
        }
        else
        {
            NN_ABORT("Unexpected Get Report Completed Event\n");
        }
    }
    else
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }
}

void BtFirmwareUpdater::GetReportFailed(int status) NN_NOEXCEPT
{
    NN_UNUSED(status);

    if (m_InternalStage == InternalUpdateStage_Error)
    {
        // エラー発生状態では何もしない
        return;
    }

    if (m_MemoryControlParameters.address == OtaAddress_Signature)
    {
        // Signature の Read をやり直す
        m_MemoryControlParameters.type = MemoryControlType_None;
        CheckSignature();
    }
    else if (m_MemoryControlParameters.address >= OtaAddress_DynamicSection[m_DynamicSectionIndex] &&
             m_MemoryControlParameters.address <= OtaAddress_DynamicSection[m_DynamicSectionIndex] + OtaSize_DynamicSection)
    {
        // 直前の Verify をやり直す
        // 完了していないので、常に success になるはず
        m_MemoryControlParameters.type  = MemoryControlType_None;
        m_MemoryControlParameters.index = m_MemoryControlParameters.previousIndex;
        ContinueVerify();
    }
}

void BtFirmwareUpdater::SetReportComplete(Result result) NN_NOEXCEPT
{
    if (m_InternalStage == InternalUpdateStage_Error)
    {
        // エラー発生状態では何もしない
        return;
    }
    else if (result.IsFailure())
    {
        // SetReport に失敗した場合は FirmwareUpdate を中止
        ErrorOccurred(ResultBtFirmwareUpdateUnknownError());
        return;
    }

    if (m_SetReportId == Report_OtaEnableFwu::Id)
    {
        // Enable Firmware Update が完了
        ProceedNextStage();
    }
    else if (m_SetReportId == Report_OtaSetupRead::Id)
    {
        // SetupRead　完了後に Read を行う
        // SetupRead 直後の ProceedMemoryControl() は必ず成功する
        NN_ABORT_UNLESS_RESULT_SUCCESS(ProceedMemoryControl());
    }
    else if (m_SetReportId == Report_OtaErase::Id)
    {
        auto memResult = ProceedMemoryControl();
        // Erase が完了したかどうか
        if (ResultBtFirmwareUpdateMemoryControlCompleted().Includes(memResult))
        {
            switch (m_InternalStage)
            {
            case InternalUpdateStage_EraseDynamicSection:
                // Dynamic Section の Erase が完了
                ProceedNextStage();
                break;
            case InternalUpdateStage_EraseFailsafeForDs1:
                // Failsafe section の Erase が完了 (DS1 更新時)
                ProceedNextStage();
                break;

            case InternalUpdateStage_EraseFailsafeForDs2:
                // Failsafe section の Erase が完了 (DS2 更新時)
                ProceedNextStage();
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
            }
        }
        else
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(memResult);
        }
    }
    else if (m_SetReportId == Report_OtaWrite::Id)
    {
        auto memResult = ProceedMemoryControl();

        // Write Memory が完了したかどうか
        if (ResultBtFirmwareUpdateMemoryControlCompleted().Includes(memResult))
        {
            ::nn::Result downloadResult;

            if (m_MemoryControlParameters.address >= OtaAddress_DynamicSection[m_DynamicSectionIndex] &&
                m_MemoryControlParameters.address <= OtaAddress_DynamicSection[m_DynamicSectionIndex] + OtaSize_DynamicSection)
            {
                downloadResult = ContinueDownload();
                // ダウンロードが完了したかどうか
                if (ResultBtFirmwareUpdateDownloadCompleted().Includes(downloadResult))
                {
                    // 全てのデータの書き込みが完了
                    ProceedNextStage();
                }
                else
                {
                    NN_ABORT_UNLESS_RESULT_SUCCESS(downloadResult);
                }
            }
            else if (m_MemoryControlParameters.address == OtaAddress_Ds2Offset)
            {
                // DS2 Offset の更新が完了
                ProceedNextStage();
            }
            else if (m_MemoryControlParameters.address == OtaAddress_Signature)
            {
                // Signature の更新が完了
                ProceedNextStage();
            }
            else
            {
                NN_ABORT("Unexpected Write memory complete event");
            }
        }
        else
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(memResult);
        }
    }
}

void BtFirmwareUpdater::FirmwareUpdateModeEnableComplete() NN_NOEXCEPT
{
    ProceedNextStage();
}

Result BtFirmwareUpdater::InitializeFirmwareUpdate(
    const FirmwareImage& image,
    nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemEvent);

    if (IsInsideFirmwareUpdateSequence() == false)
    {
        m_InternalStage = InternalUpdateStage_NotStarted;

        // ロードするファイル情報を初期化
        NN_RESULT_DO(m_Stream.Setup(image));

        // システムイベントのセット
        m_pSystemEvent = pSystemEvent;

        // パラメーターの初期化
        m_SetReportId = 0;
        m_MemoryControlParameters.type    = MemoryControlType_None;
        m_MemoryControlParameters.address = 0;
        m_MemoryControlParameters.length  = 0;
        m_MemoryControlParameters.pBuffer = nullptr;
        m_ErrorResult = ResultSuccess();
    }

    NN_RESULT_SUCCESS;
}

Result BtFirmwareUpdater::StartFirmwareUpdate() NN_NOEXCEPT
{
    if (IsInsideFirmwareUpdateSequence() == false)
    {
        m_InternalStage = InternalUpdateStage_NotStarted;
        ProceedNextStage();
    }

    NN_RESULT_SUCCESS;
}

Result BtFirmwareUpdater::AbortFirmwareUpdate(bool* pOutIsRebootRequired) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutIsRebootRequired);

    if (IsInsideFirmwareUpdateSequence() == false)
    {
        *pOutIsRebootRequired = false;
        NN_RESULT_SUCCESS;
    }

    // エラー終了させる
    ErrorOccurred(ResultBtFirmwareUpdateAborted());

    *pOutIsRebootRequired = true;

    NN_RESULT_SUCCESS;
}

Result BtFirmwareUpdater::GetFirmwareUpdateProgress(FirmwareUpdateStage* pOutStage, int* pOutProgress) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutStage);
    NN_SDK_REQUIRES_NOT_NULL(pOutProgress);

    switch (m_InternalStage)
    {
    case InternalUpdateStage_NotStarted:
        NN_RESULT_THROW(ResultBtFirmwareUpdateNotStarted());
        break;

    case InternalUpdateStage_EnableOtaFirmwareUpdate:
    case InternalUpdateStage_CheckSignature:
        // 初期の準備処理はダウンロード処理の一部とする
        *pOutStage = FirmwareUpdateStage_Download;
        *pOutProgress = 1;
        break;

    case InternalUpdateStage_EraseDynamicSection:
        // セクタのイレース処理はダウンロード処理の一部とする
        *pOutStage = FirmwareUpdateStage_Download;
        *pOutProgress = 1 + 9 * m_MemoryControlParameters.index / m_MemoryControlParameters.length;
        break;

    case InternalUpdateStage_Download:
        *pOutStage = FirmwareUpdateStage_Download;
        *pOutProgress = 10 + static_cast<int>(90 * m_Stream.GetTotalReadBytes() / m_Stream.GetImageSize());
        break;

    case InternalUpdateStage_Verify:
        *pOutStage = FirmwareUpdateStage_Verify;
        *pOutProgress = static_cast<int>(100 * m_Stream.GetTotalReadBytes() / m_Stream.GetImageSize());
        break;

    case InternalUpdateStage_EraseFailsafeForDs1:
        *pOutStage = FirmwareUpdateStage_Commit;
        *pOutProgress = 50;
        break;

    case InternalUpdateStage_EraseFailsafeForDs2:
        *pOutStage = FirmwareUpdateStage_Commit;
        *pOutProgress = 25;
        break;

    case InternalUpdateStage_UpdateDs2Offset:
        *pOutStage = FirmwareUpdateStage_Commit;
        *pOutProgress = 50;
        break;

    case InternalUpdateStage_UpdateSignature:
        *pOutStage = FirmwareUpdateStage_Commit;
        *pOutProgress = 75;
        break;

    case InternalUpdateStage_Launch:
        // Launch は現在の実装では使用されていない
        NN_ABORT("Launch is not used in current OTA implementation\n");
        break;

    case InternalUpdateStage_Completed:
        *pOutStage = FirmwareUpdateStage_Completed;
        *pOutProgress = 100;
        break;

    case InternalUpdateStage_Error:
        NN_RESULT_THROW(m_ErrorResult);
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
    }

    NN_RESULT_SUCCESS;
}

bool BtFirmwareUpdater::IsInsideFirmwareUpdateSequence() NN_NOEXCEPT
{
    if (m_InternalStage == InternalUpdateStage_NotStarted ||
    m_InternalStage == InternalUpdateStage_Error)
    {
        return false;
    }

    return true;
}

void BtFirmwareUpdater::ProceedNextStage() NN_NOEXCEPT
{
    switch (m_InternalStage)
    {
    case InternalUpdateStage_NotStarted:
        EnableOtaFirmwareUpdate();
        break;

    case InternalUpdateStage_EnableOtaFirmwareUpdate:
        CheckSignature();
        break;

    case InternalUpdateStage_CheckSignature:
        EraseDynamicSection();
        break;

    case InternalUpdateStage_EraseDynamicSection:
        DownloadFirmware();
        break;

    case InternalUpdateStage_Download:
        VerifyFirmware();
        break;

    case InternalUpdateStage_Verify:
        UpdateFailsafeSection();
        break;

    case InternalUpdateStage_EraseFailsafeForDs1:
        CompleteUpdate();
        break;

    case InternalUpdateStage_EraseFailsafeForDs2:
        UpdateDs2Offset();
        break;

    case InternalUpdateStage_UpdateDs2Offset:
        UpdateSignature();
        break;

    case InternalUpdateStage_UpdateSignature:
        CompleteUpdate();
        break;

    case InternalUpdateStage_Launch:
        // Luanch は使用しない
        NN_ABORT("Launch is not supported");
        break;

    case InternalUpdateStage_Completed:
        // Complete の次のステージは存在しないのでここには到達してはならない
        NN_ABORT("No Stage exist after complete");
        break;

    case InternalUpdateStage_Error:
        // エラーステートのの次のステージは存在しないのでここには到達してはならない
        NN_ABORT("No Stage exist after error");
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

void BtFirmwareUpdater::EnableOtaFirmwareUpdate() NN_NOEXCEPT
{
    OTA_LOG("[xcd] Enable Ota Firmware Update\n");

    m_InternalStage = InternalUpdateStage_EnableOtaFirmwareUpdate;
    m_pHidAccessor->SetFirmwareUpdateModeEnabled(true);
    EnableFwuImpl();
}

void BtFirmwareUpdater::CheckSignature() NN_NOEXCEPT
{
    OTA_LOG("[xcd] Read Signature\n");

    m_InternalStage = InternalUpdateStage_CheckSignature;
    ReadMemory(OtaAddress_Signature, OtaSize_Signature);
}

void BtFirmwareUpdater::EraseDynamicSection() NN_NOEXCEPT
{
    OTA_LOG("[xcd] Erase Dynamic Section\n");

    m_InternalStage = InternalUpdateStage_EraseDynamicSection;
    EraseMemory(OtaAddress_DynamicSection[m_DynamicSectionIndex], OtaSize_DynamicSection);
}

void BtFirmwareUpdater::DownloadFirmware() NN_NOEXCEPT
{
    OTA_LOG("[xcd] Download Firmware\n");

    m_InternalStage = InternalUpdateStage_Download;
    m_Stream.Rewind();
    ContinueDownload();
}

void BtFirmwareUpdater::VerifyFirmware() NN_NOEXCEPT
{
    OTA_LOG("[xcd] Verify Firmware\n");

    m_InternalStage = InternalUpdateStage_Verify;
    m_Stream.Rewind();
    ContinueVerify();
}

void BtFirmwareUpdater::UpdateFailsafeSection() NN_NOEXCEPT
{
    OTA_LOG("[xcd] Update Failsafe section\n");

    if (m_DynamicSectionIndex == OtaDynamicSectionIndex_1)
    {
        // Dynamic Section 1 を使うように Failsafe Section を更新
        m_InternalStage = InternalUpdateStage_EraseFailsafeForDs1;
    }
    else
    {
        // Dynamic Section 2 を使うように Failsafe Section を更新
        m_InternalStage = InternalUpdateStage_EraseFailsafeForDs2;
    }
    EraseMemory(OtaAddress_FailsafeSection, OtaSize_FailsafeSection);
}

void BtFirmwareUpdater::UpdateDs2Offset() NN_NOEXCEPT
{
    uint8_t data[OtaSize_Ds2Offset];
    SetUint32ToBuffer(data, OtaAddress_DynamicSection[OtaDynamicSectionIndex_2]);

    OTA_LOG("[xcd] Update Ds2 Offset\n");

    m_InternalStage = InternalUpdateStage_UpdateDs2Offset;
    WriteMemory(OtaAddress_Ds2Offset, OtaSize_Ds2Offset, data);
}

void BtFirmwareUpdater::UpdateSignature() NN_NOEXCEPT
{
    uint8_t data[OtaSize_Signature];
    std::memcpy(data, OtaSignature, OtaSize_Signature);

    OTA_LOG("[xcd] Update Signature\n");

    m_InternalStage = InternalUpdateStage_UpdateSignature;
    WriteMemory(OtaAddress_Signature, OtaSize_Signature, data);
}

void BtFirmwareUpdater::CompleteUpdate() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pSystemEvent);

    OTA_LOG("[xcd] OTA Completed\n");
    m_pHidAccessor->SetFirmwareUpdateModeEnabled(false);
    m_InternalStage = InternalUpdateStage_Completed;

    nn::os::SignalSystemEvent(m_pSystemEvent);
}

void BtFirmwareUpdater::Launch() NN_NOEXCEPT
{
    OTA_LOG("[xcd] Launch\n");

    m_InternalStage = InternalUpdateStage_Launch;
    LaunchImpl(0);
}

void BtFirmwareUpdater::ErrorOccurred(Result result) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pSystemEvent);
    NN_SDK_REQUIRES(result.IsFailure());

    OTA_LOG("[xcd] Error Occurred\n");

    m_pHidAccessor->SetFirmwareUpdateModeEnabled(false);
    m_InternalStage = InternalUpdateStage_Error;
    m_ErrorResult = result;

    nn::os::SignalSystemEvent(m_pSystemEvent);
}

Result BtFirmwareUpdater::ContinueDownload() NN_NOEXCEPT
{
    size_t length;
    uint8_t buffer[OtaUnitSize] = { 0 };

    auto writeOffset = m_Stream.GetTotalReadBytes();
    NN_RESULT_DO(m_Stream.Read(buffer, &length, OtaUnitSize));

    // 読み出しサイズがゼロの場合は、更新完了
    if (length == 0 ||
        m_Stream.GetTotalReadBytes() > m_Stream.GetImageSize() ||
        m_Stream.GetTotalReadBytes() > OtaSize_DynamicSection)
    {
        NN_RESULT_THROW(ResultBtFirmwareUpdateDownloadCompleted());
    }

    WriteMemory(
        OtaAddress_DynamicSection[m_DynamicSectionIndex] + static_cast<uint32_t>(writeOffset),
        static_cast<uint32_t>(length),
        buffer);

    NN_RESULT_SUCCESS;
}

Result BtFirmwareUpdater::ContinueVerify() NN_NOEXCEPT
{
    auto length = m_Stream.GetRemainBytes();

    if (length > OtaUnitSize)
    {
        length = OtaUnitSize;
    }

    if (length == 0)
    {
        NN_RESULT_THROW(ResultBtFirmwareUpdateVerifyCompleted());
    }

    ReadMemory(
        OtaAddress_DynamicSection[m_DynamicSectionIndex] + static_cast<uint32_t>(m_Stream.GetTotalReadBytes()),
        static_cast<uint16_t>(length));

    NN_RESULT_SUCCESS;
}

Result BtFirmwareUpdater::VerifyFirmwareImpl(uint8_t* pBuffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);

    // FW 更新失敗エミュレーション時は常に失敗
    if (g_IsUpdateFailureEmulationEnabled)
    {
        NN_RESULT_THROW(ResultBtFirmwareUpdateVerifyError());
    }

    size_t readLength;
    uint8_t bufferForImage[OtaUnitSize];

    m_Stream.Read(bufferForImage, &readLength, size);

    // 読み出しサイズがゼロの場合は異常
    if (readLength == 0 ||
        std::memcmp(pBuffer, bufferForImage, size) != 0)
    {
        NN_RESULT_THROW(ResultBtFirmwareUpdateVerifyError());
    }

    NN_RESULT_SUCCESS;
}

void BtFirmwareUpdater::ReadMemory(uint32_t address, uint16_t length) NN_NOEXCEPT
{
    StartMemoryControl(address, length, nullptr, MemoryControlType_Read);
}

void BtFirmwareUpdater::WriteMemory(uint32_t address, uint32_t length, uint8_t* pBuffer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);

    StartMemoryControl(address, length, pBuffer, MemoryControlType_Write);
}

void BtFirmwareUpdater::EraseMemory(uint32_t address, uint32_t length) NN_NOEXCEPT
{
    StartMemoryControl(address, length, nullptr, MemoryControlType_Erase);
}

void BtFirmwareUpdater::StartMemoryControl(uint32_t address, uint32_t length, uint8_t* pBuffer, MemoryControlType type) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_EQUAL(m_MemoryControlParameters.type, MemoryControlType_None);

    m_MemoryControlParameters.type = type;
    m_MemoryControlParameters.address = address;
    m_MemoryControlParameters.length = length;
    m_MemoryControlParameters.index = 0;
    m_MemoryControlParameters.previousIndex = 0;
    m_MemoryControlParameters.pBuffer = pBuffer;

    switch (m_MemoryControlParameters.type)
    {
    case MemoryControlType_Read:
        m_MemoryControlParameters.unitLength = OtaReadSizeMax;
        // Read はまず SetupRead コマンドを送信する
        SetupReadImpl(m_MemoryControlParameters.address, static_cast<uint16_t>(m_MemoryControlParameters.length));
        break;

    case MemoryControlType_Write:
        m_MemoryControlParameters.unitLength = OtaWriteSizeMax;
        // 最初のメモリアクセスは絶対に成功する
        NN_ABORT_UNLESS_RESULT_SUCCESS(ProceedMemoryControl());
        break;

    case MemoryControlType_Erase:
        m_MemoryControlParameters.unitLength = OtaSerialFlashSectorSize;
        // 最初のメモリアクセスは絶対に成功する
        NN_ABORT_UNLESS_RESULT_SUCCESS(ProceedMemoryControl());
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

Result BtFirmwareUpdater::ProceedMemoryControl() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(m_MemoryControlParameters.type, MemoryControlType_None);

    // 処理を行う残りの長さ
    uint32_t remainingLength = m_MemoryControlParameters.length - m_MemoryControlParameters.index;

    // 最後のバイトまで処理が完了した
    if (remainingLength == 0)
    {
        m_MemoryControlParameters.type = MemoryControlType_None;
        NN_RESULT_THROW(ResultBtFirmwareUpdateMemoryControlCompleted());
    }

    // GetReport の最大長よりも大きい場合はクランプ
    else if (remainingLength > m_MemoryControlParameters.unitLength)
    {
        remainingLength = m_MemoryControlParameters.unitLength;
    }

    // 続きを処理
    switch (m_MemoryControlParameters.type)
    {
    case MemoryControlType_Read:
        ReadImpl();
        break;
    case MemoryControlType_Write:
        WriteImpl(m_MemoryControlParameters.address + m_MemoryControlParameters.index,
                  static_cast<uint16_t>(remainingLength),
                  &(m_MemoryControlParameters.pBuffer[m_MemoryControlParameters.index]));
        break;
    case MemoryControlType_Erase:
        EraseImpl(m_MemoryControlParameters.address + m_MemoryControlParameters.index,
                  static_cast<uint16_t>(remainingLength));
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    // Index を処理したサイズだけシフト
    m_MemoryControlParameters.previousIndex = m_MemoryControlParameters.index;
    m_MemoryControlParameters.index += remainingLength;

    NN_RESULT_SUCCESS;
}

void BtFirmwareUpdater::SetReportImpl(ReportType reportType, uint8_t* pBuffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);

    m_SetReportId = pBuffer[0];
    m_pHidAccessor->SetReport(reportType, pBuffer, size, this);
}

void BtFirmwareUpdater::EnableFwuImpl() NN_NOEXCEPT
{
    uint8_t reportId = Report_OtaEnableFwu::Id;

    SetReportImpl(Report_OtaEnableFwu::Type, &reportId, 1);
}

void BtFirmwareUpdater::SetupReadImpl(uint32_t address, uint16_t length) NN_NOEXCEPT
{
    uint8_t report[Report_OtaSetupRead::Size];

    SetupOtaReport(report,
                   Report_OtaSetupRead::Id,
                   address + OtaAddress_SerialFlashOffset,
                   length,
                   nullptr,
                   0);
    SetReportImpl(Report_OtaSetupRead::Type, report, sizeof(report));
}

void BtFirmwareUpdater::ReadImpl() NN_NOEXCEPT
{
    m_pHidAccessor->GetReport(Report_OtaRead::Type, Report_OtaRead::Id, this);
}

void BtFirmwareUpdater::EraseImpl(uint32_t address, uint16_t length) NN_NOEXCEPT
{
    uint8_t report[Report_OtaErase::Size];

    SetupOtaReport(report,
                   Report_OtaErase::Id,
                   address + OtaAddress_SerialFlashOffset,
                   length,
                   nullptr,
                   0);
    SetReportImpl(Report_OtaErase::Type, report, sizeof(report));
}

void BtFirmwareUpdater::WriteImpl(uint32_t address, uint16_t length, uint8_t* pBuffer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);

    uint8_t report[Report_OtaWrite::Size];

    SetupOtaReport(report,
                   Report_OtaWrite::Id,
                   address + OtaAddress_SerialFlashOffset,
                   length,
                   pBuffer,
                   length);
    SetReportImpl(Report_OtaWrite::Type, report, length + 8);
}

void BtFirmwareUpdater::LaunchImpl(uint32_t address) NN_NOEXCEPT
{
    uint8_t report[Report_OtaLaunch::Size];

    SetupOtaReport(report,
                   Report_OtaLaunch::Id,
                   address,
                   0,
                   nullptr,
                   0);
    SetReportImpl(Report_OtaSetupRead::Type, report, sizeof(report));
}

}} // namespace nn::xcd
