﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/os/os_SystemEventTypes.h>
#include <nn/os/os_TimerEventTypes.h>
#include <nn/xcd/xcd_BleTypes.h>
#include <nn/xcd/xcd_Input.h>
#include "xcd_BleReportTypes.h"
#include "xcd_IBleCommandListener.h"
#include "detail/xcd_BleGattClientTypes.h"
#include "detail/xcd_BleHidAccessor.h"
#include "detail/xcd_IBleNhogClient.h"
#include "xcd_PalmaCommandHandler.h"

namespace nn { namespace xcd {

const int MaxBleOperationResultCount = 4;

//!< Ble デバイスの Command Handler
class BleCommandHandler final : public IBleCommandListener
{
    NN_DISALLOW_COPY(BleCommandHandler);
    NN_DISALLOW_MOVE(BleCommandHandler);

private:
    //!< Activate されているかどうか
    bool m_IsActivated;

    //!< コマンドリスナー
    IBleCommandListener* m_pListener;

    //!< コマンドハンドラの種類
    enum CommandHandlerType
    {
        CommandHandlerType_NotSupported,
        CommandHandlerType_Palma,
    };
    CommandHandlerType m_CommandHandlerType;

    //!< Palma フォーマットのコマンドハンドラ
    PalmaCommandHandler m_PalmaCommandHandler;

    //!< デバイス操作の完了を通知するイベント
    nn::os::SystemEventType* m_pOperationCompletedEvent;

    //!< コマンドの応答と結果
    struct BleResponse
    {
        BleResponseStatus        status;
        BleDeviceOperationResult result;
    };

    //!< コマンドの応答と結果をためておく配列
    BleResponse m_ResponseArray[MaxBleOperationResultCount];

    //!< ResponseArray から確保しているコマンド応答の数
    int m_ResponseCount;

    //!< ResponseArray のインデックス
    int m_ResponseIndex;

    //!< 通知するコマンドの結果の数
    int m_CompletedCount;

    //!< Response アクセスを排他するためのミューテックス
    nn::os::SdkMutexType m_ResponseMutex;

    //!< xcd内の呼び出しかどうか
    bool m_IsSystem;

    //!< コマンドの応答のタイマー
    nn::os::TimerEventType m_ResponseTimer;

public:
    BleCommandHandler() NN_NOEXCEPT;
    virtual ~BleCommandHandler() NN_NOEXCEPT NN_OVERRIDE;

    //!< Ble OutputCommand の実行完了の通知を受け取るための関数です
    virtual void NotifyCommandCompleted(BleDeviceOperationType operationType, BleDeviceOperationResultType resultType) NN_NOEXCEPT NN_OVERRIDE;

    //!< Ble OutputCommand に対して ProductType の読み込みが完了した際の通知を受け取るための関数です
    virtual void NotifyReadProductType(const BleDeviceProductType& type) NN_NOEXCEPT NN_OVERRIDE;

    //!< Ble OutputCommand に対して ModelInformation の読み込みが完了した際の通知を受け取るための関数です
    virtual void NotifyReadModelInformation(const BleDeviceModelInformation& info) NN_NOEXCEPT NN_OVERRIDE;

    //!< Ble OutputCommand に対して SensorCalibration の読み込みが完了した際の通知を受け取るための関数です
    virtual void NotifyReadSensorCalibration(const SensorCalibrationValue& value) NN_NOEXCEPT NN_OVERRIDE;

    //!< Ble OutputCommand に対して AnalogStickCalibration の読み込みが完了した際の通知を受け取るための関数です
    virtual void NotifyReadAnalogStickCalibration(const AnalogStickValidRange& value) NN_NOEXCEPT NN_OVERRIDE;

    //!< Ble OutputCommand に対して ContentUniqueCode の読み込みが完了した際の通知を受け取るための関数です
    virtual void NotifyReadContentUniqueCode(BleDeviceOperationResultType resultType, const uint8_t* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE;

    //!< Ble OutputCommand に対して Step の読み込みが完了した際の通知を受け取るための関数です
    virtual void NotifyReadStep(BleDeviceOperationResultType resultType, uint32_t step) NN_NOEXCEPT NN_OVERRIDE;

    //!< Ble OutputCommand に対して ContentState の読み込みが完了した際の通知を受け取るための関数です
    virtual void NotifyReadContentState(BleDeviceOperationResultType resultType, bool isValid) NN_NOEXCEPT NN_OVERRIDE;

    //!< Ble OutputCommand に対してデータベース識別バージョンの読み込みが完了した際の通知を受け取るための関数です
    virtual void NotifyReadDatabaseVersion(BleDeviceOperationResultType resultType, int32_t version) NN_NOEXCEPT NN_OVERRIDE;

    //!< Ble OutputCommand に対して Application セクションからのデータ読み込みが完了した際の通知を受け取るための関数です
    virtual void NotifyReadApplicationSection(BleDeviceOperationResultType resultType, const uint8_t* buffer, int32_t size, int32_t address) NN_NOEXCEPT NN_OVERRIDE;

    //!< Ble OutputCommand に対して一時停止されている FeatureSet の機能を受信した際の通知を受け取るための関数です
    virtual void NotifyRespSuspendFeatureSet(BleDeviceOperationResultType resultType, uint32_t featureFlagSet) NN_NOEXCEPT NN_OVERRIDE;

    //!< Ble OutputCommand に対して PlayLog の読み込みが完了した際の通知を受け取るための関数です
    virtual void NotifyReadPlayLog(BleDeviceOperationResultType resultType, uint16_t index, const uint8_t* buffer, int32_t size) NN_NOEXCEPT NN_OVERRIDE;

    //!< CommandHandler を起動する
    void Activate(detail::BleHidAccessor* pAccessor, BleOutputCommandFormatVersion commandVer, BleCommandResponseFormatVersion responseVer) NN_NOEXCEPT;

    //!< CommandHandler を無効にする
    void Deactivate(detail::BleHidAccessor* pAccessor) NN_NOEXCEPT;

    //!< アクティベートされているかどうか
    bool IsActivated() NN_NOEXCEPT
    {
        return m_IsActivated;
    };

    //!< デバイス操作の完了を通知するイベントを登録する
    Result RegisterOperationEvent(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT;

    //!< コマンドのリスナーをセットする
    void SetListener(IBleCommandListener* pListener) NN_NOEXCEPT;

    //!< コマンドの実行結果を取得する
    Result GetOperationResult(BleDeviceOperationResult* pOutValue) NN_NOEXCEPT;

    //!< ProductType を取得するコマンドを発行する
    Result GetProductType() NN_NOEXCEPT;

    //!< ModelInformation を取得するコマンドを発行する
    Result GetModelInformation() NN_NOEXCEPT;

    //!< Sensor Cal を取得するコマンドを発行する
    Result GetSensorCalibrationValue() NN_NOEXCEPT;

    //!< AnalogStick Cal を取得するコマンドを発行する
    Result GetAnalogStickCalibrationValue() NN_NOEXCEPT;

    //!< Activity を再生するコマンドを発行する
    Result PlayActivity(uint16_t number) NN_NOEXCEPT;

    //!< Activity を再生するコマンドを発行する
    Result PlayActivitySystem(uint16_t number) NN_NOEXCEPT;

    //!< FrMode の動作種別を設定するコマンドを発行する
    Result SetFrModeType(uint32_t frModeType) NN_NOEXCEPT;

    //!< Database Entry の書き込みを行うコマンドを発行する
    Result WriteDatabaseEntry(const BleDeviceDatabaseEntryConfig& config, const uint8_t* buffer, size_t size) NN_NOEXCEPT;

    //!< データベース識別バージョンを設定するコマンドを発行する
    Result SetDatabaseIdentificationVersion(int32_t version) NN_NOEXCEPT;

    //!< データベース識別バージョンを取得するコマンドを発行する
    Result GetDatabaseIdentificationVersion() NN_NOEXCEPT;

    //!< StepCount を0で初期化するコマンドを発行する
    Result ResetStepCount() NN_NOEXCEPT;

    //!< StepCount を取得するコマンドを発行する
    Result GetStepCount() NN_NOEXCEPT;

    //!< Step 機能を設定するコマンドを発行する
    Result EnableStepCounter(bool isEnable) NN_NOEXCEPT;

    //!< Application セクションの読み出しを行うコマンドを発行する
    Result ReadApplicationSection(int32_t address, size_t readSize) NN_NOEXCEPT;

    //!< Application セクションへ書き込みを行うコマンドを発行する
    Result WriteApplicationSection(int32_t address, const uint8_t* buffer, size_t size) NN_NOEXCEPT;

    //!< UniqueCode を読み出すコマンドを発行する
    Result ReadContentUniqueCode() NN_NOEXCEPT;

    //!< UniqueCode を無効にするコマンドを発行する
    Result SetContentUniqueCodeInvalid() NN_NOEXCEPT;

    //!< FeatureSet の機能の一時停止を行う
    Result SuspendFeature(uint32_t featureFlagSet) NN_NOEXCEPT;

    //!< FeatureSet の機能の一時停止を行う
    Result SuspendFeatureSystem(uint32_t featureFlagSet) NN_NOEXCEPT;

    //!< PlayLog を取得するコマンドを発行する
    Result ReadPlayLog(uint16_t index) NN_NOEXCEPT;

    //!< PlayLog を0で初期化するコマンドを発行する
    Result ResetPlayLog(uint16_t index) NN_NOEXCEPT;

    //!< 定常的な処理を行う関数
    void PeriodicOperation() NN_NOEXCEPT;

    //!< Database Entry の書き込みを中断するコマンドを発行する
    void CancelWriteDatabaseEntry() NN_NOEXCEPT;

private:
    //!< CommandResponse を受信した際に呼ばれるハンドラ
    static void CommandResponseHandler(void* pParser, const uint8_t* buffer, size_t size) NN_NOEXCEPT;

    //!< CommandResponse をパースする
    void ParseCommnadResponse(const uint8_t* buffer, size_t size) NN_NOEXCEPT;

    //!< ResponseArray から空き状態の Response を取得する
    Result GetEmptyResponseFromArray(BleResponse** pOutPtr) NN_NOEXCEPT;

    //!< ResponseArray から目的の Response を探索する
    BleResponse* FindResponseFromArray(BleDeviceOperationType operation) NN_NOEXCEPT;

    //!< ResponseArray から目的の Response を探索する
    BleResponse* FindResponseFromArray(BleResponseStatus status) NN_NOEXCEPT;

    //!< ResponseArray から目的の Response を削除する
    void RemoveResponseFromArray(BleResponse* pResponse) NN_NOEXCEPT;
};

}} // namespace nn::xcd
