﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/os/os_LightEventTypes.h>
#include <nn/os/os_Mutex.h>
#include <nn/pinmux/pinmux.h>
#include <nn/xcd/xcd_Device.h>
#include <nn/xcd/xcd_Sleep.h>
#include <nn/xcd/xcd_Usb.h>
#include <nn/xcd/xcd_KuinaFirmware.h>

#include "../xcd_ReportTypes.h"
#include "../xcd_IEventTask.h"
#include "../xcd_MultiWaitEvent.h"
#include "xcd_IHidListener.h"

namespace nn { namespace xcd { namespace detail{

//!< コントローラーとの UsbHid 通信を扱うドライバ
class UsbHidDriver final : public IEventTask
{
    NN_DISALLOW_MOVE(UsbHidDriver);
    NN_DISALLOW_COPY(UsbHidDriver);

private:
    //!< ドライバ内の排他制御用の mutex
    nn::os::Mutex m_Mutex;

    //!< デバイスの接続完了した際に通知するためのイベント
    nn::os::LightEventType* m_pConnectionEvent;

    //!< ドライバーがアクティベートされているかどうか
    bool m_Activated;

    //!< USB デバイスがアタッチされているかどうか
    bool m_IsAttached;

    //!< Nx コントローラーがアタッチされているかどうか
    bool m_IsNxDeviceAttached;

    //!< Nx コントローラーが Activate 中かどうか
    bool m_IsNxDeviceActivating;

    //!< USB の先の NX コントローラーが有効かどうか
    bool m_IsNxDeviceAvailable;

    //!< Nx コントローラーの USB 通信が有効化された回数
    int m_NxHidProtocolActivationCount;

    //!< 接続されたコントローラーの Bluetooth アドレス
    ::nn::bluetooth::Address m_Address;

    //!< 接続されたコントローラーのデバイスタイプ
    DeviceType m_DeviceType;

    //!< 受信済み InputReport の長さ
    uint16_t m_HidInputReportLength;

    //!< 受信済み InputReport
    uint8_t m_HidInputReportBuffer[UsbHidReportLengthMax];

    //!< パケットが受信されたかどうか
    bool m_IsPacketReceived;

    //!< 送信待ち OutputReport 用リングバッファのサイズ
    static const size_t OutputReportRingBufferCountMax = 32;

    struct HidOutputReportEntity
    {
        size_t size;
        uint8_t buffer[UsbHidReportLengthMax];
    };

    //!< 送信待ち OutputReport 用リングバッファ
    HidOutputReportEntity m_HidOutputReportBuffer[OutputReportRingBufferCountMax];

    //!< 送信待ち OutputReport 用リングバッファの書き込み位置
    uint8_t m_HidOutputReportBufferIndex;

    //!< 送信待ち OutputReport 用リングバッファ内の数
    uint8_t m_HidOutputReportBufferCount;

    //!< ExtGripOut のタイムアウトイベント
    ::nn::os::TimerEventType m_ExtGripOutTimeoutEvent;

    //!< ExtGripOut のビジー状態
    bool m_IsExtGripOutBusy;

    //!< 実行中の ExtGripStatus
    ExtGripStatus m_ProcessingExtGripStatus;

    //!< InputReport を解析する関数
    InputReportParserFunc m_pInputReportParserFunc;

    //!< InputReport 解析関数に渡す引数
    void* m_pInputReportParserArg;

    //!< USB(PID,VID)の情報
    UsbHidDeviceInfo m_UsbHidDeviceInfo;

    //!< Kuina Fw のバージョンが取得済みかどうか
    bool m_IsAvailableKuinaFirmwareVersion;

    //!< Kuina Fw のバージョン情報
    KuinaVersionData m_KuinaFirmwareVersion;

public:
    UsbHidDriver() NN_NOEXCEPT;
    ~UsbHidDriver() NN_NOEXCEPT;

    //!< Eventが通知された際に呼ばれる関数
    virtual void EventFunction(const ::nn::os::MultiWaitHolderType* pMultiWaitHolder) NN_NOEXCEPT NN_OVERRIDE;

    //!< 定常的な処理を行う関数
    virtual void PeriodicEventFunction() NN_NOEXCEPT NN_OVERRIDE;

    //!< デバイスの接続状態の監視を開始する
    void StartMonitoring(nn::os::LightEventType* pDeviceUpdateEvent) NN_NOEXCEPT;

    //!< デバイスの接続状態の監視を停止する
    void StopMonitoring() NN_NOEXCEPT;

    //!< サンプル受信時に解析する処理を登録する
    void SetSampleParserFunction(InputReportParserFunc func, void* pArg) NN_NOEXCEPT;

    //!< 指定された Usb デバイスをサポートしているか評価します
    static bool IsUsbHidSupported(UsbHidDeviceInfo deviceInfo) NN_NOEXCEPT;

    //!< USB デバイスを追加します
    Result AddDevice(UsbHidDeviceInfo deviceInfo) NN_NOEXCEPT;

    //!< USB デバイスを削除します
    Result RemoveDevice() NN_NOEXCEPT;

    //!< Nx コントローラーの情報を取得します
    Result GetDeviceInfo(::nn::bluetooth::Address* pOutAddress, DeviceType* pOutDeviceType) NN_NOEXCEPT;

    //!< NX コントローラーの Hid 通信を有効にします
    void ActivateUsbHidOnController() NN_NOEXCEPT;

    //!< NX コントローラーの Hid 通信を無効にします
    void DeactivateUsbHidOnController() NN_NOEXCEPT;

    //!< NX コントローラーの Hid 通信が有効かどうか
    bool IsNxDeviceHidActivated() NN_NOEXCEPT;

    //!< NX コントローラーの Hid プロトコルが有効化された回数を取得する
    int GetNxHidProtocolActivationCount() NN_NOEXCEPT;

    //!< NX コントローラーの Hid プロトコルが有効化された回数をリセットする
    void ResetNxHidProtocolActivationCount() NN_NOEXCEPT;

    //!< Input Report を処理します
    Result SetInputReport(uint8_t *pBuffer, size_t length) NN_NOEXCEPT;

    //!< Input Report を取得します
    size_t GetInputReport(uint8_t *pOutBuffer, size_t length) NN_NOEXCEPT;

    //!< Output Report を登録します
    Result SetOutputReport(const uint8_t *pOutBuffer, size_t length) NN_NOEXCEPT;

    //!< Output Report を取得します
    size_t GetOutputReport(uint8_t *pOutBuffer, size_t length) NN_NOEXCEPT;

    //!< 通信間隔を取得します
    ::nn::TimeSpan GetInterval() NN_NOEXCEPT;

    //!< Kuina の Fw バージョンを取得します
    Result GetKuinaVersion(KuinaVersionData* pOutMcuVersionData) NN_NOEXCEPT;

    //!< Kuina の Fw バージョンを要求します
    Result RequestKuinaVersion() NN_NOEXCEPT;

    //!< Kuina の Fw 更新モードに遷移させます
    Result SetKuinaToFirmwareUpdateMode() NN_NOEXCEPT;

    // リトライ回数をカウントする
    int m_RetryCount;

private:
    //!< Input Report を処理します
    Result SetInputReportImpl(uint8_t *pBuffer, size_t length) NN_NOEXCEPT;

    //!< Output Report を登録します
    Result SetOutputReportImpl(const uint8_t *pOutBuffer, size_t length) NN_NOEXCEPT;

    //!< CreateHidConnection コマンドを送ります
    void SendCreateHidConnection() NN_NOEXCEPT;

    //!< DeleteHidConnection コマンドを送ります
    void SendDeleteHidConnection() NN_NOEXCEPT;

    //!< StartHidData コマンドを送ります
    void SendStartHidData() NN_NOEXCEPT;

    //!< StopHidData コマンドを送ります
    void SendStopHidData() NN_NOEXCEPT;

    //!< FwVersion 取得コマンドを送ります
    void SendGetFwVersion();

    //!< Inquiry コマンドを送ります
    void SendInquiry() NN_NOEXCEPT;

    //!< Fw 更新モードへの遷移コマンドを送ります
    void SendGotoFwUpdateMode() NN_NOEXCEPT;

    //!< Reboot コマンドを送ります
    void SendReboot() NN_NOEXCEPT;

    //<! Dummy の Hid Output Report をセットする
    void SetDummyHidOutputReport() NN_NOEXCEPT;

    //!< ジョイコンとの USB 通信が切断された際の処理を行います
    void HandleNxDeviceDetach() NN_NOEXCEPT;

    //!< Output 送信バッファを初期化する
    void ResetOutputBuffer() NN_NOEXCEPT;

    //!< Data Format を初期化する
    void ResetDataFormat() NN_NOEXCEPT;
};

}}} // namespace nn::xcd::detail
