﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <mutex>
#include <nn/nn_Macro.h>  // type_traits 対応判定のため、先に必要
#if defined(NN_BUILD_CONFIG_COMPILER_SUPPORTS_STD_TYPE_TRAITS)
    #include <type_traits>
#endif
#include <nn/nn_Abort.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_StaticAssert.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/xcd/xcd_Tera.h>
#include <nn/xcd/xcd_TeraFirmware.h>
#include <nn/xcd/xcd_Result.h>
#include <nn/xcd/detail/xcd_Log.h>
#include "../xcd_ReportTypes.h"
#include "xcd_TeraUpdater.h"
#include "xcd_TeraCommon.h"

#if defined(NN_BUILD_CONFIG_COMPILER_SUPPORTS_STD_TYPE_TRAITS)

// 構造体の POD 判定
NN_STATIC_ASSERT(std::is_pod<nn::xcd::McuVersionData>::value);
NN_STATIC_ASSERT(std::is_pod<nn::xcd::McuUpdateStateInfo>::value);
NN_STATIC_ASSERT(std::is_pod<nn::xcd::detail::McuFirmwareImageInfo>::value);
NN_STATIC_ASSERT(std::is_pod<nn::xcd::detail::McuUpdateInPayload>::value);
NN_STATIC_ASSERT(std::is_pod<nn::xcd::detail::McuUpdateOutPayload>::value);

#endif  // defined(NN_BUILD_CONFIG_COMPILER_SUPPORTS_STD_TYPE_TRAITS)

NN_STATIC_ASSERT(sizeof(nn::xcd::detail::McuFirmwareImageInfo) == nn::xcd::detail::McuFirmwareImageHeaderSize);

namespace
{

// 起動処理中の進捗率
const uint16_t ProgressRateBoot = 0;

// 起動フラグ無効化時の進捗率
const uint16_t ProgressRateInvalidate = 1;

// ROM 消去開始時の進捗率
const uint16_t ProgressRateErase = 2;

// 書き込み開始時の進捗率
const uint16_t ProgressRateWriting = 17;

// CRC、ブロック数を含めて送出完了したときの進捗率
const uint16_t ProgressRateSendFinish = 98;

// 消去中に変動する進捗率
const uint16_t ProgressRateRangeForErase = ProgressRateWriting - ProgressRateErase;

// 書き込み中に変動する進捗率
const uint16_t ProgressRateRangeForWrite = ProgressRateSendFinish - ProgressRateWriting;

// すべての書き込みが終了したときの進捗率
const uint16_t ProgressRateWriteFinish = 99;

// シーケンス完了時の進捗率
const uint16_t ProgressRateComplete = 100;

// キューが空いていると見なせるコマンド送出数と完了数の差分
//  ※ キューサイズは 7。1 レポートで最大 3 コマンドを送出する想定。
const int QueueFreeThreshold = 4;

// 再送に失敗したと見なすまでの Output report 送信回数
const int ResendTimeoutCount = 10;

// 消去中の進捗表示に使用する消去予想時間
const auto StandardEraseTime = nn::TimeSpan::FromSeconds(10);

// コマンド発行から更新処理自体のタイムアウトと見なすまでの時間
const auto SequenceTimeout = nn::TimeSpan::FromSeconds(30);

// コマンドの送信失敗と見なすまでの時間
const auto CommandTimeout = nn::TimeSpan::FromSeconds(3);

}  // anonymous

// デバッグ設定の初期値は All OFF
nn::xcd::ITeraUpdater::DebugOptionSet nn::xcd::ITeraUpdater::g_DebugOptions =
    nn::xcd::ITeraUpdater::DebugOptionSet();

namespace nn { namespace xcd { namespace detail {

void TeraUpdater::Activate(DeviceType deviceType, PhaseChangeCallbackType callback, void* pCallbackArgument) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(callback);
    NN_SDK_REQUIRES_NOT_NULL(pCallbackArgument);

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    if (m_IsActive)
    {
        return;
    }

    ClearInternalState();
    m_Phase               = UpdatePhase::BootPrepare;
    m_Type                = deviceType;
    m_PhaseChangeCallback = callback;
    m_pCallbackArgument   = pCallbackArgument;
    m_ProgressRate        = 0;
    m_IsActive            = true;
}

void TeraUpdater::Deactivate() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    if (!m_IsActive)
    {
        return;
    }

    ClearInternalState();
    m_IsActive = false;
}

void TeraUpdater::ClearInternalState() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    // 完了後に進捗確認を行えるように、進捗状況は初期化しない
    m_Phase               = UpdatePhase::Stopped;
    m_Type                = DeviceType_Unknown;
    m_PhaseChangeCallback = nullptr;
    m_pCallbackArgument   = nullptr;

    ClearFirmwareUpdateInfo();
}

void TeraUpdater::ClearFirmwareUpdateInfo() NN_NOEXCEPT
{
    m_ReceivedInput.Clear();
    m_OutputData.Clear();
    m_CommandBuilder.Clear();
    for (auto& data : m_SentOutputData)
    {
        data.Clear();
    }
    m_TotalCommandCount     = 0;
    m_RestartSequenceNumber = 0;
    m_ResendWaitCount       = 0;
    m_EraseStartTick        = nn::os::Tick();
    m_LastCommandTick       = nn::os::Tick();
    m_TimeoutCounter.Stop();
}

nn::Result TeraUpdater::SetFirmwareImage(
    const FirmwareImage& image,
    UpdateMode mode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(mode == UpdateMode::Normal || mode == UpdateMode::Full);

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_SDK_REQUIRES(m_IsActive);

    NN_RESULT_DO(m_Stream.Setup(image));

    // イメージの正当性チェック
    McuFirmwareImageInfo imageInfo;
    m_Stream.GetImageInfo(&imageInfo);

    switch (m_Type)
    {
    case DeviceType_Right:
        {
            NN_RESULT_THROW_UNLESS(
                imageInfo.type == McuFirmwareImageType_JoyRight,
                nn::xcd::ResultMcuInvalidFirmwareImage());
        }
        break;
    case DeviceType_FullKey:
        {
            NN_RESULT_THROW_UNLESS(
                imageInfo.type == McuFirmwareImageType_FullKey,
                nn::xcd::ResultMcuInvalidFirmwareImage());
        }
        break;
    default:
        NN_RESULT_THROW(nn::xcd::ResultMcuInvalidFirmwareImage());
    }

    // 予約領域は All 0x00
    NN_RESULT_THROW_UNLESS(
        detail::IsAllZero(imageInfo._reserve, sizeof(imageInfo._reserve)),
        nn::xcd::ResultMcuInvalidFirmwareImage());

    m_Mode  = mode;
    m_Crc32 = detail::Crc32InitialValue;

    NN_RESULT_SUCCESS;
}

void TeraUpdater::GetRomFooter(RomFooter* pOutRomFooter) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutRomFooter);

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    // IAP 以外を更新ブロック数の計算対象とする
    uint32_t skipBytes;
    switch (m_Mode)
    {
    case UpdateMode::Normal:
        {
            // NFC Chip FW から書き込むので全域が対象
            skipBytes = 0;
        }
        break;
    case UpdateMode::Full:
        {
            // IAP 分のサイズをスキップ
            skipBytes = AddressOffsetNfcChipFirmware - AddressOffsetIap;
            NN_SDK_REQUIRES_GREATER(m_Stream.GetImageSize(), skipBytes);
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    // 更新ブロック数は -1 した値を使用
    auto customerCodeSize = m_Stream.GetImageSize() - skipBytes;
    auto blockNumber = (customerCodeSize + McuUpdateDataSizeMax - 1) / McuUpdateDataSizeMax;

    NN_XCD_TERA_UPDATER_LOG_VERBOSE("CRC32 = %08X\n", m_Crc32);

    // 更新失敗エミュレーション時は CRC を反転して Verify を失敗させる
    nn::Bit32 crc32 =
        g_DebugOptions.Test<DebugOption::UpdateFailureEmulation>() ?
        ~m_Crc32 :
        m_Crc32;

    // CRC32 と更新ブロック数はリトルエンディアンで格納
    pOutRomFooter->value[0] = static_cast<uint8_t>(crc32 & 0xFF);
    pOutRomFooter->value[1] = static_cast<uint8_t>((crc32 >> 8) & 0xFF);
    pOutRomFooter->value[2] = static_cast<uint8_t>((crc32 >> 16) & 0xFF);
    pOutRomFooter->value[3] = static_cast<uint8_t>((crc32 >> 24) & 0xFF);

    pOutRomFooter->value[4] = static_cast<uint8_t>(blockNumber & 0xFF);
    pOutRomFooter->value[5] = static_cast<uint8_t>((blockNumber >> 8) & 0xFF);
}

nn::Result TeraUpdater::Abort() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_SDK_REQUIRES(m_IsActive);

    // アップデート動作中なら強制的にエラーにする
    if (m_Phase != UpdatePhase::Stopped &&
        m_Phase != UpdatePhase::BootPrepare &&
        m_Phase != UpdatePhase::Error)
    {
        SetPhase(UpdatePhase::Error);
        NotifyPhaseChanged();
    }

    NN_RESULT_SUCCESS;
}

bool TeraUpdater::IsAborted() const NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    return m_Phase == UpdatePhase::Error;
}

bool TeraUpdater::IsReadyToSend() const NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    return m_TotalCommandCount - m_ReceivedInput.GetCompleteCount() <= QueueFreeThreshold;
}

void TeraUpdater::SetPhase(UpdatePhase phase) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    m_Phase = phase;
}

void TeraUpdater::NotifyPhaseChanged() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_SDK_REQUIRES_NOT_NULL(m_PhaseChangeCallback);

    m_PhaseChangeCallback(m_Phase, m_pCallbackArgument);
}

nn::Result TeraUpdater::GetState(McuUpdateStateInfo* pOutStateInfo) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutStateInfo);

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    pOutStateInfo->progress = static_cast<uint16_t>(m_ProgressRate);

    if (!m_IsActive)
    {
        // 開始していないときは常に終了状態
        pOutStateInfo->state = McuUpdateState_End;
        NN_RESULT_SUCCESS;
    }

    // アップデートの進捗に応じてステートを決める
    switch (m_Phase)
    {
    case UpdatePhase::BootPrepare:
    case UpdatePhase::Boot:
    case UpdatePhase::BootFinish:
        {
            pOutStateInfo->state = McuUpdateState_Boot;
        }
        break;
    case UpdatePhase::Invalidate:    // 起動フラグ無効化は消去扱い
    case UpdatePhase::Erase:
    case UpdatePhase::Erasing:
    case UpdatePhase::EraseFinish:
        {
            pOutStateInfo->state = McuUpdateState_RomErasing;
        }
        break;
    case UpdatePhase::Write:
    case UpdatePhase::Finalize:
    case UpdatePhase::Finalizing:
        {
            pOutStateInfo->state = McuUpdateState_Writing;
        }
        break;
    case UpdatePhase::Stopped:
        {
            pOutStateInfo->state = McuUpdateState_End;
        }
        break;
    case UpdatePhase::FinalizeFinish:
    case UpdatePhase::Error:
    default:
        {
            // 完了もしくは不正終了なのでリブートする
            // (更新の成否はリブート後に確認するため、この段階での通知は不要)
            pOutStateInfo->state = McuUpdateState_Reboot;
        }
        break;
    }

    NN_RESULT_SUCCESS;
}

void TeraUpdater::ParseMcuUpdateInputReport(const uint8_t* pBuffer, size_t size, uint8_t sampleNumber) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);
    NN_UNUSED(sampleNumber);

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    if (!m_IsActive)
    {
        return;
    }

    std::memcpy(&m_ReceivedInput, pBuffer, std::min(size, sizeof(m_ReceivedInput)));

    if (m_ReceivedInput.GetEventType() != McuUpdateEventType_None)
    {
        NN_XCD_TERA_UPDATER_LOG_VERBOSE("Event: %d\n", m_ReceivedInput.GetEventType());
    }

    auto previousPhase = m_Phase;
    ProcessMcuUpdateInput();

    if (previousPhase != m_Phase)
    {
        NotifyPhaseChanged();
    }
}

size_t TeraUpdater::GetMcuUpdateOutputReport(uint8_t* pOutValue, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    if (!m_IsActive ||
        IsAborted() ||
        m_ReceivedInput.HasError())
    {
        return 0;
    }

    // 再送判定
    size_t resendPayloadSize;
    if (ProcessMcuUpdateOutputResend(&resendPayloadSize, pOutValue, size))
    {
        m_LastCommandTick = nn::os::GetSystemTick();
        return resendPayloadSize;
    }

    // コマンドを規定数以上は送らない
    if (!IsReadyToSend())
    {
        return 0;
    }

    if (!m_OutputData.HasData())
    {
        return 0;
    }

    auto payloadSize = std::min(m_OutputData.payloadSize, size);
    std::memcpy(pOutValue, m_OutputData.payload, payloadSize);

#if 0
    NN_SDK_LOG("<Data: %d bytes>\n  ", (int)payloadSize);
    for (size_t i = 0; i < payloadSize; i++)
    {
        NN_SDK_LOG("%02X ", m_OutputData.payload[i]);
        if ((i + 1) % 16 == 0)
        {
            NN_SDK_LOG("\n  ");
        }
    }
    NN_SDK_LOG("\n");
#endif

    m_TotalCommandCount += m_OutputData.commandCount;
    m_LastCommandTick = nn::os::GetSystemTick();

    // 再送用に送信済みパケットを記録
    for (int i = NN_ARRAY_SIZE(m_SentOutputData) - 1; i > 0; i--)
    {
        m_SentOutputData[i] = m_SentOutputData[i - 1];
    }
    m_SentOutputData[0] = m_OutputData;

    m_OutputData.Clear();

    NN_XCD_TERA_UPDATER_LOG_VERBOSE("Total commands: %d\n", m_TotalCommandCount);

    return payloadSize;
}

void TeraUpdater::ProcessMcuUpdateInput() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    if (IsAborted())
    {
        // 中断された場合は何もしない
        return;
    }
    else if (m_Phase == UpdatePhase::BootPrepare)
    {
        // 準備中は何もしない
        return;
    }
    else if (m_ReceivedInput.HasError() || m_TimeoutCounter.IsExpired())
    {
        NN_XCD_TERA_UPDATER_LOG_VERBOSE("%s occurred\n",
            m_ReceivedInput.HasError() ? "Error" : "Timeout");
        NN_XCD_TERA_UPDATER_LOG_VERBOSE("  Total sent command: %d\n", m_TotalCommandCount);
        NN_XCD_TERA_UPDATER_LOG_VERBOSE("  Complete command  : %d\n", m_ReceivedInput.GetCompleteCount());
        NN_XCD_TERA_UPDATER_LOG_VERBOSE("  Receive command   : %d\n", m_ReceivedInput.GetReceiveCount());
        NN_XCD_TERA_UPDATER_LOG_VERBOSE("  Total sequence num: %d\n", m_CommandBuilder.GetSequenceNumber());
        NN_XCD_TERA_UPDATER_LOG_VERBOSE("  Last sequence num : %d\n", m_ReceivedInput.GetLastSequenceNumber());
        NN_XCD_TERA_UPDATER_LOG_VERBOSE("  Flags             : %02X\n", m_ReceivedInput._flags.storage);
        SetPhase(UpdatePhase::Error);
        m_TimeoutCounter.Stop();
        NotifyPhaseChanged();
        return;
    }
    else if (!IsReadyToSend())
    {
        // キューが埋まっているときは何も送らない
        return;
    }
    else if (m_OutputData.HasData())
    {
        // 送信待ちデータがあるので何もしない
        return;
    }

    Proceed();
}

bool TeraUpdater::ProcessMcuUpdateOutputResend(
    size_t* pOutPayloadSize,
    uint8_t* pOutPayload,
    size_t payloadSizeMax) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutPayloadSize);
    NN_SDK_REQUIRES_NOT_NULL(pOutPayload);

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    *pOutPayloadSize = 0;

    if (m_Phase != UpdatePhase::Erasing)
    {
        auto diffTime = (nn::os::GetSystemTick() - m_LastCommandTick).ToTimeSpan();
        if (!IsReadyToSend() && diffTime > CommandTimeout)
        {
            m_RestartSequenceNumber = m_ReceivedInput.GetLastSequenceNumber() + 1;
            m_ResendWaitCount       = ResendTimeoutCount;
            NN_XCD_TERA_UPDATER_LOG_VERBOSE("Last command may be timed out. Resending\n");
        }
    }

    // 再送が必要な状態か
    if (m_ReceivedInput.IsReportLost() &&
        m_RestartSequenceNumber == 0)
    {
        // 再送バッファを超えて乖離しないはずなので、発生した場合は中断
        auto diffNumber = m_CommandBuilder.GetSequenceNumber() - m_ReceivedInput.GetLastSequenceNumber();
        if (diffNumber >= NN_ARRAY_SIZE(m_SentOutputData))
        {
            NN_XCD_TERA_UPDATER_LOG_VERBOSE("Sequence number diff is out of range (%d)\n", diffNumber);
            Abort();
            return true;
        }

        // 未到達パケットから再送開始
        m_RestartSequenceNumber = m_ReceivedInput.GetLastSequenceNumber() + 1;
        m_ResendWaitCount       = ResendTimeoutCount;
    }

    if (m_RestartSequenceNumber <= 0)
    {
        // 再送不要
        return false;
    }

    for (auto& output : m_SentOutputData)
    {
        // 再送すべきパケットを探す
        if (output.sequenceNumber == m_RestartSequenceNumber)
        {
            auto payloadSize = std::min(output.payloadSize, payloadSizeMax);
            std::memcpy(pOutPayload, output.payload, payloadSize);
            NN_XCD_TERA_UPDATER_LOG_VERBOSE(
                "Resend sequence: %d\n",
                m_RestartSequenceNumber);
            m_RestartSequenceNumber++;

            // 再送ではタイムアウトの再カウントはせず、規定時間再送が続く場合は失敗にする

            *pOutPayloadSize = payloadSize;
            return true;
        }
    }

    if (m_ReceivedInput.GetLastSequenceNumber() ==  m_RestartSequenceNumber - 1)
    {
        // ロスト分のパケットを送出し終わったら再送終了
        NN_XCD_TERA_UPDATER_LOG_VERBOSE("Resend finished\n");
        m_RestartSequenceNumber = 0;
    }
    else if (m_ResendWaitCount <= 0)
    {
        // 再送が更にパケロスした可能性があるので、未達パケットからやり直す
        m_RestartSequenceNumber = m_ReceivedInput.GetLastSequenceNumber() + 1;
        m_ResendWaitCount       = ResendTimeoutCount;
        return true;
    }
    else
    {
        // 再送完了待ち
        m_ResendWaitCount--;
        return true;
    }

    return false;
}

void TeraUpdater::Proceed() NN_NOEXCEPT
{
    switch (m_Phase)
    {
    case UpdatePhase::Boot:
        {
            ProceedBootPhase();
        }
        break;
    case UpdatePhase::Invalidate:
        {
            ProceedInvalidatePhase();
        }
        break;
    case UpdatePhase::Erase:
    case UpdatePhase::Erasing:
        {
            ProceedErasePhase();
        }
        break;
    case UpdatePhase::Write:
        {
            ProceedWritePhase();
        }
        break;
    case UpdatePhase::Finalize:
    case UpdatePhase::Finalizing:
        {
            ProceedFinalizePhase();
        }
        break;
    default:
        // ここに入るときはコマンド完了状態
        m_TimeoutCounter.Stop();
        return;
    }

    FlushCommandList();
}

void TeraUpdater::ProceedBootPhase() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_XCD_TERA_UPDATER_LOG_VERBOSE("%s %d\n", NN_CURRENT_FUNCTION_NAME, m_Phase);

    {
        // SYNC コマンドの送信
        uint8_t sendData[] = { SofByte };
        m_CommandBuilder.Append(sendData, sizeof(sendData));
    }

    SetPhase(UpdatePhase::BootFinish);
}

void TeraUpdater::ProceedInvalidatePhase() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_XCD_TERA_UPDATER_LOG_VERBOSE("%s %d\n", NN_CURRENT_FUNCTION_NAME, m_Phase);

    {
        // 書き込みコマンドの送信
        uint8_t sendData[] = { SofByte, WriteCommand, WriteCommand ^ 0xFF };
        m_CommandBuilder.Append(sendData, sizeof(sendData));
    }

    {
        // 起動フラグアドレスの送信
        auto startAddress = FlashRomStartAddress + AddressOffsetActivationFlag;
        uint8_t sendData[5];
        SetBigEndianValue(sendData, sizeof(uint32_t), startAddress);
        sendData[sizeof(sendData) - 1] = CalcCheckSum(sendData, sizeof(sendData) - 1);
        m_CommandBuilder.Append(sendData, sizeof(sendData));
    }

    {
        // 起動フラグ消去データの送信
        uint8_t sendData[ActivationFlagSize + 2] = {};
        sendData[0] = static_cast<uint8_t>(ActivationFlagSize - 1);
        sendData[sizeof(sendData) - 1] = CalcCheckSum(sendData, sizeof(sendData) - 1);
        m_CommandBuilder.Append(sendData, sizeof(sendData));
    }

    SetPhase(UpdatePhase::Erase);
}

void TeraUpdater::ProceedErasePhase() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_XCD_TERA_UPDATER_LOG_VERBOSE("%s %d\n", NN_CURRENT_FUNCTION_NAME, m_Phase);

    if (m_Phase == UpdatePhase::Erasing)
    {
        // 消去経過時間に応じて進捗率を進める
        // 規定時間以上かかる場合は、上限で一旦進捗が止まったように見える
        auto diffTime = (nn::os::GetSystemTick() - m_EraseStartTick).ToTimeSpan();
        uint16_t rate = static_cast<uint16_t>(
            ProgressRateRangeForErase * diffTime.GetMilliSeconds() / StandardEraseTime.GetMilliSeconds());
        m_ProgressRate =
            ProgressRateErase + std::min(rate, ProgressRateRangeForErase);

        // 送出済みの消去コマンドまで全て処理されたら消去完了
        if (m_ReceivedInput.GetCompleteCount() == m_TotalCommandCount)
        {
            SetPhase(UpdatePhase::EraseFinish);
        }
        else if (m_ReceivedInput.GetCompleteCount() > m_TotalCommandCount)
        {
            // 送出済みコマンド数以上に処理されることはないので異常
            Abort();
        }
        return;
    }

    // 消去対象セクタの情報
    const int* EraseSectors;
    int eraseSectorCount;
    switch (m_Mode)
    {
    case UpdateMode::Normal:
        {
            EraseSectors = EraseSectorsCustomer;
            eraseSectorCount = sizeof(EraseSectorsCustomer) / sizeof(EraseSectorsCustomer[0]);
        }
        break;
    case UpdateMode::Full:
        {
            EraseSectors = EraseSectorsFull;
            eraseSectorCount = sizeof(EraseSectorsFull) / sizeof(EraseSectorsFull[0]);
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    {
        // 消去コマンドの送信
        uint8_t sendData[] = { SofByte, EraseCommand, EraseCommand ^ 0xFF };
        m_CommandBuilder.Append(sendData, sizeof(sendData));

    }

    {
        // 送信するセクタ数は -1 する
        int sectorCountForSend = eraseSectorCount - 1;

        // 消去セクタ数の送信
        uint8_t sendData[3];
        SetBigEndianValue(sendData, 2, sectorCountForSend);
        sendData[2] = CalcCheckSum(sendData, sizeof(sendData) - 1);
        m_CommandBuilder.Append(sendData, sizeof(sendData));
    }

    {
        // 消去セクタ番号の送信
        size_t sendDataSize = eraseSectorCount * 2 + 1;
        uint8_t sendData[EraseSectorCountMax * 2 + 1];
        for (int i = 0; i < eraseSectorCount; i++)
        {
            // 各セクタ番号は 16bit Big endian で格納
            SetBigEndianValue(sendData + i * 2, 2, EraseSectors[i]);
        }
        sendData[sendDataSize - 1] = CalcCheckSum(sendData, sendDataSize - 1);
        m_CommandBuilder.Append(sendData, sendDataSize);
    }

    m_ProgressRate = ProgressRateErase;
    SetPhase(UpdatePhase::Erasing);
}

void TeraUpdater::ProceedWritePhase() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_XCD_TERA_UPDATER_LOG_VERBOSE("%s %d\n", NN_CURRENT_FUNCTION_NAME, m_Phase);

    // 書き込み先アドレスの送信
    auto startAddress = FlashRomStartAddress;
    switch (m_Mode)
    {
    case UpdateMode::Normal:
        {
            startAddress += AddressOffsetNfcChipFirmware;
        }
        break;
    case UpdateMode::Full:
        {
            startAddress += AddressOffsetIap;
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    auto nextAddress = static_cast<uint32_t>(startAddress + m_Stream.GetTotalReadBytes());

    {
        // 書き込みコマンドの送信
        uint8_t sendData[] = { SofByte, WriteCommand, WriteCommand ^ 0xFF };
        m_CommandBuilder.Append(sendData, sizeof(sendData));
    }

    {
        // 書き込みアドレスの送信
        uint8_t sendData[5];
        SetBigEndianValue(sendData, sizeof(uint32_t), nextAddress);
        sendData[sizeof(sendData) - 1] = CalcCheckSum(sendData, sizeof(sendData) - 1);
        m_CommandBuilder.Append(sendData, sizeof(sendData));
    }

    {
        // 書き込むデータの送信
        uint8_t sendData[McuUpdateDataSizeMax + 2];
        size_t dataBytes;
        auto result = FetchNextWriteData(&dataBytes, sendData, sizeof(sendData), nextAddress);
        if (result.IsFailure())
        {
            // FW イメージ読み込み中にエラー
            Abort();
            return;
        }
    }

    m_ProgressRate =
        ProgressRateWriting +
        std::min<uint16_t>(
            static_cast<uint16_t>(m_Stream.GetTotalReadBytes() * ProgressRateRangeForWrite / m_Stream.GetImageSize()),
            ProgressRateRangeForWrite);

    if (m_Stream.IsEof())
    {
        // 全データの書き込みが完了したのでファイナライズへ
        SetPhase(UpdatePhase::Finalize);
    }
}

nn::Result TeraUpdater::FetchNextWriteData(
    size_t* pOutBytes,
    uint8_t* pOutBuffer,
    size_t bufferSize,
    uint32_t nextAddress) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutBytes);
    NN_SDK_REQUIRES_NOT_NULL(pOutBuffer);

    // 実装ミス以外で引っかかることはないが、これを満たさないと配列の範囲外アクセスになるので abort
    NN_ABORT_UNLESS_GREATER_EQUAL(bufferSize, McuUpdateDataSizeMax + 2);

    size_t dataBytes;
    m_Stream.Read(&pOutBuffer[1], &dataBytes, McuUpdateDataSizeMax);

    *pOutBytes = dataBytes;
    if (dataBytes == 0)
    {
        NN_RESULT_SUCCESS;
    }

    // 先頭バイトは有効なデータサイズ - 1
    pOutBuffer[0] = static_cast<uint8_t>(dataBytes - 1);

    // 送信サイズに満たない分は、CRC32 計算のためにパディングデータで埋める
    auto remainSize = McuUpdateDataSizeMax - dataBytes;
    if (remainSize > 0)
    {
        std::memset(&pOutBuffer[dataBytes + 1], PaddingByte, remainSize);
    }

    // 書き込み先が IAP 外なら CRC32 を更新
    if (nextAddress >= FlashRomStartAddress + AddressOffsetNfcChipFirmware)
    {
        m_Crc32 = detail::CalcCrc32(&pOutBuffer[1], McuUpdateDataSizeMax, m_Crc32);
    }

    if (dataBytes % 2 != 0)
    {
        // 送信サイズが奇数のときはパディングを入れて 1 バイト余分に送る
        dataBytes++;
    }

    // チェックサムを付与して送信予約
    pOutBuffer[dataBytes + 1] = CalcCheckSum(pOutBuffer, dataBytes + 1);
    m_CommandBuilder.Append(pOutBuffer, dataBytes + 2);

    NN_RESULT_SUCCESS;
}

void TeraUpdater::ProceedFinalizePhase() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_XCD_TERA_UPDATER_LOG_VERBOSE("%s %d\n", NN_CURRENT_FUNCTION_NAME, m_Phase);

    if (m_Phase == UpdatePhase::Finalizing)
    {
        // 送出した全コマンドが処理されたら完了
        if (m_ReceivedInput.GetCompleteCount() >= m_TotalCommandCount)
        {
            m_ProgressRate = ProgressRateWriteFinish;
            NN_XCD_TERA_UPDATER_LOG_VERBOSE("Finalize complete!!!\n");
            SetPhase(UpdatePhase::FinalizeFinish);
        }
        return;
    }

    {
        // 書き込みコマンドの送信
        uint8_t sendData[] = { SofByte, WriteCommand, WriteCommand ^ 0xFF };
        m_CommandBuilder.Append(sendData, sizeof(sendData));
    }

    {
        // 書き込み先アドレスの送信
        auto startAddress = FlashRomStartAddress + AddressOffsetCrc32;
        uint8_t sendData[5];
        SetBigEndianValue(sendData, sizeof(uint32_t), startAddress);
        sendData[sizeof(sendData) - 1] = CalcCheckSum(sendData, sizeof(sendData) - 1);
        m_CommandBuilder.Append(sendData, sizeof(sendData));
    }

    {
        // CRC32 + データブロック数の送信
        RomFooter romFooter;
        GetRomFooter(&romFooter);

        uint8_t sendData[McuFirmwareFooterSize + 2];
        sendData[0] = static_cast<uint8_t>(McuFirmwareFooterSize - 1);
        std::memcpy(sendData + 1, romFooter.value, McuFirmwareFooterSize);
        sendData[sizeof(sendData) - 1] = CalcCheckSum(sendData, sizeof(sendData) - 1);
        m_CommandBuilder.Append(sendData, sizeof(sendData));
    }

    m_ProgressRate = ProgressRateSendFinish;

    SetPhase(UpdatePhase::Finalizing);
}

void TeraUpdater::FlushCommandList() NN_NOEXCEPT
{
    if (m_CommandBuilder.IsEmpty())
    {
        return;
    }

    m_OutputData.commandCount = m_CommandBuilder.GetCommandCount();

    uint8_t sendData[McuUpdateOutReportSize_Payload];
    size_t dataSize;
    m_OutputData.sequenceNumber = m_CommandBuilder.CreateReport(&dataSize, sendData, sizeof(sendData));
    m_OutputData.SetPayload(sendData, dataSize);
    NN_XCD_TERA_UPDATER_LOG_VERBOSE("SequenceNumber=%d\n", m_CommandBuilder.GetSequenceNumber());

    m_TimeoutCounter.Start(SequenceTimeout);
}

void TeraUpdater::StartSyncWithBootLoader() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_SDK_REQUIRES(m_IsActive);

    if (IsAborted())
    {
        return;
    }

    NN_SDK_REQUIRES_EQUAL(m_Phase, UpdatePhase::BootPrepare);

    NN_XCD_TERA_UPDATER_LOG_VERBOSE("%s\n", NN_CURRENT_FUNCTION_NAME);

    m_ProgressRate = ProgressRateBoot;
    SetPhase(UpdatePhase::Boot);

    ClearFirmwareUpdateInfo();
    m_LastCommandTick = nn::os::GetSystemTick();
}

void TeraUpdater::StartEraseRom() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_SDK_REQUIRES(m_IsActive);

    if (IsAborted())
    {
        return;
    }

    NN_SDK_REQUIRES_EQUAL(m_Phase, UpdatePhase::BootFinish);

    NN_XCD_TERA_UPDATER_LOG_VERBOSE("%s\n", NN_CURRENT_FUNCTION_NAME);

    // 起動フラグの消去から開始する
    m_ProgressRate   = ProgressRateInvalidate;
    m_EraseStartTick = nn::os::GetSystemTick();
    SetPhase(UpdatePhase::Invalidate);
}

void TeraUpdater::StartWriteRom() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_SDK_REQUIRES(m_IsActive);

    if (IsAborted())
    {
        return;
    }

    NN_SDK_REQUIRES_EQUAL(m_Phase, UpdatePhase::EraseFinish);

    NN_XCD_TERA_UPDATER_LOG_VERBOSE("%s\n", NN_CURRENT_FUNCTION_NAME);

    m_ProgressRate = ProgressRateWriting;
    SetPhase(UpdatePhase::Write);
}

void TeraUpdater::Finish() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_SDK_REQUIRES(m_IsActive);
    NN_SDK_REQUIRES(
        m_Phase == UpdatePhase::FinalizeFinish ||
        m_Phase == UpdatePhase::Error);

    NN_XCD_TERA_UPDATER_LOG_VERBOSE("%s\n", NN_CURRENT_FUNCTION_NAME);

    // 最後まで書き込みが完了していれば 100% にする
    if (m_ProgressRate == ProgressRateWriteFinish)
    {
        m_ProgressRate = ProgressRateComplete;
    }

    NN_XCD_TERA_UPDATER_LOG("Total sent command: %d\n", m_TotalCommandCount);

    SetPhase(UpdatePhase::Stopped);
}

}}} // namespace nn::xcd::detail
