﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/xcd/detail/xcd_Log.h>
#include "xcd_TeraBaseParser.h"

//#define VERBOSE

/*
 * VERBOSE モード時のみ、引数に指定された文字列を、Tera のログとして出力します。
 */
#ifdef VERBOSE

#ifdef NN_BUILD_CONFIG_COMPILER_VC
#define NN_XCD_DETAIL_PARSER_LOG(...)           NN_DETAIL_XCD_INFO("[TeraMcu] " ##__VA_ARGS__)
#else
#define NN_XCD_DETAIL_PARSER_LOG(format, ...)   NN_DETAIL_XCD_INFO("[TeraMcu] " format, ##__VA_ARGS__)
#endif

#else  // VERBOSE

#define NN_XCD_DETAIL_PARSER_LOG(...)           static_cast<void>(0)

#endif  // VERBOSE

namespace nn { namespace xcd { namespace detail {

bool TeraBaseParser::ParseCommonInputReport(
    CommonResponseType* pOutResponse,
    const uint8_t* pBuffer,
    size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutResponse);
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);
    NN_UNUSED(size);

    CommonResponseType common = {};
    common.result        = pBuffer[detail::McuCommonInputOffsetByte_ResultCode];
    common.command       = pBuffer[detail::McuCommonInputOffsetByte_Command];
    common.version.major =
        (pBuffer[detail::McuCommonInputOffsetByte_MajorVersion] << 8) |
         pBuffer[detail::McuCommonInputOffsetByte_MajorVersion + 1];
    common.version.minor =
        (pBuffer[detail::McuCommonInputOffsetByte_MinorVersion] << 8) |
         pBuffer[detail::McuCommonInputOffsetByte_MinorVersion + 1];
    common.state         = pBuffer[detail::McuCommonInputOffsetByte_State];

    // state が 0 (NOP) の場合は異常なので無視
    if (common.state == 0)
    {
        NN_XCD_DETAIL_PARSER_LOG("State = NOP is invalid. Ignored.\n");
        return false;
    }

    if (pOutResponse->result  != common.result ||
        pOutResponse->command != common.command ||
        pOutResponse->state   != common.state)
    {
        NN_XCD_DETAIL_PARSER_LOG(
            "\n  result: %d, command: %d, version: %d.%d, state: %d\n",
            common.result,
            common.command,
            common.version.major,
            common.version.minor,
            common.state);
    }

    *pOutResponse = std::move(common);

    return true;
}

bool TeraBaseParser::ParseIapInputReport(
    uint8_t* pOutResultCode,
    const uint8_t* pBuffer,
    size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutResultCode);
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);
    NN_UNUSED(size);

    auto resultCode = pBuffer[detail::McuIapInputOffsetByte_ResultCode];
    auto invertedResultCode = pBuffer[detail::McuIapInputOffsetByte_InvertedResultCode];
    if (resultCode != (~invertedResultCode & 0xFF))
    {
        // IAP の応答が不正
        NN_XCD_DETAIL_PARSER_LOG(
            "Invalid IAP result (%02X %02X). Ignored.\n",
            resultCode,
            invertedResultCode);
        return false;
    }

    *pOutResultCode = resultCode;

    return true;
}

}}}
