﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <vector>
#include <nn/nn_Result.h>
#include <nn/nn_Macro.h>
#include <nn/os/os_LightEventTypes.h>
#include <nn/psm/psm_SystemProcess.h>

#include "../xcd_ActivationCount.h"
#include "../xcd_IEventTask.h"
#include "../xcd_MultiWaitEvent.h"

namespace nn { namespace xcd { namespace detail {

const int PowerSupplyManagerSessionCountMax = 2;

//!< 本体の電源管理を行うためのクラスです。
class PowerSupplyManager final : public IEventTask
{
private:
    //!< 電源状態に変化があった場合に通知するイベントを管理する構造体
    struct PsmEventHolder
    {
        ::nn::os::LightEventType* pEventType;
        bool isUsed;
    };

    //!< psm のセッション
    ::nn::psm::Session m_Session;

    //!< psm から通知を受けるイベント
    PsmEventHolder m_PsmEvents[PowerSupplyManagerSessionCountMax];

    //!< PowerSupplyManager のアクティベートカウント
    ActivationCount m_ActivationCount;

    //!< 電源共有の有効カウント
    ActivationCount m_PowerSupplyCount;

    //!< 電源状態の変更があった際に通知を受け取るためのオブジェクト
    ::nn::os::SystemEventType m_PowerUpdateEvent;

    //!< 本体の電源状態
    bool m_IsConsolePowered;

public:
    PowerSupplyManager() NN_NOEXCEPT;
    virtual ~PowerSupplyManager() NN_NOEXCEPT NN_OVERRIDE;

    //!<  充電制御機能を有効にします。
    Result Activate() NN_NOEXCEPT;

    //!<  充電制御機能を有効にします。
    Result Deactivate() NN_NOEXCEPT;

    //!<  充電のために電源供給を有効にします。
    Result EnablePowerSupply() NN_NOEXCEPT;

    //!<  充電のために電源供給を無効にします。
    Result DisablePowerSupply() NN_NOEXCEPT;

    //!< 本体の充電状態に変化があった場合に、通知するイベントを登録します。
    Result BindPowerStateChangeEvent(::nn::os::LightEventType* pEvent) NN_NOEXCEPT;

    //!< 本体の充電状態に変化があった場合に、通知するイベントを登録します。
    void UnbindPowerStateChangeEvent(::nn::os::LightEventType* pEvent) NN_NOEXCEPT;

    //!< 本体の給電状態を取得します。
    bool IsPowered() NN_NOEXCEPT;

    //!< Eventが通知された際に呼ばれる関数
    virtual void EventFunction(const ::nn::os::MultiWaitHolderType* pMultiWaitHolder) NN_NOEXCEPT NN_OVERRIDE;

    //!< 定常的な処理を行う関数
    virtual void PeriodicEventFunction() NN_NOEXCEPT NN_OVERRIDE;

private:
    //!< 本体電源状態の更新
    void UpdatePowerState() NN_NOEXCEPT;
};

}}} // namespace nn::xcd::detail
