﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/os/os_LightEvent.h>
#include <nn/psm/psm.h>
#include <nn/psm/psm_System.h>
#include <nn/psm/psm_SystemProcess.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/xcd/xcd_ResultForPrivate.h>

#include "../xcd_TaskManager.h"
#include "xcd_PowerSupplyManager-os.horizon.h"


namespace nn { namespace xcd { namespace detail {

PowerSupplyManager::PowerSupplyManager() NN_NOEXCEPT
{
    // 何もしない
}

PowerSupplyManager::~PowerSupplyManager() NN_NOEXCEPT
{
    // 何もしない
}

Result PowerSupplyManager::Activate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsMax(), ResultActivationUpperLimitOver());

    if (m_ActivationCount.IsZero())
    {
        ::nn::psm::Initialize();
        ::nn::psm::OpenSession(&m_Session);
        ::nn::psm::BindStateChangeEvent(&m_PowerUpdateEvent, &m_Session);

        GetTaskManager().RegisterPeriodicTask(this);

        UpdatePowerState();
    }
    ++m_ActivationCount;

    NN_RESULT_SUCCESS;
}

Result PowerSupplyManager::Deactivate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsZero(), ResultActivationLowerLimitOver());

    --m_ActivationCount;

    if (m_ActivationCount.IsZero())
    {
        GetTaskManager().UnregisterPeriodicTask(this);

        ::nn::psm::UnbindStateChangeEvent(&m_Session);
        ::nn::psm::CloseSession(&m_Session);

        ::nn::psm::Finalize();
    }

    NN_RESULT_SUCCESS;
}

Result PowerSupplyManager::EnablePowerSupply() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!m_ActivationCount.IsZero());
    NN_RESULT_THROW_UNLESS(!m_PowerSupplyCount.IsMax(), ResultPowerSupplyEnableUppwerLimitOver());

    if (m_PowerSupplyCount.IsZero())
    {
        ::nn::psm::AcquireControllerPowerSupply();
    }
    ++m_PowerSupplyCount;

    NN_RESULT_SUCCESS;
}

Result PowerSupplyManager::DisablePowerSupply() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!m_ActivationCount.IsZero());
    NN_RESULT_THROW_UNLESS(!m_PowerSupplyCount.IsZero(), ResultPowerSupplyEnableLowerLimitOver());

    --m_PowerSupplyCount;
    if (m_PowerSupplyCount.IsZero())
    {
        ::nn::psm::ReleaseControllerPowerSupply();
    }

    NN_RESULT_SUCCESS;
}

Result PowerSupplyManager::BindPowerStateChangeEvent(::nn::os::LightEventType* pEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!m_ActivationCount.IsZero());

    for (auto& psmEvent : m_PsmEvents)
    {
        if (psmEvent.isUsed == false)
        {
            psmEvent.pEventType = pEvent;
            psmEvent.isUsed = true;
            NN_RESULT_SUCCESS;
        }
    }

    NN_RESULT_THROW(ResultPowerSupplySessionCountMax());
}

void PowerSupplyManager::UnbindPowerStateChangeEvent(::nn::os::LightEventType* pEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!m_ActivationCount.IsZero());

    for (auto& psmEvent : m_PsmEvents)
    {
        if (psmEvent.isUsed == true && psmEvent.pEventType == pEvent)
        {
            psmEvent.pEventType = nullptr;
            psmEvent.isUsed = false;
        }
    }
}

bool PowerSupplyManager::IsPowered() NN_NOEXCEPT
{
    return m_IsConsolePowered;
}

void PowerSupplyManager::EventFunction(const ::nn::os::MultiWaitHolderType* pMultiWaitHolder) NN_NOEXCEPT
{
    NN_UNUSED(pMultiWaitHolder);
}

void PowerSupplyManager::PeriodicEventFunction() NN_NOEXCEPT
{
    if (::nn::os::TryWaitSystemEvent(&m_PowerUpdateEvent))
    {
        ::nn::os::ClearSystemEvent(&m_PowerUpdateEvent);
        UpdatePowerState();
        for (auto& psmEvent : m_PsmEvents)
        {
            if (psmEvent.isUsed == true)
            {
                NN_ABORT_UNLESS_NOT_NULL(psmEvent.pEventType);
                ::nn::os::SignalLightEvent(psmEvent.pEventType);
            }
        }
    }
}

void PowerSupplyManager::UpdatePowerState() NN_NOEXCEPT
{
    auto psmPowerState = ::nn::psm::GetChargerType();
    switch(psmPowerState)
    {
    case ::nn::psm::ChargerType_EnoughPower:
    case ::nn::psm::ChargerType_LowPower:
        m_IsConsolePowered = true;
        break;
    case ::nn::psm::ChargerType_NotSupported:
    case ::nn::psm::ChargerType_Unconnected:
        m_IsConsolePowered = false;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

}}} // namespace nn::xcd::detail
