﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

namespace nn { namespace xcd { namespace detail {

//!< Nwcp のペイロードサイズから全体の Uart パケットのサイズを計算する
inline size_t CalculateUartPacketFullSize(size_t payloadSize)
{
    return payloadSize + 5;
}

//!< Nwcp に接続されるデバイスの種類
enum NwcpDeviceType
{
    NwcpDeviceType_JoyRight,
    NwcpDeviceType_JoyLeft,
    NwcpDeviceType_Fifty,
    NwcpDeviceType_Unknown,
};

// Nwcp のデータフォーマットの定義
const int NwcpSize_HciHeaderSize         = 5;   // Hci 部分の共通のヘッダサイズ
const int NwcpByte_HciIndicator          = 0;   // Hci の Indicator
const int NwcpByte_HciConnectionHandle   = 1;   // Hhi の Connection Handle
const int NwcpByte_HciLength             = 3;   // Hci の Length
const int NwcpSize_HciLength             = 2;   // Hci の Length のサイズ

const int NwcpSize_NwcpHeaderSize        = 7;   // Nwcp 部分の共通のヘッダサイズ
const int NwcpByte_NnIndicator           = 5;   // Nwcp 内の Indicator
const int NwcpByte_NwcpOpCode            = 6;   // Nwcp の OpCode
const int NwcpByte_NwcpPayloadLength     = 7;   // Nwcp の PayLoad の Length
const int NwcpByte_NwcpStatus            = 9;   // Nwcp の Status
const int NwcpByte_NwcpDataCrc           = 10;  // Nwcp の Data CRC
const int NwcpByte_NwcpHeaderCrc         = 11;  // Nwcp の Header CRC
const int NwcpByte_NwcpPayload           = 12;  // Nwcp の Payload

//!< Nwcp の 識別子
const uint8_t NwcpIndicator = 0x19;

//!< Nwcp の種類
const uint8_t NwcpType_Out[] = {0x01, 0x03};
const uint8_t NwcpType_In[] = {0x81, 0x03};

//!< Nwcp の NN Indicator の定義
const uint8_t NwcpNnIndicator_SetReportSuccess = 0x00;
const uint8_t NwcpNnIndicator_GetReportResult  = 0xA3;
const uint8_t NwcpNnIndicator_Sync             = 0xA5;
const uint8_t NwcpNnIndicator_DeviceCommand    = 0x91;
const uint8_t NwcpNnIndicator_DeviceResult     = 0x94;
const uint8_t NwcpNnIndicator_Hid              = 0x92;
const uint8_t NwcpNnIndicator_SetReport        = 0x53;
const uint8_t NwcpNnIndicator_GetReport        = 0x43;
const uint8_t NwcpNnIndicator_FiftyData        = 0x9a;                       //!< Data(Fifty向け) の OpCode

//!< Magic Packet
const uint8_t NwcpMagicPacket[] =  { 0xA1, 0xA2, 0xA3, 0xA4};
const uint8_t NwcpSize_MagicPacket = sizeof(NwcpMagicPacket);

//!< Sync Message
const uint8_t NwcpSync[] =         { NwcpNnIndicator_Sync, 0x02, 0x01, 0x7E, 0x00, 0x00, 0x00 };
const uint8_t NwcpSize_SyncPayload = sizeof(NwcpSync);
const uint8_t NwcpSyncResponse[] = { NwcpNnIndicator_Sync, 0x02, 0x02, 0x7D, 0x00, 0x00, 0x00 };
const uint8_t NwcpSize_SyncResponsePayload = sizeof(NwcpSyncResponse);

//!< Reset Message
const uint8_t NwcpReset[] =         { NwcpNnIndicator_Sync, 0x02, 0x03, 0x7C, 0x00, 0x00, 0x00 };
const uint8_t NwcpSize_ResetPayload = sizeof(NwcpSync);

//!< DeviceCommand 共通の定義
enum NwcpStatus : uint8_t
{
    NwcpStatus_Success = 0x00,    //!< Nwcp Device Command の処理が成功した
    NwcpStatus_Failed = 0x01,     //!< Nwcp Device Command の処理が失敗した(暫定)
};

// DeviceCommand Inquiry
const uint8_t NwcpOpCode_Inquiry = 0x01;                            //!< Inquiry の OpCode
const uint8_t NwcpSize_InquiryResponsePayload = 0x08;               //!< Inquiry の Response の Payload 長
const uint8_t NwcpByte_InquiryDeviceType = NwcpByte_NwcpPayload;    //!< Inquiry Response 内の DeviceType
const uint8_t NwcpByte_InquiryBdAddr = NwcpByte_NwcpPayload + 1;    //!< Inquiry Response 内の BdAddr

// DeviceCommand WriteChargerSetting
const uint8_t NwcpOpCode_WriteChargerSettings = 0x02;               //!< WriteChargerSettings の OpCode
const uint8_t NwcpSize_WriteChargerSettingsCommandPayload = 0x04;   //!< WriteChargerSettings の Response の Payload 長
const uint8_t NwcpSize_WriteChargerSettingsResponsePayload = 0x00;  //!< WriteChargerSettings の Response の Payload 長
const uint8_t NwcpByte_WriteChargerSettings = NwcpByte_NwcpPayload; //!< WriteChargerSettings の 設定位置
const uint8_t NwcpChargerSettingsMask_Enable = 0x04;                //!< 充電を開始する
const uint8_t NwcpChargerSettingsMask_100mA  = 0x08;                //!< 充電を 100mA で行う
const uint8_t NwcpChargerSettingsMask_200mA  = 0x10;                //!< 充電を 200mA で行う

// DeviceCommand UpdateUartBaudRate
const uint8_t NwcpOpCode_UpdateUartBaudRate = 0x20;                     //!< UpdateUartBaudRate の OpCode
const uint8_t NwcpByte_UpdateUartBaudRateValue = NwcpByte_NwcpPayload;  //!< UpdateUartBaudRate で設定するインターバル値
const uint8_t NwcpSize_UpdateUartBaudRatePayload = 0x08;                //!< UpdateUartBaudRate の Payload 長

// DeviceCommand ReadAndClearWakeupReason
const uint8_t NwcpOpCode_ReadWakeUpReason = 0x07;                   //!< Read and Clear Wakeup Reason の OpCode
const uint8_t NwcpSize_ReadWakeUpReasonResponsePayload = 0x04;      //!< Read and Clear Wakeup Reason の Response の Payload 長
const uint8_t NwcpByte_WakeUpReason       = NwcpByte_NwcpPayload;   //!< スリープ復帰要因
const Bit8    NwcpWakeUpReasonBitMask_Home = 0x01;                  //!< スリープ復帰要因 の BitMask : Home押し
const Bit8    NwcpWakeUpReasonBitMask_BatteryLevelChange = 0x02;    //!< スリープ復帰要因 の BitMask : 電池残量変化

// DeviceCommand CreateHidConnection
const uint8_t NwcpOpCode_CreateHidConnection = 0x10;                //!< CreateHidConenction の OpCode

// DeviceCommand DisconnectHid
const uint8_t NwcpOpCode_DisconnectHid = 0x11;                      //!< DisconnectHidConenction の OpCode

// DeviceCommand Pairing
const uint8_t NwcpOpCode_Pairing = 0x18;                             //!< Pairing の OpCode
const uint8_t NwcpByte_PairingBdAddr        = NwcpByte_NwcpPayload;  //!< UpdateUartBaudRate で設定するインターバル値
const uint8_t NwcpSize_PairingBdAddr        = 0x06;                  //!< UpdateUartBaudRate の Payload 長
const uint8_t NwcpByte_PairingClassOfDevice = NwcpByte_PairingBdAddr + NwcpSize_PairingBdAddr;
                                                                     //!< UpdateUartBaudRate で設定するインターバル値
const uint8_t NwcpSize_PairingClassOfDevice = 0x03;                  //!< UpdateUartBaudRate の Payload 長
const uint8_t NwcpByte_PairingDeviceName    = NwcpByte_PairingClassOfDevice + NwcpSize_PairingClassOfDevice;
                                                                     //!< UpdateUartBaudRate で設定するインターバル値
const uint8_t NwcpSize_PairingDeviceName    = 0x14;                  //!< UpdateUartBaudRate の Payload 長
const uint8_t NwcpByte_PairingFeatureSet    = NwcpByte_PairingDeviceName + NwcpSize_PairingDeviceName;
                                                                     //!< UpdateUartBaudRate で設定するインターバル値
const uint8_t NwcpSize_PairingFeatureSet    = 0x01;                  //!< UpdateUartBaudRate の Payload 長
const uint8_t NwcpSize_Pairing              = 0x20;

const uint8_t NwcpByte_PairingLinkKey       = NwcpByte_NwcpPayload;
                                                                     //!< UpdateUartBaudRate で設定するインターバル値
const uint8_t NwcpSize_PairingLinkKey       = 0x10;                  //!< UpdateUartBaudRate の Payload 長
enum NwcpPairingStatus : uint8_t
{
    NwcpPairingStatus_Initiate = 1,
    NwcpPairingStatus_LinkKey = 2,
    NwcpPairingStatus_Ack = 3,
};

// DeviceCommand SetHidInterval
const uint8_t NwcpOpCode_SetHidInterval = 0x12;                     //!< SetHidInterval の OpCode
const uint8_t NwcpByte_SetHidIntervalValue = NwcpByte_NwcpPayload;  //!< SetHidInterval で設定するインターバル値
const uint8_t NwcpSize_SetHidIntervalPayload = 0x04;                //!< SetHidInterval の Payload 長

// Nwcp Hid
const uint8_t NwcpHidByte_ReportId = NwcpByte_NwcpPayload;          //!< Report Id を格納するバイト

// Hid POLL Packet
const uint8_t NwcpHidReportId_Poll = 0x1f;                          //!< 何もしないための Hid Report
const uint8_t NwcpHidSize_Poll = 0x01;                              //!< 何もしないための Hid Report サイズ

// SetReport
const int NwcpSize_SetReportPayloadMax = 257;                   //!< SetReport の Payload 最大長 256 Byte + 1Byte(Indicator)

// GetReport
const int NwcpSize_GetReportPayload = 0x02;                     //!< GetReport の Payload 長さ
const int NwcpSize_GetReportPayloadMax = 0xff;                  //!< GetReport の Payload 最大長

// DeviceCommand Fifty Data
const uint8_t NwcpSize_FiftyDataResponsePayload = 0x0c;               //!< Data(Fifty向け) の Response の Payload 長
const uint8_t NwcpByte_FiftyDataPayload = NwcpByte_NwcpPayload;       //!< Data(Fifty向け) Response 内の DeviceType

}}} // namespace nn::xcd
