﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/bluetooth/bluetooth_Api.h>
#include <nn/btm/btm.h>
#include <nn/btm/btm_Result.h>
#include <nn/os/os_LightEvent.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/os/os_Thread.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/xcd/xcd_Result.h>
#include <nn/xcd/xcd_ResultForPrivate.h>
#include <nn/xcd/detail/xcd_Log.h>

#include "xcd_BluetoothHidTask-hardware.nx.h"
#include "xcd_BluetoothUtil-hardware.nx.h"
#include "../xcd_DeviceHandleGenerator.h"
#include "../xcd_TaskManager.h"

#if 0
#define BT_DUMP_LOG(...) NN_DETAIL_XCD_INFO(__VA_ARGS__)
#else
#define BT_DUMP_LOG(...)
#endif

namespace nn { namespace xcd { namespace detail {

namespace
{
    void PrintBluetoothAddress(nn::bluetooth::BluetoothAddress& address)
    {
        BT_DUMP_LOG("::nn::bluetooth::Address:%02x %02x %02x %02x %02x %02x", address.address[0], address.address[1], address.address[2], address.address[3], address.address[4], address.address[5]);
    }

    ::nn::TimeSpan GetTimeSpanFromSniffMode(::nn::btm::SniffMode& sniff)
    {
        switch (sniff)
        {
        case ::nn::btm::SniffMode_5ms:
            return ::nn::TimeSpan::FromMilliSeconds(5);
        case ::nn::btm::SniffMode_10ms:
            return ::nn::TimeSpan::FromMilliSeconds(10);
        case ::nn::btm::SniffMode_15ms:
            return ::nn::TimeSpan::FromMilliSeconds(15);
        default:
            return ::nn::TimeSpan(0);
        }
    }

    // BTM の SlotMode を内部用の値に変換
    int ConvertBtmSlotModeToInternal(::nn::btm::SlotMode btmSlotMode) NN_NOEXCEPT
    {
        switch (btmSlotMode)
        {
        case ::nn::btm::SlotMode_Active:
            // Active は 0xff を返す
            return 0xff;
        case ::nn::btm::SlotMode_2:
            return 2;
        case ::nn::btm::SlotMode_4:
            return 4;
        case ::nn::btm::SlotMode_6:
            return 6;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    // 内部用の SlotMode を BTM の SlotMode に変換
    ::nn::btm::SlotMode ConvertInternalSlotModeToBtm(int slotMode) NN_NOEXCEPT
    {
        switch (slotMode)
        {
        case 2:
            return ::nn::btm::SlotMode_2;
        case 4:
            return ::nn::btm::SlotMode_4;
        case 6:
            return ::nn::btm::SlotMode_6;
        case 0xff:
            return ::nn::btm::SlotMode_Active;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    const int SetSlotModeRetryTime = 30;   //!< SetSlotMode に失敗した際に Retry するまでの時間 (msec)
}

void BluetoothHidTask::EventFunction(const ::nn::os::MultiWaitHolderType* pMultiWaitHolder) NN_NOEXCEPT
{
    if (pMultiWaitHolder == m_HidEvent.GetWaitId())
    {
        HandleBluetoothEvent();
    }
}

void BluetoothHidTask::PeriodicEventFunction() NN_NOEXCEPT
{
    if (nn::os::TryWaitSystemEvent(&m_BtmEvent))
    {
        nn::os::ClearSystemEvent(&m_BtmEvent);
        HandleBtmEvent();
        return;
    }

    if (nn::os::TryWaitTimerEvent(&m_BtmRetryEvent))
    {
        nn::os::ClearTimerEvent(&m_BtmRetryEvent);
        HandleSetSlotModeRetry();
        return;
    }
}

void BluetoothHidTask::HandleBluetoothEvent() NN_NOEXCEPT
{
    nn::bluetooth::HidEventType eventType;
    uint8_t buffer[nn::bluetooth::BUFFER_SIZE_OF_HID_OUT];

    nn::bluetooth::HidGetReportEventInfo(&eventType, &buffer[0], nn::bluetooth::BUFFER_SIZE_OF_HID_OUT);

    if (eventType == nn::bluetooth::EventFromGetReportCallback)
    {
        nn::bluetooth::BluetoothAddress address = reinterpret_cast<nn::bluetooth::InfoFromGetReportCallback*>(&buffer[0])->bluetoothAddress;
        auto pReport = reinterpret_cast<const nn::bluetooth::InfoFromGetReportCallback*>(&buffer[0]);
        auto descryptor = GetDescryptorFromAddress(address);
        if (descryptor != nullptr)
        {
            descryptor->accessor.ParseInputReport(&pReport->rptData.reportData.data[0], pReport->rptData.reportData.length);
        }
    }
    else if (eventType == nn::bluetooth::EventFromSetReportStatusCallback)
    {
        Result result = ResultSuccess();
        nn::bluetooth::BluetoothAddress address = reinterpret_cast<nn::bluetooth::InfoFromSetReportStatusCallback*>(&buffer[0])->bluetoothAddress;
        auto status = reinterpret_cast<nn::bluetooth::InfoFromSetReportStatusCallback*>(&buffer[0])->status;
        auto descryptor = GetDescryptorFromAddress(address);
        if (status != nn::bluetooth::BTHH_OK)
        {
            NN_DETAIL_XCD_ERROR("SetReport Failed %d\n", status);
            result = ResultHidSetReportFailed();
        }
        if (descryptor != nullptr)
        {
            descryptor->accessor.SetSetReportResult(result);
        }
    }
    else if (eventType == nn::bluetooth::EventFromGetReportStatusCallback)
    {
        nn::bluetooth::BluetoothAddress address = reinterpret_cast<nn::bluetooth::InfoFromGetReportStatusCallback*>(&buffer[0])->bluetoothAddress;
        auto pBuffer =  &((reinterpret_cast<nn::bluetooth::InfoFromGetReportStatusCallback*>(&buffer[0]))->data[0]);
        auto status = (reinterpret_cast<nn::bluetooth::InfoFromGetReportStatusCallback*>(&buffer[0]))->status;
        auto length =  reinterpret_cast<nn::bluetooth::InfoFromGetReportStatusCallback*>(&buffer[0])->length;

        auto descryptor = GetDescryptorFromAddress(address);
        if (descryptor != nullptr)
        {
            if (status == nn::bluetooth::BTHH_OK)
            {
                descryptor->accessor.SetGetReportData(pBuffer, length);
            }
            else
            {
                NN_DETAIL_XCD_ERROR("GetReport Failed %d\n", status);
                descryptor->accessor.NotifyGetReportFailed(status);
            }
        }
    }
}

void BluetoothHidTask::HandleBtmEvent() NN_NOEXCEPT
{
    nn::btm::DeviceConditionList list;
    // 現在の DeviceCondition 情報を取得
    nn::btm::GetConnectedDeviceCondition(&list);

    // deviceCount が COUNT_OF_DC_LIST より大きい場合は、リストが異常な状態となっているので、カウントを 0 に初期化
    if (list.deviceCount > nn::btm::COUNT_OF_DC_LIST)
    {
        list.deviceCount = 0;
    }

    HandleDetachedDevices(list);
    HandleAttachedDevices(list);
    HandleSlotModeUpdate(list);
    m_LinkCountMax = static_cast<int>(list.deviceCountCapacity);
}

void BluetoothHidTask::HandleSetSlotModeRetry() NN_NOEXCEPT
{
    nn::btm::DeviceConditionList list;
    // 現在の DeviceCondition 情報を取得
    nn::btm::GetConnectedDeviceCondition(&list);

    HandleSlotModeUpdate(list);
}

void BluetoothHidTask::HandleDetachedDevices(nn::btm::DeviceConditionList& list) NN_NOEXCEPT
{
    // 切断されたデバイスの探索
    for (auto& descryptor : m_Devices)
    {
        if (descryptor.accessor.IsActivated() == true)
        {
            bool isDetached = true;

            for (int i = 0; i < list.deviceCount; ++i)
            {
                if (IsSameBluetoothAddress(descryptor.address, list.device[i].bdAddress))
                {
                    isDetached = false;
                    break;
                }
            }

            if (isDetached == true)
            {
                if (RemoveDevice(descryptor.address) == true)
                {
                    // 上位のデバイス情報の更新を通知
                    nn::os::SignalLightEvent(m_pUpdatedEvent);
                }
            }
        }
    }
}

void BluetoothHidTask::HandleAttachedDevices(nn::btm::DeviceConditionList& list) NN_NOEXCEPT
{
    // 接続されたデバイスの探索
    for (int i = 0; i < list.deviceCount; ++i)
    {
        if (list.device[i].profile == nn::btm::Profile_Hid &&
            IsNintendoGamepad(list.device[i].bdName))
        {
            NN_DETAIL_XCD_INFO("Device[%d] Sniff:%d SlotSize:%d\n", i,
                                                                    list.device[i].hidDeviceCondition.sniffMode,
                                                                    list.device[i].hidDeviceCondition.slotMode);
            bool isAttached = true;

            for (auto& descryptor : m_Devices)
            {
                if (descryptor.accessor.IsActivated() == true &&
                    IsSameBluetoothAddress(descryptor.address, list.device[i].bdAddress))
                {
                    isAttached = false;
                    break;
                }
            }

            if (isAttached == true)
            {
                // DeviceType は使わないので、 Unknown をセットしても問題ない
                // Hid 通信の GetDeviceInfo で 最終的なデバイス種別を確定するため、BTではデバイス種別による分岐をしない
                if (RegisterDevice(list.device[i].bdAddress, DeviceType_Unknown) == true)
                {
                    // Default の SlotMode は 2 Slot
                    ChangeSlotMode(list.device[i].bdAddress, nn::btm::SlotMode_2);
                    nn::os::SignalLightEvent(m_pUpdatedEvent);
                }
            }
        }
    }
}

void BluetoothHidTask::HandleSlotModeUpdate(nn::btm::DeviceConditionList& list) NN_NOEXCEPT
{
    for (auto &descryptor : m_Devices)
    {
        if (descryptor.accessor.IsActivated())
        {
            // BdAddr を探索して、Slot モードを最新の設定に更新
            for (int i = 0; i < list.deviceCount; ++i)
            {
                auto& device = list.device[i];
                if (IsSameBluetoothAddress(descryptor.address, device.bdAddress) == true)
                {
                    // 通信周期を設定
                    descryptor.accessor.SetInterval(
                        (device.hidDeviceCondition.sniffMode != ::nn::btm::SniffMode_Active),
                        GetTimeSpanFromSniffMode(device.hidDeviceCondition.sniffMode));

                    bool isSlotModeChanged = false;
                    if (descryptor.targetSlotMode == device.hidDeviceCondition.slotMode &&
                        descryptor.isChangingSlotMode == true)
                    {
                        // 指定された SlotMode への変更完了
                        descryptor.isChangingSlotMode = false;
                        isSlotModeChanged             = true;
                    }

                    // SlotMode が自動で変更された、または指定されたモードに変更完了した場合は完了通知を出す
                    if (descryptor.currentSlotMode != device.hidDeviceCondition.slotMode ||
                        isSlotModeChanged == true)
                    {
                        descryptor.currentSlotMode = device.hidDeviceCondition.slotMode;

                        // SlotMode の変更を通知
                        if (descryptor.pEvent != nullptr)
                        {
                            ::nn::os::SignalLightEvent(descryptor.pEvent);
                        }
                    }

                    BT_DUMP_LOG("[xcd] ");
                    PrintBluetoothAddress(device.bdAddress);
                    BT_DUMP_LOG("  SlotMode: %d\n", descryptor.currentSlotMode);
                    break;
                }
            }
        }
    }

    // Bluetoot Mode を更新
    m_BluetoothMode = list.bluetoothMode;
    BT_DUMP_LOG("[xcd] Bluetooth Mode %d\n", m_BluetoothMode);

    // SlotModeの更新
    SetSlotModeToBtm(false);
}

BluetoothHidTask::BluetoothHidTask() NN_NOEXCEPT :
    m_BluetoothMode(::nn::btm::BluetoothMode_Auto),
    m_LinkCountMax(0)
{
    // 何もしない
}

BluetoothHidTask::~BluetoothHidTask() NN_NOEXCEPT
{
    // 何もしない
}

nn::Result BluetoothHidTask::Activate(nn::os::LightEventType* pUpdatedEvent) NN_NOEXCEPT
{
    m_pUpdatedEvent = pUpdatedEvent;

    m_CurrentBluetoothSettings = BluetoothSettings_6xUp3Slot;

    nn::os::InitializeTimerEvent(&m_BtmRetryEvent, nn::os::EventClearMode_ManualClear);

    // BTM の初期化
    nn::btm::InitializeBtmInterface();

    // BTM にイベントをセット
    nn::btm::RegisterSystemEventForConnectedDeviceCondition(&m_BtmEvent);

    nn::bluetooth::InitializeBluetoothDriver();
    NN_RESULT_DO(nn::bluetooth::RegisterHidReportEvent(m_HidEvent.GetBase()));

    GetTaskManager().RegisterPeriodicTask(this);
    GetTaskManager().RegisterEvent(&m_HidEvent, this);

    // 起動時に一度 Btm のパラメーターを読み出しておく
    HandleBtmEvent();

    m_IsSuspended = false;

    NN_RESULT_SUCCESS;
}

void BluetoothHidTask::Deactivate() NN_NOEXCEPT
{
    GetTaskManager().UnregisterEvent(&m_HidEvent);
    GetTaskManager().UnregisterPeriodicTask(this);

    nn::bluetooth::FinalizeBluetoothDriver();

    nn::btm::FinalizeBtmInterface();

    nn::os::FinalizeTimerEvent(&m_BtmRetryEvent);
}

bool BluetoothHidTask::RegisterDevice(nn::bluetooth::BluetoothAddress address, DeviceType deviceType) NN_NOEXCEPT
{
    // リストに追加
    for(auto& descryptor : m_Devices)
    {
        if(descryptor.accessor.IsActivated() == false)
        {
            int vendorId = 0x57E;
            int productId = 0x2006;

            // 接続直後はアクティブ
            descryptor.currentSlotMode = nn::btm::SlotMode_Active;
            descryptor.isChangingSlotMode = false;

            descryptor.deviceInfo.address = ConvertToCommonBluetoothAddress(address);
            descryptor.deviceInfo.deviceHandle = DeviceHandleGenerator::Get().GetDeviceHandle();
            descryptor.deviceInfo.vid = vendorId;
            descryptor.deviceInfo.pid = productId;
            descryptor.deviceInfo.interfaceType = InterfaceType_Bluetooth;
            descryptor.deviceInfo.deviceType = deviceType;
            descryptor.pEvent = nullptr;
            descryptor.accessor.Activate(descryptor.deviceInfo.deviceHandle, address);
            descryptor.address = address;
            return true;
        }
    }

    return false;
}

bool BluetoothHidTask::RemoveDevice(nn::bluetooth::BluetoothAddress address) NN_NOEXCEPT
{
    auto descryptor = GetDescryptorFromAddress(address);
    if (descryptor == nullptr)
    {
        return false;
    }
    ClearBluetoothAddress(&descryptor->address);
    descryptor->accessor.Deactivate();
    return true;
}

void BluetoothHidTask::ChangeSlotMode(nn::bluetooth::BluetoothAddress address, nn::btm::SlotMode slotMode) NN_NOEXCEPT
{
    auto descryptor = GetDescryptorFromAddress(address);
    if (descryptor == nullptr)
    {
        return;
    }
    BT_DUMP_LOG("[xcd] Change Slot Mode ");
    PrintBluetoothAddress(descryptor->address);
    BT_DUMP_LOG(" SlotMode : %d\n", slotMode);

    descryptor->targetSlotMode     = slotMode;
    descryptor->isChangingSlotMode = true;
}

void BluetoothHidTask::SetSlotModeToBtm(bool forceUpdate) NN_NOEXCEPT
{
    nn::btm::DeviceSlotModeList slotModeList;
    slotModeList.deviceCount = 0;

    bool needsUpdate   = forceUpdate;
    auto bluetoothMode = ::nn::btm::BluetoothMode_Auto;

    for (auto &descryptor : m_Devices)
    {
        if (descryptor.accessor.IsActivated())
        {
            // もし 1 つでも SlotMode_Active のものがあれば、Active に変更
            if (descryptor.targetSlotMode == ::nn::btm::SlotMode_Active)
            {
                bluetoothMode = ::nn::btm::BluetoothMode_Active;
            }

            if (descryptor.currentSlotMode != descryptor.targetSlotMode)
            {
                needsUpdate |= true;
            }

            // SlotModeList に期待するスロット設定を追加
            slotModeList.device[slotModeList.deviceCount].bdAddress = descryptor.address;
            slotModeList.device[slotModeList.deviceCount].slotMode = descryptor.targetSlotMode;
            slotModeList.deviceCount++;
        }
    }

    BT_DUMP_LOG("[xcd] Update SlotMode\n");

    // BluetoothMode の切り替え
    if (m_BluetoothMode != bluetoothMode)
    {
        BT_DUMP_LOG("    -> Change Bt Mode %d\n", bluetoothMode);
        auto result = ::nn::btm::SetBluetoothMode(bluetoothMode);
        if (::nn::btm::ResultInvalidUsecase::Includes(result))
        {
            // InvalidUsecase の場合は Active が使えない
            // この状況になるのはローカル通信中で、ローカル通信中は 4 slot になっているはず
            BT_DUMP_LOG("    -> InvalidUsecase. Using 4-slot mode.\n");
            m_BluetoothMode = ::nn::btm::BluetoothMode_Auto;

            // 4 slot を指定したことにする
            for (int i = 0; i < slotModeList.deviceCount; i++)
            {
                auto& device = slotModeList.device[i];
                if (device.slotMode == ::nn::btm::SlotMode_Active)
                {
                    device.slotMode = ::nn::btm::SlotMode_4;
                }
            }

            for (auto &descryptor : m_Devices)
            {
                if (descryptor.accessor.IsActivated() &&
                    descryptor.targetSlotMode == ::nn::btm::SlotMode_Active)
                {
                    descryptor.targetSlotMode = ::nn::btm::SlotMode_4;
                }
            }

            needsUpdate = true;
        }
        else
        {
            if (result.IsFailure())
            {
                BT_DUMP_LOG("    -> Busy result:%d\n", result.GetDescription());
                nn::os::StartOneShotTimerEvent(&m_BtmRetryEvent, nn::TimeSpan::FromMilliSeconds(SetSlotModeRetryTime));
            }

            return;
        }
    }

    // 各コントローラーの Slot 数の変更
    if (needsUpdate)
    {
        BT_DUMP_LOG("    -> Update SlotMode\n");
        auto result = nn::btm::SetSlotMode(&slotModeList);
        if (result.IsFailure())
        {
            if (nn::btm::ResultInvalidUsecase::Includes(result))
            {
                // InvalidUsecase の場合は変更不可能なのでリトライはしない
                BT_DUMP_LOG("    -> InvalidUsecase. Skipped.\n");

                // 変更中フラグを解除
                CancelChangingSlotMode();
            }
            else
            {
                BT_DUMP_LOG("    -> Busy result:%d\n", result.GetDescription());
                nn::os::StartOneShotTimerEvent(&m_BtmRetryEvent, nn::TimeSpan::FromMilliSeconds(SetSlotModeRetryTime));
            }
        }
    }
}

void BluetoothHidTask::CancelChangingSlotMode() NN_NOEXCEPT
{
    for (auto &descryptor : m_Devices)
    {
        if (descryptor.accessor.IsActivated() &&
            descryptor.isChangingSlotMode)
        {
            descryptor.isChangingSlotMode = false;
            if (descryptor.pEvent != nullptr)
            {
                ::nn::os::SignalLightEvent(descryptor.pEvent);
            }
        }
    }
}

BluetoothHidTask::HidDeviceDescryptor* BluetoothHidTask::GetDescryptorFromAddress(const nn::bluetooth::BluetoothAddress address) NN_NOEXCEPT
{
    // リストに追加
    for(auto& descryptor : m_Devices)
    {
        if(descryptor.accessor.IsActivated() == true)
        {
            if (IsSameBluetoothAddress(descryptor.address, address) == true)
            {
                return &descryptor;
            }
        }
    }

    return nullptr;
}

size_t BluetoothHidTask::GetDevices(HidDeviceInfo* pOutValue, size_t deviceCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    size_t returnCount = 0;

    for(auto& descryptor : m_Devices)
    {
        if(descryptor.accessor.IsActivated() == true)
        {
            pOutValue[returnCount] = descryptor.deviceInfo;
            returnCount++;

            if(returnCount == deviceCount)
            {
                break;
            }
        }
    }

    return returnCount;
}

HidAccessor* BluetoothHidTask::GetHidAccessor(DeviceHandle deviceHandle) NN_NOEXCEPT
{
    for(int i = 0; i < HidDeviceCountMax; i++)
    {
        if(m_Devices[i].accessor.GetDeviceHandle() == deviceHandle)
        {
            return &m_Devices[i].accessor;
        }
    }

    // みつからなかった
    return nullptr;
}

void BluetoothHidTask::DetachDevice(DeviceHandle deviceHandle) NN_NOEXCEPT
{
    for (int i = 0; i < HidDeviceCountMax; i++)
    {
        if (m_Devices[i].accessor.GetDeviceHandle() == deviceHandle)
        {
            // Busy だった場合にエラー
            // Sleep 遷移時くらいしか使う想定がないので、暫定実装
            nn::btm::HidDisconnect(&(m_Devices[i].address));
        }
    }
}

Result BluetoothHidTask::SetBluetoothSettings(BluetoothSettings bluetoothSettings, nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT
{
    m_CurrentBluetoothSettings = bluetoothSettings;

    NN_RESULT_SUCCESS;
}

BluetoothSettings BluetoothHidTask::GetBluetoothSettings() NN_NOEXCEPT
{
    return m_CurrentBluetoothSettings;
}

void BluetoothHidTask::SetSlotSizeEvent(DeviceHandle deviceHandle, ::nn::os::LightEventType* pEvent) NN_NOEXCEPT
{
    for (int i = 0; i < HidDeviceCountMax; i++)
    {
        if (m_Devices[i].accessor.GetDeviceHandle() == deviceHandle)
        {
            m_Devices[i].pEvent = pEvent;
        }
    }
}

Result BluetoothHidTask::SetSlotSize(DeviceHandle deviceHandle, int slotSize) NN_NOEXCEPT
{
    for (int i = 0; i < HidDeviceCountMax; i++)
    {
        if (m_Devices[i].accessor.GetDeviceHandle() == deviceHandle)
        {
            auto btmSlotMode = ConvertInternalSlotModeToBtm(slotSize);

            if (m_Devices[i].targetSlotMode == btmSlotMode ||
                m_Devices[i].isChangingSlotMode)
            {
                // 既に変更処理中
                NN_RESULT_SUCCESS;
            }

            ChangeSlotMode(m_Devices[i].address, btmSlotMode);
            SetSlotModeToBtm(true);
        }
    }

    NN_RESULT_SUCCESS;
}

int BluetoothHidTask::GetSlotSize(DeviceHandle deviceHandle) NN_NOEXCEPT
{
    for (int i = 0; i < HidDeviceCountMax; i++)
    {
        if (m_Devices[i].accessor.GetDeviceHandle() == deviceHandle)
        {
            if (m_Devices[i].isChangingSlotMode)
            {
                // 変更中は -1 を返す
                return -1;
            }

            return ConvertBtmSlotModeToInternal(m_Devices[i].currentSlotMode);
        }
    }

    return 0;
}

int BluetoothHidTask::GetTargetSlotSize(DeviceHandle deviceHandle) NN_NOEXCEPT
{
    for (int i = 0; i < HidDeviceCountMax; i++)
    {
        if (m_Devices[i].accessor.GetDeviceHandle() == deviceHandle)
        {
            return ConvertBtmSlotModeToInternal(m_Devices[i].targetSlotMode);
        }
    }

    return 0;
}

bool BluetoothHidTask::IsChangingSlotSize(DeviceHandle deviceHandle) NN_NOEXCEPT
{
    for (int i = 0; i < HidDeviceCountMax; i++)
    {
        if (m_Devices[i].accessor.GetDeviceHandle() == deviceHandle)
        {
            return m_Devices[i].isChangingSlotMode;
        }
    }

    return false;
}

void BluetoothHidTask::Suspend() NN_NOEXCEPT
{
    if (m_IsSuspended == true)
    {
        return;
    }

    GetTaskManager().UnregisterPeriodicTask(this);
    GetTaskManager().UnregisterEvent(&m_HidEvent);

    m_IsSuspended = true;
}


void BluetoothHidTask::Resume() NN_NOEXCEPT
{
    if (m_IsSuspended == false)
    {
        return;
    }

    // Bluetooth のデバイスリストを削除
    for (auto& device : m_Devices)
    {
        if (device.accessor.IsActivated() == true)
        {
            ClearBluetoothAddress(&device.address);
            device.accessor.Deactivate();
        }
    }
    // デバイスリストの更新を通知
    nn::os::SignalLightEvent(m_pUpdatedEvent);

    GetTaskManager().RegisterEvent(&m_HidEvent, this);
    GetTaskManager().RegisterPeriodicTask(this);

    m_IsSuspended = false;
}

int BluetoothHidTask::GetMaxLinks() NN_NOEXCEPT
{
    return m_LinkCountMax;
}

}}} // namespace nn::xcd::detail
