﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/bluetooth/bluetooth_LeTypes.h>
#include <nn/btm/btm_Api.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_Thread.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/xcd/xcd_ResultForPrivate.h>
#include <nn/xcd/detail/xcd_Log.h>

#include "xcd_BleHidAccessorImpl-hardware.nx.h"

namespace nn { namespace xcd { namespace detail {

namespace {

}

BleHidAccessorImpl::BleHidAccessorImpl() NN_NOEXCEPT
    : m_ConnectionHandle(nn::bluetooth::BleInvalidConnectionHandle)
    , m_pInputReportParserFunc(nullptr)
    , m_pInputReportParserArg(nullptr)
    , m_pListener(nullptr)
    , m_Activated(false)
    , m_ServiceNum(0)
    , m_ConfigureMtuValue(nn::bluetooth::BleMtuDefault)
    , m_IsGattClientActivated(false)
{
    m_RetryFlags.Reset();
}

BleHidAccessorImpl::~BleHidAccessorImpl() NN_NOEXCEPT
{
    // 何もしない
}

void BleHidAccessorImpl::Activate(uint32_t connectionHandle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_EQUAL(m_Activated, false);

    ::nn::os::InitializeTimerEvent(&m_BtmRetryEvent, ::nn::os::EventClearMode_AutoClear);
    m_RetryFlags.Reset();

    m_ConnectionHandle = connectionHandle;
    m_Activated = true;
}

void BleHidAccessorImpl::Deactivate() NN_NOEXCEPT
{
    StopInputReport();
    DeactivateCommandResponse();

    m_NhogClient.Deactivate();
    m_NbatClient.Deactivate();

    ::nn::os::FinalizeTimerEvent(&m_BtmRetryEvent);
    m_Activated = false;
    m_IsGattClientActivated = false;
    m_pListener = nullptr;
}

void BleHidAccessorImpl::StartInputReport(BleInputReportParserFunc func, void* pArg) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(func);
    NN_SDK_REQUIRES_NOT_NULL(pArg);

    m_pInputReportParserFunc = func;
    m_pInputReportParserArg = pArg;

    m_NhogClient.EnableInputReportNotification(nullptr, true);
}

void BleHidAccessorImpl::StopInputReport() NN_NOEXCEPT
{
    m_NhogClient.EnableInputReportNotification(nullptr, false);

    m_pInputReportParserFunc = nullptr;
    m_pInputReportParserArg = nullptr;
}

void BleHidAccessorImpl::ActivateCommandResponse(BleCommandResponseParserFunc func, void* pArg) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(func);
    NN_SDK_REQUIRES_NOT_NULL(pArg);

    m_pCommandResponseParserFunc = func;
    m_pCommandResponseParserArg = pArg;

    m_NhogClient.EnableCommandResponseIndication(nullptr, true);
}

void BleHidAccessorImpl::DeactivateCommandResponse() NN_NOEXCEPT
{
    m_NhogClient.EnableCommandResponseNotification(nullptr, false);

    m_pCommandResponseParserFunc = nullptr;
    m_pCommandResponseParserArg = nullptr;
}

void BleHidAccessorImpl::DetachDevice() NN_NOEXCEPT
{
    nn::btm::BleDisconnect(m_ConnectionHandle);
}

void BleHidAccessorImpl::SetListener(IBleCommandListener* pListener) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pListener);

    m_pListener = pListener;
}

void BleHidAccessorImpl::ConfigureBleMtu(uint16_t mtu) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_EQUAL(m_Activated, true);

    m_ConfigureMtuValue = mtu;
    SetRetryTimerIfFail(nn::btm::ConfigureBleMtu(m_ConnectionHandle, m_ConfigureMtuValue), &m_BtmRetryEvent, m_RetryFlags, BtmRetryFlag::ConfigureBleMtu::Index);
}

void BleHidAccessorImpl::GetDeviceAddress(::nn::bluetooth::Address* pAddress) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAddress);
    *pAddress = m_Address;
}

BleNhogClient* BleHidAccessorImpl::GetBleNhogClient() NN_NOEXCEPT
{
    if (m_NhogClient.IsActivated())
    {
        return &m_NhogClient;
    }

    return nullptr;
}

BleNbatClient* BleHidAccessorImpl::GetBleNbatClient() NN_NOEXCEPT
{
    if (m_NbatClient.IsActivated())
    {
        return &m_NbatClient;
    }

    return nullptr;
}

void BleHidAccessorImpl::SetDeviceAddress(const ::nn::bluetooth::Address& address) NN_NOEXCEPT
{
    m_Address = address;
}

void BleHidAccessorImpl::EnumerateGattServices() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_EQUAL(m_Activated, true);

    m_ServiceNum = nn::btm::GetGattServices(m_Services, nn::bluetooth::GattAttributeCountMaxClient, m_ConnectionHandle);
    NN_DETAIL_XCD_INFO("%s: ConnHandle=%d, ServiceNum=%d\n", __FUNCTION__, m_ConnectionHandle, m_ServiceNum);
}

bool BleHidAccessorImpl::ActivateGattClient() NN_NOEXCEPT
{
    // どれか1 つでもActivete に成功したら、HID が接続を奪う
    bool result = ActivateNhogClient();
    result |= ActivateNbatClient();
    if(result)
    {
        SetRetryTimerIfFail(nn::btm::BleOverrideConnection(m_ConnectionHandle), &m_BtmRetryEvent, m_RetryFlags, BtmRetryFlag::OverrideConnection::Index);
        m_IsGattClientActivated = true;
    }

    return result;
}

void BleHidAccessorImpl::HandleMtuConfigCallback() NN_NOEXCEPT
{
    if (m_pListener != nullptr)
    {
        m_pListener->NotifyConfigureMtuCompleted(nn::btm::GetBleMtu(m_ConnectionHandle));
    }
}

void BleHidAccessorImpl::HandleGattOperationCallback(nn::bluetooth::InfoFromLeGattOperationCallback* pResult) NN_NOEXCEPT
{
    // InputReport の Notification であればパーサに通知する
    if (pResult->operation == nn::bluetooth::GattOperationType_Notify && pResult->charcteristicUuid == InputReportCharacteristic.Uuid)
    {
        if (m_pInputReportParserFunc != nullptr)
        {
            m_pInputReportParserFunc(m_pInputReportParserArg, pResult->value, pResult->length);
        }
        return;
    }

    // CommandResponse の Notification/Indication であればパーサに通知する
    if ((pResult->operation == nn::bluetooth::GattOperationType_Notify || pResult->operation == nn::bluetooth::GattOperationType_Indicate)
        && pResult->charcteristicUuid == CommandResponseCharacteristic.Uuid)
    {
        if (m_pCommandResponseParserFunc != nullptr)
        {
            m_pCommandResponseParserFunc(m_pCommandResponseParserArg, pResult->value, pResult->length);
        }
        return;
    }

    // Nintendo Hid over Gatt でハンドルできる場合
    if (m_NhogClient.IsActivated() && m_NhogClient.GattOperationCompletedHandler(pResult))
    {
        return;
    }


    // Nintendo Battery Service でハンドルできる場合
    if (m_NbatClient.IsActivated() && m_NbatClient.GattOperationCompletedHandler(pResult))
    {
        return;
    }
}

void BleHidAccessorImpl::PeriodicOperation() NN_NOEXCEPT
{
    if (nn::os::TryWaitTimerEvent(&m_BtmRetryEvent))
    {
        HandleBtmRetryEvent();
    }

    // Nintend Hid over Gatt クライアントの処理
    if (m_NhogClient.IsActivated() == true)
    {
        m_NhogClient.Proceed();
    }

    // Nintend Battery Service クライアントの処理
    if (m_NbatClient.IsActivated() == true)
    {
        m_NbatClient.Proceed();
    }
}

bool BleHidAccessorImpl::ActivateNhogClient() NN_NOEXCEPT
{
    // すでに Activate されていたら何もしない
    if (m_NhogClient.IsActivated())
    {
        return true;
    }

    nn::btm::user::GattService service;
    if (GetGattService(&service, NhogService.Uuid))
    {
        m_NhogClient.Activate(m_ConnectionHandle, service);
        return true;
    }
    return false;
}

bool BleHidAccessorImpl::ActivateNbatClient() NN_NOEXCEPT
{
    // すでに Activate されていたら何もしない
    if (m_NbatClient.IsActivated())
    {
        return true;
    }

    nn::btm::user::GattService service;
    if (GetGattService(&service, NbatService.Uuid))
    {
        m_NbatClient.Activate(m_ConnectionHandle, service);
        return true;
    }
    return false;
}

bool BleHidAccessorImpl::GetGattService(nn::btm::user::GattService* pOutService, const nn::bluetooth::GattAttributeUuid& uuid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutService);
    return nn::btm::GetGattService(pOutService, m_ConnectionHandle, uuid);
}

void BleHidAccessorImpl::HandleBtmRetryEvent() NN_NOEXCEPT
{
    m_RetryFlags.Reset<BtmRetryFlag::Retrying>();

    if (m_RetryFlags.Test<BtmRetryFlag::OverrideConnection>())
    {
        m_RetryFlags.Reset<BtmRetryFlag::OverrideConnection>();
        SetRetryTimerIfFail(nn::btm::BleOverrideConnection(m_ConnectionHandle), &m_BtmRetryEvent, m_RetryFlags, BtmRetryFlag::OverrideConnection::Index);
    }
    if (m_RetryFlags.Test<BtmRetryFlag::ConfigureBleMtu>())
    {
        m_RetryFlags.Reset<BtmRetryFlag::ConfigureBleMtu>();
        SetRetryTimerIfFail(nn::btm::ConfigureBleMtu(m_ConnectionHandle, m_ConfigureMtuValue), &m_BtmRetryEvent, m_RetryFlags, BtmRetryFlag::ConfigureBleMtu::Index);
    }
}

}}} // namespace nn::xcd::detail
