﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/bluetooth/bluetooth_LeTypes.h>
#include <nn/btm/btm_Types.h>
#include <nn/os/os_ThreadTypes.h>
#include <nn/os/os_LightEventTypes.h>
#include <nn/os/os_SystemEventTypes.h>
#include <nn/os/os_MultipleWaitTypes.h>
#include <nn/xcd/xcd_BleDevice.h>
#include "xcd_BleHidAccessorImpl-hardware.nx.h"
#include "xcd_BleNhogClient-hardware.nx.h"
#include "xcd_BtmUtil-hardware.nx.h"
#include "xcd_LinkMonitorTypes.h"
#include "../xcd_IEventTask.h"
#include "../xcd_MultiWaitEvent.h"

namespace nn { namespace xcd { namespace detail{

class BleCentralTask final : public IEventTask
{
    NN_DISALLOW_MOVE(BleCentralTask);
    NN_DISALLOW_COPY(BleCentralTask);

private:
    //!< BTM からのスキャン結果更新イベント
    nn::os::SystemEventType m_ScanEvent;

    //!< BTM からの接続イベント
    nn::os::SystemEventType m_ConnectionEvent;

    //!< BTM からの MTU 設定完了イベント
    nn::os::SystemEventType m_MtuConfigEvent;

    //!< BTM からのサービス発見イベント
    nn::os::SystemEventType m_ServiceDiscoveryEvent;

    //!< BLE GATT オペレーション完了イベント
    MultiWaitSystemEvent m_GattOperationEvent;

    //!< デバイス接続監視の終了を通知するイベント
    nn::os::LightEventType* m_pUpdatedEvent;

    //!< BTM のリトライをするためのタイマー
    nn::os::TimerEventType m_BtmRetryEvent;

    //!< BTM のリトライで使用するフラグの集合
    BtmRetryFlagSet m_RetryFlags;

    //!< BLE デバイスのリスト
    struct BleDevice
    {
        BleHidAccessorImpl accessor;
        ::nn::bluetooth::Address address;
        bool pendingPairing;
    };
    BleDevice m_Devices[::nn::bluetooth::BleConnectionCountMaxClient];

    //!< ConnectionState のリスト
    struct ConnectionStateList
    {
        uint8_t deviceCount;
        nn::bluetooth::BleConnectionInfo state[::nn::bluetooth::BleConnectionCountMaxClient];  // 読み出す台数は bluetooth の最大接続数
    };

    //!< BleCentralTask で扱う GATT サービス UUID のリスト
    enum GattServiceIndex
    {
        GattServiceIndex_Nhog,
        GattServiceIndex_Nbat,

        GattServiceIndex_Max
    };
    static const nn::bluetooth::GattAttributeUuid GattServiceUuidList[GattServiceIndex_Max];

    //!< Palma の接続に用いる Advertise Packet のパラメーター
    nn::btm::user::BleAdvFilterForGeneral m_AdvertiseParameter;

    //!< サスペンド中かどうか
    bool m_IsSuspended;

    //!< 接続されているデバイス数
    int m_DeviceCount;

    //!< Palma の全デバイススキャンが有効かどうか
    bool m_AllowPalmaScanAll;

    //!< 登録済み Palma のスキャンが有効かどうか
    bool m_AllowPalmaScanRegistered;

public:
    BleCentralTask() NN_NOEXCEPT;
    virtual ~BleCentralTask() NN_NOEXCEPT NN_OVERRIDE;

    nn::Result Activate(nn::os::LightEventType* pUpdatedEvent) NN_NOEXCEPT;
    void Deactivate() NN_NOEXCEPT;

    //!< Eventが通知された際に呼ばれる関数
    virtual void EventFunction(const ::nn::os::MultiWaitHolderType* pMultiWaitHolder) NN_NOEXCEPT NN_OVERRIDE;

    //!< 定常的な処理を行う関数
    virtual void PeriodicEventFunction() NN_NOEXCEPT NN_OVERRIDE;

    //!< BLE の処理を Suspend する
    void Suspend() NN_NOEXCEPT;

    //!< BLE の処理を Resume する
    void Resume() NN_NOEXCEPT;

    //!< 接続済みのデバイスのハンドルを取得する
    size_t GetConnectionHandleList(uint32_t* pOutList, size_t deviceCount) NN_NOEXCEPT;

    //!< ハンドルから BleHidAccessor を取得する
    BleHidAccessor* GetBleHidAccessor(uint32_t connectionHandle) NN_NOEXCEPT;

    //!< 全ての Palma のスキャンの有効無効を切り替える
    void SetIsPalmaAllConnectable(bool connectable) NN_NOEXCEPT;

    //!< 登録済みの Palma のスキャンの有効無効を切り替える
    void SetIsPalmaPairedConnectable(bool connectable) NN_NOEXCEPT;

    //!< 指定されたデバイスとペアリングする
    Result PairPalma(const ::nn::bluetooth::Address& address) NN_NOEXCEPT;

private:
    //!< BTM からのスキャン結果更新イベントをハンドリングする
    void HandleScanEvent() NN_NOEXCEPT;

    //!< BTM からの接続イベントをハンドリングする
    void HandleConnectionEvent() NN_NOEXCEPT;

    //!< BTM からの MTU 設定完了イベントをハンドリングする
    void HandleMtuConfigEvent() NN_NOEXCEPT;

    //!< BTM からのサービス発見イベントをハンドリングする
    void HandleServiceDiscoveryEvent() NN_NOEXCEPT;

    //!< GATT オペレーション完了イベントをハンドリングする
    void HandleGattOperationEvent() NN_NOEXCEPT;

    //!< 切断されたデバイスを探索して、切断されていれば Remove する
    void HandleDetachedDevices(const ConnectionStateList* pList) NN_NOEXCEPT;

    //!< 接続されたデバイスを探索して、接続されていれば Register する
    void HandleAttachedDevices(const ConnectionStateList* pList) NN_NOEXCEPT;

    //!< 接続されたデバイスを登録します
    bool RegisterDevice(const nn::bluetooth::Address& address, uint32_t connectionHandle) NN_NOEXCEPT;

    //!< 切断されたデバイスを削除します
    bool RemoveDevice(uint32_t connectionHandle) NN_NOEXCEPT;

    //!< GATT クライアントの定常的な処理を行う関数
    void PeriodicGattClientOperation() NN_NOEXCEPT;

    //!< ConnectionHandle から BleDevice を取得します
    BleDevice* GetBleDeviceFromConnectionHandle(uint32_t connectionHandle) NN_NOEXCEPT;

    //!< スキャンの状態の更新を行う
    void UpdateScanStateAll() NN_NOEXCEPT;

    //!< 全ての Palma のスキャンの有効無効を切り替える
    void UpdateGeneralScanState() NN_NOEXCEPT;

    //!< 登録済みの Palma のスキャンの有効無効を切り替える
    void UpdateAutoConnectState() NN_NOEXCEPT;

    //!< 登録処理が必要な Palma の登録を行う
    void TryPairPalma() NN_NOEXCEPT;

    //!< BTM のリトライをハンドリングする
    void HandleBtmRetryEvent() NN_NOEXCEPT;
};

}}} // namespace nn::xcd::detail
