﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <memory>
#include <mutex>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Windows.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/os/os_Thread.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/xcd/xcd_Device.h>
#include <nn/xcd/xcd_DeviceState.h>
#include "xcd_AardvarkAccessor-os.win32.h"
#include "xcd_TeraCommon.h"
#include "../xcd_SerialFlashMap.h"
#include "../xcd_CommandTypes.h"
#include "../xcd_ReportTypes.h"

namespace nn { namespace xcd { namespace detail {

// XXX: 適当な値
const TCHAR * const AardvarkDevicePath = _T("aardvark_usb_device");

namespace {

// GPIO 切り替え時の適当なウェイト
const auto GpioWait = nn::TimeSpan::FromMicroSeconds(10);

// リセットパルス期間 (300ns 以上)
const auto ResetPulseTime = nn::TimeSpan::FromNanoSeconds(350);

// リセット後のウェイト (74.5ms 以上。80ms 程度推奨)
const auto AfterResetWait = nn::TimeSpan::FromMilliSeconds(80);

// SPI 送受信間隔
const auto TransferInterval = nn::TimeSpan::FromMilliSeconds(15);

// 通信サイズ反映のためのウェイト
// (BT 環境での通信間隔を再現するために使用)
const auto SpiTimingAdjustWait = nn::TimeSpan::FromMicroSeconds(200);

// 通常の SPI ビットレート
//  ※ 製品仕様では 12MHz だが、Aardvark は 8MHz が上限
//    (BT の 15msec インターバルが支配的なので、これの影響はほとんどないはず)
const int NormalSpiBitrateKhz = 8000;

// FW アップデートで使用する SPI ビットレート
const int UpdateSpiBitrateKhz = 8000;

// SPI 通信データサイズ (先頭の mode バイトを除いたサイズ)
const size_t SpiTransferBytes = 359;

/*
 * ACK 返送を行うコマンドか判定
 */
bool IsAckRequiredCommand(uint8_t commandId) NN_NOEXCEPT
{
    if (commandId == Command_PairingOut.Id ||
        commandId == Command_SetDataFormat.Id ||
        commandId == Command_Page.Id ||
        commandId == Command_Reset.Id ||
        commandId == Command_SerialFlashWrite.Id ||
        commandId == Command_SerialFlashSectorErase.Id ||
        commandId == Command_McuResume.Id ||
        commandId == Command_McuReset.Id ||
        commandId == Command_SetIndicatorLed.Id ||
        commandId == Command_SensorSleep.Id ||
        commandId == Command_SensorConfig.Id ||
        commandId == Command_SensorWrite.Id ||
        commandId == Command_MotorEnable.Id ||
        commandId == Command_Shipment.Id)
    {
        return true;
    }
    else
    {
        return false;
    }
}

}  // anonymous

#if defined(NN_BUILD_CONFIG_COMPILER_VC)
    // C4351: 配列の初期化が規定値で行われる旨の警告を抑止
    #pragma warning(push)
    #pragma warning(disable: 4351)
#endif

AardvarkAccessor::AardvarkAccessor() NN_NOEXCEPT :
    m_AardvarkHandle(0),
    m_PeriodicEvent(),
    m_BufferMutex(false),
    m_DummyHidBuffer(),
    m_DummyInSize(0),
    m_DummyOutSize(0),
    m_OutputBuffer(),
    m_SendSize(0),
    m_SampleNumber(0)
{
    // 何もしない
}

#if defined(NN_BUILD_CONFIG_COMPILER_VC)
    #pragma warning(pop)
#endif

AardvarkAccessor::~AardvarkAccessor() NN_NOEXCEPT
{
    // 何もしない
}

void AardvarkAccessor::Activate(DeviceHandle deviceHandle, aardvark::DeviceHandle aardvarkHandle) NN_NOEXCEPT
{
    m_AardvarkHandle = aardvarkHandle;
    NN_SDK_ASSERT_GREATER(m_AardvarkHandle, 0);

    SetupDummyInput(DummyType::Normal);
    m_SendSize     = 0;
    m_SampleNumber = 0;

    // Aardvak 使用時は HidHandle を使用しないため、0を入れておく
    Win32HidAccessor::Activate(deviceHandle, 0);
}

void AardvarkAccessor::Deactivate() NN_NOEXCEPT
{
    Win32HidAccessor::Deactivate();
    FinalizeDevice();
}

void AardvarkAccessor::ThreadFunctionImpl() NN_NOEXCEPT
{
    InitializeDevice();
    SetupSpi();

    nn::os::InitializeTimerEvent(&m_PeriodicEvent, nn::os::EventClearMode_ManualClear);

    nn::os::MultiWaitHolderType periodicHolder;
    nn::os::MultiWaitHolderType terminateHolder;
    nn::os::MultiWaitType multiWait;
    nn::os::InitializeMultiWaitHolder(&periodicHolder, &m_PeriodicEvent);
    nn::os::InitializeMultiWaitHolder(&terminateHolder, &m_TerminateEvent);
    nn::os::InitializeMultiWait(&multiWait);
    nn::os::LinkMultiWaitHolder(&multiWait, &periodicHolder);
    nn::os::LinkMultiWaitHolder(&multiWait, &terminateHolder);

    nn::os::StartPeriodicTimerEvent(&m_PeriodicEvent, TransferInterval, TransferInterval);

    while (NN_STATIC_CONDITION(true))
    {
        auto* pHolder = nn::os::WaitAny(&multiWait);

        if (pHolder == &terminateHolder)
        {
            break;
        }

        NN_SDK_ASSERT_EQUAL(pHolder, &periodicHolder);
        nn::os::ClearTimerEvent(&m_PeriodicEvent);

        std::lock_guard<nn::os::Mutex> lock(m_BufferMutex);

        m_ReceivedSize = 0;
        if (m_SendSize > 0)
        {
            int result;

            // 通信サイズを MCU に伝えるために 1 バイト送信
            result = detail::aardvark::WriteSpi(
                reinterpret_cast<char*>(&m_Buffer[m_DummyInSize]),
                reinterpret_cast<char*>(&m_OutputBuffer[m_DummyOutSize]),
                1,
                m_AardvarkHandle);
            if (result < 1)
            {
                NN_SDK_LOG("[xcd: aardvark] SPI transfer1 failed (%d)\n", result);
                continue;
            }

            // 通信サイズを決定
            const int BtModeCountMax = sizeof(BluetoothTransferSizeList) / sizeof(BluetoothTransferSizeList[0]);
            int btMode = (m_OutputBuffer[m_DummyOutSize] >> 4) & 0x0F;
            NN_ABORT_UNLESS_LESS(btMode, BtModeCountMax);

            const auto& TransferSize = BluetoothTransferSizeList[btMode];

            // 通信サイズを MCU が認識するために 200usec 待つ (仕様)
            nn::os::SleepThread(SpiTimingAdjustWait);

            // 残りを送受信
            result = detail::aardvark::WriteSpi(
                reinterpret_cast<char*>(&m_Buffer[m_DummyInSize]),
                reinterpret_cast<char*>(&m_OutputBuffer[m_DummyOutSize + 1]),
                SpiTransferBytes,  // CC2.0 以降、通信データサイズは固定
                m_AardvarkHandle);
            std::memset(m_OutputBuffer, 0, TransferSize.send);
            m_SendSize = 0;

            if (result < 0)
            {
                NN_SDK_LOG("[xcd: aardvark] SPI transfer2 failed (%d)\n", result);
                continue;
            }
            m_ReceivedSize += TransferSize.receive;
        }

        if (m_DummyInSize > 0)
        {
            std::memcpy(m_Buffer, m_DummyHidBuffer, m_DummyInSize);
            m_ReceivedSize += m_DummyInSize;
            m_DummyInSize = 0;
        }

        if (m_pInputReportParserFunc != nullptr)
        {
            m_pInputReportParserFunc(m_pInputReportParserArg, m_Buffer, m_ReceivedSize);
        }
    }

    nn::os::StopTimerEvent(&m_PeriodicEvent);

    nn::os::UnlinkMultiWaitHolder(&terminateHolder);
    nn::os::UnlinkMultiWaitHolder(&periodicHolder);
    nn::os::FinalizeMultiWaitHolder(&terminateHolder);
    nn::os::FinalizeMultiWaitHolder(&periodicHolder);
    nn::os::FinalizeMultiWait(&multiWait);

    nn::os::FinalizeTimerEvent(&m_PeriodicEvent);
}

void AardvarkAccessor::SetupDummyInput(DummyType type, uint8_t commandId, uint32_t address) NN_NOEXCEPT
{
    std::memset(m_DummyHidBuffer, 0, sizeof(m_DummyHidBuffer));

    // HID Sample number
    m_DummyHidBuffer[InputReportByte_SampleNumber] = m_SampleNumber;
    m_SampleNumber = static_cast<uint8_t>((m_SampleNumber + 1) & 0xFF);

    // 電池残量は常に最大を装う
    m_DummyHidBuffer[InputReportByte_Status] =
        static_cast<uint8_t>(nn::xcd::BatteryLevel_High << InputReportBit_BatteryLeast);

    switch (type)
    {
    case DummyType::Normal:
        {
            // 定常通信のフリ (実質何もしない)
            m_DummyHidBuffer[ReportByte_ReportId] = Report_BasicIn::Id;

            m_DummyInSize  = CommandInReportByte_Payload;
            m_DummyOutSize = CommandOutReportByte_Payload;
        }
        break;

    case DummyType::Ack:
        {
            // ACK を返す
            m_DummyHidBuffer[ReportByte_ReportId] = Report_CommandIn::Id;

            auto* pPayload = &m_DummyHidBuffer[CommandInReportByte_Payload];
            std::memset(pPayload, 0, CommandInReportSize_Payload);
            pPayload[CommandByte_CommandId]  = Command_Ack.Id;
            pPayload[CommandInByte_OutputId] = commandId;
            pPayload[AckByte_Result] = AckResultType_Success;

            m_DummyInSize  = CommandInReportByte_Payload + CommandInReportSize_Payload;
            m_DummyOutSize = CommandOutReportByte_Payload;
        }
        break;

    case DummyType::GetDeviceInfo:
        {
            // UKYO を装う。MAC ADDR は 0xA5 0x5A の繰り返し
            m_DummyHidBuffer[ReportByte_ReportId] = Report_CommandIn::Id;

            auto* pPayload = &m_DummyHidBuffer[CommandInReportByte_Payload];
            std::memset(pPayload, 0, CommandInReportSize_Payload);
            pPayload[CommandByte_CommandId]     = Command_DeviceInfo.Id;
            pPayload[CommandInByte_OutputId]    = Command_GetDeviceInfo.Id;
            pPayload[DeviceInfoByte_DeviceType] = DeviceInfoDeviceType_RightJoy;
            for (int i = DeviceInfoByte_BdAddr; i < DeviceInfoByte_SensorType; ++i)
            {
                pPayload[i] = (i & 1) ? 0x5A : 0xA5;
            }
            m_DummyInSize  = CommandInReportByte_Payload + CommandInReportSize_Payload;
            m_DummyOutSize = CommandOutReportByte_Payload;
        }
        break;

    case DummyType::LrButtonElapsedTime:
        {
            // LR ボタンは押下されていないものとする
            m_DummyHidBuffer[ReportByte_ReportId] = Report_CommandIn::Id;

            auto* pPayload = &m_DummyHidBuffer[CommandInReportByte_Payload];
            std::memset(pPayload, 0, CommandInReportSize_Payload);
            pPayload[CommandByte_CommandId]  = Command_LRButtonElapsedTime.Id;
            pPayload[CommandInByte_OutputId] = Command_LRButtonDetection.Id;

            // ElapsedTime は all 0
            pPayload[LRButtonElapsedTimeByte_L  + 0] = 0;
            pPayload[LRButtonElapsedTimeByte_L  + 1] = 0;
            pPayload[LRButtonElapsedTimeByte_R  + 0] = 0;
            pPayload[LRButtonElapsedTimeByte_R  + 1] = 0;
            pPayload[LRButtonElapsedTimeByte_ZL + 0] = 0;
            pPayload[LRButtonElapsedTimeByte_ZL + 1] = 0;
            pPayload[LRButtonElapsedTimeByte_ZR + 0] = 0;
            pPayload[LRButtonElapsedTimeByte_ZR + 1] = 0;

            m_DummyInSize  = CommandInReportByte_Payload + CommandInReportSize_Payload;
            m_DummyOutSize = CommandOutReportByte_Payload;
        }
        break;

    case DummyType::SerialFlashRead:
        {
            // シリアルフラッシュ読み込みを装う。アドレス以外の情報は All 0
            m_DummyHidBuffer[ReportByte_ReportId] = Report_CommandIn::Id;

            auto* pPayload = &m_DummyHidBuffer[CommandInReportByte_Payload];
            std::memset(pPayload, 0, CommandInReportSize_Payload);
            pPayload[CommandByte_CommandId]  = Command_SerialFlashData.Id;
            pPayload[CommandInByte_OutputId] = Command_SerialFlashRead.Id;
            pPayload[SerialFlashDataByte_Address + 0] = static_cast<uint8_t>(address & 0xFF);
            pPayload[SerialFlashDataByte_Address + 1] = static_cast<uint8_t>((address >> 8) & 0xFF);
            pPayload[SerialFlashDataByte_Address + 2] = static_cast<uint8_t>((address >> 16) & 0xFF);
            pPayload[SerialFlashDataByte_Address + 3] = static_cast<uint8_t>((address >> 24) & 0xFF);
            m_DummyInSize  = CommandInReportByte_Payload + CommandInReportSize_Payload;
            m_DummyOutSize = CommandOutReportByte_Payload;
        }
        break;

    case DummyType::McuIn:
        {
            // McuIn のフリをする
            m_DummyHidBuffer[ReportByte_ReportId] = Report_McuIn::Id;

            m_DummyInSize  = McuInReportByte_Payload;
            m_DummyOutSize = McuOutReportByte_Payload;
        }
        break;

    case DummyType::McuRead:
        {
            // McuRead のフリをする
            m_DummyHidBuffer[ReportByte_ReportId] = Report_CommandIn::Id;

            auto* pPayload = &m_DummyHidBuffer[CommandInReportByte_Payload];
            std::memset(pPayload, 0, CommandInReportSize_Payload);
            pPayload[CommandByte_CommandId]  = Command_McuData.Id;
            pPayload[CommandInByte_OutputId] = Command_McuWrite.Id;

            // ペイロードのデータは (現在は) 上位レイヤーで無視しているので、何も入れなくて良い

            m_DummyInSize  = CommandInReportByte_Payload + CommandInReportSize_Payload;
            m_DummyOutSize = CommandOutReportByte_Payload + 1;  // cmd_id 分の 1 バイトをスキップする
        }
        break;

    default: NN_UNEXPECTED_DEFAULT;
    }
} // NOLINT(impl/function_size)

void AardvarkAccessor::SetupDummyInput(DummyType type, uint8_t commandId) NN_NOEXCEPT
{
    SetupDummyInput(type, commandId, 0x00);
}

void AardvarkAccessor::SetupDummyInput(DummyType type) NN_NOEXCEPT
{
    SetupDummyInput(type, 0, 0x00);
}

void AardvarkAccessor::InitializeDevice() NN_NOEXCEPT
{
    // 操作対象の GPIO のみを出力にする
    detail::aardvark::SetMode(m_AardvarkHandle, detail::aardvark::Mode_SpiAndGpio);
    detail::aardvark::SetGpioDirection(
        m_AardvarkHandle,
        detail::aardvark::TeraGpioPins_Reset | detail::aardvark::TeraGpioPins_SystemBoot);
}

void AardvarkAccessor::FinalizeDevice() NN_NOEXCEPT
{
    // GPIO を入力に戻す
    detail::aardvark::SetGpioDirection(
        m_AardvarkHandle,
        detail::aardvark::TeraGpioPins_None);
    detail::aardvark::Close(m_AardvarkHandle);
    m_AardvarkHandle = 0;
}

void AardvarkAccessor::ResetMcu(bool isSystemBoot) NN_NOEXCEPT
{
    auto bootPin = isSystemBoot ?
        detail::aardvark::TeraGpioPins_SystemBoot :
        detail::aardvark::TeraGpioPins_None;

    // ブートモードを設定
    detail::aardvark::SetGpioPins(
        m_AardvarkHandle,
        detail::aardvark::TeraGpioPins_Reset | bootPin);
    nn::os::SleepThread(GpioWait);

    // リセットをかける
    detail::aardvark::SetGpioPins(
        m_AardvarkHandle,
        detail::aardvark::TeraGpioPins_None | bootPin);

    // リセットがかかるまで待つ
    nn::os::SleepThread(ResetPulseTime);

    // リセットを解除
    detail::aardvark::SetGpioPins(
        m_AardvarkHandle,
        detail::aardvark::TeraGpioPins_Reset | bootPin);
}

void AardvarkAccessor::SetupSpi() NN_NOEXCEPT
{
    // SPI モードは固定
    detail::aardvark::SpiConfig config = {};
    config.polarity = detail::aardvark::SpiPolarity_ActiveLow;
    config.phase    = detail::aardvark::SpiPhase_SampleSetup;
    config.bitOrder = detail::aardvark::SpiBitOrder_MsbFirst;
    detail::aardvark::ConfigureSpi(m_AardvarkHandle, config);
    detail::aardvark::SetSpiBitrate(m_AardvarkHandle, NormalSpiBitrateKhz);
    detail::aardvark::SetSpiMode(m_AardvarkHandle, detail::aardvark::SpiMode_Master);
    detail::aardvark::SetSpiMasterSsPolarity(m_AardvarkHandle, detail::aardvark::SpiMasterSsPolarity_ActiveLow);
}

//!< コントローラに送るOutputReportをセットする
void AardvarkAccessor::SetOutputReport(const uint8_t* pBuffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);
    NN_SDK_REQUIRES(IsActivated());
    NN_SDK_REQUIRES(size < sizeof(m_OutputBuffer) + 1);

    if (size > 0)
    {
        std::lock_guard<nn::os::Mutex> lock(m_BufferMutex);

        // Report_XXX::Id や Command_XXX.Id は constexpr ではないので、switch は使えない
        if (pBuffer[0] == Report_CommandOut::Id)
        {
            // UKYO (BT MCU) との通信
            m_SendSize = 0;
            const int CommandIdOffset = 0x0A;
            auto commandId = pBuffer[CommandIdOffset];
            if (commandId == Command_GetDeviceInfo.Id)
            {
                SetupDummyInput(DummyType::GetDeviceInfo);
                return;
            }
            else if (commandId == Command_LRButtonDetection.Id)
            {
                SetupDummyInput(DummyType::LrButtonElapsedTime);
                return;
            }
            else if (commandId == Command_SerialFlashRead.Id)
            {
                // FlashRead するアドレスを取得
                uint32_t address = pBuffer[CommandIdOffset + SerialFlashReadByte_Address + 0];
                address |= pBuffer[CommandIdOffset + SerialFlashReadByte_Address + 1] << 8;
                address |= pBuffer[CommandIdOffset + SerialFlashReadByte_Address + 2] << 16;
                address |= pBuffer[CommandIdOffset + SerialFlashReadByte_Address + 3] << 24;
                SetupDummyInput(DummyType::SerialFlashRead, commandId, address);
                return;
            }
            else if (commandId == Command_McuWrite.Id)
            {
                SetupDummyInput(DummyType::McuRead);

                // McuOut と同様のデータを送る
                SetOutputDataForMcu(pBuffer, size);
                return;
            }
            else if (commandId == Command_McuReset.Id)
            {
                ResetMcu(false);
                SetupDummyInput(DummyType::Ack, commandId);
                return;
            }
            else if (commandId == Command_McuResume.Id)
            {
                auto mode = pBuffer[CommandIdOffset + 1];
                switch (mode)
                {
                case 0:  // Suspend
                    SetStandbyCommandForMcu();
                    break;
                case 1:  // Resume
                    ResetMcu(false);
                    break;
                default:
                    NN_UNEXPECTED_DEFAULT;
                }

                SetupDummyInput(DummyType::Ack, commandId);
                return;
            }
            else if (IsAckRequiredCommand(commandId))
            {
                // ACK 系のコマンドは何もせず ACK 応答する
                SetupDummyInput(DummyType::Ack, commandId);
                return;
            }

            NN_SDK_LOG("[xcd: aardvark] Unsupported command: 0x%02X\n", commandId);
        }
        else if (pBuffer[0] == Report_McuOut::Id)
        {
            // Tera との通信
            SetupDummyInput(DummyType::McuIn);
            SetOutputDataForMcu(pBuffer, size);
            return;
        }

        // 実質何もしない
        SetupDummyInput(DummyType::Normal);
        m_SendSize = 0;
    }
}

void AardvarkAccessor::SetOutputDataForMcu(const uint8_t* pBuffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);

    std::memcpy(m_OutputBuffer, pBuffer, size);
    if (size < sizeof(m_OutputBuffer))
    {
        std::memset(m_OutputBuffer + size, 0, sizeof(m_OutputBuffer) - size);
    }
    m_SendSize = size;
}

void AardvarkAccessor::SetStandbyCommandForMcu() NN_NOEXCEPT
{
    // TERA_CONTROL(Standby) を生成
    uint8_t commandBuffer[McuWriteSize_Data] = {};
    commandBuffer[McuCommonOutputOffsetByte_ResultCode] =
        CreateMcuPacketModeByte(
            BluetoothMode_McuWrite,
            CommandProtocol_Common);
    commandBuffer[McuCommonOutputOffsetByte_Command] = CommonCommandId_TeraControl;
    commandBuffer[McuCommonOutputOffsetByte_State]   = InternalMcuState_Standby;
    commandBuffer[McuWriteSize_Data - 1] =
        CalcCrc8(commandBuffer + 1, McuWriteSize_Data - 2);

    SetOutputDataForMcu(commandBuffer, sizeof(commandBuffer));
}

size_t AardvarkAccessor::GetInputReport(uint8_t* pOutValue, size_t size) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_BufferMutex);

    return Win32HidAccessor::GetInputReport(pOutValue, size);
}

}}} // namespace nn::xcd::detail
