﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>

#include <Windows.h>

#include "aardvark_Types-os.win32.h"

namespace nn { namespace xcd { namespace detail { namespace aardvark {

/**
 * @brief   Aardvark ドライバの DLL に関連する情報を扱うクラス。
 */
class Driver
{
    NN_DISALLOW_COPY(Driver);
    NN_DISALLOW_MOVE(Driver);

public:
    static Driver& GetInstance() NN_NOEXCEPT
    {
        static Driver s_Instance;
        return s_Instance;
    }

    bool Initialize() NN_NOEXCEPT;
    void Finalize() NN_NOEXCEPT;
    bool IsInitialized() NN_NOEXCEPT
    {
        return m_DllHandle != nullptr;
    }
    void AssertInitialized() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(IsInitialized(), "Aardvark is not initialized");
    }

    DeviceHandle Open(int portNumber) NN_NOEXCEPT;
    int Close(DeviceHandle handle) NN_NOEXCEPT;
    int GetConfig(DeviceHandle handle) NN_NOEXCEPT;
    int SetConfig(DeviceHandle handle, Mode mode) NN_NOEXCEPT;

    // SPI
    int ConfigureSpi(DeviceHandle handle, const SpiConfig& config) NN_NOEXCEPT;
    int SetSpiMode(DeviceHandle handle, SpiMode mode) NN_NOEXCEPT;

    // For SPI master
    int GetSpiBitrate(DeviceHandle handle) NN_NOEXCEPT;
    int SetSpiBitrate(DeviceHandle handle, int bitrateKhz) NN_NOEXCEPT;
    int SetSpiMasterSsPolarity(DeviceHandle handle, SpiMasterSsPolarity polarity) NN_NOEXCEPT;
    int WriteSpi(
        char* pOutReceivedData,
        size_t receiveBytes,
        const char* pWriteData,
        size_t writeDataBytes,
        DeviceHandle handle) NN_NOEXCEPT;

    // For SPI slave
    int ReadSpi(size_t* pOutDoneBytes, char* pOutReadData, size_t readDataBytes, DeviceHandle handle) NN_NOEXCEPT;

    // GPIO
    int SetGpioDirection(DeviceHandle handle, nn::Bit8 directionMask) NN_NOEXCEPT;
    int SetGpioPullup(DeviceHandle handle, nn::Bit8 pullupMask) NN_NOEXCEPT;
    int GetGpioPins(DeviceHandle handle) NN_NOEXCEPT;
    int SetGpioPins(DeviceHandle handle, nn::Bit8 pinStatus) NN_NOEXCEPT;

private:
    typedef DeviceHandle (*OpenFuncType)(int);
    typedef int (*CloseFuncType)(DeviceHandle);
    typedef int (*ConfigFuncType)(DeviceHandle, Mode);

    typedef int (*SpiBitrateFuncType)(DeviceHandle, int);
    typedef int (*SpiConfigFuncType)(DeviceHandle, int, int, int);
    typedef int (*SetSpiSlaveFuncType)(DeviceHandle);
    typedef int (*SetSpiMasterSsPolarityFuncType)(DeviceHandle, int);
    typedef int (*ReadSpiFuncType)(DeviceHandle, uint16_t, char*);
    typedef int (*WriteSpiFuncType)(DeviceHandle, uint16_t, const char*, uint16_t, char*);

    typedef int (*SetGpioDirectionFuncType)(DeviceHandle, nn::Bit8);
    typedef int (*SetGpioPullupFuncType)(DeviceHandle, nn::Bit8);
    typedef int (*GetGpioFuncType)(DeviceHandle);
    typedef int (*SetGpioFuncType)(DeviceHandle, nn::Bit8);

    struct DllFunctions
    {
        // Port
        OpenFuncType                    open;
        CloseFuncType                   close;
        ConfigFuncType                  configure;

        // SPI
        SpiBitrateFuncType              spiBitrate;
        SpiConfigFuncType               spiConfigure;
        SetSpiSlaveFuncType             enableSpiSlave;
        SetSpiSlaveFuncType             disableSpiSlave;
        SetSpiMasterSsPolarityFuncType  setMasterSsPolarity;
        ReadSpiFuncType                 readSpi;
        WriteSpiFuncType                writeSpi;

        // GPIO
        SetGpioDirectionFuncType        setGpioDirection;
        SetGpioPullupFuncType           setGpioPullup;
        GetGpioFuncType                 getGpio;
        SetGpioFuncType                 setGpio;
    };

private:
    // Singleton
    Driver() NN_NOEXCEPT :
        m_DllHandle(nullptr),
        m_Functions()
    {}

private:
    HINSTANCE    m_DllHandle;
    DllFunctions m_Functions;
};

}}}}
