﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Abort.h>
#include <nn/wlan/wlan_ScanResultReader.h>
#include <nn/wlan/wlan_BssDescriptionReader.h>

namespace nn {
namespace wlan {

/*
 * BssDescriptionReaderBase
 */

const Bit8 BeaconDescriptionReader::WpaOui[MacAddress::OuiSize] = {0x00, 0x50, 0xF2};

size_t BssDescriptionReaderBase::GetLength() const NN_NOEXCEPT
{
    const BssInfo* pHeader = reinterpret_cast<const BssInfo*>(m_pBuffer);
    if( pHeader != NULL )
    {
        return pHeader->bssSize;
    }
    else
    {
        return 0;
    }
}

int16_t  BssDescriptionReaderBase::GetRssi() const NN_NOEXCEPT
{
    const BssInfo* pHeader = reinterpret_cast<const BssInfo*>(m_pBuffer);
    if( pHeader != NULL )
    {
        return pHeader->rssi;
    }
    else
    {
        return 0;
    }
}

LinkLevel  BssDescriptionReaderBase::GetLinkLevel() const NN_NOEXCEPT
{
    const BssInfo* pHeader = reinterpret_cast<const BssInfo*>(m_pBuffer);
    if( pHeader != NULL )
    {
        return static_cast<LinkLevel>(pHeader->linkLevel);
    }
    else
    {
        return LinkLevel_0;
    }
}

int16_t BssDescriptionReaderBase::GetChannel() const NN_NOEXCEPT
{
    const BssInfo* pHeader = reinterpret_cast<const BssInfo*>(m_pBuffer);
    if( pHeader != NULL )
    {
        return pHeader->channel;
    }
    else
    {
        return 0;
    }
}

MacAddress BssDescriptionReaderBase::GetBssid() const NN_NOEXCEPT
{
    const BssInfo* pHeader = reinterpret_cast<const BssInfo*>(m_pBuffer);
    if( pHeader != NULL )
    {
        return MacAddress(pHeader->bssid);
    }
    else
    {
        return nn::wlan::MacAddress::CreateZeroMacAddress();
    }
}


/*
 * BeaconDescriptionReader
 */

BeaconDescriptionReader::BeaconDescriptionReader(const BssDescription* pBuffer) NN_NOEXCEPT
: BssDescriptionReaderBase(pBuffer)
{
    const BssInfo* pHeader = reinterpret_cast<const BssInfo*>(pBuffer);
    if(pHeader == NULL)
    {
        return;
    }
}

//! Beacon情報要素の数を取得します。
uint32_t BeaconDescriptionReader::GetIeCount() const NN_NOEXCEPT
{
    const BssInfo* pHeader = reinterpret_cast<const BssInfo*>(m_pBuffer);
    if(pHeader != NULL)
    {
        return pHeader->ieCount;
    }
    else
    {
        return 0;
    }
}

//! SSIDを取得します。Ssidクラスのコピーが渡されます。
Ssid BeaconDescriptionReader::GetSsid() const NN_NOEXCEPT
{
    const BssInfo* pHeader = reinterpret_cast<const BssInfo*>(m_pBuffer);
    if(pHeader != NULL)
    {
        return Ssid(pHeader->ssid, pHeader->ssidLength);
    }
    else
    {
        return Ssid();
    }
}

//! Beacon Intervalを取得します。
uint16_t BeaconDescriptionReader::GetInterval() const NN_NOEXCEPT
{
    const BssInfo* pHeader = reinterpret_cast<const BssInfo*>(m_pBuffer);
    if(pHeader != NULL)
    {
        return pHeader->beaconPeriod;
    }
    else
    {
        return 0;
    }
}

//! BeaconのCapability Informationを取得します。
Bit16 BeaconDescriptionReader::GetCapabilityInformation() const NN_NOEXCEPT
{
    const BssInfo* pHeader = reinterpret_cast<const BssInfo*>(m_pBuffer);
    if(pHeader != NULL)
    {
        return pHeader->capability;
    }
    else
    {
        return 0;
    }
}

const Bit8* BeaconDescriptionReader::GetIe(Bit8 elementId) const NN_NOEXCEPT
{
    const BssInfo* pHeader = reinterpret_cast<const BssInfo*>(m_pBuffer);
    if(pHeader == NULL)
    {
        return NULL;
    }

    const Bit8* pReturnIe = NULL;
    const Bit8* pIeHead = reinterpret_cast<const Bit8*>(m_pBuffer) + sizeof(BssInfo);

    if (pIeHead == NULL)
    {
        NN_SDK_LOG("Couldn't Get IE.\n");
        return NULL;
    }

    const Bit8* pIeEnd = pIeHead + pHeader->bssSize - sizeof(BssInfo);

    for( const Bit8* pIe = pIeHead;
         pIe + sizeof(InfoElementReader) + reinterpret_cast<const InfoElementReader*>(pIe)->GetLength() <= pIeEnd;
         pIe += sizeof(InfoElementReader) + reinterpret_cast<const InfoElementReader*>(pIe)->GetLength()
         )
    {
        const InfoElementReader *pIeHeader = reinterpret_cast<const InfoElementReader*>(pIe);
        if ( pIeHeader->GetElementId() == elementId )
        {
            pReturnIe = pIe;
        }
    }
    return pReturnIe;
}

const VendorInfoElementReader* BeaconDescriptionReader::GetVendorIeWithOui(const Bit8 pOui[MacAddress::OuiSize]) const NN_NOEXCEPT
{
    const BssInfo* pHeader = reinterpret_cast<const BssInfo*>(m_pBuffer);
    if(pHeader == NULL)
    {
        return NULL;
    }

    const uint8_t OuiOffset = 2;
    const Bit8* pReturnIe = NULL;
    const Bit8* pIeHead = reinterpret_cast<const Bit8*>(m_pBuffer) + sizeof(BssInfo);

    if (pOui == NULL || pIeHead == NULL)
    {
        NN_SDK_LOG("Couldn't Get IE.\n");
        return NULL;
    }

    const Bit8* pIeEnd = pIeHead + pHeader->bssSize - sizeof(BssInfo);

    //NN_SDK_LOG("VIE Head %p VIE End %p\n", pIeHead, pIeEnd);

    for( const Bit8* pIe = pIeHead;
         pIe + sizeof(InfoElementReader) + reinterpret_cast<const InfoElementReader*>(pIe)->GetLength() <= pIeEnd;
         pIe += sizeof(InfoElementReader) + reinterpret_cast<const InfoElementReader*>(pIe)->GetLength()
         )
    {
        const InfoElementReader *pIeHeader = reinterpret_cast<const InfoElementReader*>(pIe);
        //NN_SDK_LOG("Check IE [%p] EID[%d] Len[%d]\n", pIeHeader, pIeHeader->GetElementId(), pIeHeader->GetLength());
        if ( pIeHeader->GetElementId() == ElementId_Vendor )
        {
            // OUI をマッチング
            if (0 == std::memcmp(&pIe[OuiOffset], pOui, nn::wlan::MacAddress::OuiSize))
            {
                pReturnIe = pIe;
            }
        }
    }
    return reinterpret_cast<const VendorInfoElementReader*>(pReturnIe);
}

const VendorInfoElementReader* BeaconDescriptionReader::GetVendorIeWithOuiAndType(const Bit8 pOui[MacAddress::OuiSize], Bit8 type) const NN_NOEXCEPT
{
    const BssInfo* pHeader = reinterpret_cast<const BssInfo*>(m_pBuffer);
    if(pHeader == NULL)
    {
        return NULL;
    }

    const uint8_t OuiOffset = 2;
    const uint8_t TypeOffset = OuiOffset + nn::wlan::MacAddress::OuiSize;
    const Bit8* pReturnIe = NULL;
    const Bit8* pIeHead = reinterpret_cast<const Bit8*>(m_pBuffer) + sizeof(BssInfo);

    if (pOui == NULL || pIeHead == NULL)
    {
        NN_SDK_LOG("Couldn't Get IE.\n");
        return NULL;
    }

    const Bit8* pIeEnd = pIeHead + pHeader->bssSize - sizeof(BssInfo);

    //NN_SDK_LOG("VIE Head %p VIE End %p\n", pIeHead, pIeEnd);

    for( const Bit8* pIe = pIeHead;
         pIe + sizeof(InfoElementReader) + reinterpret_cast<const InfoElementReader*>(pIe)->GetLength() <= pIeEnd;
         pIe += sizeof(InfoElementReader) + reinterpret_cast<const InfoElementReader*>(pIe)->GetLength()
         )
    {
        const InfoElementReader *pIeHeader = reinterpret_cast<const InfoElementReader*>(pIe);
        //NN_SDK_LOG("Check IE [%p] EID[%d] Len[%d]\n", pIeHeader, pIeHeader->GetElementId(), pIeHeader->GetLength());
        if ( pIeHeader->GetElementId() == ElementId_Vendor )
        {
            // OUI と type をマッチング
            if( 0 == std::memcmp(&pIe[OuiOffset], pOui, nn::wlan::MacAddress::OuiSize) &&
                    pIe[TypeOffset] == type )
            {
                pReturnIe = pIe;
            }
        }
    }

    return reinterpret_cast<const VendorInfoElementReader*>(pReturnIe);
}

const VendorInfoElementReader* BeaconDescriptionReader::GetVendorIeWithOuiAndProtocolId(const Bit8 pOui[MacAddress::OuiSize], const Bit8 (&protocolId)[2]) const NN_NOEXCEPT
{
    const BssInfo* pHeader = reinterpret_cast<const BssInfo*>(m_pBuffer);
    if(pHeader == NULL)
    {
        return NULL;
    }

    const uint8_t OuiOffset = 2;
    const uint8_t ProtocolIdOffset = OuiOffset + nn::wlan::MacAddress::OuiSize;
    const Bit8* pReturnIe = NULL;
    const Bit8* pIeHead = reinterpret_cast<const Bit8*>(m_pBuffer) + sizeof(BssInfo);

    if (pOui == NULL || pIeHead == NULL)
    {
        NN_SDK_LOG("Couldn't Get IE.\n");
        return NULL;
    }

    const Bit8* pIeEnd = pIeHead + pHeader->bssSize - sizeof(BssInfo);

    //NN_SDK_LOG("VIE Head %p VIE End %p\n", pIeHead, pIeEnd);

    for( const Bit8* pIe = pIeHead;
         pIe + sizeof(InfoElementReader) + reinterpret_cast<const InfoElementReader*>(pIe)->GetLength() <= pIeEnd;
         pIe += sizeof(InfoElementReader) + reinterpret_cast<const InfoElementReader*>(pIe)->GetLength()
         )
    {
        const InfoElementReader *pIeHeader = reinterpret_cast<const InfoElementReader*>(pIe);
        //NN_SDK_LOG("Check IE [%p] EID[%d] Len[%d]\n", pIeHeader, pIeHeader->GetElementId(), pIeHeader->GetLength());
        if ( pIeHeader->GetElementId() == ElementId_Vendor )
        {
            // OUI と Protocol ID をマッチング
            if(     0 == std::memcmp(&pIe[OuiOffset], pOui, nn::wlan::MacAddress::OuiSize) &&
                    pIe[ProtocolIdOffset] == protocolId[0] &&
                    pIe[ProtocolIdOffset + 1] == protocolId[1] )
            {
                pReturnIe = pIe;
            }
        }
    }

    return reinterpret_cast<const VendorInfoElementReader*>(pReturnIe);
}

const WpaInfoElementReader<WifiInfoElementReader>* BeaconDescriptionReader::GetWpaIe() const NN_NOEXCEPT
{
    return reinterpret_cast<const WpaInfoElementReader<WifiInfoElementReader>*>(
            GetVendorIeWithOuiAndType(WpaOui, 0x01)
            );
}

const WpaInfoElementReader<InfoElementReader>* BeaconDescriptionReader::GetWpa2Ie() const NN_NOEXCEPT
{
    return reinterpret_cast<const WpaInfoElementReader<InfoElementReader>*>(
            GetIe(ElementId_Rsn)
            );
}

const InfoElementReader* BeaconDescriptionReader::GetWpsIe() const NN_NOEXCEPT
{
    return reinterpret_cast<const InfoElementReader*>(
            GetVendorIeWithOuiAndType(WpaOui, 0x04)
            );
}


} // end of namespace wlan
} // end of namespace nn

