﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "visrv_MemoryManagement.h"

#include <type_traits>
#include <nn/lmem/lmem_UnitHeap.h>
#include "visrv_Config.h"
#include "visrv_Log.h"

#include "client/visrv_ClientObject.h"
#include "client/visrv_ClientDisplayHolder.h"
#include "client/visrv_ClientLayerHolder.h"


namespace nn{ namespace visrv{
    namespace {

        template<typename T, int N>
        class UnitHeap
        {
            NN_DISALLOW_COPY(UnitHeap);
            NN_DISALLOW_MOVE(UnitHeap);
        public:
            UnitHeap() NN_NOEXCEPT
            {
            }

            void Initialize() NN_NOEXCEPT
            {
                m_HeapHandle = nn::lmem::CreateUnitHeap(
                    &m_Storage,
                    sizeof(m_Storage),
                    sizeof(T),
                    nn::lmem::CreationOption_ThreadSafe,
                    NN_ALIGNOF(T),
                    &m_HeapHead
                    );
                NN_SDK_ASSERT_EQUAL(nn::lmem::GetUnitHeapAllocatableCount(m_HeapHandle), N);
            }

            void Finalize() NN_NOEXCEPT
            {
                nn::lmem::DestroyUnitHeap(m_HeapHandle);
            }

            void* Allocate() NN_NOEXCEPT
            {
                return nn::lmem::AllocateFromUnitHeap(m_HeapHandle);
            }

            void Free(void* p) NN_NOEXCEPT
            {
                nn::lmem::FreeToUnitHeap(m_HeapHandle, p);
            }

            void Dump(const char* name) NN_NOEXCEPT
            {
                int allocatable = nn::lmem::GetUnitHeapAllocatableCount(m_HeapHandle);
                int allocated = nn::lmem::GetUnitHeapAllocatedCount(m_HeapHandle);
                NN_UNUSED(allocatable);
                NN_UNUSED(allocated);
                NN_VISRV_LOG_MEM("%s % 3d/% 3d\n", name, allocated, allocated + allocatable);
            }

        private:
            nn::lmem::HeapHandle     m_HeapHandle;
            nn::lmem::HeapCommonHead m_HeapHead;
            typename std::aligned_storage<sizeof(T) * N, NN_ALIGNOF(T)>::type m_Storage;
        };

        static const int TotalClientCountMax  = ClientProcessCountMax;
        static const int TotalDisplayCountMax = DisplayCountPerClientMax * ClientProcessCountMax;

        UnitHeap<client::ClientObject       , TotalClientCountMax > g_ClientObjectHeap;
        UnitHeap<client::ClientDisplayHolder, TotalDisplayCountMax> g_ClientDisplayHolderHeap;
    }

    void DumpMemory() NN_NOEXCEPT
    {
#ifdef NN_VISRV_ENABLE_MEMORY_DUMP
        NN_VISRV_LOG_MEM("---- memory ----\n");
        g_ClientObjectHeap       .Dump("ClientObjectHeap       ");
        g_ClientDisplayHolderHeap.Dump("ClientDisplayHolderHeap");
        NN_VISRV_LOG_MEM("----------------\n");
#endif
    }

    void InitializeMemory() NN_NOEXCEPT
    {
        g_ClientObjectHeap.Initialize();
        g_ClientDisplayHolderHeap.Initialize();
    }

    void FinalizeMemory() NN_NOEXCEPT
    {
        g_ClientDisplayHolderHeap.Finalize();
        g_ClientObjectHeap.Finalize();
    }

    void* AllocateClientObject() NN_NOEXCEPT
    {
        auto p = g_ClientObjectHeap.Allocate();
        DumpMemory();
        return p;
    }

    void FreeClientObject(void* p) NN_NOEXCEPT
    {
        g_ClientObjectHeap.Free(p);
        DumpMemory();
    }

    void* AllocateClientDisplayHolder() NN_NOEXCEPT
    {
        auto p = g_ClientDisplayHolderHeap.Allocate();
        DumpMemory();
        return p;
    }

    void FreeClientDisplayHolder(void* p) NN_NOEXCEPT
    {
        g_ClientDisplayHolderHeap.Free(p);
        DumpMemory();
    }
}}
