﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "visrv_SyncpointWaiter.h"

#include <nn/util/util_ScopeExit.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/vi/vi_Result.h>
#include "../visrv_Log.h"

#include <sync/sync.h>
#include <nvnflinger_service.h>

#define NN_VISRV_LOG_SYNCPOINT_DEV(format,...) \
    NN_VISRV_LOG_SYNCPOINT("(%d)" format, nn::os::GetThreadId(nn::os::GetCurrentThread()), __VA_ARGS__);


namespace nn{ namespace visrv{ namespace native{
    SyncpointWaiter g_SyncpointWaiter;

    //-------------------------------------------------------

#define NN_VISRV_SYNCPOINTENTRY_INITIALIZERLIST \
    m_State(detail::SyncpointEntryState_Free)   \
    , m_Sync()              \
    , m_ExpiredEvent()      \
    , m_pWaiter(nullptr)    \
    , m_WaiterPrivate()

#define NN_VISRV_SYNCPOINTENTRY_INITIALIZE_COMMON   \
        nn::os::InitializeEvent(&m_ExpiredEvent, false, nn::os::EventClearMode_ManualClear);

    SyncpointEntry::SyncpointEntry() NN_NOEXCEPT
        : NN_VISRV_SYNCPOINTENTRY_INITIALIZERLIST
    {
        NN_VISRV_SYNCPOINTENTRY_INITIALIZE_COMMON;
    }

    SyncpointEntry::SyncpointEntry(const NvRmFence& fence) NN_NOEXCEPT
        : NN_VISRV_SYNCPOINTENTRY_INITIALIZERLIST
    {
        NN_VISRV_SYNCPOINTENTRY_INITIALIZE_COMMON;
        Reset(fence);
    }

    SyncpointEntry::SyncpointEntry(const android::sp<android::Fence>& fence) NN_NOEXCEPT
        : NN_VISRV_SYNCPOINTENTRY_INITIALIZERLIST
    {
        NN_VISRV_SYNCPOINTENTRY_INITIALIZE_COMMON;
        Reset(fence);
    }

    SyncpointEntry::~SyncpointEntry() NN_NOEXCEPT
    {
        Cancel();
        nn::os::FinalizeEvent(&m_ExpiredEvent);
    }

    void SyncpointEntry::Reset() NN_NOEXCEPT
    {
        Cancel();
        m_State = detail::SyncpointEntryState_Free;
        m_Sync.Clear();
        m_pWaiter = nullptr;
        nn::os::ClearEvent(&m_ExpiredEvent);
        std::memset(&m_WaiterPrivate, 0, sizeof(m_WaiterPrivate));
    }

    void SyncpointEntry::Reset(const NvRmFence& fence) NN_NOEXCEPT
    {
        Reset();
        m_Sync.count = 1;
        m_Sync.fenceList[0] = NativeRmFence(fence);
    }

    void SyncpointEntry::Reset(const android::sp<android::Fence>& fence) NN_NOEXCEPT
    {
        Reset();
        m_Sync.ReadFrom(fence);
    }


    void SyncpointEntry::Wait() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(
            m_State == detail::SyncpointEntryState_Queued ||
            m_State == detail::SyncpointEntryState_Waiting ||
            m_State == detail::SyncpointEntryState_Expired);

        nn::os::WaitEvent(&m_ExpiredEvent);
    }

    bool SyncpointEntry::TryWait() NN_NOEXCEPT
    {
        return nn::os::TryWaitEvent(&m_ExpiredEvent);
    }

    void SyncpointEntry::Cancel() NN_NOEXCEPT
    {
        if(!m_pWaiter)
        {
            return;
        }
        m_pWaiter->Cancel(this);
    }

    void SyncpointEntry::InitializeWaitHolder(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pHolder);
        nn::os::InitializeMultiWaitHolder(pHolder, &m_ExpiredEvent);
    }

    void SyncpointEntry::FinalizeWaitHolder(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pHolder);
        nn::os::FinalizeMultiWaitHolder(pHolder);
    }

    //----------------------------------------------------------------------------

    const uintptr_t SyncpointWaiter::DequeueableTag;
    const uintptr_t SyncpointWaiter::CancelRequestTag;
    const uintptr_t SyncpointWaiter::ExpiringEventTagMin;
    const uintptr_t SyncpointWaiter::ExpiringEventTagMax;

    void SyncpointWaiter::Initialize() NN_NOEXCEPT
    {
        m_Queue.Initialize();
        m_EventManager.Initialize();

        nn::os::InitializeMutex(&m_CancelRequestMutex, false , 0);
        nn::os::InitializeEvent(&m_CancelRequestEvent, false, nn::os::EventClearMode_ManualClear);
        nn::os::InitializeEvent(&m_CancelCompleteEvent, true, nn::os::EventClearMode_ManualClear);
        m_pCancelRequestEntry = nullptr;

        m_Queue.InitializeDequeueMultiWaitHolder(&m_DequeueableWaitHolder);
        m_DequeueableWaitHolder.userData = DequeueableTag;

        nn::os::InitializeMultiWaitHolder(&m_CancelRequestWaitHolder, &m_CancelRequestEvent);
        m_CancelRequestWaitHolder.userData = CancelRequestTag;

        nn::os::InitializeMultiWait(&m_MultiWait);
        nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_CancelRequestWaitHolder);

        m_IsDequeueableLinked = false;
        UpdateDequeueableLinkImpl();
    }

    void SyncpointWaiter::Finalize() NN_NOEXCEPT
    {
        nn::os::UnlinkAllMultiWaitHolder(&m_MultiWait);
        m_IsDequeueableLinked = false;

        nn::os::FinalizeMultiWait(&m_MultiWait);
        nn::os::FinalizeMultiWaitHolder(&m_CancelRequestWaitHolder);
        m_Queue.FinalizeDequeueMultiWaitHolder(&m_DequeueableWaitHolder);

        nn::os::FinalizeEvent(&m_CancelCompleteEvent);
        nn::os::FinalizeEvent(&m_CancelRequestEvent);
        nn::os::FinalizeMutex(&m_CancelRequestMutex);

        m_EventManager.Finalize();
        m_Queue.Finalize();
    }

    void SyncpointWaiter::Enqueue(SyncpointEntry* pEntry) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pEntry);
        NN_SDK_REQUIRES_EQUAL(pEntry->m_State, detail::SyncpointEntryState_Free);
        NN_SDK_REQUIRES(pEntry->m_pWaiter == nullptr);

        //NN_VISRV_LOG_SYNCPOINT_DEV("Enqueue %d-%d\n", pEntry->m_Fence.syncpointId, pEntry->m_Fence.value);

        pEntry->m_State = detail::SyncpointEntryState_Queued;
        pEntry->m_pWaiter = this;
        m_Queue.Enqueue(pEntry);
    }

    void SyncpointWaiter::Cancel(SyncpointEntry* pEntry) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pEntry);
        NN_SDK_REQUIRES(pEntry->m_pWaiter == this);

        nn::os::LockMutex(&m_CancelRequestMutex);
        NN_UTIL_SCOPE_EXIT{ nn::os::UnlockMutex(&m_CancelRequestMutex); };

        NN_SDK_ASSERT(nn::os::TryWaitEvent(&m_CancelCompleteEvent));
        NN_SDK_ASSERT(!nn::os::TryWaitEvent(&m_CancelRequestEvent));

        nn::os::ClearEvent(&m_CancelCompleteEvent);
        m_pCancelRequestEntry = pEntry;
        nn::os::SignalEvent(&m_CancelRequestEvent);

        nn::os::WaitEvent(&m_CancelCompleteEvent);
        m_pCancelRequestEntry = nullptr;
    }



    void SyncpointWaiter::Run() NN_NOEXCEPT
    {
        for(;;)
        {
            auto pHolder = nn::os::WaitAny(&m_MultiWait);
            auto tag = pHolder->userData;

            if(tag >= ExpiringEventTagMin && tag <= ExpiringEventTagMax)
            {
                ProcessExpiredImpl(pHolder);
            }
            else
            {
                switch(tag)
                {
                case DequeueableTag:
                    ProcessDequeueImpl();
                    break;
                case CancelRequestTag:
                    ProcessCancelRequestImpl();
                    break;
                default:
                    NN_VISRV_LOG_SYNCPOINT_ERR("Unknown tag %lld(0x%llX)\n", static_cast<uint64_t>(pHolder->userData), static_cast<uint64_t>(pHolder->userData));
                    break;
                }
            }
        }
    }

    void SyncpointWaiter::ProcessDequeueImpl() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(!m_EventManager.IsFull());

        SyncpointEntry* pEntry = nullptr;
        m_Queue.Dequeue(&pEntry);

        // Ignore canceled syncpoint
        if(pEntry == nullptr)
        {
            NN_VISRV_LOG_SYNCPOINT_DEV("AlreadyCanceled\n", 0);
            return;
        }

        NN_SDK_ASSERT_EQUAL(pEntry->m_State, detail::SyncpointEntryState_Queued);
        NN_SDK_ASSERT_MINMAX(pEntry->m_Sync.count, 0, native::NativeRmSync::FenceCountMax);

        pEntry->m_WaiterPrivate.nextFence = 0;
        if(!KickNextFenceWaitImpl(pEntry))
        {
            UpdateDequeueableLinkImpl();
            return;
        }

        // 待つものが無ければシグナルする
        pEntry->m_State = detail::SyncpointEntryState_Expired;
        nn::os::SignalEvent(&pEntry->m_ExpiredEvent);
    }

    void SyncpointWaiter::ProcessExpiredImpl(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT
    {
        SyncpointEntry* pEntry = nullptr;

        // シグナル済の処理
        {
            int slotIndex = static_cast<int>(pHolder->userData - ExpiringEventTagMin);
            NN_SDK_ASSERT_RANGE(slotIndex, 0, detail::SyncpointEventManager::Size);
            pEntry = m_EventManager.GetEntry(slotIndex);
            NN_SDK_ASSERT_EQUAL(pEntry->m_State, detail::SyncpointEntryState_Waiting);

            //NN_VISRV_LOG_SYNCPOINT_DEV("Expired %d-%d\n", pEntry->m_Fence.syncpointId, pEntry->m_Fence.value);
            nn::os::UnlinkMultiWaitHolder(pHolder);
            m_EventManager.ReleaseWaitSlot(slotIndex);
            pEntry->m_WaiterPrivate.waitSlot = -1;
        }

        // 次の fence を待つ
        if(!KickNextFenceWaitImpl(pEntry))
        {
            return;
        }

        // 待つものが無ければシグナルする
        UpdateDequeueableLinkImpl();
        pEntry->m_State = detail::SyncpointEntryState_Expired;
        nn::os::SignalEvent(&pEntry->m_ExpiredEvent);
    }

    bool SyncpointWaiter::KickNextFenceWaitImpl(SyncpointEntry* pEntry) NN_NOEXCEPT
    {
        for(int i = pEntry->m_WaiterPrivate.nextFence; i < pEntry->m_Sync.count; i++)
        {
            //NN_VISRV_LOG_SYNCPOINT_DEV("Dequeue %d-%d\n", pEntry->m_Fence.syncpointId, pEntry->m_Fence.value);
            auto& fence = pEntry->m_Sync.fenceList[i];

            int slotIndex = -1;
            if(fence.syncpointId == NVRM_INVALID_SYNCPOINT_ID ||
                m_EventManager.AcquireWaitSlot(&slotIndex, pEntry->m_Sync.fenceList[i], pEntry)
                )
            {
                //NN_VISRV_LOG_SYNCPOINT_DEV("AlreadyExpired %d-%d\n", pEntry->m_Fence.syncpointId, pEntry->m_Fence.value);
                continue;
            }

            auto pHolder = m_EventManager.GetMultiWaitHolder(slotIndex);
            pHolder->userData = ExpiringEventTagMin + slotIndex;
            NN_SDK_ASSERT_MINMAX(pHolder->userData, ExpiringEventTagMin, ExpiringEventTagMax);

            nn::os::LinkMultiWaitHolder(&m_MultiWait, pHolder);
            pEntry->m_State     = detail::SyncpointEntryState_Waiting;
            pEntry->m_WaiterPrivate.nextFence = i + 1;
            pEntry->m_WaiterPrivate.waitSlot  = slotIndex;
            return false;
        }

        // 全部の fence が終わった
        pEntry->m_WaiterPrivate.nextFence = 0;
        pEntry->m_WaiterPrivate.waitSlot  = 0;
        return true;
    }

    void SyncpointWaiter::ProcessCancelRequestImpl() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(!nn::os::TryWaitEvent(&m_CancelCompleteEvent));
        NN_SDK_ASSERT(nn::os::TryWaitEvent(&m_CancelRequestEvent));

        auto pEntry = m_pCancelRequestEntry;
        if(pEntry->m_State == detail::SyncpointEntryState_Queued)
        {
            NN_VISRV_LOG_SYNCPOINT_DEV("Cancel(Queued) %d-%d / %d-%d / %d-%d / %d-%d\n",
                pEntry->m_FenceList[0].syncpointId, pEntry->m_FenceList[0].value,
                pEntry->m_FenceList[1].syncpointId, pEntry->m_FenceList[1].value,
                pEntry->m_FenceList[2].syncpointId, pEntry->m_FenceList[2].value,
                pEntry->m_FenceList[3].syncpointId, pEntry->m_FenceList[3].value
            );
            bool isCanceled = m_Queue.Cancel(pEntry);
            NN_SDK_ASSERT(isCanceled == true);
            NN_UNUSED(isCanceled);
            pEntry->m_State = detail::SyncpointEntryState_Canceled;
        }
        else if(pEntry->m_State == detail::SyncpointEntryState_Waiting)
        {
            NN_VISRV_LOG_SYNCPOINT_DEV("Cancel(Waiting) %d-%d / %d-%d / %d-%d / %d-%d\n",
                pEntry->m_FenceList[0].syncpointId, pEntry->m_FenceList[0].value,
                pEntry->m_FenceList[1].syncpointId, pEntry->m_FenceList[1].value,
                pEntry->m_FenceList[2].syncpointId, pEntry->m_FenceList[2].value,
                pEntry->m_FenceList[3].syncpointId, pEntry->m_FenceList[3].value
            );
            // 待ち中のイベントを解除
            NN_SDK_ASSERT_RANGE(pEntry->m_WaiterPrivate.waitSlot, 0, detail::SyncpointEventManager::Size);
            auto pHolder = m_EventManager.GetMultiWaitHolder(pEntry->m_WaiterPrivate.waitSlot);
            nn::os::UnlinkMultiWaitHolder(pHolder);
            // イベントを返却
            m_EventManager.CancelWaitSlot(pEntry->m_WaiterPrivate.waitSlot);
            pEntry->m_State = detail::SyncpointEntryState_Canceled;
            pEntry->m_WaiterPrivate.nextFence = 0;
            pEntry->m_WaiterPrivate.waitSlot  = 0;
        }

        UpdateDequeueableLinkImpl();
        m_pCancelRequestEntry = nullptr;
        nn::os::ClearEvent(&m_CancelRequestEvent);
        nn::os::SignalEvent(&m_CancelCompleteEvent);
    }

    void SyncpointWaiter::UpdateDequeueableLinkImpl() NN_NOEXCEPT
    {
        if(!m_EventManager.IsFull())
        {
            if(!m_IsDequeueableLinked)
            {
                nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_DequeueableWaitHolder);
                m_IsDequeueableLinked = true;
            }
        }
        else
        {
            if(m_IsDequeueableLinked)
            {
                nn::os::UnlinkMultiWaitHolder(&m_DequeueableWaitHolder);
                m_IsDequeueableLinked = false;
            }
        }
        NN_SDK_ASSERT(!m_EventManager.IsFull() == m_IsDequeueableLinked);
    }

}}}
