﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "visrv_TransactionIGraphicBufferProducer.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_FormatString.h>
#include <nn/vi/vi_Result.h>
#include "../visrv_BinderFunctions.h"
#include "../visrv_ParcelIo.h"
#include "../../visrv_Config.h"
#include "../../visrv_Log.h"
#include <binder/Parcel.h>
#include <gui/IGraphicBufferProducer.h>

#define NN_VISRV_CHECK_INTERFACE(parcel)    \
    NN_RESULT_THROW_UNLESS(parcel.enforceInterface(InterfaceDescriptor::GetIGraphicBufferProducerDescriptor()), nn::vi::ResultBroken());

#define NN_VISRV_WRITE_INTERFACE(parcel)    \
    NN_RESULT_THROW_UNLESS(parcel.writeInterfaceToken(InterfaceDescriptor::GetIGraphicBufferProducerDescriptor()) == android::NO_ERROR, nn::vi::ResultOperationFailed());


#define NN_VISRV_OPEN_PARCEL(varParcel,varPos,buff,size) \
    android::Parcel varParcel;  \
    size_t varPos;              \
    NN_RESULT_THROW_UNLESS(ParcelIo::OpenParcel(&varParcel, &varPos, buff, size) == android::NO_ERROR, nn::vi::ResultOperationFailed());    \
    NN_UTIL_SCOPE_EXIT{ ParcelIo::CloseParcel(&varParcel); };

#define NN_VISRV_WRITE_PARCEL(pOutSize,buff,buffSize,parcel)  \
    ParcelIo::WriteParcel(pOutSize, buff, buffSize, &parcel);   \
    NN_RESULT_THROW_UNLESS(*pOutSize > 0, nn::vi::ResultOperationFailed());


namespace nn{ namespace visrv{ namespace native{ namespace detail{
    NN_STATIC_ASSERT(sizeof(TransactionIGraphicBufferProducer::Rect) == sizeof(android::Rect));
    NN_STATIC_ASSERT(sizeof(TransactionIGraphicBufferProducer::QueueBufferInput) == sizeof(android::IGraphicBufferProducer::QueueBufferInput));
    NN_STATIC_ASSERT(sizeof(TransactionIGraphicBufferProducer::QueueBufferOutput) == sizeof(android::IGraphicBufferProducer::QueueBufferOutput));

    void TransactionIGraphicBufferProducer::Rect::Read(android::Parcel& parcel) NN_NOEXCEPT
    {
        this->left   = parcel.readInt32();
        this->top    = parcel.readInt32();
        this->right  = parcel.readInt32();
        this->bottom = parcel.readInt32();
    }

    void TransactionIGraphicBufferProducer::Rect::Write(android::Parcel& parcel) NN_NOEXCEPT
    {
        parcel.writeInt32(this->left);
        parcel.writeInt32(this->top);
        parcel.writeInt32(this->right);
        parcel.writeInt32(this->bottom);
    }

    void TransactionIGraphicBufferProducer::QueueBufferInput::ReadFrom(android::Parcel& parcel) NN_NOEXCEPT
    {
        auto native = reinterpret_cast<android::IGraphicBufferProducer::QueueBufferInput*>(this);
        parcel.read(*native);
    }

    void TransactionIGraphicBufferProducer::QueueBufferInput::WriteTo(android::Parcel& parcel) const NN_NOEXCEPT
    {
        auto native = reinterpret_cast<const android::IGraphicBufferProducer::QueueBufferInput*>(this);
        parcel.write(*native);
    }

    void TransactionIGraphicBufferProducer::QueueBufferOutput::ReadFrom(android::Parcel& parcel) NN_NOEXCEPT
    {
        auto src = parcel.readInplace(sizeof(*this));
        if(src)
        {
            std::memcpy(this, src, sizeof(*this));
        }
    }

    void TransactionIGraphicBufferProducer::QueueBufferOutput::WriteTo(android::Parcel& parcel) const NN_NOEXCEPT
    {
        auto dst = parcel.writeInplace(sizeof(*this));
        if(dst)
        {
            std::memcpy(dst, this, sizeof(*this));
        }
    }

    //----------------------------------------
    // Decode
    //----------------------------------------

    nn::Result TransactionIGraphicBufferProducer::DecodeRequestBufferRequest(RequestBufferRequest* pOutValue, const void* p, size_t size) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutValue);
        RequestBufferRequest value = {};

        NN_VISRV_OPEN_PARCEL(request, pos, p, size);
        NN_VISRV_CHECK_INTERFACE(request);
        value.bufferIndex = request.readInt32();

        NN_RESULT_THROW_UNLESS(request.errorCheck() == android::NO_ERROR, nn::vi::ResultOperationFailed());
        *pOutValue = value;
        NN_RESULT_SUCCESS;
    }

    nn::Result TransactionIGraphicBufferProducer::DecodeRequestBufferReply(RequestBufferReply* pOutValue, const void* p, size_t size) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutValue);
        RequestBufferReply value = {};

        NN_VISRV_OPEN_PARCEL(reply, pos, p, size);
        int32_t hasBuffer = reply.readInt32();
        if(hasBuffer)
        {
            value.buf = new android::GraphicBuffer();
            if(reply.read(*value.buf) != android::NO_ERROR)
            {
                value.buf.clear();
            }
        }
        else
        {
            value.buf.clear();
        }
        value.result = reply.readInt32();

        NN_RESULT_THROW_UNLESS(reply.errorCheck() == android::NO_ERROR, nn::vi::ResultOperationFailed());
        *pOutValue = value;
        NN_RESULT_SUCCESS;
    }

    nn::Result TransactionIGraphicBufferProducer::DecodeSetBufferCountRequest(SetBufferCountRequest* pOutValue, const void* p, size_t size) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutValue);
        SetBufferCountRequest value = {};

        NN_VISRV_OPEN_PARCEL(request, pos, p, size);
        NN_VISRV_CHECK_INTERFACE(request);
        value.count = request.readInt32();

        NN_RESULT_THROW_UNLESS(request.errorCheck() == android::NO_ERROR, nn::vi::ResultOperationFailed());
        *pOutValue = value;
        NN_RESULT_SUCCESS;
    }

    nn::Result TransactionIGraphicBufferProducer::DecodeSetBufferCountReply(SetBufferCountReply* pOutValue, const void* p, size_t size) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutValue);
        SetBufferCountReply value = {};

        NN_VISRV_OPEN_PARCEL(reply, pos, p, size);
        value.result = reply.readInt32();

        NN_RESULT_THROW_UNLESS(reply.errorCheck() == android::NO_ERROR, nn::vi::ResultOperationFailed());
        *pOutValue = value;
        NN_RESULT_SUCCESS;
    }

    nn::Result TransactionIGraphicBufferProducer::DecodeConnectRequest(ConnectRequest* pOutValue, const void* p, size_t size) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutValue);
        ConnectRequest value = {};

        NN_VISRV_OPEN_PARCEL(request, pos, p, size);
        NN_VISRV_CHECK_INTERFACE(request);
        value.hasListener     = request.readInt32();
        value.api             = request.readInt32();
        value.controlledByApp = request.readInt32();

        NN_RESULT_THROW_UNLESS(request.errorCheck() == android::NO_ERROR, nn::vi::ResultOperationFailed());
        *pOutValue = value;
        NN_RESULT_SUCCESS;
    }

    nn::Result TransactionIGraphicBufferProducer::DecodeConnectReply(ConnectReply* pOutValue, const void* p, size_t size) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutValue);
        ConnectReply value = {};

        NN_VISRV_OPEN_PARCEL(reply, pos, p, size);
        value.output.ReadFrom(reply);
        value.result = reply.readInt32();

        NN_RESULT_THROW_UNLESS(reply.errorCheck() == android::NO_ERROR, nn::vi::ResultOperationFailed());
        *pOutValue = value;
        NN_RESULT_SUCCESS;
    }

    nn::Result TransactionIGraphicBufferProducer::DecodeDisconnectRequest(DisconnectRequest* pOutValue, const void* p, size_t size) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutValue);
        DisconnectRequest value = {};

        NN_VISRV_OPEN_PARCEL(request, pos, p, size);
        NN_VISRV_CHECK_INTERFACE(request);
        value.api = request.readInt32();

        NN_RESULT_THROW_UNLESS(request.errorCheck() == android::NO_ERROR, nn::vi::ResultOperationFailed());
        *pOutValue = value;
        NN_RESULT_SUCCESS;
    }

    nn::Result TransactionIGraphicBufferProducer::DecodeDisconnectReply(DisconnectReply* pOutValue, const void* p, size_t size) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutValue);
        DisconnectReply value = {};

        NN_VISRV_OPEN_PARCEL(reply, pos, p, size);
        value.result = reply.readInt32();

        NN_RESULT_THROW_UNLESS(reply.errorCheck() == android::NO_ERROR, nn::vi::ResultOperationFailed());
        *pOutValue = value;
        NN_RESULT_SUCCESS;
    }

    nn::Result TransactionIGraphicBufferProducer::DecodeDequeueBufferRequest(DequeueBufferRequest* pOutValue, const void* p, size_t size) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutValue);
        DequeueBufferRequest value = {};

        NN_VISRV_OPEN_PARCEL(request, pos, p, size);
        NN_VISRV_CHECK_INTERFACE(request);
        value.async  = request.readInt32();
        value.w      = request.readInt32();
        value.h      = request.readInt32();
        value.format = request.readInt32();
        value.usage  = request.readInt32();

        NN_RESULT_THROW_UNLESS(request.errorCheck() == android::NO_ERROR, nn::vi::ResultOperationFailed());
        *pOutValue = value;
        NN_RESULT_SUCCESS;
    }

    nn::Result TransactionIGraphicBufferProducer::DecodeDequeueBufferReply(DequeueBufferReply* pOutValue, const void* p, size_t size) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutValue);
        DequeueBufferReply value = {};

        NN_VISRV_OPEN_PARCEL(reply, pos, p, size);
        value.buf    = reply.readInt32();
        int32_t hasFence = reply.readInt32();
        if(hasFence)
        {
            value.fence = new android::Fence();
            reply.read(*value.fence);
        }
        else
        {
            value.fence = android::Fence::NO_FENCE;
        }
        value.result = reply.readInt32();

        NN_RESULT_THROW_UNLESS(reply.errorCheck() == android::NO_ERROR, nn::vi::ResultOperationFailed());
        *pOutValue = value;
        NN_RESULT_SUCCESS;
    }

    nn::Result TransactionIGraphicBufferProducer::DecodeQueueBufferRequest(QueueBufferRequest* pOutValue, const void* p, size_t size) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutValue);
        QueueBufferRequest value = {};

        NN_VISRV_OPEN_PARCEL(request, pos, p, size);
        NN_VISRV_CHECK_INTERFACE(request);
        value.buf    = request.readInt32();
        value.input.ReadFrom(request);

        NN_RESULT_THROW_UNLESS(request.errorCheck() == android::NO_ERROR, nn::vi::ResultOperationFailed());
        *pOutValue = value;
        NN_RESULT_SUCCESS;
    }

    nn::Result TransactionIGraphicBufferProducer::DecodeQueueBufferReply(QueueBufferReply* pOutValue, const void* p, size_t size) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutValue);
        QueueBufferReply value = {};

        NN_VISRV_OPEN_PARCEL(reply, pos, p, size);
        value.output.ReadFrom(reply);
        value.result = reply.readInt32();

        NN_RESULT_THROW_UNLESS(reply.errorCheck() == android::NO_ERROR, nn::vi::ResultOperationFailed());
        *pOutValue = value;
        NN_RESULT_SUCCESS;
    }

    nn::Result TransactionIGraphicBufferProducer::DecodeSetPreallocatedBufferRequest(SetPreallocatedBufferRequest* pOutValue, const void* p, size_t size) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutValue);
        SetPreallocatedBufferRequest value = {};

        NN_VISRV_OPEN_PARCEL(request, pos, p, size);
        NN_VISRV_CHECK_INTERFACE(request);
        value.slot   = request.readInt32();
        int32_t hasBuffer = request.readInt32();
        if(hasBuffer)
        {
            value.buffer = new android::GraphicBuffer();
            request.read(*value.buffer);
        }

        NN_RESULT_THROW_UNLESS(request.errorCheck() == android::NO_ERROR, nn::vi::ResultOperationFailed());
        *pOutValue = value;
        NN_RESULT_SUCCESS;
    }

    nn::Result TransactionIGraphicBufferProducer::DecodeSetPreallocatedBufferReply(SetPreallocatedBufferReply* pOutValue, const void* p, size_t size) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutValue);
        SetPreallocatedBufferReply value = {};

        NN_VISRV_OPEN_PARCEL(reply, pos, p, size);
        value.result = reply.readInt32();

        NN_RESULT_THROW_UNLESS(reply.errorCheck() == android::NO_ERROR, nn::vi::ResultOperationFailed());
        *pOutValue = value;
        NN_RESULT_SUCCESS;
    }

    nn::Result TransactionIGraphicBufferProducer::DecodeQueryRequest(QueryRequest* pOutValue, const void* p, size_t size) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutValue);
        QueryRequest value = {};

        NN_VISRV_OPEN_PARCEL(request, pos, p, size);
        NN_VISRV_CHECK_INTERFACE(request);
        value.what = request.readInt32();

        NN_RESULT_THROW_UNLESS(request.errorCheck() == android::NO_ERROR, nn::vi::ResultOperationFailed());
        *pOutValue = value;
        NN_RESULT_SUCCESS;
    }

    nn::Result TransactionIGraphicBufferProducer::DecodeQueryReply(QueryReply* pOutValue, const void* p, size_t size) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutValue);
        QueryReply value = {};

        NN_VISRV_OPEN_PARCEL(reply, pos, p, size);
        value.value  = reply.readInt32();
        value.result = reply.readInt32();

        NN_RESULT_THROW_UNLESS(reply.errorCheck() == android::NO_ERROR, nn::vi::ResultOperationFailed());
        *pOutValue = value;
        NN_RESULT_SUCCESS;
    }


    //----------------------------------------
    // Encode
    //----------------------------------------

    nn::Result TransactionIGraphicBufferProducer::EncodeConnectRequest(size_t* pOutSize, void* buff, size_t buffSize, const ConnectRequest& value) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutSize);
        NN_SDK_ASSERT_NOT_NULL(buff);

        android::Parcel parcel;
        NN_VISRV_WRITE_INTERFACE(parcel);
        parcel.writeInt32(0); // hasListener
        parcel.writeInt32(value.api);
        parcel.writeInt32(value.controlledByApp);

        NN_VISRV_WRITE_PARCEL(pOutSize, buff, buffSize, parcel);
        NN_RESULT_SUCCESS;
    }

    nn::Result TransactionIGraphicBufferProducer::EncodeDisconnectRequest(size_t* pOutSize, void* buff, size_t buffSize, const DisconnectRequest& value) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutSize);
        NN_SDK_ASSERT_NOT_NULL(buff);

        android::Parcel parcel;
        NN_VISRV_WRITE_INTERFACE(parcel);
        parcel.writeInt32(value.api);

        NN_VISRV_WRITE_PARCEL(pOutSize, buff, buffSize, parcel);
        NN_RESULT_SUCCESS;
    }

    nn::Result TransactionIGraphicBufferProducer::EncodeQueueBufferRequest(size_t* pOutSize, void* buff, size_t buffSize, const QueueBufferRequest& value) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutSize);
        NN_SDK_ASSERT_NOT_NULL(buff);

        android::Parcel parcel;
        NN_VISRV_WRITE_INTERFACE(parcel);
        parcel.writeInt32(value.buf);
        value.input.WriteTo(parcel);

        NN_VISRV_WRITE_PARCEL(pOutSize, buff, buffSize, parcel);
        NN_RESULT_SUCCESS;
    }

    nn::Result TransactionIGraphicBufferProducer::EncodeQueueBufferReply(size_t* pOutSize, void* buff, size_t buffSize, const QueueBufferReply& value) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutSize);
        NN_SDK_ASSERT_NOT_NULL(buff);

        android::Parcel parcel;
        value.output.WriteTo(parcel);
        parcel.writeInt32(value.result);

        NN_VISRV_WRITE_PARCEL(pOutSize, buff, buffSize, parcel);
        NN_RESULT_SUCCESS;
    }

    //----------------------------------------
    // ToString
    //----------------------------------------

#ifdef NN_VISRV_ENABLE_TOSTRING

#define NN_VISRV_TOSTRING_START(buff,size)  \
    char*        nn_visrv_tostring_p     = buff;        \
    char*        nn_visrv_tostring_p_beg = buff;        \
    char*const   nn_visrv_tostring_p_end = buff + size; \
    const size_t nn_visrv_tostring_size  = size;

#define NN_VISRV_TOSTRING_FORMAT(...)   \
    nn_visrv_tostring_p += nn::util::SNPrintf(nn_visrv_tostring_p, nn_visrv_tostring_p_end - nn_visrv_tostring_p, __VA_ARGS__); \
    if(nn_visrv_tostring_p >= nn_visrv_tostring_p_end)  \
    {                                                   \
        return nn_visrv_tostring_size - 1;              \
    }

#define NN_VISRV_TOSTRING_APPEND(obj)   \
    nn_visrv_tostring_p += (obj).ToString(nn_visrv_tostring_p, nn_visrv_tostring_p_end - nn_visrv_tostring_p);   \
    if(nn_visrv_tostring_p >= nn_visrv_tostring_p_end)  \
    {                                                   \
        return nn_visrv_tostring_size - 1;              \
    }

#define NN_VISRV_TOSTRING_RETURN()  \
    return nn_visrv_tostring_p - nn_visrv_tostring_p_beg;

#else
#define NN_VISRV_TOSTRING_START(buff,size) NN_UNUSED(buff); NN_UNUSED(size);
#define NN_VISRV_TOSTRING_FORMAT(...)
#define NN_VISRV_TOSTRING_APPEND(obj)
#define NN_VISRV_TOSTRING_RETURN() return 0;
#endif

    size_t TransactionIGraphicBufferProducer::Rect::ToString(char* buff, size_t size) const NN_NOEXCEPT
    {
        NN_VISRV_TOSTRING_START(buff, size);
        NN_VISRV_TOSTRING_FORMAT("Rect{left=%d;top=%d;right=%d;bottom=%d}",
            this->left, this->top, this->right, this->bottom);
        NN_VISRV_TOSTRING_RETURN();
    }

    size_t TransactionIGraphicBufferProducer::QueueBufferInput::ToString(char* buff, size_t size) const NN_NOEXCEPT
    {
        NN_VISRV_TOSTRING_START(buff, size);
        NN_VISRV_TOSTRING_FORMAT("QueueBufferInput{timestamp=%lld;isAutoTimestamp=%d;crop=", this->timestamp, this->isAutoTimestamp);
        NN_VISRV_TOSTRING_APPEND(this->crop);
        NN_VISRV_TOSTRING_FORMAT(";scalingMode=%d;transform=%d;flags=%X;async=%d;interval=%d;hasFence=%d}",
            this->scalingMode, this->transform, this->flags, this->async, this->interval,
            (this->fence != nullptr && this->fence->isValid())
        );
        NN_VISRV_TOSTRING_RETURN();
    }

    size_t TransactionIGraphicBufferProducer::QueueBufferOutput::ToString(char* buff, size_t size) const NN_NOEXCEPT
    {
        NN_VISRV_TOSTRING_START(buff, size);
        NN_VISRV_TOSTRING_FORMAT("QueueBufferOutput{width=%d;height=%d;transformHint=%d;numPendingBuffers=%d}",
            this->width, this->height, this->transformHint, this->numPendingBuffers);
        NN_VISRV_TOSTRING_RETURN();
    }

    //----------------------------------

    size_t TransactionIGraphicBufferProducer::RequestBufferRequest::ToString(char* buff, size_t size) const NN_NOEXCEPT
    {
        NN_VISRV_TOSTRING_START(buff, size);
        NN_VISRV_TOSTRING_FORMAT("RequestBufferRequest{bufferIndex=%d}", this->bufferIndex);
        NN_VISRV_TOSTRING_RETURN();
    }

    size_t TransactionIGraphicBufferProducer::RequestBufferReply::ToString(char* buff, size_t size) const NN_NOEXCEPT
    {
        NN_VISRV_TOSTRING_START(buff, size);
        NN_VISRV_TOSTRING_FORMAT("RequestBufferReply{hasBuffer=%d;result=%d}",
            this->buf != nullptr, this->result);
        NN_VISRV_TOSTRING_RETURN();
    }

    size_t TransactionIGraphicBufferProducer::ConnectRequest::ToString(char* buff, size_t size) const NN_NOEXCEPT
    {
        NN_VISRV_TOSTRING_START(buff, size);
        NN_VISRV_TOSTRING_FORMAT("ConnectRequest{hasListener=%d;api=%d;controlledByApp=%d}",
            this->hasListener, this->api, this->controlledByApp);
        NN_VISRV_TOSTRING_RETURN();
    }

    size_t TransactionIGraphicBufferProducer::ConnectReply::ToString(char* buff, size_t size) const NN_NOEXCEPT
    {
        NN_VISRV_TOSTRING_START(buff, size);
        NN_VISRV_TOSTRING_FORMAT("ConnectReply{output=");
        NN_VISRV_TOSTRING_APPEND(this->output);
        NN_VISRV_TOSTRING_FORMAT(";result=%d}", this->result);
        NN_VISRV_TOSTRING_RETURN();
    }

    size_t TransactionIGraphicBufferProducer::DisconnectRequest::ToString(char* buff, size_t size) const NN_NOEXCEPT
    {
        NN_VISRV_TOSTRING_START(buff, size);
        NN_VISRV_TOSTRING_FORMAT("DisconnectRequest{api=%d}", this->api);
        NN_VISRV_TOSTRING_RETURN();
    }

    size_t TransactionIGraphicBufferProducer::DisconnectReply::ToString(char* buff, size_t size) const NN_NOEXCEPT
    {
        NN_VISRV_TOSTRING_START(buff, size);
        NN_VISRV_TOSTRING_FORMAT("DisconnectReply{result=%d}", this->result);
        NN_VISRV_TOSTRING_RETURN();
    }

    size_t TransactionIGraphicBufferProducer::DequeueBufferRequest::ToString(char* buff, size_t size) const NN_NOEXCEPT
    {
        NN_VISRV_TOSTRING_START(buff, size);
        NN_VISRV_TOSTRING_FORMAT("DequeueBufferRequest{async=%d;w=%d;h=%d;format=%d;usage=%X}", this->async, this->w, this->h, this->format, this->usage);
        NN_VISRV_TOSTRING_RETURN();
    }

    size_t TransactionIGraphicBufferProducer::DequeueBufferReply::ToString(char* buff, size_t size) const NN_NOEXCEPT
    {
        NN_VISRV_TOSTRING_START(buff, size);
        NN_VISRV_TOSTRING_FORMAT("DequeueBufferReply{buf=%d;hasFence=%d;result=%d}",
            this->buf,
            (this->fence != nullptr && this->fence->isValid()),
            this->result
        );
        NN_VISRV_TOSTRING_RETURN();
    }

    size_t TransactionIGraphicBufferProducer::QueueBufferRequest::ToString(char* buff, size_t size) const NN_NOEXCEPT
    {
        NN_VISRV_TOSTRING_START(buff, size);
        NN_VISRV_TOSTRING_FORMAT("QueueBufferRequest{buf=%d;input=", this->buf);
        NN_VISRV_TOSTRING_APPEND(this->input);
        NN_VISRV_TOSTRING_FORMAT("}");
        NN_VISRV_TOSTRING_RETURN();
    }

    size_t TransactionIGraphicBufferProducer::QueueBufferReply::ToString(char* buff, size_t size) const NN_NOEXCEPT
    {
        NN_VISRV_TOSTRING_START(buff, size);
        NN_VISRV_TOSTRING_FORMAT("QueueBufferReply{output=");
        NN_VISRV_TOSTRING_APPEND(this->output);
        NN_VISRV_TOSTRING_FORMAT(";result=%d}", this->result);
        NN_VISRV_TOSTRING_RETURN();
    }

    size_t TransactionIGraphicBufferProducer::SetPreallocatedBufferRequest::ToString(char* buff, size_t size) const NN_NOEXCEPT
    {
        NN_VISRV_TOSTRING_START(buff, size);
        NN_VISRV_TOSTRING_FORMAT("SetPreallocatedBufferRequest{slot=%d;hasBuffer=%d}", this->slot, this->buffer != nullptr);
        NN_VISRV_TOSTRING_RETURN();
    }

    size_t TransactionIGraphicBufferProducer::SetPreallocatedBufferReply::ToString(char* buff, size_t size) const NN_NOEXCEPT
    {
        NN_VISRV_TOSTRING_START(buff, size);
        NN_VISRV_TOSTRING_FORMAT("SetPreallocatedBufferReply{result=%d}", this->result);
        NN_VISRV_TOSTRING_RETURN();
    }

#ifdef NN_VISRV_ENABLE_TOSTRING
    // see window.h
    static const char* GetQueryKeyName(int32_t key) NN_NOEXCEPT
    {
        switch(key)
        {
        case NATIVE_WINDOW_WIDTH:                     return "WIDTH";
        case NATIVE_WINDOW_HEIGHT:                    return "HEIGHT";
        case NATIVE_WINDOW_FORMAT:                    return "FORMAT";
        case NATIVE_WINDOW_MIN_UNDEQUEUED_BUFFERS:    return "MIN_UNDEQUEUED_BUFFERS";
        case NATIVE_WINDOW_QUEUES_TO_WINDOW_COMPOSER: return "QUEUES_TO_WINDOW_COMPOSER";
        case NATIVE_WINDOW_CONCRETE_TYPE:             return "CONCRETE_TYPE";
        case NATIVE_WINDOW_DEFAULT_WIDTH:             return "DEFAULT_WIDTH";
        case NATIVE_WINDOW_DEFAULT_HEIGHT:            return "DEFAULT_HEIGHT";
        case NATIVE_WINDOW_TRANSFORM_HINT:            return "TRANSFORM_HINT";
        case NATIVE_WINDOW_CONSUMER_RUNNING_BEHIND:   return "CONSUMER_RUNNING_BEHIND";
        case NATIVE_WINDOW_CONSUMER_USAGE_BITS:       return "CONSUMER_USAGE_BITS";
        case NATIVE_WINDOW_STICKY_TRANSFORM:          return "STICKY_TRANSFORM";
        default: return "Unknown";
        }
    }
#endif

    size_t TransactionIGraphicBufferProducer::QueryRequest::ToString(char* buff, size_t size) const NN_NOEXCEPT
    {
        NN_VISRV_TOSTRING_START(buff, size);
        NN_VISRV_TOSTRING_FORMAT("QueryRequest{what=%d(%s)}", this->what, GetQueryKeyName(this->what));
        NN_VISRV_TOSTRING_RETURN();
    }


    size_t TransactionIGraphicBufferProducer::QueryReply::ToString(char* buff, size_t size) const NN_NOEXCEPT
    {
        NN_VISRV_TOSTRING_START(buff, size);
        NN_VISRV_TOSTRING_FORMAT("QueryReply{value=%d,result=%d}", this->value, this->result);
        NN_VISRV_TOSTRING_RETURN();
    }


}}}}
