﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "visrv_Android.h"

#include <hardware/hwcomposer_defs.h>
#include <nn/vi/vi_ScalingMode.h>
#include <nn/vi/vi_PixelFormat.h>
#include <nn/vi/vi_RgbRange.h>
#include <nn/vi/vi_PowerState.h>
#include <nn/vi/vi_LayerSettings.h>
#include <nn/vi/vi_DisplayModeInfo.h>
#include <nn/vi/vi_DisplayModeInfoInternal.h>
#include <nn/vi/vi_CmuMode.h>
#include "visrv_Edid.h"

namespace nn{ namespace visrv{ namespace master{ namespace detail{

    int ConvertScalingMode(nn::vi::ScalingMode mode) NN_NOEXCEPT
    {
        switch (mode)
        {
        case nn::vi::ScalingMode_None:
            return NATIVE_WINDOW_SCALING_MODE_NO_SCALE_CROP;
        case nn::vi::ScalingMode_Exact:
            return NATIVE_WINDOW_SCALING_MODE_FREEZE;
        case nn::vi::ScalingMode_FitToLayer:
            return NATIVE_WINDOW_SCALING_MODE_SCALE_TO_WINDOW;
        case nn::vi::ScalingMode_ScaleAndCrop:
            return NATIVE_WINDOW_SCALING_MODE_SCALE_CROP;
        case nn::vi::ScalingMode_PreserveAspectRatio:
            return NATIVE_WINDOW_SCALING_MODE_SCALE_NO_CROP;
        default: NN_UNEXPECTED_DEFAULT;
        }
    }

    void ConvertDisplayMode(nn::vi::DisplayModeInfo* pOutInfo, const android::DisplayInfo& info) NN_NOEXCEPT
    {
        pOutInfo->width = info.w;
        pOutInfo->height = info.h;
        pOutInfo->refreshRate = info.fps;
        // TODO: how to query stereo mode?
        pOutInfo->mode = nn::vi::StereoMode_None;
    }

    bool ConvertDisplayMode(nn::vi::DisplayModeInfoInternal* pOutInfo, const android::DisplayInfo& info) NN_NOEXCEPT
    {
        ConvertDisplayMode(&pOutInfo->clientInfo, info);
        return ConvertAspectRatio(&pOutInfo->ratio, info);
    }

    bool ConvertRgbRangeToVi(nn::vi::RgbRange* pOutRange, int range) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutRange);

        switch( range )
        {
        case DISPLAY_RGB_Limited:
            *pOutRange = nn::vi::RgbRange_Limited;
            break;
        case DISPLAY_RGB_Full:
            *pOutRange = nn::vi::RgbRange_Full;
            break;
        default:
            return false;
        }

        return true;
    }

    int ConvertRgbRangeToAndroid(nn::vi::RgbRange range) NN_NOEXCEPT
    {
        switch ( range )
        {
        case nn::vi::RgbRange_Limited:
            return DISPLAY_RGB_Limited;
        case nn::vi::RgbRange_Full:
            return DISPLAY_RGB_Full;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    int ConvertPowerState(nn::vi::PowerState state) NN_NOEXCEPT
    {
        switch( state )
        {
        case nn::vi::PowerState_Off:
            return HWC_POWER_MODE_OFF;
        case nn::vi::PowerState_Blank:
            return HWC_POWER_MODE_PREPARE;
        case nn::vi::PowerState_On:
            return HWC_POWER_MODE_NORMAL;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    uint32_t ConvertLayerSettings(nn::vi::LayerSettings settings) NN_NOEXCEPT
    {
        uint32_t flags = 0;

        if( settings.Test<nn::vi::LayerFlags::Opaque>() )
        {
            flags |= android::ISurfaceComposerClient::eOpaque;
        }

        return flags;
    }

    DisplayCMUMode ConvertCmuMode(nn::vi::CmuMode mode) NN_NOEXCEPT
    {
        switch( mode )
        {
        case nn::vi::CmuMode_Disabled:
            return DISPLAY_CMU_Disabled;
        case nn::vi::CmuMode_Default:
            return DISPLAY_CMU_Normal;
        case nn::vi::CmuMode_InvertColor:
            return DISPLAY_CMU_ColorInvert;
        case nn::vi::CmuMode_HighContrast:
            return DISPLAY_CMU_HighContrast;
        case nn::vi::CmuMode_Grayscale:
            return DISPLAY_CMU_GrayScale;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    bool ConvertCmuMode(nn::vi::CmuMode* pOutMode, DisplayCMUMode mode) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutMode);

        switch( mode )
        {
        case DISPLAY_CMU_Disabled:
            *pOutMode = nn::vi::CmuMode_Disabled;
            break;
        case DISPLAY_CMU_Normal:
            *pOutMode = nn::vi::CmuMode_Default;
            break;
        case DISPLAY_CMU_ColorInvert:
            *pOutMode = nn::vi::CmuMode_InvertColor;
            break;
        case DISPLAY_CMU_HighContrast:
            *pOutMode = nn::vi::CmuMode_HighContrast;
            break;
        case DISPLAY_CMU_GrayScale:
            *pOutMode = nn::vi::CmuMode_Grayscale;
            break;
        default:
            return false;
        }

        return true;
    }

    bool ConvertAspectRatio(nn::vi::AspectRatio* pOutRatio, const android::DisplayInfo& info) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutRatio);

        switch( info.aspectRatioFlag )
        {
        case HWC_DISPLAY_ASPECT_RATIO_4_3:
            *pOutRatio = nn::vi::AspectRatio_4_3;
            break;
        case HWC_DISPLAY_ASPECT_RATIO_16_9:
            *pOutRatio = nn::vi::AspectRatio_16_9;
            break;
        case HWC_DISPLAY_ASPECT_RATIO_64_27:
            *pOutRatio = nn::vi::AspectRatio_64_27;
            break;
        case HWC_DISPLAY_ASPECT_RATIO_256_135:
            *pOutRatio = nn::vi::AspectRatio_256_135;
            break;
        default:
            // No flag or unknown value...
            // Last ditch effort to find a mode based on dimensions
            return ConvertAspectRatio(pOutRatio, static_cast<float>(info.w) / info.h);
        }

        return true;
    }

    bool operator==(const nn::vi::DisplayModeInfo& viInfo, const android::DisplayInfo& androidInfo) NN_NOEXCEPT
    {
        // TODO: how to query stereo mode?
        return androidInfo.w == viInfo.width &&
               androidInfo.h == viInfo.height &&
               androidInfo.fps == viInfo.refreshRate;
    }

    bool operator==(const android::DisplayInfo& androidInfo, const nn::vi::DisplayModeInfo& viInfo) NN_NOEXCEPT
    {
        return viInfo == androidInfo;
    }

}}}}


