﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "visrv_PresentationTracer.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/os/os_SystemEventApi.h>
#include <nn/sf/sf_NativeHandle.h>
#include <nn/vi/vi_Result.h>
#include "visrv_PresentationTracerPool.h"
#include "../visrv_Log.h"

namespace nn{ namespace visrv{ namespace client{

    namespace detail{

        bool PresentationPreAcquireFenceWait::IsActive() const NN_NOEXCEPT
        {
            return m_CallbackFunction != nullptr;
        }

        void PresentationPreAcquireFenceWait::Initialize(uintptr_t multiWaitTag) NN_NOEXCEPT
        {
            m_Entry.InitializeWaitHolder(&m_Holder);
            m_Holder.userData = multiWaitTag;
            m_pPresentationEntry = nullptr;
            m_CallbackFunction = nullptr;
            m_CallbackUserPtr  = nullptr;
        }

        void PresentationPreAcquireFenceWait::Finalize() NN_NOEXCEPT
        {
            Deactivate();
            m_Entry.FinalizeWaitHolder(&m_Holder);
        }

        void PresentationPreAcquireFenceWait::Activate(
            PresentationEntry* pEntry,
            void (*function)(PresentationEntry*, void*),
            void* arg
        ) NN_NOEXCEPT
        {
            Deactivate();

            auto fence = pEntry->request.input.fence;
            m_Entry.Reset(fence);
            native::g_SyncpointWaiter.Enqueue(&m_Entry);

            m_pPresentationEntry = pEntry;
            m_CallbackFunction = function;
            m_CallbackUserPtr = arg;

            g_PresentationTracerPool.LinkMultiWaitImpl(&m_Holder);
        }

        void PresentationPreAcquireFenceWait::Deactivate() NN_NOEXCEPT
        {
            if(!IsActive())
            {
                return;
            }

            g_PresentationTracerPool.UnlinkMultiWaitImpl(&m_Holder);
            m_Entry.Reset();
            m_pPresentationEntry = nullptr;
            m_CallbackFunction = nullptr;
            m_CallbackUserPtr = nullptr;
        }

        void PresentationPreAcquireFenceWait::ProcessSignal(nn::os::MultiWaitHolderType* pHolder, bool isUnlinked) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES(IsActive());
            NN_SDK_REQUIRES_EQUAL(pHolder, &m_Holder);

            auto pEntry = m_pPresentationEntry;
            auto f = m_CallbackFunction;
            auto arg = m_CallbackUserPtr;

            // deactivate
            if(!isUnlinked)
            {
                g_PresentationTracerPool.UnlinkMultiWaitImpl(&m_Holder);
            }
            m_Entry.Reset();
            m_pPresentationEntry = nullptr;
            m_CallbackFunction = nullptr;
            m_CallbackUserPtr = nullptr;

            f(pEntry, arg);
        }
    } // namespace detail

    //----------------------------------

    void PresentationTracer::Initialize(uintptr_t multiWaitTag) NN_NOEXCEPT
    {
        m_Mode = Mode_Idle;
        m_RecordedPresentationQueue.Clear();
        m_PreAcquireFenceWait.Initialize(multiWaitTag);
        nn::os::CreateSystemEvent(&m_AllFencesExpiredEvent, nn::os::EventClearMode_ManualClear, true);
    }

    void PresentationTracer::Finalize() NN_NOEXCEPT
    {
        m_PreAcquireFenceWait.Finalize();
        m_Mode = Mode_Idle;
        m_RecordedPresentationQueue.Clear();
        nn::os::SignalSystemEvent(&m_AllFencesExpiredEvent);
        nn::os::DestroySystemEvent(&m_AllFencesExpiredEvent);
    }

    void PresentationTracer::GetAllFencesExpiredEvent(nn::sf::NativeHandle& outHandle) NN_NOEXCEPT
    {
        outHandle = nn::sf::NativeHandle(
            nn::os::GetReadableHandleOfSystemEvent(&m_AllFencesExpiredEvent),
            false
        );
    }

    void PresentationTracer::SetMode(Mode value) NN_NOEXCEPT
    {
        NN_VISRV_LOG_PRESENTTRACE("changing mode %d -> %d\n", m_Mode, value);
        m_PreAcquireFenceWait.Deactivate();

        m_Mode = value;

        UpdatePreAcquireFenceWaitStateImpl();
    }

    nn::Result PresentationTracer::RecordQueueBufferRequest(
        int32_t handle,
        uint32_t code,
        const void* requestBuffer,
        size_t requestBufferSize,
        uint32_t flags
    ) NN_NOEXCEPT
    {
        if(m_Mode != Mode_Recording)
        {
            NN_RESULT_SUCCESS;
        }

        PresentationEntry e = {};
        e.handle = handle;
        e.code = code;
        e.flags = flags;
        NN_RESULT_DO(native::detail::TransactionIGraphicBufferProducer::DecodeQueueBufferRequest(&e.request, requestBuffer, requestBufferSize));

        if(m_RecordedPresentationQueue.IsFull())
        {
            PresentationEntry discarding = {};
            m_RecordedPresentationQueue.TryDequeue(&discarding);
        }

        NN_ABORT_UNLESS_RESULT_SUCCESS(m_RecordedPresentationQueue.TryEnqueue(e));
        NN_VISRV_LOG_PRESENTTRACE("recorded %d\n", m_RecordedPresentationQueue.GetCount());

        NN_RESULT_SUCCESS;
    }

    void PresentationTracer::ProcessMultiWaitSignalImpl(nn::os::MultiWaitHolderType* pHolder, bool isUnlinked) NN_NOEXCEPT
    {
        m_PreAcquireFenceWait.ProcessSignal(pHolder, isUnlinked);
    }

    PresentationTracer::PresentationEntry* PresentationTracer::FindPresentedEntryWithPendingPreAcquireFenceImpl() NN_NOEXCEPT
    {
        // 有効な fence を持つ最後のエントリーを探す。
        // syncpoint value は単調増加するはずなので最後から待った方が最適化しやすい。

        PresentationEntry* pEntry = nullptr;

        m_RecordedPresentationQueue.Foreach([&](PresentationEntry& e)->void
        {
            auto fence = e.request.input.fence;
            if(fence != nullptr && fence->isValid())
            {
                pEntry = &e;
            }
        });

        return pEntry;
    }

    void PresentationTracer::UpdatePreAcquireFenceWaitStateImpl() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(!m_PreAcquireFenceWait.IsActive());

        bool areAllFencesExpired = false;
        if(m_Mode == Mode_FenceWait)
        {
            auto pEntry = FindPresentedEntryWithPendingPreAcquireFenceImpl();
            if(pEntry)
            {
                NN_VISRV_LOG_PRESENTTRACE("waiting for pre-acquire fence expired\n");
                m_PreAcquireFenceWait.Activate(
                    pEntry,
                    [](PresentationEntry* p, void* arg)->void
                    {
                        // シグナルした fence を削除
                        p->request.input.fence = android::Fence::NO_FENCE;

                        // 更新
                        reinterpret_cast<PresentationTracer*>(arg)->UpdatePreAcquireFenceWaitStateImpl();
                    },
                    this
                );
            }
            else
            {
                NN_VISRV_LOG_PRESENTTRACE("all pre-acquire fences are expired\n");
                areAllFencesExpired = true;
            }
        }
        else
        {
            // NOTE: Mode == Waiting でない場合シグナルしない
        }

        if(areAllFencesExpired)
        {
            nn::os::SignalSystemEvent(&m_AllFencesExpiredEvent);
        }
        else
        {
            nn::os::ClearSystemEvent(&m_AllFencesExpiredEvent);
        }
    }

}}}
