﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "visrv_ClientObject.h"

#include <nn/nn_SdkAssert.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/os/os_SdkSystemEventApi.h>

#include <nn/vi/vi_Result.h>
#include "../visrv_Log.h"
#include "../visrv_MemoryManagement.h"
#include "../visrv_ResourceIdManagement.h"
#include "../master/detail/visrv_Globals.h"
#include "../master/detail/visrv_DisplayManager.h"
#include "../native/visrv_SerializeNativeWindow.h"

#include "visrv_ClientUtility.h"
#include "visrv_LayerPool.h"
#include "visrv_PresentationTracerPool.h"
#include "visrv_SystemEventSplitter.h"

namespace nn{ namespace visrv{ namespace client{
    nn::Result ClientObject::OpenDisplay(nn::vi::DisplayId* pOutDisplayId, const char* name) NN_NOEXCEPT
    {
        return OpenDisplay(pOutDisplayId, name, m_Constants.GetPolicyLevel());
    }

    nn::Result ClientObject::OpenDisplay(nn::vi::DisplayId* pOutDisplayId, const char* name, nn::vi::PolicyLevelType policyLevel) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_NOT_NULL(pOutDisplayId);
        NN_VISRV_CHECK_ALIVE();

        NN_RESULT_THROW_UNLESS(
            name != nullptr && name[0] != '\0',
            nn::vi::ResultNotFound()
        );
        NN_RESULT_THROW_UNLESS(
            nn::util::Strnlen(name, sizeof(nn::vi::DisplayName)) < static_cast<int>(sizeof(nn::vi::DisplayName)),
            nn::vi::ResultNotFound()
        );

        // check if already opened
        {
            auto pDisplayHolder = ClientUtility::LookupDisplayHolderByName(&m_DisplayHolderList, name);
            if(pDisplayHolder != nullptr)
            {
                NN_RESULT_THROW(nn::vi::ResultAlreadyOpened());
            }
        }

        // check static limit
        NN_RESULT_THROW_UNLESS(
            m_DisplayHolderList.size() < DisplayCountPerClientMax,
            nn::vi::ResultDenied()
        );

        master::Display* pDisplay = master::detail::g_DisplayManager->Open(name, policyLevel);
        NN_RESULT_THROW_UNLESS(
            pDisplay != nullptr,
            nn::vi::ResultNotFound()
        );

        auto displayId = static_cast<nn::vi::DisplayId>(AcquireResourceId());

        // register display to client
        {
            auto pDisplayHolder = new(AllocateClientDisplayHolder()) ClientDisplayHolder;
            NN_SDK_ASSERT_NOT_NULL(pDisplayHolder); // enough memory should be reserved
            pDisplayHolder->pDisplay    = pDisplay;
            pDisplayHolder->displayId   = displayId;
            pDisplayHolder->policyLevel = policyLevel;
            nn::util::Strlcpy(pDisplayHolder->name.value, name, sizeof(nn::vi::DisplayName));
            std::memset(&pDisplayHolder->displayEventInfo, 0, sizeof(pDisplayHolder->displayEventInfo));
            m_DisplayHolderList.push_back(*pDisplayHolder);
        }

        *pOutDisplayId = displayId;
        NN_VISRV_LOG_OPENCLOSE("Display opened #%lld\n", displayId);

        NN_RESULT_SUCCESS;
    }

    nn::Result ClientObject::CloseDisplay(nn::vi::DisplayId displayId) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_ALIVE();
        NN_VISRV_FIND_CLIENT_DISPLAY(pDisplayHolder, displayId);

        // destroy all layers on the display
        // NOTE: on non-manager clients, only stray layers are created on their displays.
        while(pDisplayHolder->layerList.size() > 0)
        {
            auto& layer = pDisplayHolder->layerList.front();
            if(layer.state == ClientLayerHolderState_Bound || layer.state == ClientLayerHolderState_Unbound)
            {
                DestroyManagedLayer(layer.layerId);
            }
            else if(layer.state == ClientLayerHolderState_Stray)
            {
                DestroyStrayLayer(layer.layerId);
            }
        }

        client::g_HotplugEventSplitter.Get()->RemoveSplittedEvents(displayId);
        client::g_VsyncEventSplitter.Get()->RemoveSplittedEvents(displayId);
        client::g_ErrorReportEventSplitter.Get()->RemoveSplittedEvents(displayId);

        master::detail::g_DisplayManager->Close(pDisplayHolder->pDisplay);
        ReleaseResourceId(static_cast<uint64_t>(pDisplayHolder->displayId));

        m_DisplayHolderList.erase(m_DisplayHolderList.iterator_to(*pDisplayHolder));
        pDisplayHolder->~ClientDisplayHolder();
        FreeClientDisplayHolder(&(*pDisplayHolder));
        NN_VISRV_LOG_OPENCLOSE("Display closed #%lld\n", displayId);

        NN_RESULT_SUCCESS;
    }

    namespace {
        void SetupCreatedLayerHolder(
            ClientLayerHolder* pLayerHolder,
            ClientLayerHolderState createState,
            nn::applet::AppletResourceUserId userAruid,
            master::Layer* pLayer,
            ClientDisplayHolder* pDisplayHolder,
            nn::vi::LayerId layerId
        ) NN_NOEXCEPT
        {
            pLayerHolder->state          = createState;
            pLayerHolder->userAruid      = userAruid;
            pLayerHolder->pLayer         = pLayer;
            pLayerHolder->pDisplayHolder = pDisplayHolder;
            pLayerHolder->layerId        = layerId;
            pLayerHolder->pPresentationTracer = nullptr;
            pLayerHolder->layerTexturePresentingEvent.isInitialized = false;
        }

        void ResetDestroyedLayerHolder(
            ClientLayerHolder* pLayerHolder
        ) NN_NOEXCEPT
        {
            pLayerHolder->state          = ClientLayerHolderState_Empty;
            pLayerHolder->userAruid      = nn::applet::AppletResourceUserId::GetInvalidId();
            pLayerHolder->pLayer         = nullptr;
            pLayerHolder->pDisplayHolder = nullptr;
            pLayerHolder->layerId        = 0;
            if(pLayerHolder->pPresentationTracer)
            {
                g_PresentationTracerPool.Release(pLayerHolder->pPresentationTracer);
                pLayerHolder->pPresentationTracer = nullptr;
            }
            if(pLayerHolder->layerTexturePresentingEvent.isInitialized)
            {
                nn::os::DestroySystemEvent(&pLayerHolder->layerTexturePresentingEvent.event);
                pLayerHolder->layerTexturePresentingEvent.isInitialized = false;
            }
        }

        nn::Result CreatePoolLayer(
            nn::visrv::client::ClientLayerHolder** ppOutLayerHolder,
            nn::visrv::client::ClientDisplayHolder* pDisplayHolder,
            nn::vi::LayerSettingsType settings,
            nn::applet::AppletResourceUserId userAruid,
            nn::visrv::client::ClientLayerHolderState createState
        ) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(ppOutLayerHolder);
            NN_SDK_REQUIRES_NOT_NULL(pDisplayHolder);
            NN_SDK_REQUIRES(createState == ClientLayerHolderState_Unbound || createState == ClientLayerHolderState_Stray);

            bool isSuccess = false;
            {
                nn::vi::LayerSettings flags;
                flags._storage[0] = settings;

                // create master layer
                master::Layer* pLayer = nullptr;
                NN_RESULT_DO(
                    pDisplayHolder->pDisplay->CreateLayer(
                        &pLayer,
                        flags,
                        pDisplayHolder->policyLevel,
                        userAruid
                    )
                );
                NN_UTIL_SCOPE_EXIT
                {
                    if(!isSuccess)
                    {
                        pDisplayHolder->pDisplay->DestroyLayer(pLayer);
                    }
                };

                // acquire layerId
                auto layerId = static_cast<nn::vi::LayerId>(AcquireResourceId());
                NN_UTIL_SCOPE_EXIT
                {
                    if (!isSuccess)
                    {
                        ReleaseResourceId(static_cast<uint64_t>(layerId));
                    }
                };

                // setup LayerHolder and register to display
                auto pLayerHolder = std::find_if(
                    g_LayerPool.begin(),
                    g_LayerPool.end(),
                    [](LayerPool::reference v){ return v.state == ClientLayerHolderState_Empty; }
                );
                NN_RESULT_THROW_UNLESS(
                    pLayerHolder != g_LayerPool.end(),
                    nn::vi::ResultOperationFailed()
                );
                SetupCreatedLayerHolder(pLayerHolder, createState, userAruid, pLayer, pDisplayHolder, layerId);
                pDisplayHolder->layerList.push_back(*pLayerHolder);
                NN_UTIL_SCOPE_EXIT
                {
                    if(!isSuccess)
                    {
                        pDisplayHolder->layerList.erase(pDisplayHolder->layerList.iterator_to(*pLayerHolder));
                        ResetDestroyedLayerHolder(pLayerHolder);
                    }
                };

                isSuccess = true;
                *ppOutLayerHolder = pLayerHolder;
            }

            NN_RESULT_SUCCESS;
        }

        nn::Result DestroyPoolLayer(
            ClientLayerHolder* pLayerHolder
        ) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(pLayerHolder);
            NN_SDK_REQUIRES_EQUAL(pLayerHolder->pBoundClient, nullptr);

            auto pDisplayHolder = pLayerHolder->pDisplayHolder;
            // unregister from display
            pDisplayHolder->layerList.erase(pDisplayHolder->layerList.iterator_to(*pLayerHolder));
            // release layerId
            ReleaseResourceId(static_cast<uint64_t>(pLayerHolder->layerId));
            // destroy master layer
            pDisplayHolder->pDisplay->DestroyLayer(pLayerHolder->pLayer);
            // reset LayerHolder
            ResetDestroyedLayerHolder(pLayerHolder);

            NN_RESULT_SUCCESS;
        }

        nn::Result BindPoolLayer(
            size_t* pOutNativeWindowDataSize,
            void* pNativeWindowDataBuffer,
            size_t nativeWindowDataBufferSize,
            ClientObject* pClientObject,
            ClientLayerHolder* pLayerHolder,
            bool isProxyNameExchangeEnabled,
            const nn::vi::ProxyName& clientProxyName
        ) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(pClientObject);
            NN_SDK_REQUIRES(
                pLayerHolder->state == ClientLayerHolderState_Unbound ||
                pLayerHolder->state == ClientLayerHolderState_Stray
            );
            NN_SDK_REQUIRES_EQUAL(pLayerHolder->pBoundClient, nullptr);

            bool isSuccess = false;
            {
                const nn::vi::DisplayId displayId = pLayerHolder->pDisplayHolder->displayId;
                const nn::vi::LayerId   layerId   = pLayerHolder->layerId;

                // register to client
                auto pLayerHolderList = ClientObjectAccessor::GetLayerHolderList(pClientObject);
                pLayerHolderList->push_back(*pLayerHolder);
                pLayerHolder->pBoundClient = pClientObject;
                NN_UTIL_SCOPE_EXIT
                {
                    if(!isSuccess)
                    {
                        pLayerHolderList->erase(pLayerHolderList->iterator_to(*pLayerHolder));
                        pLayerHolder->pBoundClient = nullptr;
                    }
                };

                // serialize native window
                int32_t driverHandle = 0;
                size_t dataSize = 0;
                NN_RESULT_DO(
                    native::SerializeNativeWindow(
                        &driverHandle,
                        &dataSize,
                        pNativeWindowDataBuffer,
                        nativeWindowDataBufferSize,
                        pLayerHolder->pLayer,
                        isProxyNameExchangeEnabled,
                        clientProxyName
                    )
                );

                // register native window to binder
                auto pBinderTable = ClientObjectAccessor::GetBinderTable(pClientObject);
                auto pTableEntry = pBinderTable->AddEntry(
                    pLayerHolder->pLayer->m_GraphicBufferProducer->asBinder(),
                    native::BinderClass_IGraphicBufferProducer,
                    driverHandle,
                    displayId,
                    layerId
                );
                NN_RESULT_THROW_UNLESS(
                    pTableEntry != nullptr,
                    nn::vi::ResultOperationFailed()
                );
                NN_UTIL_SCOPE_EXIT
                {
                    if(!isSuccess)
                    {
                        pTableEntry->Finalize();
                    }
                };

                // update state
                if(pLayerHolder->state == ClientLayerHolderState_Unbound)
                {
                    pLayerHolder->state = ClientLayerHolderState_Bound;
                }

                isSuccess = true;
                *pOutNativeWindowDataSize = dataSize;
            }
            NN_RESULT_SUCCESS;
        }

        nn::Result UnbindPoolLayer(
            ClientLayerHolder* pLayerHolder
        ) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(pLayerHolder);
            NN_SDK_REQUIRES(
                pLayerHolder->state == ClientLayerHolderState_Bound ||
                pLayerHolder->state == ClientLayerHolderState_Stray
            );

            auto pClientObject = pLayerHolder->pBoundClient;

            auto layerId = pLayerHolder->layerId;

            pLayerHolder->pLayer->m_Client.setConductorLayer(pLayerHolder->pLayer->m_Handle, /*isConductor*/ false);
            //{
            //    NN_VISRV_LOG_WARN("setConductorLayer(#%d, false) on unbind\n", static_cast<int>(layerId));
            //}

            // unregister driver handles
            auto pBinderTable = ClientObjectAccessor::GetBinderTable(pClientObject);
            pBinderTable->RemoveHandlesOnLayer(layerId);

            // unregister from client
            auto pLayerHolderList = ClientObjectAccessor::GetLayerHolderList(pClientObject);
            pLayerHolderList->erase(pLayerHolderList->iterator_to(*pLayerHolder));
            pLayerHolder->pBoundClient = nullptr;

            if(pLayerHolder->state == ClientLayerHolderState_Bound)
            {
                pLayerHolder->state = ClientLayerHolderState_Unbound;
            }

            NN_RESULT_SUCCESS;
        }
    }

    nn::Result ClientObject::CreateManagedLayer(
        nn::vi::LayerId* pOutLayerId,
        nn::vi::DisplayId displayId,
        nn::vi::LayerSettingsType settings,
        nn::applet::AppletResourceUserId userAruid
        ) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_NOT_NULL(pOutLayerId);
        NN_VISRV_CHECK_ALIVE();

        NN_VISRV_FIND_CLIENT_DISPLAY(pDisplayHolder, displayId);

        bool isSuccess = false;
        {
            // create layer on pool
            ClientLayerHolder* pLayerHolder = nullptr;
            NN_RESULT_DO(
                CreatePoolLayer(
                    &pLayerHolder,
                    pDisplayHolder,
                    settings,
                    userAruid,
                    ClientLayerHolderState_Unbound
                )
            );
            NN_UTIL_SCOPE_EXIT
            {
                if(!isSuccess)
                {
                    DestroyPoolLayer(pLayerHolder);
                }
            };

            NN_SDK_ASSERT_NOT_NULL(pLayerHolder);

            isSuccess = true;
            *pOutLayerId = pLayerHolder->layerId;
            NN_VISRV_LOG_OPENCLOSE("ManagedLayer created #%lld\n", pLayerHolder->layerId);
        }

        NN_RESULT_SUCCESS;
    }

    nn::Result ClientObject::BindManagedLayer(
        size_t* pOutNativeWindowDataSize,
        void* pNativeWindowDataBuffer,
        size_t nativeWindowDataBufferSize,
        nn::vi::LayerId layerId,
        const char* displayName,
        nn::applet::AppletResourceUserId aruid
        ) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_NOT_NULL(pOutNativeWindowDataSize);
        NN_VISRV_CHECK_NOT_NULL(pNativeWindowDataBuffer);
        NN_VISRV_CHECK_ALIVE();

        auto pLayerHolder = std::find_if(
            g_LayerPool.begin(),
            g_LayerPool.end(),
            [layerId](const ClientLayerHolder& v){ return (v.state != ClientLayerHolderState_Empty) && (v.layerId == layerId); }
        );
        NN_RESULT_THROW_UNLESS(
            (pLayerHolder != g_LayerPool.end()) &&
            (pLayerHolder->state == ClientLayerHolderState_Unbound) &&
            (nn::util::Strncmp(pLayerHolder->pDisplayHolder->name.value, displayName, sizeof(nn::vi::DisplayName)) == 0) &&
            (pLayerHolder->userAruid == aruid),
            nn::vi::ResultOperationFailed()
        );

        NN_RESULT_DO(BindPoolLayer(
            pOutNativeWindowDataSize,
            pNativeWindowDataBuffer,
            nativeWindowDataBufferSize,
            this,
            pLayerHolder,
            m_Constants.IsProxyNameExchangeEnabled(),
            m_Constants.GetExchangeProxyName()
        ));

        NN_VISRV_LOG_OPENCLOSE("ManagedLayer bound #%lld\n", layerId);
        NN_RESULT_SUCCESS;
    }

    nn::Result ClientObject::UnbindManagedLayer(
        nn::vi::LayerId layerId
    ) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_ALIVE();
        NN_VISRV_FIND_CLIENT_LAYER(pLayerHolder, layerId);

        NN_RESULT_DO(UnbindPoolLayer(pLayerHolder));
        NN_VISRV_LOG_OPENCLOSE("ManagedLayer unbound #%lld\n", layerId);

        NN_RESULT_SUCCESS;
    }

    nn::Result ClientObject::CreateStrayLayer(
        nn::vi::LayerId* pOutLayerId,
        size_t* pOutNativeWindowDataSize,
        void* pNativeWindowDataBuffer,
        size_t nativeWindowDataBufferSize,
        nn::vi::DisplayId displayId,
        nn::vi::LayerSettingsType settings
        ) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_NOT_NULL(pOutLayerId);
        NN_VISRV_CHECK_NOT_NULL(pOutNativeWindowDataSize);
        NN_VISRV_CHECK_NOT_NULL(pNativeWindowDataBuffer);
        NN_VISRV_CHECK_ALIVE();

        NN_VISRV_FIND_CLIENT_DISPLAY(pDisplayHolder, displayId);

        // check static limit
        NN_RESULT_THROW_UNLESS(
            pDisplayHolder->layerList.size() < LayerCountPerDisplayMax,
            nn::vi::ResultDenied()
        );
        // check policy limit
        {
            auto& info = pDisplayHolder->GetDisplayInfo();;
            NN_RESULT_THROW_UNLESS(
                info.HasValidName(),
                nn::vi::ResultDenied()
            );
            auto& policy = info.GetPolicy();
            if(policy.HasLayerLimit())
            {
                NN_RESULT_THROW_UNLESS(
                    pDisplayHolder->layerList.size() < policy.GetLayerCountMax(),
                    nn::vi::ResultDenied()
                );
            }
        }

        bool isSuccess = false;
        {
            // create layer on pool
            ClientLayerHolder* pLayerHolder = nullptr;
            NN_RESULT_DO(
                CreatePoolLayer(
                    &pLayerHolder,
                    pDisplayHolder,
                    settings,
                    nn::applet::AppletResourceUserId::GetInvalidId(),
                    ClientLayerHolderState_Stray
                )
            );
            NN_UTIL_SCOPE_EXIT
            {
                if(!isSuccess)
                {
                    DestroyPoolLayer(pLayerHolder);
                }
            };

            NN_SDK_ASSERT_NOT_NULL(pLayerHolder);

            // bind the layer immediately
            NN_RESULT_DO(
                BindPoolLayer(
                    pOutNativeWindowDataSize,
                    pNativeWindowDataBuffer,
                    nativeWindowDataBufferSize,
                    this,
                    pLayerHolder,
                    m_Constants.IsProxyNameExchangeEnabled(),
                    m_Constants.GetExchangeProxyName()
                )
            );
            NN_UTIL_SCOPE_EXIT
            {
                if(!isSuccess)
                {
                    UnbindPoolLayer(pLayerHolder);
                }
            };

            isSuccess = true;
            *pOutLayerId = pLayerHolder->layerId;
            NN_VISRV_LOG_OPENCLOSE("StrayLayer created #%lld\n", pLayerHolder->layerId);
        }

        NN_RESULT_SUCCESS;
    }

    nn::Result ClientObject::DestroyManagedLayer(
        nn::vi::LayerId layerId
        ) NN_NOEXCEPT
    {
        // only manager can destroy managed layers
        NN_RESULT_THROW_UNLESS(
            m_Constants.GetPermission() == ClientPermission_Manager,
            nn::vi::ResultDenied()
        );
        NN_VISRV_CHECK_ALIVE();
        NN_VISRV_FIND_CLIENT_LAYER(pLayerHolder, layerId);

        NN_RESULT_THROW_UNLESS(
            pLayerHolder->state == ClientLayerHolderState_Unbound || pLayerHolder->state == ClientLayerHolderState_Bound,
            nn::vi::ResultDenied()
        );

        if(pLayerHolder->state == ClientLayerHolderState_Bound)
        {
            NN_RESULT_DO(UnbindPoolLayer(pLayerHolder));
        }

        NN_SDK_ASSERT_EQUAL(pLayerHolder->state, ClientLayerHolderState_Unbound);

        NN_RESULT_DO(DestroyPoolLayer(pLayerHolder));
        NN_VISRV_LOG_OPENCLOSE("ManagedLayer destroyed #%lld\n", layerId);

        NN_RESULT_SUCCESS;
    }

    nn::Result ClientObject::DestroyStrayLayer(
        nn::vi::LayerId layerId
        ) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_ALIVE();
        NN_VISRV_FIND_CLIENT_LAYER(pLayerHolder, layerId);

        NN_RESULT_THROW_UNLESS(
            pLayerHolder->state == ClientLayerHolderState_Stray,
            nn::vi::ResultDenied()
        );

        // stray layer can be destroyed by its owner
        NN_RESULT_DO(UnbindPoolLayer(pLayerHolder));
        NN_RESULT_DO(DestroyPoolLayer(pLayerHolder));
        NN_VISRV_LOG_OPENCLOSE("StrayLayer destroyed #%lld\n", layerId);

        NN_RESULT_SUCCESS;
    }

}}}
